/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using Tasklets;

namespace zcd
{
    public errordomain SerializerError {
        GENERIC
    }

    internal bool endianness_network;

    /** The interface ISerializable has to be implemented by any
      * class whose instances we want to be able to serialize and
      * transmit over the net.
      * The class that we want to be serializeable needs to know how
      * to put in a Variant its data and retrieve it from a Variant.
      * But the code that wants to serialize an instance of those classes
      * will use the methods serialize and deserialize, which treat
      * simple uchar arrays.
      */
    public interface ISerializable : Object
    {
        public abstract Variant serialize_to_variant();
        public abstract void deserialize_from_variant(Variant v) throws SerializerError;
        public uchar[] serialize()
        {
            Variant vtuple = serialize_to_variant();
            string s = this.get_type().name();
            Variant v1 = new Variant("(sv)", s, vtuple);
            uchar[] ser = null;
            if (!endianness_network) v1 = v1.byteswap();
            size_t siz = v1.get_size();
            ser = new uchar[siz];
            v1.store(ser);
            return ser;
        }
        public static ISerializable deserialize(uchar[] ser) throws SerializerError
        {
            BufferOwner owner = new BufferOwner(ser);
            VariantType vt = new VariantType("(sv)");
            Variant v2 = Variant.new_from_data<BufferOwner>(vt, owner.buf, false, owner);
            if (!endianness_network) v2 = v2.byteswap();
            string typename;
            Variant vtuple;
            v2.get("(sv)", out typename, out vtuple);
            if (typename == null || typename == "") throw new SerializerError.GENERIC("Typename not specified.");
            Type type = Type.from_name(typename);
            if (type == 0) throw new SerializerError.GENERIC(@"Typename '$typename' is unknown.");
            if (! type.is_a(typeof(ISerializable))) throw new SerializerError.GENERIC(@"Typename '$typename' is not serializable.");
            Object obj = Object.new(type);
            ISerializable retval = (ISerializable)obj;
            retval.deserialize_from_variant(vtuple);
            return retval;
        }
        public static bool is_instance_of(Type t1, uchar []ser)
        {
            BufferOwner owner = new BufferOwner(ser);
            VariantType vt = new VariantType("(sv)");
            Variant v2 = Variant.new_from_data<BufferOwner>(vt, owner.buf, false, owner);
            if (!endianness_network) v2 = v2.byteswap();
            string typename;
            Variant vtuple;
            v2.get("(sv)", out typename, out vtuple);
            Type t2 = Type.from_name(typename);
            return t2.is_a(t1);
        }
        public static string typename(uchar []ser)
        {
            BufferOwner owner = new BufferOwner(ser);
            VariantType vt = new VariantType("(sv)");
            Variant v2 = Variant.new_from_data<BufferOwner>(vt, owner.buf, false, owner);
            if (!endianness_network) v2 = v2.byteswap();
            string typename;
            Variant vtuple;
            v2.get("(sv)", out typename, out vtuple);
            return typename;
        }
    }

    public class BufferOwner : Object
    {
        private uchar[] _buf;
        public uchar[] buf {
            get {
                return _buf;
            }
        }
        public BufferOwner(uchar[] seed)
        {
            _buf = new uchar[seed.length];
            Posix.memcpy((void*)_buf, (void*)seed, (size_t)seed.length*sizeof(uchar));
        }
    }

    /** Serializable classes for basic types **/

    /** Return value for "void"
      */
    public class SerializableNone : Object, ISerializable
    {
        public Variant serialize_to_variant()
        {
            Variant v = Serializer.int_to_variant(0);
            return v;
        }
        
        public void deserialize_from_variant(Variant v)
        {
        }
    }

    /** Serializable class for int
      */
    public class SerializableInt : Object, ISerializable
    {
        private int _i;
        public SerializableInt(int i)
        {
            _i = i;
        }
        public int i {
            get {
                return _i;
            }
        }

        public Variant serialize_to_variant()
        {
            Variant v = Serializer.int_to_variant(_i);
            return v;
        }
        public void deserialize_from_variant(Variant v)
        {
            _i = Serializer.variant_to_int(v);
        }
    }

    /** Serializable class for bool
      */
    public class SerializableBool : Object, ISerializable
    {
        private int _i;
        public SerializableBool(bool b)
        {
            _i = b ? 1 : 0;
        }
        public bool b {
            get {
                return _i == 1;
            }
        }

        public Variant serialize_to_variant()
        {
            Variant v = Serializer.int_to_variant(_i);
            return v;
        }
        public void deserialize_from_variant(Variant v)
        {
            _i = Serializer.variant_to_int(v);
        }
    }

    /** Serializable class for string
      */
    public class SerializableString : Object, ISerializable
    {
        private string _s;
        public SerializableString(string s)
        {
            _s = s;
        }
        public string s {
            get {
                return _s;
            }
        }

        public Variant serialize_to_variant()
        {
            Variant v = Serializer.string_to_variant(_s);
            return v;
        }
        public void deserialize_from_variant(Variant v)
        {
            _s = Serializer.variant_to_string(v);
        }
    }

    /** Serializable class for uint8[]
      */
    public class SerializableBuffer : Object, ISerializable
    {
        private uint8[] _buffer;
        public SerializableBuffer(uint8[] buffer)
        {
            _buffer = buffer;
        }
        public uint8[] buffer {
            get {
                return _buffer;
            }
        }

        public Variant serialize_to_variant()
        {
            return Serializer.uchar_array_to_variant((uchar[])_buffer);
        }
        public void deserialize_from_variant(Variant v)
        {
            _buffer = (uint8[])Serializer.variant_to_uchar_array(v);
        }
    }

    /** Serializable class for List<string>
      */
    public class ListString : Object, ISerializable
    {
        public Gee.List<string> backed;

        public ListString()
        {
            backed = new ArrayList<string>();
        }

        public ListString.with_backer(Gee.List<string> backed)
        {
            this.backed = backed;
        }

        public int size {
            get {
                return backed.size;
            }
        }

        public Iterator<string> iterator()
        {
            return backed.iterator();
        }

        public void add(string el)
        {
            backed.add(el);
        }

        public uchar[] hash_for_signature()
        {
            if (backed.size == 0) return new uchar[] {0};
            uchar[] ret = {1};
            foreach (string s in backed)
                foreach (uchar u in s.data)
                    ret += u;
            return ret;
        }

        public Variant serialize_to_variant()
        {
            if (backed.size == 0) return Serializer.int_to_variant(0);
            string[] toar = new string[backed.size];
            for (int i = 0; i < backed.size; i++) toar[i] = backed[i];
            return Serializer.string_array_to_variant(toar);
        }

        public void deserialize_from_variant(Variant v)
        {
            string vt = v.get_type_string();
            backed = new ArrayList<string>();
            if (vt != "i")
            {
                string[] toar = Serializer.variant_to_string_array(v);
                for (int i = 0; i < toar.length; i++) backed.add(toar[i]);
            }
        }
    }

    /** Serializable class for List<ISerializable>
      *
      * This class is used for passing or returning a list of objects, all of the
      *  same type and that implement ISerializable.
      * With this class, the objects contained in the list are serialized when we
      *  call "serialize()" on the list, and are deserialized when we call deserialize()
      *  for the list.
      * A different approach has to be used if we need to
      *   § serialize a list of serializable objects, obtaining a
      *     list of uchar[]
      *   § put the list of uchar[] in a serializable container
      *   § serialize the container
      *   § deserialize the container WITHOUT deserializing the objects
      * This situation is needed for remotable methods in optional peer to peer
      *  services; because we need to pass the arguments to hops that might not
      *  implement the service and might not have knowledge of the classes involved.
      * For this see an example of SerializedList in the serializer_tester_1 module.
      * Such a class is RemoteCall, in module messages.
      */
    public class ListISerializable : Object, ISerializable
    {
        public Gee.List<ISerializable> backed;

        private static bool not_impl_equal_func (Object? a, Object? b)
        {
            string astr = "null";
            if (a != null) astr = a.get_type().name();
            string bstr = "null";
            if (b != null) bstr = b.get_type().name();
            error(@"equal_func not implemented: compare between $(astr) and $(bstr)");
        }
        public ListISerializable()
        {
            backed = new ArrayList<ISerializable>((EqualDataFunc)not_impl_equal_func);
        }

        public ListISerializable.with_backer(Gee.List<ISerializable> backed)
        {
            this.backed = backed;
        }

        public int size {
            get {
                return backed.size;
            }
        }

        public Iterator<ISerializable> iterator()
        {
            return backed.iterator();
        }

        public void add(ISerializable el)
        {
            backed.add(el);
        }

        public Variant serialize_to_variant()
        {
            if (backed.size == 0) return Serializer.int_to_variant(0);
            Variant[] toar = new Variant[backed.size];
            for (int i = 0; i < backed.size; i++)
            {
                ISerializable el = (ISerializable)backed[i];
                toar[i] = el.serialize_to_variant();
            }
            Variant v0 = Serializer.variant_array_to_variant(toar);
            Variant v1 = Serializer.string_to_variant(backed[0].get_type().name());
            return Serializer.tuple_to_variant(v0, v1);
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            string vt = v.get_type_string();
            backed = new ArrayList<ISerializable>((EqualDataFunc)not_impl_equal_func);
            if (vt != "i")
            {
                Variant v0;
                Variant v1;
                Serializer.variant_to_tuple(v, out v0, out v1);
                Variant[] toar = Serializer.variant_to_variant_array(v0);
                string typename = Serializer.variant_to_string(v1);
                if (typename == null || typename == "") throw new SerializerError.GENERIC("Typename not specified.");
                Type type = Type.from_name(typename);
                if (type == 0) throw new SerializerError.GENERIC(@"Typename '$typename' is unknown.");
                if (! type.is_a(typeof(ISerializable))) throw new SerializerError.GENERIC(@"Typename '$typename' is not serializable.");
                for (int i = 0; i < toar.length; i++)
                {
                    ISerializable el = (ISerializable)Object.new(type);
                    el.deserialize_from_variant(toar[i]);
                    backed.add(el);
                }
            }
        }
    }

    namespace Serializer
    {
        public void init()
        {
            // Check if host endianness is the same as the network standard (big)
            endianness_network = Posix.ntohs(1) == 1;
            // Register serializable types
            typeof(RemoteCall).class_peek();
            typeof(RemotableException).class_peek();
            typeof(TCPRequest).class_peek();
            typeof(UDPMessage).class_peek();
            typeof(UDPPayload).class_peek();
            typeof(SerializableNone).class_peek();
            typeof(SerializableInt).class_peek();
            typeof(SerializableBool).class_peek();
            typeof(SerializableString).class_peek();
            typeof(SerializableBuffer).class_peek();
            typeof(ListString).class_peek();
            typeof(ListISerializable).class_peek();
        }

        public Variant int_to_variant(int i)
        {
            return i; // this does not leak memory (tested with valac-0.12)
        }

        public int variant_to_int(Variant v)
        {
            return (int)v; // this does not leak memory (tested with valac-0.12)
        }

        public Variant int64_to_variant(int64 i)
        {
            return i; // this does not leak memory (tested with valac-0.12)
        }

        public int64 variant_to_int64(Variant v)
        {
            return (int64)v; // this does not leak memory (tested with valac-0.12)
        }

        public Variant uchar_to_variant(uchar y)
        {
            return y; // this does not leak memory (tested with valac-0.12)
        }

        public uchar variant_to_uchar(Variant v)
        {
            return (uchar)v; // this does not leak memory (tested with valac-0.12)
        }

        public Variant string_to_variant(string s)
        {
            return s; // this does not leak memory (tested with valac-0.12)
        }

        public string variant_to_string(Variant v)
        {
            // simply doing:   return (string)v;
            // would have leaked memory (tested with valac-0.12)
            string t2;
            assert(v.get_type_string() == "s");
            v.get("s", out t2);
            return t2;
        }

        public Variant int_array_to_variant(int[] ai)
        {
            return ai; // this does not leak memory (tested with valac-0.12)
        }

        public int[] variant_to_int_array(Variant v)
        {
            // simply doing:   return (int[])v;
            // would have leaked memory (tested with valac-0.12)
            // simply doing:   v.get("ai",...);
            // would have returned a wrong size of the array (tested with valac-0.12)
            int[] ret = {};
            VariantIter iter = new VariantIter(v);
            Variant? v0 = iter.next_value();
            while (v0 != null)
            {
                int i = (int)v0;
                ret += i;
                v0 = iter.next_value();
            }
            return ret;
        }

        public Variant uchar_array_to_variant(uchar[] auc)
        {
            // TODO Here we have a performance bottleneck, reported by a profiler.

            return auc; // this does not leak memory (tested with valac-0.12)
        }

        public uchar[] variant_to_uchar_array(Variant v)
        {
            // TODO Here we have a performance bottleneck, reported by a profiler.

            // simply doing:   return (uchar[])v;
            // would have leaked memory (tested with valac-0.12)
            // simply doing:   v.get("ay",...);
            // would have returned a wrong size of the array (tested with valac-0.12)
            uchar[] ret = {};
            VariantIter iter = new VariantIter(v);
            Variant? v0 = iter.next_value();
            while (v0 != null)
            {
                uchar i = (uchar)v0;
                ret += i;
                v0 = iter.next_value();
            }
            return ret;
        }

        public Variant string_array_to_variant(string[] ars)
        {
            return ars; // this does not leak memory (tested with valac-0.12)
        }

        public string[] variant_to_string_array(Variant v)
        {
            // simply doing:   return (string[])v;
            // would have leaked memory (tested with valac-0.12)
            // simply doing:   v.get("as",...);
            // would have returned a wrong size of the array (tested with valac-0.12)
            string[] ret = {};
            VariantIter iter = new VariantIter(v);
            Variant? v0 = iter.next_value();
            while (v0 != null)
            {
                // this would leak: string s = (string)v0;
                // using g_variant_get s now points to a string
                string s;
                assert(v0.get_type_string() == "s");
                v0.get("s", out s);
                ret += s;
                v0 = iter.next_value();
            }
            return ret;
        }

        public Variant variant_array_to_variant(Variant[] av)
        {
            return av; // this does not leak memory (tested with valac-0.12)
        }

        public Variant[] variant_to_variant_array(Variant v)
        {
            // simply doing:   return (Variant[])v;
            // would have leaked memory (tested with valac-0.12)
            // simply doing:   v.get("av",...);
            // would have returned a wrong size of the array (tested with valac-0.12)
            Variant[] ret = {};
            VariantIter iter = new VariantIter(v);
            Variant? v0 = iter.next_value();
            while (v0 != null)
            {
                Variant v1 = v0.get_variant();
                ret += v1;
                v0 = iter.next_value();
            }
            return ret;
        }

        public Variant tuple_to_variant(Variant v0, Variant v1)
        {
            return new Variant("(vv)", v0, v1); // this does not leak memory (tested with valac-0.12)
        }

        public void variant_to_tuple(Variant v, out Variant v0, out Variant v1)
        {
            v.get("(vv)", out v0, out v1); // this does not leak memory (tested with valac-0.12)
        }

        public Variant tuple_to_variant_3(Variant v0, Variant v1, Variant v2)
        {
            return new Variant("(vvv)", v0, v1, v2); // this does not leak memory (tested with valac-0.12)
        }

        public void variant_to_tuple_3(Variant v, out Variant v0, out Variant v1, out Variant v2)
        {
            v.get("(vvv)", out v0, out v1, out v2); // this does not leak memory (tested with valac-0.12)
        }

        public Variant tuple_to_variant_4(Variant v0, Variant v1, Variant v2, Variant v3)
        {
            return new Variant("(vvvv)", v0, v1, v2, v3); // this does not leak memory (tested with valac-0.12)
        }

        public void variant_to_tuple_4(Variant v, out Variant v0, out Variant v1, out Variant v2, out Variant v3)
        {
            v.get("(vvvv)", out v0, out v1, out v2, out v3); // this does not leak memory (tested with valac-0.12)
        }

        public Variant tuple_to_variant_5(Variant v0, Variant v1, Variant v2, Variant v3, Variant v4)
        {
            return new Variant("(vvvvv)", v0, v1, v2, v3, v4); // this does not leak memory (tested with valac-0.12)
        }

        public void variant_to_tuple_5(Variant v, out Variant v0, out Variant v1, out Variant v2, out Variant v3, out Variant v4)
        {
            v.get("(vvvvv)", out v0, out v1, out v2, out v3, out v4); // this does not leak memory (tested with valac-0.12)
        }

    }
}

