/*
 *  This file is part of librpc.
 *  (c) Copyright 2013 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  librpc is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  librpc is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with librpc.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;

public class RpcDesign
{
    private static bool version = false;
    private static bool force_ovw = false;
    private static bool only_first_pass = false;
    private static bool only_second_pass = false;
    private static string? input_file = null;
    private static string? stub_file = null;
    private static string? skel_file = null;
    private static string? temp_file = null;
    private const string def_input_file = "rpc_interfaces.vala";
    private const string def_stub_file = "generated_stub.vala";
    private const string def_skel_file = "generated_skeleton.vala";
    private const string def_temp_file = "output.rpcdef";
    private const GLib.OptionEntry[] options = {
        {"version", 0, 0, OptionArg.NONE, ref version, "Display version number", null},
        {"force", 'f', 0, OptionArg.NONE, ref force_ovw, "Force overwirte", null},
        {"firstpass", '1', 0, OptionArg.NONE, ref only_first_pass, "Only first pass", null},
        {"secondpass", '2', 0, OptionArg.NONE, ref only_second_pass, "Only second pass", null},
        {"interface", 'i', 0, OptionArg.FILENAME, ref input_file, "Interface file (input). Default is " + def_input_file, null},
        {"stub", 'u', 0, OptionArg.FILENAME, ref stub_file, "Stub classes file. Default is " + def_stub_file, null},
        {"skeleton", 'k', 0, OptionArg.FILENAME, ref skel_file, "Skeleton classes file. Default is " + def_skel_file, null},
        {"temp-file", 0, 0, OptionArg.FILENAME, ref temp_file, "Temporary definition file. Default is " + def_temp_file, null},
        { null }
	};

    public static int main(string[] args)
    {
        OptionContext oc = new OptionContext("- Parses interface and produce stub and skeleton classes.");
        oc.add_main_entries(options, null);
        try {
            oc.parse(ref args);
        }
        catch (OptionError e) {
            stdout.printf(@"Error parsing options: $(e.message)\n");
            return 1;
        }

		if (version) {
			stdout.printf(@"$(Config.PACKAGE_NAME) $(Config.PACKAGE_VERSION)\n");
			return 0;
		}

        if (only_first_pass && only_second_pass) {
            stdout.printf("Cannot use flags firstpass and secondpass together.\n");
            return 1;
        }

        if (input_file == null) input_file = def_input_file;
        if (stub_file == null) stub_file = def_stub_file;
        if (skel_file == null) skel_file = def_skel_file;
        if (temp_file == null) temp_file = def_temp_file;

        if (! only_second_pass)
        {
            if (! FileUtils.test(input_file, FileTest.EXISTS))
            {
                stderr.printf(@"Source file $(input_file) not found.\n");
                return 1;
            }
            if (!force_ovw && FileUtils.test(temp_file, FileTest.EXISTS))
            {
                stderr.printf(@"Destination file $(temp_file) exists.\n");
                return 1;
            }
            FirstPass.firstpass(input_file, temp_file);
        }

        if (! only_first_pass)
        {
            if (! FileUtils.test(temp_file, FileTest.EXISTS))
            {
                stderr.printf(@"Source file $(temp_file) not found.\n");
                return 1;
            }
            if (!force_ovw && FileUtils.test(stub_file, FileTest.EXISTS))
            {
                stderr.printf(@"Destination file $(stub_file) exists.\n");
                return 1;
            }
            if (!force_ovw && FileUtils.test(skel_file, FileTest.EXISTS))
            {
                stderr.printf(@"Destination file $(skel_file) exists.\n");
                return 1;
            }
            SecondPass.secondpass(temp_file, stub_file, skel_file);
        }

        return 0;
    }
}

