/*
 *  This file is part of librpc.
 *  (c) Copyright 2013 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  librpc is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  librpc is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with librpc.  If not, see <http://www.gnu.org/licenses/>.
 */
using Gee;

namespace SecondPass
{

    class Property
    {
        public string propname;
        public string classname;
    }

    class Argument
    {
        public string argname;
        public string classname;
    }

    class Method
    {
        public string returntype;
        public string name;
        public Argument[] args;
        public bool pass_caller = false;
        public string[] errors;
    }

    string? namespace_guess;
    string[] serializable_types;
    HashMap<string, ArrayList<string>> errordefs;
    string[] roots;
    string[] remotes;
    HashMap<string, ArrayList<Property>> properties;
    HashMap<string, ArrayList<Method>> methods;
    HashMap<string, ArrayList<Method>> methods_rmt;

    string output;
    string closepar;

    public void secondpass(string if_name, string stub_name, string skel_name)
    {
        read_file_if(if_name);

        output = "";
        write_file_skeleton();
        write_file(skel_name, output);
        output = "";
        write_file_stub();
        write_file(stub_name, output);
    }

    void stub_property_firstpart(Property prop)
    {
        wr_line(
@"      private Remote$(prop.classname) _$(prop.propname) = null;"
        );
        wr_line(
@"      public unowned I$(prop.classname) _$(prop.propname)_getter()"
        );
        wr_line(
@"      {"
        );
        wr_line(
@"          if (_$(prop.propname) == null)"
        );
        wr_line(
@"          {"
        );
        wr_line(
@"              _$(prop.propname) = new Remote$(prop.classname)();"
        );
    }

    void stub_property_secondpart(Property prop)
    {
        wr_line(
@"          }"
        );
        wr_line(
@"          return _$(prop.propname);"
        );
        wr_block(
"""
        }

"""
        );
    }

    void stub_declare_method(Method met)
    {
        string str_throws = "";
        foreach (string err in met.errors) if (errordefs.keys.contains(err))
            str_throws += @", $(err)";
        string str_args = "";
        string str_args_next = "";
        foreach (Argument arg in met.args)
        {
            str_args += @"$(str_args_next)$(arg.classname) $(arg.argname)";
            str_args_next = ", ";
        }
        if (met.pass_caller)
            str_args += @"$(str_args_next)CallerInfo? _rpc_caller=null";
        wr_line(
@"      public $(met.returntype) $(met.name)($(str_args)) throws RPCError$(str_throws)"
        );
    }

    void stub_add_parameters(Argument[] args)
    {
        foreach (Argument arg in args)
        {
            if (arg.classname.length > 1 && arg.classname.substring(arg.classname.length-1) == "?")
            {
                // nullable... what?
                string nullclassname = arg.classname.substring(0, arg.classname.length-1);
                if (nullclassname in serializable_types)
                {
                    wr_line(
@"          if ($(arg.argname) == null) rc.add_parameter(new SerializableNone());"
                    );
                    wr_line(
@"          else rc.add_parameter($(arg.argname));"
                    );
                }
                else if (nullclassname == "int")
                {
                    wr_line(
@"          if ($(arg.argname) == null) rc.add_parameter(new SerializableNone());"
                    );
                    wr_line(
@"          else rc.add_parameter(new SerializableInt($(arg.argname)));"
                    );
                }
                else if (nullclassname == "bool")
                {
                    wr_line(
@"          if ($(arg.argname) == null) rc.add_parameter(new SerializableNone());"
                    );
                    wr_line(
@"          else rc.add_parameter(new SerializableBool($(arg.argname)));"
                    );
                }
                else if (nullclassname == "string")
                {
                    wr_line(
@"          if ($(arg.argname) == null) rc.add_parameter(new SerializableNone());"
                    );
                    wr_line(
@"          else rc.add_parameter(new SerializableString($(arg.argname)));"
                    );
                }
            }
            else if (arg.classname == "ISerializable")
            {
                wr_line(
@"          rc.add_parameter($(arg.argname));"
                );
            }
            else if (arg.classname in serializable_types)
            {
                wr_line(
@"          rc.add_parameter($(arg.argname));"
                );
            }
            else if (arg.classname == "int")
            {
                wr_line(
@"          rc.add_parameter(new SerializableInt($(arg.argname)));"
                );
            }
            else if (arg.classname == "bool")
            {
                wr_line(
@"          rc.add_parameter(new SerializableBool($(arg.argname)));"
                );
            }
            else if (arg.classname == "string")
            {
                wr_line(
@"          rc.add_parameter(new SerializableString($(arg.argname)));"
                );
            }
            else if (arg.classname == "Gee.List<string>")
            {
                wr_line(
@"          ListString _$(arg.argname) = new ListString.with_backer($(arg.argname));"
                );
                wr_line(
@"          rc.add_parameter(_$(arg.argname));"
                );
            }
            else if (arg.classname.length > 9 && arg.classname.substring(0, 9) == "Gee.List<")
            {
                wr_line(
@"          ListISerializable _$(arg.argname) = new ListISerializable.with_backer($(arg.argname));"
                );
                wr_line(
@"          rc.add_parameter(_$(arg.argname));"
                );
            }
        }
    }

    void stub_launch_method_firstpart(Method met)
    {
        if (met.returntype == "void")
        {
            wr_line(
@"          filter_exception("
            );
        }
        else if (met.returntype.length > 1 && met.returntype.substring(met.returntype.length-1) == "?")
        {
            // nullable... whatever
            wr_block(
"""
            ISerializable ret =
                filter_exception(
"""
            );
        }
        else if (met.returntype == "ISerializable")
        {
            wr_line(
@"          return"
            );
            wr_line(
@"              filter_exception("
            );
        }
        else if (met.returntype in serializable_types)
        {
            wr_line(
@"          return ($(met.returntype))"
            );
            wr_line(
@"              filter_exception("
            );
        }
        else if (met.returntype == "int")
        {
            wr_line(
@"          return ((SerializableInt)"
            );
            wr_line(
@"              filter_exception("
            );
        }
        else if (met.returntype == "bool")
        {
            wr_line(
@"          return ((SerializableBool)"
            );
            wr_line(
@"              filter_exception("
            );
        }
        else if (met.returntype == "string")
        {
            wr_line(
@"          return ((SerializableString)"
            );
            wr_line(
@"              filter_exception("
            );
        }
        else if (met.returntype == "Gee.List<string>")
        {
            wr_line(
@"          ListString ret = (ListString)"
            );
            wr_line(
@"              filter_exception("
            );
        }
        else if (met.returntype.length > 9 && met.returntype.substring(0, 9) == "Gee.List<")
        {
            wr_line(
@"          ListISerializable ret = (ListISerializable)"
            );
            wr_line(
@"              filter_exception("
            );
        }

        closepar = "";
        foreach (string err in met.errors) if (errordefs.keys.contains(err))
        {
            wr_line(
@"              filter_exception_$(err)("
            );
            closepar += ")";
        }
    }

    void stub_launch_method_secondpart(Method met)
    {
        if (met.returntype == "void")
        {
            wr_line(
@"          );"
            );
        }
        else if (met.returntype.length > 1 && met.returntype.substring(met.returntype.length-1) == "?")
        {
            // nullable... what?
            string nullclassname = met.returntype.substring(0, met.returntype.length-1);
            if (nullclassname in serializable_types)
            {
                wr_block(
"""
            );
            if (ret.get_type().is_a(typeof(SerializableNone))) return null;
"""
                );
                wr_line(
@"          else return ($(nullclassname))ret;"
                );
            }
            else if (nullclassname == "int")
            {
                wr_block(
"""
            );
            if (ret.get_type().is_a(typeof(SerializableNone))) return null;
"""
                );
                wr_line(
@"          else return ((SerializableInt)ret).i;"
                );
            }
            else if (nullclassname == "bool")
            {
                wr_block(
"""
            );
            if (ret.get_type().is_a(typeof(SerializableNone))) return null;
"""
                );
                wr_line(
@"          else return ((SerializableBool)ret).b;"
                );
            }
            else if (nullclassname == "string")
            {
                wr_block(
"""
            );
            if (ret.get_type().is_a(typeof(SerializableNone))) return null;
"""
                );
                wr_line(
@"          else return ((SerializableString)ret).s;"
                );
            }
        }
        else if (met.returntype == "ISerializable")
        {
            wr_line(
@"          );"
            );
        }
        else if (met.returntype in serializable_types)
        {
            wr_line(
@"          );"
            );
        }
        else if (met.returntype == "int")
        {
            wr_line(
@"          )).i;"
            );
        }
        else if (met.returntype == "bool")
        {
            wr_line(
@"          )).b;"
            );
        }
        else if (met.returntype == "string")
        {
            wr_line(
@"          )).s;"
            );
        }
        else if (met.returntype.length > 9 && met.returntype.substring(0, 9) == "Gee.List<")
        {
            string listof = met.returntype.substring(9, met.returntype.length-10);
            wr_line(
@"          );"
            );
            wr_line(
@"          return (Gee.List<$(listof)>)ret.backed;"
            );
        }
    }

    void write_file_stub()
    {
        wr_block(
"""
/**  Stub classes generated by rpcdesign.
  */

using Gee;
using zcd;

"""
        );
        if (namespace_guess != null)
        {
            wr(@"namespace $(namespace_guess)\n");
            wr_block(
"""
{
"""
            );
        }
        foreach (string errordom in errordefs.keys)
        {
            wr_line(
@"  public ISerializable filter_exception_$(errordom)(ISerializable ret) throws $(errordom)"
            );
            wr_block(
"""
    {
        if (ret.get_type().is_a(typeof(RemotableException)))
        {
            RemotableException e = (RemotableException)ret;
"""
            );
            wr_line(
@"          if (e.domain == \"$(errordom)\")"
            );
            wr_line(
@"          {"
            );
            foreach (string errorcode in errordefs[errordom])
            {
                wr_line(
@"              if (e.code == \"$(errorcode)\") throw new $(errordom).$(errorcode)(e.message);"
                );
            }
            wr_block(
"""
            }
        }
        return ret;
    }

"""
            );
        }
        wr_block(
"""
    /** The following classes are used to perform
      *  RPC call using the following form:
      *
      *    var x = remote_instance.property1.property2.method0(p1, p2, p3);
      *
      *  instead of:
      *
      *    RemoteCall rc = new RemoteCall();
      *    rc.method_name = "property1.property2.method0";
      *    rc.add_parameter(p1);
      *    rc.add_parameter(p2);
      *    rc.add_parameter(p3);
      *    var x = (SerClass)remote_instance.rmt(rc);
      *
      * The actual implementation of method rmt is a duty of a derived class.
      */

"""
        );
        foreach (string root in roots)
        {
            wr_line(
@"  public abstract class $(root)FakeRmt : Object, I$(root)RootDispatcher, FakeRmt"
            );
            wr_line(
@"  {"
            );
            foreach (Property prop in properties[root]) if (prop.classname in remotes)
            {
                stub_property_firstpart(prop);
                wr_line(
@"              _$(prop.propname).root = this;"
                );
                wr_line(
@"              _$(prop.propname).accumulated = \"$(prop.propname)\";"
                );
                stub_property_secondpart(prop);
            }
            foreach (Method met in methods_rmt[root])
            {
                string str_args = "";
                string str_values = "";
                string str_args_next = "";
                foreach (Argument arg in met.args)
                {
                    str_args += @"$(str_args_next)$(arg.classname) $(arg.argname)";
                    if (arg.classname == "int")
                    {
                        str_values += str_args_next + "$(" + arg.argname + ")";
                    }
                    str_args_next = ",";
                }
                wr_line(
@"      public I$(met.returntype) $(met.name)($(str_args))"
                );
                wr_line(
@"      {"
                );
                wr_line(
@"          Remote$(met.returntype) ret = new Remote$(met.returntype)();"
                );
                wr_line(
@"          ret.root = this;"
                );
                wr_line(
@"          ret.accumulated = @\"$(met.name)($(str_values))\";"
                );
                wr_block(
"""
            return ret;
        }

"""
                );
            }
            foreach (Method met in methods[root])
            {
                stub_declare_method(met);
                wr_block(
"""
        {
            RemoteCall rc = new RemoteCall();
"""
                );
                wr_line(
@"          rc.method_name = \"$(met.name)\";"
                );

                stub_add_parameters(met.args);
                stub_launch_method_firstpart(met);
                wr_line(
@"              this.rmt(rc)$(closepar)"
                );
                stub_launch_method_secondpart(met);

                wr_block(
"""
        }

"""
                );
            }
            wr_block(
"""
        public abstract ISerializable rmt(RemoteCall data) throws RPCError;
    }
"""
            );
        }

        foreach (string remote in remotes)
        {
            wr_block(
"""

"""
            );
            wr_line(
@"  public class Remote$(remote) : Object, I$(remote)"
            );
            wr_block(
"""
    {
        public weak FakeRmt root;
        public string accumulated = "";

"""
            );
            foreach (Property prop in properties[remote]) if (prop.classname in remotes)
            {
                stub_property_firstpart(prop);
                wr_line(
@"              _$(prop.propname).root = this.root;"
                );
                wr_line(
@"              _$(prop.propname).accumulated = accumulated + \".$(prop.propname)\";"
                );
                stub_property_secondpart(prop);
            }
            foreach (Method met in methods_rmt[remote])
            {
                string str_args = "";
                string str_values = "";
                string str_args_next = "";
                foreach (Argument arg in met.args)
                {
                    str_args += @"$(str_args_next)$(arg.classname) $(arg.argname)";
                    if (arg.classname == "int")
                    {
                        str_values += str_args_next + "$(" + arg.argname + ")";
                    }
                    str_args_next = ",";
                }
                wr_line(
@"      public I$(met.returntype) $(met.name)($(str_args))"
                );
                wr_line(
@"      {"
                );
                wr_line(
@"          Remote$(met.returntype) ret = new Remote$(met.returntype)();"
                );
                wr_line(
@"          ret.root = root;"
                );
                wr_line(
@"          ret.accumulated = accumulated + @\"$(met.name)($(str_values))\";"
                );
                wr_block(
"""
            return ret;
        }

"""
                );
            }
            foreach (Method met in methods[remote])
            {
                stub_declare_method(met);
                wr_block(
"""
        {
            RemoteCall rc = new RemoteCall();
"""
                );
                wr_line(
@"          rc.method_name = accumulated + \".$(met.name)\";"
                );

                stub_add_parameters(met.args);
                stub_launch_method_firstpart(met);
                wr_line(
@"              root.rmt(rc)$(closepar)"
                );
                stub_launch_method_secondpart(met);

                wr_block(
"""
        }

"""
                );
            }
            wr_block(
"""
    }
"""
            );
        }

        foreach (string root in roots)
        {
            wr_block(
"""

"""
            );
            wr_line(
@"  /** An implementation of $(root)FakeRmt that sends a message via TCP."
            );
            wr_line(
@"    */"
            );
            wr_line(
@"  public class $(root)TCPClient : $(root)FakeRmt"
            );
            wr_block(
"""
    {
        private TCPClient inner;
"""
            );
            wr_line(
@"      public $(root)TCPClient(string dest_addr, uint16? dest_port=null, string? my_addr=null, bool wait_response=true)"
            );
            wr_block(
"""
        {
            inner = new TCPClient(dest_addr, dest_port, my_addr, wait_response);
        }

        public override ISerializable rmt(RemoteCall data) throws RPCError
        {
            return inner.rmt(data);
        }

        public uint16 dest_port {
            get {
                return inner.dest_port;
            }
        }

        public string dest_addr {
            get {
                return inner.dest_addr;
            }
        }

        public bool calling {
            get {
                return inner.calling;
            }
        }

        public bool retry_connect {
            get {
                return inner.retry_connect;
            }
            set {
                inner.retry_connect = value;
            }
        }

        public void close() throws RPCError
        {
            inner.close();
        }
    }

"""
            );
            wr_line(
@"  /** An implementation of $(root)FakeRmt that sends a message "
            );
            wr_block(
"""
      *  to its neighbours in broadcast via UDP.
      */
"""
            );
            wr_line(
@"  public class $(root)BroadcastClient : $(root)FakeRmt"
            );
            wr_block(
"""
    {
        private BroadcastClient inner;
"""
            );
            wr_line(
@"      public $(root)BroadcastClient(ISerializable broadcast_id,"
            );
            wr_block(
"""
                                             string[] devs,
                                             PrepareForAcknowledgements? prep=null,
                                             uint16 port=269)
        {
            inner = new BroadcastClient(broadcast_id, devs, prep, port);
        }

        public override ISerializable rmt(RemoteCall data) throws RPCError
        {
            return inner.rmt(data);
        }

        public uint16 port {
            get {
                return inner.dest_port;
            }
        }
    }

"""
            );
            wr_line(
@"  /** An implementation of $(root)FakeRmt that sends requests to"
            );
            wr_block(
"""
      *  a set of local dispatchers. Simulates a BroadcastClient.
      */
"""
            );
            wr_line(
@"  public class $(root)PseudoBroadcastClient : $(root)FakeRmt"
            );
            wr_block(
"""
    {
        private PseudoBroadcastClient inner;
"""
            );
            wr_line(
@"      public $(root)PseudoBroadcastClient(Gee.List<RPCDispatcher> rpcdispatchers)"
            );
            wr_block(
"""
        {
            inner = new PseudoBroadcastClient(rpcdispatchers);
        }

        public override ISerializable rmt(RemoteCall data) throws RPCError
        {
            return inner.rmt(data);
        }
    }

"""
            );
            wr_line(
@"  /** An implementation of $(root)FakeRmt that sends a message"
            );
            wr_block(
"""
      *  to a neighbour via UDP.
      */
"""
            );
            wr_line(
@"  public class $(root)NeighbourClient : $(root)FakeRmt"
            );
            wr_block(
"""
    {
        private NeighbourClient inner;
"""
            );
            wr_line(
@"      public $(root)NeighbourClient(ISerializable unicast_id, string[] devs, uint16? port=null,  bool wait_response=true)"
            );
            wr_block(
"""
        {
            inner = new NeighbourClient(unicast_id, devs, port, wait_response);
        }

        public override ISerializable rmt(RemoteCall data) throws RPCError
        {
            return inner.rmt(data);
        }

        public uint16 port {
            get {
                return inner.dest_port;
            }
        }
    }

"""
            );
            wr_line(
@"  /** An implementation of $(root)FakeRmt that sends requests to a"
            );
            wr_block(
"""
      *  local dispatcher. Simulates a NeighbourClient.
      */
"""
            );
            wr_line(
@"  public class $(root)PseudoNeighbourClient : $(root)FakeRmt"
            );
            wr_block(
"""
    {
        private PseudoNeighbourClient inner;
"""
            );
            wr_line(
@"      public $(root)PseudoNeighbourClient(RPCDispatcher rpcdispatcher, bool wait_response=true)"
            );
            wr_block(
"""
        {
            inner = new PseudoNeighbourClient(rpcdispatcher, wait_response);
        }

        public override ISerializable rmt(RemoteCall data) throws RPCError
        {
            return inner.rmt(data);
        }
    }

"""
            );
            wr_line(
@"  /** An implementation of $(root)FakeRmt that sends a message (usually without waiting for"
            );
            wr_block(
"""
      *  a result) to two remote objects.
      */
"""
            );
            wr_line(
@"  public class Couple$(root)FakeRmt : $(root)FakeRmt"
            );
            wr_line(
@"  {"
            );
            wr_line(
@"      private $(root)FakeRmt f1;"
            );
            wr_line(
@"      private $(root)FakeRmt f2;"
            );
            wr_line(
@"      public Couple$(root)FakeRmt($(root)FakeRmt f1, $(root)FakeRmt f2)"
            );
            wr_block(
"""
        {
            this.f1 = f1;
            this.f2 = f2;
        }

        public override ISerializable rmt(RemoteCall data) throws RPCError
        {
            try {f1.rmt(data);} catch (RPCError e) {}
            try {f2.rmt(data);} catch (RPCError e) {}
            return new SerializableNone();
        }
    }
"""
            );
        }

        if (namespace_guess != null)
        {
            wr_block(
"""
}
"""
            );
        }
    }

    void write_file_skeleton()
    {
        wr_block(
"""
/**  Skeleton classes generated by rpcdesign.
  */

using Gee;
using zcd;

"""
        );
        if (namespace_guess != null)
        {
            wr(@"namespace $(namespace_guess)\n");
            wr_block(
"""
{
"""
            );
        }
        foreach (string root in roots)
        {
            wr_line(
@"  public class $(root)Dispatcher : RPCDispatcher"
            );
            wr_block(
"""
    {
"""
            );
            wr_line(
@"      private I$(root)RootDispatcher root;"
            );
            wr_line(
@"      public $(root)Dispatcher(I$(root)RootDispatcher root)"
            );
            wr_block(
"""
        {
            this.root = root;
        }

"""
            );
            if (! errordefs.keys.is_empty)
            {
                wr_block(
"""
        /** This method adds the feature that a model-specific exception is made serializable.
          */
        public override ISerializable remoteexception_dispatch(Object? caller, RemoteCall data) throws Error
        {
            try
            {
                return _dispatch(caller, data);
            }
"""
                );
                foreach (string errdom in errordefs.keys)
                {
                    wr_line(
@"          catch ($(errdom) e)"
                    );
                    wr_block(
"""
            {
                RemotableException re = new RemotableException();
                re.message = e.message;
"""
                    );
                    wr_line(
@"              re.domain = \"$(errdom)\";");
                    foreach (string errcode in errordefs[errdom])
                    {
                        wr_line(
@"              if (e is $(errdom).$(errcode)) re.code = \"$(errcode)\";");
                    }
                    wr_block(
"""
                return re;
            }
"""
                    );
                }
                wr_block(
"""
        }

"""
                );
            }
            wr_block(
"""
        /** This method could be called *directly* for a dispatcher that does not need to transform
          * an exception into a remotable.
          */
        public override ISerializable _dispatch(Object? caller, RemoteCall data) throws Error
        {
            string[] pieces = data.method_name.split(".");
"""
            );
            string impl = recurse_remote_members(root);
            wr(impl);
            wr_block(
"""
            return base._dispatch(caller, data);
        }
    }
"""
            );
        }

        if (namespace_guess != null)
        {
            wr_block(
"""
}
"""
            );
        }
    }

    string recurse_remote_members(string classname, int level = 0, string accumulated = "")
    {
        string ret = "";
        foreach (Property p in properties[classname])
        {
            ret += rec_line(
@"          if (pieces[$(level)] == \"$(p.propname)\")"
            );
            ret += rec_block(
"""
            {
"""
            );
            ret += indentlines(recurse_remote_members(p.classname, level+1, @"$(p.propname)."));
            ret += rec_block(
"""
            }
"""
            );
        }
        foreach (Method m in methods_rmt[classname])
        {
            int len = m.name.length + 1;
            ret += rec_line(
@"          if (pieces[$(level)].length > $(len) && pieces[$(level)].substring(0, $(len)) == \"$(m.name)(\")"
            );
            ret += rec_line(
@"          {"
            );
            ret += rec_line(
@"              string remaining = pieces[$(level)].substring($(len));"
            );
            ret += rec_line(
@"              string curarg = \"\";"
            );
            foreach (Argument arg in m.args)
            {
                if (arg.classname == "int")
                {
                    ret += rec_block(
"""
                while (remaining.substring(0, 1) in "1234567890")
                {
                    curarg += remaining.substring(0, 1);
                    remaining = remaining.substring(1);
                }
                remaining = remaining.substring(1);
"""
                    );
                    ret += rec_line(
@"              int $(arg.argname) = int.parse(curarg);"
                    );
                }
            }
            
            string str_args = "";
            string str_args_next = "";
            foreach (Argument arg in m.args)
            {
                str_args += @"$(str_args_next)$(arg.argname)";
                str_args_next = ", ";
            }

            ret += indentlines(recurse_remote_members(m.returntype, level+1, @"$(m.name)($(str_args))."));
            ret += rec_block(
"""
            }
"""
            );
        }
        foreach (Method m in methods[classname])
        {
            ret += rec_line(
@"          if (pieces[$(level)] == \"$(m.name)\")"
            );
            ret += rec_block(
"""
            {
"""
            );
            ret += rec_line(
@"              if (pieces.length != $(level+1))"
            );
            ret += rec_block(
"""
                    throw new RPCError.MALFORMED_PACKET(
"""
            );
            ret += rec_line(
@"                      \"$(accumulated)$(m.name) is a function.\");"
            );
            ret += rec_line(
@"              if (data.parameters.size != $(m.args.length))"
            );
            ret += rec_block(
"""
                    throw new RPCError.MALFORMED_PACKET(
"""
            );
            if (m.args.length == 0)
                ret += rec_line(
@"                      \"$(accumulated)$(m.name) wants no parameters.\");"
                );
            else if (m.args.length == 1)
                ret += rec_line(
@"                      \"$(accumulated)$(m.name) wants 1 parameter.\");"
                );
            else
                ret += rec_line(
@"                      \"$(accumulated)$(m.name) wants $(m.args.length) parameters.\");"
                );
            int argcount = -1;
            foreach (Argument arg in m.args)
            {
                argcount++;
                ret += rec_line(
@"              ISerializable iser$(argcount) = data.parameters[$(argcount)];"
                );
                if (arg.classname.length > 1 && arg.classname.substring(arg.classname.length-1) == "?")
                {
                    // nullable... what?
                    string nullclassname = arg.classname.substring(0, arg.classname.length-1);
                    if (nullclassname in serializable_types)
                    {
                        ret += rec_line(
@"                  if (! iser$(argcount).get_type().is_a(typeof($(nullclassname))) && ! iser$(argcount).get_type().is_a(typeof(SerializableNone)))"
                        );
                        ret += rec_line(
@"                      throw new RPCError.MALFORMED_PACKET("
                        );
                        ret += rec_line(
@"                          \"$(accumulated)$(m.name) parameter $(argcount+1) is not a $(nullclassname)?.\");"
                        );
                        ret += rec_line(
@"                  $(nullclassname)? $(arg.argname) = null;"
                        );
                        ret += rec_line(
@"                  if (iser$(argcount).get_type().is_a(typeof($(nullclassname)))) $(arg.argname) = ($(nullclassname)) iser$(argcount);"
                        );
                    }
                    else if (nullclassname == "int")
                    {
                        ret += rec_line(
@"                  if (! iser$(argcount).get_type().is_a(typeof(SerializableInt)) && ! iser$(argcount).get_type().is_a(typeof(SerializableNone)))"
                        );
                        ret += rec_line(
@"                      throw new RPCError.MALFORMED_PACKET("
                        );
                        ret += rec_line(
@"                          \"$(accumulated)$(m.name) parameter $(argcount+1) is not a int?.\");"
                        );
                        ret += rec_line(
@"                  int? $(arg.argname) = null;"
                        );
                        ret += rec_line(
@"                  if (iser$(argcount).get_type().is_a(typeof(SerializableInt))) $(arg.argname) = ((SerializableInt) iser$(argcount)).i;"
                        );
                    }
                    else if (nullclassname == "bool")
                    {
                        ret += rec_line(
@"                  if (! iser$(argcount).get_type().is_a(typeof(SerializableBool)) && ! iser$(argcount).get_type().is_a(typeof(SerializableNone)))"
                        );
                        ret += rec_line(
@"                      throw new RPCError.MALFORMED_PACKET("
                        );
                        ret += rec_line(
@"                          \"$(accumulated)$(m.name) parameter $(argcount+1) is not a bool?.\");"
                        );
                        ret += rec_line(
@"                  bool? $(arg.argname) = null;"
                        );
                        ret += rec_line(
@"                  if (iser$(argcount).get_type().is_a(typeof(SerializableBool))) $(arg.argname) = ((SerializableBool) iser$(argcount)).b;"
                        );
                    }
                    else if (nullclassname == "string")
                    {
                        ret += rec_line(
@"                  if (! iser$(argcount).get_type().is_a(typeof(SerializableString)) && ! iser$(argcount).get_type().is_a(typeof(SerializableNone)))"
                        );
                        ret += rec_line(
@"                      throw new RPCError.MALFORMED_PACKET("
                        );
                        ret += rec_line(
@"                          \"$(accumulated)$(m.name) parameter $(argcount+1) is not a string?.\");"
                        );
                        ret += rec_line(
@"                  string? $(arg.argname) = null;"
                        );
                        ret += rec_line(
@"                  if (iser$(argcount).get_type().is_a(typeof(SerializableString))) $(arg.argname) = ((SerializableString) iser$(argcount)).s;"
                        );
                    }
                }
                else if (arg.classname == "ISerializable")
                {
                    ret += rec_line(
@"              if (! iser$(argcount).get_type().is_a(typeof($(arg.classname))))"
                    );
                    ret += rec_line(
@"                  throw new RPCError.MALFORMED_PACKET("
                    );
                    ret += rec_line(
@"                      \"$(accumulated)$(m.name) parameter $(argcount+1) is not a $(arg.classname).\");"
                    );
                    ret += rec_line(
@"              ISerializable $(arg.argname) = iser$(argcount);"
                    );
                }
                else if (arg.classname in serializable_types)
                {
                    ret += rec_line(
@"              if (! iser$(argcount).get_type().is_a(typeof($(arg.classname))))"
                    );
                    ret += rec_line(
@"                  throw new RPCError.MALFORMED_PACKET("
                    );
                    ret += rec_line(
@"                      \"$(accumulated)$(m.name) parameter $(argcount+1) is not a $(arg.classname).\");"
                    );
                    ret += rec_line(
@"              $(arg.classname) $(arg.argname) = ($(arg.classname))iser$(argcount);"
                    );
                }
                else if (arg.classname == "int")
                {
                    ret += rec_line(
@"              if (! iser$(argcount).get_type().is_a(typeof(SerializableInt)))"
                    );
                    ret += rec_line(
@"                  throw new RPCError.MALFORMED_PACKET("
                    );
                    ret += rec_line(
@"                      \"$(accumulated)$(m.name) parameter $(argcount+1) is not a int.\");"
                    );
                    ret += rec_line(
@"              int $(arg.argname) = ((SerializableInt)iser$(argcount)).i;"
                    );
                }
                else if (arg.classname == "bool")
                {
                    ret += rec_line(
@"              if (! iser$(argcount).get_type().is_a(typeof(SerializableBool)))"
                    );
                    ret += rec_line(
@"                  throw new RPCError.MALFORMED_PACKET("
                    );
                    ret += rec_line(
@"                      \"$(accumulated)$(m.name) parameter $(argcount+1) is not a bool.\");"
                    );
                    ret += rec_line(
@"              bool $(arg.argname) = ((SerializableBool)iser$(argcount)).b;"
                    );
                }
                else if (arg.classname == "string")
                {
                    ret += rec_line(
@"              if (! iser$(argcount).get_type().is_a(typeof(SerializableString)))"
                    );
                    ret += rec_line(
@"                  throw new RPCError.MALFORMED_PACKET("
                    );
                    ret += rec_line(
@"                      \"$(accumulated)$(m.name) parameter $(argcount+1) is not a string.\");"
                    );
                    ret += rec_line(
@"              string $(arg.argname) = ((SerializableString)iser$(argcount)).s;"
                    );
                }
                else if (arg.classname == "Gee.List<string>")
                {
                    ret += rec_line(
@"              if (! iser$(argcount).get_type().is_a(typeof(ListString)))"
                    );
                    ret += rec_line(
@"                  throw new RPCError.MALFORMED_PACKET("
                    );
                    ret += rec_line(
@"                      \"$(accumulated)$(m.name) parameter $(argcount+1) is not a List<string>.\");"
                    );
                    ret += rec_line(
@"              ListString _$(arg.argname) = (ListString)iser$(argcount);"
                    );
                    ret += rec_line(
@"              Gee.List<string> $(arg.argname) = _$(arg.argname).backed;"
                    );
                }
                else if (arg.classname.length > 9 && arg.classname.substring(0, 9) == "Gee.List<")
                {
                    ret += rec_line(
@"              if (! iser$(argcount).get_type().is_a(typeof(ListISerializable)))"
                    );
                    ret += rec_line(
@"                  throw new RPCError.MALFORMED_PACKET("
                    );
                    ret += rec_line(
@"                      \"$(accumulated)$(m.name) parameter $(argcount+1) is not a List<NIP>.\");"
                    );
                    ret += rec_line(
@"              ListISerializable _$(arg.argname) = (ListISerializable)iser$(argcount);"
                    );
                    ret += rec_line(
@"              Gee.List<NIP> $(arg.argname) = (Gee.List<NIP>)_$(arg.argname).backed;"
                    );
                }
            }
            string str_args = "";
            string str_args_next = "";
            foreach (Argument arg in m.args)
            {
                str_args += @"$(str_args_next)$(arg.argname)";
                str_args_next = ", ";
            }
            if (m.pass_caller)
                str_args += @"$(str_args_next)(CallerInfo)caller";
            string str_command = @"root.$(accumulated)$(m.name)($(str_args))";
            if (m.returntype == "void")
            {
                ret += rec_line(
@"              $(str_command);"
                );
                ret += rec_line(
@"              return new SerializableNone();"
                );
            }
            else if (m.returntype.length > 1 && m.returntype.substring(m.returntype.length-1) == "?")
            {
                // nullable... what?
                string nullclassname = m.returntype.substring(0, m.returntype.length-1);
                if (nullclassname in serializable_types)
                {
                    ret += rec_line(
@"              $(nullclassname)? ret = $(str_command);"
                    );
                    ret += rec_line(
@"              if (ret == null) return new SerializableNone();"
                    );
                    ret += rec_line(
@"              return ret;"
                    );
                }
                else if (nullclassname == "int")
                {
                    ret += rec_line(
@"              int? ret = $(str_command);"
                    );
                    ret += rec_line(
@"              if (ret == null) return new SerializableNone();"
                    );
                    ret += rec_line(
@"              return new SerializableInt(ret);"
                    );
                }
                else if (nullclassname == "bool")
                {
                    ret += rec_line(
@"              bool? ret = $(str_command);"
                    );
                    ret += rec_line(
@"              if (ret == null) return new SerializableNone();"
                    );
                    ret += rec_line(
@"              return new SerializableBool(ret);"
                    );
                }
                else if (nullclassname == "string")
                {
                    ret += rec_line(
@"              string? ret = $(str_command);"
                    );
                    ret += rec_line(
@"              if (ret == null) return new SerializableNone();"
                    );
                    ret += rec_line(
@"              return new SerializableString(ret);"
                    );
                }
            }
            else if (m.returntype == "ISerializable")
            {
                ret += rec_line(
@"              return $(str_command);"
                );
            }
            else if (m.returntype in serializable_types)
            {
                ret += rec_line(
@"              return $(str_command);"
                );
            }
            else if (m.returntype == "int")
            {
                ret += rec_line(
@"              return new SerializableInt($(str_command));"
                );
            }
            else if (m.returntype == "bool")
            {
                ret += rec_line(
@"              return new SerializableBool($(str_command));"
                );
            }
            else if (m.returntype == "string")
            {
                ret += rec_line(
@"              return new SerializableString($(str_command));"
                );
            }
            else if (m.returntype == "Gee.List<string>")
            {
                ret += rec_line(
@"              Gee.List<string> _ret = $(str_command);"
                );
                ret += rec_block(
"""
                ListString ret = new ListString.with_backer(_ret);
                return ret;
"""
                );
            }
            else if (m.returntype.length > 9 && m.returntype.substring(0, 9) == "Gee.List<")
            {
                string listof = m.returntype.substring(9, m.returntype.length-10);
                ret += rec_line(
@"              Gee.List<$(listof)> _ret = $(str_command);"
                );
                ret += rec_block(
"""
                ListISerializable ret = new ListISerializable.with_backer(_ret);
                return ret;
"""
                );
            }

            ret += rec_block(
"""
            }
"""
            );
        }
        return ret;
    }

    string rec_block(string m)
    {
        // remove first NL
        return m.substring(1);
    }

    string rec_line(string m)
    {
        // add 2 blanks and final NL
        return "  " + m + "\n";
    }

    string indentlines(owned string x, string prefix = "    ")
    {
        if (x.length == 0) return x;
        bool finaleol = x.substring(x.length-1) == "\n";
        if (finaleol) x = x.substring(0, x.length-1);

        Regex regex = new Regex ("^");
        x = regex.replace (x, x.length, 0, prefix);
        Regex regex1 = new Regex ("\n");
        x = regex1.replace (x, x.length, 0, "\n" + prefix);

        if (finaleol) x = x + "\n";
        return x;
    }

    void wr_block(string m)
    {
        // remove first NL
        wr(m.substring(1));
    }

    void wr_line(string m)
    {
        // add 2 blanks and final NL
        wr("  " + m + "\n");
    }

    void wr(string m)
    {
        output += m;
    }

    void read_file_if(string if_name)
    {
        string[] lines = read_file(if_name);
        lines += "";
        namespace_guess = null;
        serializable_types = {};
        errordefs = new HashMap<string, ArrayList<string>>();
        roots = {};
        remotes = {};
        properties = new HashMap<string, ArrayList<Property>>();
        methods = new HashMap<string, ArrayList<Method>>();
        methods_rmt = new HashMap<string, ArrayList<Method>>();

        int c = 0;
        if (lines[c] != "errors:") malformed(c);
        while (Regex.match_simple("""^ [a-zA-Z_]""", lines[c+1]))
        {
            c++;
            string dom = get_keyword(lines[c]);
            errordefs[dom] = new ArrayList<string>();
            while (Regex.match_simple("""^  [a-zA-Z_]""", lines[c+1]))
            {
                c++;
                string code = get_keyword(lines[c]);
                errordefs[dom].add(code);
            }
        }
        c++;
        if (lines[c] != "serializables:") malformed(c);
        while (Regex.match_simple("""^ [a-zA-Z_]""", lines[c+1]))
        {
            c++;
            string ser = get_keyword(lines[c]);
            serializable_types += ser;
        }
        serializable_types += "RemoteCall";
        c++;
        if (lines[c] != "dispatchers:") malformed(c);
        while (Regex.match_simple("""^ [a-zA-Z_]""", lines[c+1]))
        {
            c++;
            string root = get_keyword(lines[c]);
            roots += root;
            properties[root] = new ArrayList<Property>();
            methods[root] = new ArrayList<Method>();
            methods_rmt[root] = new ArrayList<Method>();
            c++;
            if (lines[c] != "  properties:") malformed(c);
            while (Regex.match_simple("""^   [a-zA-Z_]""", lines[c+1]))
            {
                c++;
                Property p = new Property();
                string[] prop = get_two_keywords(lines[c]);
                p.classname = prop[0];
                p.propname = prop[1];
                properties[root].add(p);
            }
            c++;
            if (lines[c] != "  methods_rmt:") malformed(c);
            while (Regex.match_simple("""^   [a-zA-Z_]""", lines[c+1]))
            {
                c++;
                Method m = new Method();
                string[] meth = get_two_keywords(lines[c]);
                m.returntype = meth[0];
                m.name = meth[1];
                m.args = {};
                m.errors = {};
                methods_rmt[root].add(m);
                c++;
                if (lines[c] != "    arguments:") malformed(c);
                while (Regex.match_simple("""^     [a-zA-Z_]""", lines[c+1]))
                {
                    c++;
                    Argument a = new Argument();
                    string[] arg = get_two_keywords(lines[c]);
                    a.classname = arg[0];
                    a.argname = arg[1];
                    m.args += a;
                }
            }
            c++;
            if (lines[c] != "  methods:") malformed(c);
            while (Regex.match_simple("""^   [a-zA-Z_]""", lines[c+1]))
            {
                c++;
                Method m = new Method();
                string[] meth = get_two_keywords(lines[c]);
                m.returntype = meth[0];
                m.name = meth[1];
                m.args = {};
                m.errors = {};
                methods[root].add(m);
                c++;
                if (lines[c] != "    arguments:") malformed(c);
                while (Regex.match_simple("""^     [a-zA-Z_]""", lines[c+1]))
                {
                    c++;
                    Argument a = new Argument();
                    string[] arg = get_two_keywords(lines[c]);
                    a.classname = arg[0];
                    a.argname = arg[1];
                    m.args += a;
                }
                c++;
                if (lines[c] == "    pass_caller")
                {
                    m.pass_caller = true;
                    c++;
                }
                if (lines[c] != "    throws:") malformed(c);
                while (Regex.match_simple("""^     [a-zA-Z_]""", lines[c+1]))
                {
                    c++;
                    string err = get_keyword(lines[c]);
                    m.errors += err;
                }
            }
        }
        c++;
        if (lines[c] != "remoteclasses:") malformed(c);
        while (Regex.match_simple("""^ [a-zA-Z_]""", lines[c+1]))
        {
            c++;
            string remote = get_keyword(lines[c]);
            remotes += remote;
            properties[remote] = new ArrayList<Property>();
            methods[remote] = new ArrayList<Method>();
            methods_rmt[remote] = new ArrayList<Method>();
            c++;
            if (lines[c] != "  properties:") malformed(c);
            while (Regex.match_simple("""^   [a-zA-Z_]""", lines[c+1]))
            {
                c++;
                Property p = new Property();
                string[] prop = get_two_keywords(lines[c]);
                p.classname = prop[0];
                p.propname = prop[1];
                properties[remote].add(p);
            }
            c++;
            if (lines[c] != "  methods_rmt:") malformed(c);
            while (Regex.match_simple("""^   [a-zA-Z_]""", lines[c+1]))
            {
                c++;
                Method m = new Method();
                string[] meth = get_two_keywords(lines[c]);
                m.returntype = meth[0];
                m.name = meth[1];
                m.args = {};
                m.errors = {};
                methods_rmt[remote].add(m);
                c++;
                if (lines[c] != "    arguments:") malformed(c);
                while (Regex.match_simple("""^     [a-zA-Z_]""", lines[c+1]))
                {
                    c++;
                    Argument a = new Argument();
                    string[] arg = get_two_keywords(lines[c]);
                    a.classname = arg[0];
                    a.argname = arg[1];
                    m.args += a;
                }
            }
            c++;
            if (lines[c] != "  methods:") malformed(c);
            while (Regex.match_simple("""^   [a-zA-Z_]""", lines[c+1]))
            {
                c++;
                Method m = new Method();
                string[] meth = get_two_keywords(lines[c]);
                m.returntype = meth[0];
                m.name = meth[1];
                m.args = {};
                m.errors = {};
                methods[remote].add(m);
                c++;
                if (lines[c] != "    arguments:") malformed(c);
                while (Regex.match_simple("""^     [a-zA-Z_]""", lines[c+1]))
                {
                    c++;
                    Argument a = new Argument();
                    string[] arg = get_two_keywords(lines[c]);
                    a.classname = arg[0];
                    a.argname = arg[1];
                    m.args += a;
                }
                c++;
                if (lines[c] == "    pass_caller")
                {
                    m.pass_caller = true;
                    c++;
                }
                if (lines[c] != "    throws:") malformed(c);
                while (Regex.match_simple("""^     [a-zA-Z_]""", lines[c+1]))
                {
                    c++;
                    string err = get_keyword(lines[c]);
                    m.errors += err;
                }
            }
        }
        c++;
        if (lines[c] != "")
        {
            if (lines[c].length < 12 && lines[c].substring(0, 11) != "namespace: ") malformed(c);
            namespace_guess = lines[c].substring(11);
        }
    }

    string get_keyword(string s)
    {
        MatchInfo m;
        Regex r = new Regex("""[a-zA-Z0-9_]+""");
        if (! r.match(s, 0, out m)) return "";
        return m.fetch(0);
    }

    string[] get_two_keywords(string s)
    {
        MatchInfo m;
        // A classname can contain letters numbers and underscore.
        // It can be prefixed with a namespace.
        // It can contain generics (supports only Gee.List<xxxx>)
        // It can be nullable.
        Regex r = new Regex("""[a-zA-Z0-9_.<>\??]+ [a-zA-Z0-9_]+""");
        if (! r.match(s, 0, out m)) return " ".split(" ");
        return m.fetch(0).split(" ");
    }

    void malformed(int c)
    {
        stderr.printf(@"temp file is malformed. Line $(c).\n");
        Process.exit (1);
    }

    string[] read_file(string path)
    {
        string[] ret = new string[0];
        if (FileUtils.test(path, FileTest.EXISTS))
        {
            try
            {
                string contents;
                assert(FileUtils.get_contents(path, out contents));
                ret = contents.split("\n");
            }
            catch (FileError e) {error("%s: %d: %s".printf(e.domain.to_string(), e.code, e.message));}
        }
        return ret;
    }

    void write_file(string path, string contents)
    {
        try
        {
            assert(FileUtils.set_contents(path, contents));
        }
        catch (FileError e) {error("%s: %d: %s".printf(e.domain.to_string(), e.code, e.message));}
    }
}
