/**
 * WebAdmin
 * Copyright (C) 2006 Netwosix Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Linking WebAdmin statically or dynamically with other modules is making
 * a combined work based on WebAdmin. Thus, the terms and conditions of the 
 * GNU General Public License cover the whole combination.
 *
 * In addition, as a special exception, the copyright holders of
 * WebAdmin give you permission to combine WebAdmin with free software
 * programs or libraries that are released under the GNU LGPL and with
 * code included in the standard release of OpenSSL under the OpenSSL
 * License and SSLeay License (or modified versions of such code, with
 * unchanged licenses). You may copy and distribute such a system
 * following the terms of the GNU GPL for WebAdmin and the licenses of
 * the other code concerned, provided that you include the source code of
 * that other code when and as the GNU GPL requires distribution of
 * source code.
 *
 * Note that people who make modified versions of WebAdmin are not obligated
 * to grant this special exception for their modified versions; 
 * it is their choice whether to do so. The GNU General Public License 
 * gives permission to release a modified version without this exception; 
 * this exception also makes it possible to release a modified 
 * version which carries forward this exception.
 */

#ifndef AGENTD_RESPONSE_H_
#define AGENTD_RESPONSE_H_


/**
 * Type of response
 */
enum _AgentdResponseType
{

	AGENTD_RESPONSE_OK                 = 0, /* everything is ok                */
	AGENTD_RESPONSE_AGENT_NOT_FOUND    = 1, /* agent not found                 */
	AGENTD_RESPONSE_AGENT_FAILED       = 2, /* agent end in error              */
	AGENTD_RESPONSE_AGENTD_FAILED      = 3, /* agentd could not call agent     */
	AGENTD_RESPONSE_INVALID_AGENT_DATA = 4  /* the agent xml data is not valir */

};

typedef int AgentdResponseType;


/**
 * Structure representing a response
 */
struct _AgentdResponse
{

	struct _AgentdResponse * prev;
	struct _AgentdResponse * next;

	char                   * id;         /* request id              */
	AgentdResponseType       type;       /* type of response        */
	char                   * agent_name; /* name of requested agent */
	char                   * agent_data; /* data for agent          */

};

typedef struct _AgentdResponse AgentdResponse;


/**
 */
#define agentd_response_set_id(response,v)                   (response)->id = al_strdup(v)


/**
 */
#define agentd_response_get_id(response)                    (response)->id


/**
 */
#define agentd_response_set_agent_name(response,v)          (response)->agent_name = al_strdup(v)


/**
 */
#define agentd_response_get_agent_name(response)            (response)->agent_name


/**
 */
#define agentd_response_set_agent_data(response,v)          (response)->agent_data = al_strdup(v)


/**
 */
#define agentd_response_get_agent_data(response)            (response)->agent_data


/**
 */
#define agentd_response_set_type(response,v)                (response)->type = v


/**
 */
#define agentd_response_get_type(response)                  (response)->type


/**
 * Structure representing a packet with one or many response.
 */
struct _AgentdResponsePacket
{

	struct _AgentdResponse * responses; /* list of response in the packet */

};

typedef struct _AgentdResponsePacket AgentdResponsePacket;


/**
 * Create a new response.
 *
 * @return the new response.
 */
AgentdResponse * agentd_response_new(void);


/**
 * Delete a response.
 *
 * @param response response to free
 */
void agentd_response_delete(AgentdResponse * response);


/**
 * Create a new response packet.
 *
 * @return the new response packet.
 */
AgentdResponsePacket * agentd_response_packet_new(void);


/**
 * Delete a response packet and all response in list.
 *
 * @param response_packet response packet to free
 */
void agentd_response_packet_delete(AgentdResponsePacket * response_packet);


/**
 * Add a response to a response packet.
 *
 * @param response_packet response packet
 * @param response response to add
 */
void agentd_response_packet_response_add(AgentdResponsePacket * response_packet, AgentdResponse * response);


/**
 * Create a new response basend on a request.
 * This process the request.
 *
 * @param request request who response is based
 *
 * @return the new response
 */
AgentdResponse * agentd_response_from_request(const AgentdRequest * request, AlXml * config);


/**
 * Create a new response packet basend on a request packet.
 * This process every request in the packet.
 *
 * @param request_packet request packet who response is based
 *
 * @return the new response packet
 */
AgentdResponsePacket * agentd_response_packet_from_request_packet(const AgentdRequestPacket * request_packet, AlXml * config);


/**
 * Dump the response packet to xml.
 *
 * @param response_packet response packet to dump
 *
 * @return response packet in xml
 */
AlXmlList * agentd_response_packet_to_xml(const AgentdResponsePacket * response_packet);


/**
 * Dump the response packet to xml.
 *
 * @param response_packet response packet to dump
 *
 * @return response packet in xml
 */
AlString * agentd_response_packet_to_xml_string(const AgentdResponsePacket * response_packet);


/**
 * Dump the response to xml.
 *
 * @param response response to dump
 *
 * @return response in xml node
 */
AlXmlNode * agentd_response_to_xml(const AgentdResponse * response);


/**
 * Create a new response packet from xml string.
 *
 * @param xml xml string
 *
 * @return a new response packet or NULL if xml is not valid.
 */
AgentdResponsePacket * agentd_response_packet_from_xml_string(const AlString * xml);


/**
 * Create a new response packet from xml.
 *
 * @param xml xml
 *
 * @return a new response packet or NULL if xml is not valid.
 */
AgentdResponsePacket * agentd_response_packet_from_xml(AlXml * xml);


/**
 * Create a new response from xml.
 *
 * @param node xml node
 *
 * @return a new response
 */
AgentdResponse * agentd_response_from_xml(AlXmlNode * node);


#endif /* AGENTD_RESPONSE_H_ */
