/**
 * WebAdmin
 * Copyright (C) 2006 Netwosix Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Linking WebAdmin statically or dynamically with other modules is making
 * a combined work based on WebAdmin. Thus, the terms and conditions of the 
 * GNU General Public License cover the whole combination.
 *
 * In addition, as a special exception, the copyright holders of
 * WebAdmin give you permission to combine WebAdmin with free software
 * programs or libraries that are released under the GNU LGPL and with
 * code included in the standard release of OpenSSL under the OpenSSL
 * License and SSLeay License (or modified versions of such code, with
 * unchanged licenses). You may copy and distribute such a system
 * following the terms of the GNU GPL for WebAdmin and the licenses of
 * the other code concerned, provided that you include the source code of
 * that other code when and as the GNU GPL requires distribution of
 * source code.
 *
 * Note that people who make modified versions of WebAdmin are not obligated
 * to grant this special exception for their modified versions; 
 * it is their choice whether to do so. The GNU General Public License 
 * gives permission to release a modified version without this exception; 
 * this exception also makes it possible to release a modified 
 * version which carries forward this exception.
 */


#include "al.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>


#define AL_SOCKET_EOP                "EOP"
#define AL_SOCKET_EOF                "EOF"


/**
 * Delete the connection manager.
 *
 * @param connection_manager connection manager to delete
 */
void al_socket_connection_manager_delete(AlConnectionManager * connection_manager);


/**
 * Delete the connection.
 *
 * @param connection connection to delete
 */
void al_socket_connection_delete(AlConnection * connection);


/**
 * Wait for a new connection.
 *
 * @param connection_manager connection manager
 *
 * @return the new connection or NULL if not found.
 */
AlConnection * al_socket_connection_manager_wait(AlConnectionManager * connection_manager);


/**
 * Close the socket.
 *
 * @param connection connection to close
 */
void al_socket_connection_close(AlConnection * connection);


/**
 * Read a request.
 *
 * @param connection_manager connection manager
 *
 * @return the new packet or NULL if not found.
 */
AlString * al_socket_connection_read(AlConnection * connection);


/**
 * Send a response.
 *
 * @param connection_manager connection manager
 * @param packet packet to send
 */
void al_socket_connection_send(AlConnection * connection, const AlString * response);


/**
 * Connect to a server.
 *
 * @param connection_manager the connection manager
 * @param server server to connect
 *
 * @return the new connection or NULL if fail
 */
AlConnection * al_socket_connection_manager_connect(AlConnectionManager * connection_manager, const char * server);


/**
 */
void al_socket_connection_delete(AlConnection * connection);


/**
 */
AlConnection * al_socket_connection_new(AlConnectionManager * connection_manager);


/**
 */
struct _AlSocketManager
{
	int                init;

	int                sock;
	int                data;
	int                on;
	struct sockaddr_in name;
	unsigned long      ioctlarg;
	int                port;

};

typedef struct _AlSocketManager AlSocketManager;


/**
 */
struct _AlSocket
{

	int                  sock;
	struct sockaddr_in   sn;
	socklen_t            snsize;
	struct hostent     * hostname;

};

typedef struct _AlSocket AlSocket;


/**
 */
AlConnectionManager * al_socket_connection_manager_new(int port)
{
	AlSocketManager     * socket_manager = al_new(AlSocketManager);
	AlConnectionManager * connection = al_connection_manager_new(al_socket_connection_manager_wait,
								     al_socket_connection_manager_delete,
								     al_socket_connection_manager_connect,
								     1,
								     socket_manager);
								     
	
	socket_manager->init = 0;
	socket_manager->data = 0;
 	socket_manager->on = 1;
	socket_manager->ioctlarg = 0;
	socket_manager->port = port;


	return connection;
}


/**
 */
void al_socket_connection_manager_delete(AlConnectionManager * connection_manager)
{
	AlSocketManager * socket_manager;


	al_return_if_fail (connection_manager);

	socket_manager = al_connection_manager_get_data(connection_manager, AlSocketManager);
	al_delete(socket_manager);

}


/**
 */
AlConnection * al_socket_connection_new(AlConnectionManager * connection_manager)
{
	AlSocket     * sock = al_new(AlSocket);
	AlConnection * connection = al_connection_new(connection_manager,
						      al_socket_connection_read,
						      al_socket_connection_send,
						      al_socket_connection_close,
						      al_socket_connection_delete,
						      sock);
	
	sock->snsize = sizeof(sock->sn);
	sock->hostname = NULL;
	sock->sock = 0;
	
	return connection;
}


/**
 */
void al_socket_connection_delete(AlConnection * connection)
{
	AlSocket * socket;

	al_return_if_fail(connection);

	socket = al_connection_get_data(connection, AlSocket);

	al_delete(socket);
}


/**
 */
AlConnection * al_socket_connection_manager_wait(AlConnectionManager * connection_manager)
{
	AlConnection     * ret = NULL;
	AlSocket         * ret_sock = NULL;
	AlSocketManager  * socket_manager;



	al_return_val_if_fail(connection_manager, NULL);

	socket_manager = al_connection_manager_get_data(connection_manager, AlSocketManager);

	if (!socket_manager->init)
	{

		socket_manager->sock = socket(PF_INET, SOCK_STREAM, 0);
		if (socket_manager->sock == -1)
		{
			syslog(LOG_ERR, "Could not create the socket.\n");
			return NULL;
		}

		/* Give the socket a name. */
		socket_manager->name.sin_family = AF_INET;
		socket_manager->name.sin_port = htons (socket_manager->port);
		socket_manager->name.sin_addr.s_addr = htonl (INADDR_ANY);
		setsockopt(socket_manager->sock, SOL_SOCKET, SO_REUSEADDR, &(socket_manager->on), sizeof(socket_manager->on));
		//ioctl(sock, FIONBIO, &ioctlarg);
		if (bind (socket_manager->sock, (struct sockaddr *) &(socket_manager->name), sizeof (socket_manager->name)) < 0)
		{
			syslog (LOG_ERR, "Could not bind the server. Error %d\n", errno);
			return NULL;
		}

		if (listen (socket_manager->sock, 1) < 0)
		{
			syslog (LOG_ERR, "Could not start the listener.\n");
			return NULL;
		}



		syslog(LOG_NOTICE, "Acceptiong connections on port %d\n", socket_manager->port);

		socket_manager->init = 1;
	}

	ret = al_socket_connection_new(connection_manager);
	ret_sock = al_connection_get_data(ret, AlSocket);

	ret_sock->sock = accept(socket_manager->sock, (struct sockaddr *)&(ret_sock->sn), &(ret_sock->snsize));

	if(ret_sock->sock == -1)
	{
		al_connection_delete(ret);

		syslog(LOG_NOTICE, "Close connections.\n");

		ret = NULL;
	}
	else
	{
		ret_sock->hostname = gethostbyaddr(&(ret_sock->sn.sin_addr), sizeof(ret_sock->sn.sin_addr), AF_INET);
		if (ret_sock->hostname) syslog(LOG_NOTICE, "Accepted new connection from %s.\n", ret_sock->hostname->h_name);
	}
	

	return ret;
}


/**
 */
void al_socket_connection_close(AlConnection * connection)
{
	AlSocket    * socket;


	al_return_if_fail(connection);

	socket = al_connection_get_data(connection, AlSocket);

	if (socket->sock > 0)
	{
		if (socket->hostname && socket->hostname->h_name)
			syslog(LOG_NOTICE, "Close connection with %s.\n", socket->hostname->h_name);
		else
			syslog(LOG_NOTICE, "Close connection with unknown.\n");

		shutdown(socket->sock, SHUT_RDWR);
	}

	socket->sock = 0;
}


/**
 */
AlString * al_socket_connection_read(AlConnection * connection)
{
	AlString    * ret = NULL;
	AlString    * s = NULL;
	AlSocket    * socket;
	const int     len = 3;
	char          buf[len];
	int           rec;


	al_return_val_if_fail(connection, NULL);

	socket = al_connection_get_data(connection, AlSocket);
	s = al_string_new();
	
	while ((rec = recv(socket->sock, buf, sizeof(char) * len, 0)) > 0)
	{

		al_string_append_chars(s, buf, rec);

		if (al_string_last_match_and_remove(s, AL_SOCKET_EOP))
		{
			// End of request packet
			al_log (("Received EOP\n"));
			ret = s;
			break;
		}
		else if (al_string_last_match_and_remove(s, AL_SOCKET_EOF))
		{
			// End of trasmisson
			al_log (("Received EOF\n"));
			al_socket_connection_close(connection);
			ret = NULL;
			break;
		}

	}


	if (ret == NULL)
	{
		al_string_delete(s);
		syslog(LOG_NOTICE, "Close connection from %s.\n", (socket->hostname) ? socket->hostname->h_name : "unknown");
		al_socket_connection_close(connection);
	}

	return ret;
}


/**
 */
void al_socket_connection_send(AlConnection * connection, const AlString * response)
{
	AlSocket    * socket;


	al_return_if_fail(connection && response);

	socket = al_connection_get_data(connection, AlSocket);

	if (send(socket->sock, al_string_get(response), al_string_len(response), 0) == -1)
	{
		syslog(LOG_ERR, "Could not send data %s\n", al_string_get(response));
	}

	if (send(socket->sock, AL_SOCKET_EOP, strlen(AL_SOCKET_EOP), 0) == -1)
	{
		syslog(LOG_ERR, "Could not send data %s\n", AL_SOCKET_EOP);
	}

}


/**
 */
AlConnection * al_socket_connection_manager_connect(AlConnectionManager * connection_manager, const char * server)
{
	AlSocketManager      * socket_manager;
	AlConnection         * ret;
	AlSocket             * ret_sock;
	struct hostent       * pent;



	al_return_val_if_fail(connection_manager && server, NULL);
	
	socket_manager = al_connection_manager_get_data(connection_manager, AlSocketManager);

	ret = al_socket_connection_new(connection_manager);
	ret_sock = al_connection_get_data(ret, AlSocket);
	
	ret_sock->sn.sin_family = AF_INET;

	/* Figure out the internet address. */
	ret_sock->sn.sin_addr.s_addr = inet_addr(server);
	
	if (ret_sock->sn.sin_addr.s_addr == INADDR_NONE)
	{
		pent = gethostbyname(server);
		if (pent)
		{
			ret_sock->sn.sin_addr.s_addr = *((unsigned long *) (pent->h_addr));
		}
		else
		{
			al_log (("Could not resolve the host %s\n", server));
			al_connection_delete(ret);
			return NULL;
		}
	}

	ret_sock->sock = socket(PF_INET, SOCK_STREAM, 0);
	if (ret_sock->sock == -1)
	{
		al_log(("Could not create the socket.\n"));
		al_connection_delete(ret);
		return NULL;
	}

	/* Figure out the port name. */
	ret_sock->sn.sin_port =  htons(socket_manager->port);

	if (connect(ret_sock->sock, (struct sockaddr*)&ret_sock->sn, ret_sock->snsize) == -1)
	{
		al_log (("Could not connect to %s at port %d\n", server, socket_manager->port));
		al_connection_delete(ret);
		return NULL;
	}

	
	return ret;
}
