/* Packet.java
 * Copyright (C) 2007, 2008 Gustav Behm <gbehm (at) kth.se>
 *
 * This file is part of Netzack.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */


package se.kth.netzack;
import se.kth.netzack.*;
import java.net.*;
import java.lang.*;
import java.io.*;
import java.util.*;
import java.nio.*;



public class Packet {
    private Peer from;
    private Message msg;

    private PacketBuffer pbuf;

    Packet() {
	pbuf = Server.getPacketBuffer();
    }


    Packet(Peer f, Message m, Object[] vars) {
	this();


	from = f;
	msg = m;

	pbuf.buffer.putChar(m.get());

	for(int i = 0; i < vars.length; i++) {
	    Object var = vars[i];
	    if(var instanceof Integer) {
		pbuf.buffer.putInt(((Integer)var).intValue());
	    }
	    else if(var instanceof Double) {
		pbuf.buffer.putDouble(((Double)var).doubleValue());
	    }
	    else if(var instanceof String) {
		char[] str = ((String)var).toCharArray();
		for(int j = 0; j < str.length; j++) {
		    pbuf.buffer.putChar(str[j]);
		}
	    }
	    else if(var instanceof Direction) {
		pbuf.buffer.putInt((int)((Direction)var).get());
	    }
	    else if(var instanceof Boolean) {
		if(((Boolean)var).booleanValue())
		    pbuf.buffer.put((byte)1);
		else
		    pbuf.buffer.put((byte)0);
	    }
	    else if(var instanceof Serializable) {
		try{
		    ByteArrayOutputStream baStream = new ByteArrayOutputStream();
		    ObjectOutputStream oStream = new ObjectOutputStream(baStream);

		    oStream.writeObject(var);
		    oStream.flush();

		    pbuf.buffer.put(baStream.toByteArray());
		}
		catch(IOException e) {
		    Netzack.debug(e);
		}
	    }
	    else {
		Netzack.debug("Trying to send unsupported data: " + var.toString());
	    }

	}

	pbuf.buffer.limit(pbuf.buffer.position());
	pbuf.buffer.rewind();
    }



    public Message readMessage() {
	return Message.generate(pbuf.buffer.getChar());
    }

    public int readInt() {
	return pbuf.buffer.getInt();
    }

    public boolean readBoolean() {
	byte b = pbuf.buffer.get();
	if(b == 1)
	    return true;

	return false;
    }

    public double readDouble() {
	return pbuf.buffer.getDouble();
    }

    public String readString() {
       CharBuffer cbuf = pbuf.buffer.asCharBuffer();
       return cbuf.toString();
    }

    public Object readObject() {
	if(pbuf.buffer.remaining() > 0) {
	    try {
		ByteArrayInputStream bStream = new ByteArrayInputStream(pbuf.buffer.array(), pbuf.buffer.position(), pbuf.buffer.remaining());
		ObjectInputStream oStream = new ObjectInputStream(bStream);

		return oStream.readObject();
	    }
	    catch(IOException e) {
		Netzack.debug(e);
	    }
	    catch(ClassNotFoundException e) {
		Netzack.debug(e);
	    }
	}

	return null;
    }


    /**
     * Receive over an UDP connection.
     */
    public DatagramPacket receive(DatagramSocket socket) throws IOException {
	DatagramPacket recv = new DatagramPacket(pbuf.buffer.array(), pbuf.buffer.limit());

	socket.receive(recv);

	pbuf.buffer.rewind();

	msg = readMessage();

	return recv;
    }



    /**
     * Receive over a TCP connection.
     */
    public boolean receive(Peer p, Socket socket) throws IOException {
	InputStream input = socket.getInputStream();

	ByteBuffer bbuf = ByteBuffer.allocate(4);
	int len = 0;
	for(int i = 0; i < 3; i++) {
	    len += input.read(bbuf.array(), len, 4 - len);
	    if(len == -1)
		return false;
	    if(len == 4)
		break;
	}
	if(len != 4)
	    throw new IOException();

	int length = bbuf.getInt();

	if(length <= 0)
	    return false;


	len = 0;
	for(int i = 0; i < 3; i++) {
	    len += input.read(pbuf.buffer.array(), len, length - len);
	    if(len == -1)
		return false;
	    if(len == length)
		break;
	}

	pbuf.buffer.rewind();
	pbuf.buffer.limit(length);

	msg = readMessage();
	from = p;

	return true;
    }

    public void dispatch(Server server) {
	pbuf.buffer.rewind();
	server.dispatch(this);
    }

    public void send(Socket socket) throws IOException {
	OutputStream output = socket.getOutputStream();

	byte[] bytes = pbuf.buffer.array();

	ByteBuffer bbuf = ByteBuffer.allocate(4);
	bbuf.putInt(pbuf.buffer.limit());
	output.write(bbuf.array());

	output.write(bytes, 0, pbuf.buffer.limit());
	output.flush();
    }

    public void send(DatagramSocket socket) throws IOException {
	DatagramPacket packet = new DatagramPacket(pbuf.buffer.array(), pbuf.buffer.limit(), Server.GROUP, Server.PUBLIC_PORT);

	socket.send(packet);
    }

    public void send(DatagramSocket socket, SocketAddress addr) throws IOException {
	DatagramPacket packet = new DatagramPacket(pbuf.buffer.array(), pbuf.buffer.limit(), addr);

	socket.send(packet);
    };







    public void release() {
	pbuf.release();
    }

    public boolean isFrom(Peer f) {
	return (f == from);
    }

    public void from(Peer f) {
	from = f;
    }

    public void message(Message m) {
	msg = m;
    }

    protected void finalize() {
	pbuf.release();
    }


    public Peer getFrom() {
	return from;
    }

    public Message getMsg() {
	return msg;
    }





    static Packet dirPacket(int id, Direction dir, double x, double y, double a) {
	return new Packet(null, Message.DIR, new Object[] {(Object)new Integer(id), dir, (Object)new Double(x), (Object)new Double(y), (Object)new Double(a)});
    }

    static Packet posPacket(int id, int x, int y, double a, int rgb, int score, String name) {
	return new Packet(null, Message.POS, new Object[] {(Object)new Integer(id), (Object)new Integer(x), (Object)new Integer(y), (Object)new Double(a), (Object)new Integer(rgb), (Object)new Integer(score), (String)name});
    }


    static Packet echoPacket() {
	return new Packet(null, Message.ECHO, new Object[] {});
    }
    static Packet echoPacket(int id, boolean replyWithAddress) {
	return new Packet(null, Message.ECHO, new Object[] {(Object)new Integer(id), (Object)new Boolean(replyWithAddress)});
    }
    static Packet echoPacket(int port, byte[] addr) {
	return new Packet(null, Message.ECHO, new Object[] {(Object)new Integer(port), addr});
    }


    static Packet helloPacket(int id) {
	return new Packet(null, Message.HELLO, new Object[] {(Object)new Integer(id)});
    }

    static Packet reconnectPacket(int id, int to, int port, byte[] addr) {
	return new Packet(null, Message.RECONNECT, new Object[] {(Object)new Integer(id), (Object)new Integer(to), (Object)new Integer(port), (Object)addr});
    }

    static Packet reconnectPacket(int id, int to) {
	return new Packet(null, Message.RECONNECT, new Object[] {(Object)new Integer(id), (Object)new Integer(to)});
    }

    static Packet idPacket(int newId, int id) {
	return new Packet(null, Message.ID, new Object[] {(Object)new Integer(newId), (Object)new Integer(id)});
    }

    static Packet roundPacket(int id, boolean start) {
	return new Packet(null, Message.ROUND, new Object[] {(Object)new Integer(id), (Object)new Boolean(start)});
    }

    static Packet loopPacket() {
	return new Packet(null, Message.LOOP, new Object[] {});
    }
}
