/* $Id: basicxdr.c,v 1.8 2004/12/22 23:15:03 ali Exp $
 * Copyright (C) 2001, 2002, 2003  Slash'EM Development Team
 * Copyright (C) 2004  J. Ali Harlow
 *
 * This file is part of NetHack Proxy.
 *
 * NetHack Proxy is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of the
 * License, or (at your option) any later version.
 *
 * NetHack Proxy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with NetHack Proxy; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307   
 * USA
 *
 * Alternatively (at your option) you may instead choose to redistribute
 * and/or modify NetHack Proxy under the terms of the NetHack General
 * Public License.
 *
 * You should have receieved a copy of the NetHack General Public License
 * along with NetHack Proxy; if not, download a copy from
 * http://www.nethack.org/common/license.html
 */

#include "config.h"
#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
extern char *malloc(), *calloc(), *realloc();
#endif
#include "compat.h"
#include <nhproxy/system.h>
#include <nhproxy/xdr.h>

/*
 * This module implements the NhExt version of XDR according to RFC 1014.
 * It is backwards compatible with the code produced by rpcgen, and can
 * be replaced by that for testing purposes on systems which support it.
 * NetHack Proxy uses the extra facilities implemented here and is not
 * therefore itself compatible with rpcgen.
 * Rpcgen is included in the freely available RPCSRC 4.0 from Sun
 * Microsystems, which can be downloaded in 17 shar files from
 * ftp://bcm.tmc.edu/nfs/
 */

/*
 * Note: Although this routine is called _long and takes a pointer to a
 * long datum, it always reads and writes 32-bit values, regardless of
 * the size of the long type. This might cause problems on a platform
 * where long is smaller than 32 bits (but then NetHack doesn't support
 * such platforms anyway). On platforms with longs larger than 32 bits,
 * NetHack only needs the least significant 32 bits, and this function
 * will correctly encode and decode this quantity.
 */

nhproxy_bool_t
nhproxy_xdr_long(xdrs, datum)
NhProxyXdr *xdrs;
long *datum;
{
    int retval = TRUE;
    unsigned char buf[4];
    if (xdrs->x_op == NHPROXY_XDR_ENCODE) {
	buf[0] = *datum >> 24 & 0xff;
	buf[1] = *datum >> 16 & 0xff;
	buf[2] = *datum >> 8 & 0xff;
	buf[3] = *datum & 0xff;
	retval = xdrs->x_write(xdrs, buf, 4);
    } else if (xdrs->x_op == NHPROXY_XDR_DECODE) {
	retval = xdrs->x_read(xdrs, buf, 4);
	*datum =
	  (long)buf[0] << 24 | (long)buf[1] << 16 | (long)buf[2] << 8 | buf[3];
    } else if (xdrs->x_op == NHPROXY_XDR_COUNT)
	xdrs->x_pos += 4;
    return retval;
}

nhproxy_bool_t
nhproxy_xdr_u_long(xdrs, datum)
NhProxyXdr *xdrs;
unsigned long *datum;
{
    int retval;
    long l;
    l = *datum;
    retval = nhproxy_xdr_long(xdrs, &l);
    *datum = (unsigned long)l;
    return retval;
}

nhproxy_bool_t
nhproxy_xdr_short(xdrs, datum)
NhProxyXdr *xdrs;
short *datum;
{
    int retval;
    long l;
    l = *datum;
    retval = nhproxy_xdr_long(xdrs, &l);
    *datum = (short)l;
    return retval;
}

nhproxy_bool_t
nhproxy_xdr_u_short(xdrs, datum)
NhProxyXdr *xdrs;
unsigned short *datum;
{
    int retval;
    long l;
    l = *datum;
    retval = nhproxy_xdr_long(xdrs, &l);
    *datum = (unsigned short)l;
    return retval;
}

nhproxy_bool_t
nhproxy_xdr_int(xdrs, datum)
NhProxyXdr *xdrs;
int *datum;
{
    int retval;
    long l;
    l = *datum;
    retval = nhproxy_xdr_long(xdrs, &l);
    *datum = (int)l;
    return retval;
}

nhproxy_bool_t
nhproxy_xdr_u_int(xdrs, datum)
NhProxyXdr *xdrs;
unsigned int *datum;
{
    int retval;
    long l;
    l = *datum;
    retval = nhproxy_xdr_long(xdrs, &l);
    *datum = (unsigned int)l;
    return retval;
}

nhproxy_bool_t
nhproxy_xdr_char(xdrs, datum)
NhProxyXdr *xdrs;
char *datum;
{
    int retval;
    long l;
    l = *datum;
    retval = nhproxy_xdr_long(xdrs, &l);
    *datum = (char)l;
    return retval;
}

nhproxy_bool_t
nhproxy_xdr_u_char(xdrs, datum)
NhProxyXdr *xdrs;
unsigned char *datum;
{
    int retval;
    long l;
    l = *datum;
    retval = nhproxy_xdr_long(xdrs, &l);
    *datum = (unsigned char)l;
    return retval;
}

nhproxy_bool_t
nhproxy_xdr_enum(xdrs, datum)
NhProxyXdr *xdrs;
int *datum;
{
    int retval;
    long l;
    l = *datum;
    retval = nhproxy_xdr_long(xdrs, &l);
    *datum = (int)l;
    return retval;
}

nhproxy_bool_t
nhproxy_xdr_bool(xdrs, datum)
NhProxyXdr *xdrs;
nhproxy_bool_t *datum;
{
    int retval;
    long l;
    l = !!*datum;
    retval = nhproxy_xdr_long(xdrs, &l);
    *datum = (nhproxy_bool_t)l;
    return retval;
}

/*
 * A common routine for nhproxy_xdr_string() and nhproxy_xdr_bytes()
 */

static nhproxy_bool_t
nhproxy_xdr_bytestring(xdrs, datum, len, maxlen, ext)
NhProxyXdr *xdrs;
char **datum;
unsigned int *len;
unsigned int maxlen, ext;
{
    int retval;
    long zero = 0;
    if (xdrs->x_op == NHPROXY_XDR_ENCODE) {
	if (*len > maxlen)
	    return FALSE;
	retval = nhproxy_xdr_u_int(xdrs, len);
	retval &= xdrs->x_write(xdrs, *datum, *len);
	if (*len & 3)
	    retval &= xdrs->x_write(xdrs, &zero, 4 - (*len & 3));
    } else if (xdrs->x_op == NHPROXY_XDR_DECODE) {
	retval = nhproxy_xdr_u_int(xdrs, len);
	if (retval) {
	    if (*len > maxlen)
		return FALSE;
	    if (!*datum) {
		*datum = (char *)malloc(MAX(NHPROXY_XDR_RNDUP(*len),
		  *len + ext));
		if (!*datum)
		    return FALSE;
	    }
	    retval &= xdrs->x_read(xdrs, *datum, NHPROXY_XDR_RNDUP(*len));
	}
    } else if (xdrs->x_op == NHPROXY_XDR_FREE) {
	free(*datum);
	*datum = NULL;
	retval = TRUE;
    } else if (xdrs->x_op == NHPROXY_XDR_COUNT) {
	xdrs->x_pos += 4 + NHPROXY_XDR_RNDUP(*len);
	retval = TRUE;
    }
    return retval;
}

nhproxy_bool_t
nhproxy_xdr_string(xdrs, datum, maxsize)
NhProxyXdr *xdrs;
char **datum;
unsigned int maxsize;
{
    int retval;
    unsigned int slen;
    if (xdrs->x_op == NHPROXY_XDR_ENCODE || xdrs->x_op == NHPROXY_XDR_COUNT)
	slen = strlen(*datum);
    retval = nhproxy_xdr_bytestring(xdrs, datum, &slen, maxsize, 1);
    if (retval && xdrs->x_op == NHPROXY_XDR_DECODE)
	(*datum)[slen] = '\0';
    return retval;
}

nhproxy_bool_t
nhproxy_xdr_bytes(xdrs, datum, len, maxlen)
NhProxyXdr *xdrs;
char **datum;
unsigned int *len;
unsigned int maxlen;
{
    return nhproxy_xdr_bytestring(xdrs, datum, len, maxlen, 0);
}

nhproxy_bool_t
nhproxy_xdr_wrapstring(xdrs, datum)
NhProxyXdr *xdrs;
char **datum;
{
    return nhproxy_xdr_string(xdrs, datum, (unsigned int)-1);
}

nhproxy_bool_t
nhproxy_xdr_vector(xdrs, addr, len, size, codec)
NhProxyXdr *xdrs;
char *addr;
unsigned int len;
unsigned int size;
nhproxy_bool_t (*codec)(NhProxyXdr *, nhproxy_genericptr_t);
{
    for(; len > 0; len--) {
	if (!(*codec)(xdrs, addr))
	    return FALSE;
	addr += size;
    }
    return TRUE;
}

nhproxy_bool_t
nhproxy_xdr_array(xdrs, addr, len, maxlen, size, codec)
NhProxyXdr *xdrs;
char **addr;
unsigned int *len;
unsigned int maxlen;
unsigned int size;
nhproxy_bool_t (*codec)(NhProxyXdr *, nhproxy_genericptr_t);
{
    int retval;
    long slen;
    if (!nhproxy_xdr_u_int(xdrs, len))
	return FALSE;
    slen = *len;
    if (xdrs->x_op != NHPROXY_XDR_FREE && slen > maxlen)
	return FALSE;
    if (xdrs->x_op == NHPROXY_XDR_DECODE && !*addr && slen) {
	*addr = (char *)calloc(slen, size);
	if (!*addr)
	    return FALSE;
    }
    retval = nhproxy_xdr_vector(xdrs, *addr, slen, size, codec);
    if (xdrs->x_op == NHPROXY_XDR_FREE) {
	free(*addr);
	*addr = NULL;
    }
    return retval;
}
