/* $Id: clientcb.c,v 1.19 2004/12/22 23:15:03 ali Exp $
 * Copyright (C) 2002, 2003  Slash'EM Development Team
 * Copyright (C) 2004  J. Ali Harlow
 *
 * This file is part of NetHack Proxy.
 *
 * NetHack Proxy is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of the
 * License, or (at your option) any later version.
 *
 * NetHack Proxy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with NetHack Proxy; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307   
 * USA
 *
 * Alternatively (at your option) you may instead choose to redistribute
 * and/or modify NetHack Proxy under the terms of the NetHack General
 * Public License.
 *
 * You should have receieved a copy of the NetHack General Public License
 * along with NetHack Proxy; if not, download a copy from
 * http://www.nethack.org/common/license.html
 */

#include "config.h"
#include "compat.h"
#include <nhproxy/system.h>
#include <nhproxy/xdr.h>
#include <nhproxy/common.h>
#include <nhproxy/clientcb.h>

#ifndef SIZE
#define SIZE(array)	(sizeof(array) / sizeof(*(array)))
#endif

extern NhProxyIO *nhproxy_clnt_log;

void
nhproxy_cb_display_inventory()
{
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] display_inventory()\n", serial);
    }
    (void)nhproxy_rpc(NHPROXY_EXT_CID_DISPLAY_INVENTORY, 0, 0);
}

int
nhproxy_cb_dlbh_fopen(name, mode)
const char *name, *mode;
{
    int retval;
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] dlbh_fopen(%#s, %#s)\n", serial, name, mode);
    }
    if (!nhproxy_rpc(NHPROXY_EXT_CID_DLBH_FOPEN, 2, NHPROXY_STRING(name),
      NHPROXY_STRING(mode), 1, NHPROXY_INT_PTR(retval)))
	retval = -1;
    if (nhproxy_clnt_log)
	nhproxy_io_printf(nhproxy_clnt_log, "rpc [%u] dlbh_fopen = %d\n",
	  serial, retval);
    return retval;
}

char *
nhproxy_cb_dlbh_fgets(buf, len, fh)
char *buf;
int len, fh;
{
    char *retval, *line = (char *)0;
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] dlbh_fgets(%d, %d)\n", serial, len, fh);
    }
    if (!nhproxy_rpc(NHPROXY_EXT_CID_DLBH_FGETS, 2, NHPROXY_INT(len),
      NHPROXY_INT(fh), 1, NHPROXY_STRING_PTR(line))) {
	free(line);
	return (char *)0;
    }
    if (*line) {
	strncpy(buf, line, len - 1);
	buf[len - 1] = '\0';
	retval = buf;
    }
    else
	retval = (char *)0;
    free(line);
    if (nhproxy_clnt_log) {
	if (retval)
	    nhproxy_io_printf(nhproxy_clnt_log, "rpc [%u] dlbh_fgets = %#s\n",
	      serial, retval);
	else
	    nhproxy_io_printf(nhproxy_clnt_log, "rpc [%u] dlbh_fgets = NULL\n",
	      serial);
    }
    return retval;
}

int
nhproxy_cb_dlbh_fread(buf, size, no, fh)
char *buf;
int size, no, fh;
{
    int retval, offset = 0, nb;
    char *buffer = (char *)0;
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] dlbh_fread(%p, %d, %d, %d)\n", serial, buf, size, no, fh);
    }
    if (!nhproxy_rpc(NHPROXY_EXT_CID_DLBH_FREAD, 2, NHPROXY_INT(size * no),
      NHPROXY_INT(fh), 2, NHPROXY_INT_PTR(retval),
      NHPROXY_BYTES_PTR(buffer, nb))) {
	free(buffer);
	return -1;
    }
    if (nb)
	memcpy(buf, buffer, nb);
    free(buffer);
    while (!retval && nb % size) {
	/*
	 * The dblh_fread() callback has no concept of item sizes
	 * (it acts more like read() in this respect than fread()).
	 * If we get a sucessful read which is short then we shouldn't
	 * discard the remainder unless we have reached end of file.
	 */
	offset += nb;
	if (!nhproxy_rpc(NHPROXY_EXT_CID_DLBH_FREAD,
	  2, NHPROXY_INT(size * no - offset), NHPROXY_INT(fh),
	  2, NHPROXY_INT_PTR(retval), NHPROXY_BYTES_PTR(buffer, nb)))
	    nb = 0;
	if (nb)
	    memcpy(buf + offset, buffer, nb);
	free(buffer);
    }
    retval = retval ? -1 : (offset + nb) / size;
    if (nhproxy_clnt_log)
	nhproxy_io_printf(nhproxy_clnt_log, "rpc [%u] dlbh_fread = %d\n",
	  serial, retval);
    return retval;
}

int
nhproxy_cb_dlbh_fwrite(buf, size, no, fh)
char *buf;
int size, no, fh;
{
    int retval;
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] dlbh_fwrite(%p, %d, %d, %d)\n", serial, buf, size, no, fh);
    }
    if (!nhproxy_rpc(NHPROXY_EXT_CID_DLBH_FWRITE,
      2, NHPROXY_INT(fh), NHPROXY_BYTES(buf, size * no),
      1, NHPROXY_INT_PTR(retval)))
	retval = -1;
    if (nhproxy_clnt_log)
	nhproxy_io_printf(nhproxy_clnt_log, "rpc [%u] dlbh_fwrite = %d\n",
	  serial, retval ? -1 : no);
    return retval ? -1 : no;
}

int
nhproxy_cb_dlbh_fclose(fh)
int fh;
{
    int retval;
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] dlbh_fclose(%d)\n", serial, fh);
    }
    if (!nhproxy_rpc(NHPROXY_EXT_CID_DLBH_FCLOSE, 1, NHPROXY_INT(fh),
      1, NHPROXY_INT_PTR(retval)))
	retval = -1;
    if (nhproxy_clnt_log)
	nhproxy_io_printf(nhproxy_clnt_log, "rpc [%u] dlbh_fclose = %d\n",
	  serial, retval);
    return retval;
}

char *
nhproxy_cb_dlbh_fmd5sum(name)
const char *name;
{
    int retval;
    char *digest = (char *)0;
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] dlbh_fmd5sum(%#s)\n", serial, name);
    }
    if (!nhproxy_rpc(NHPROXY_EXT_CID_DLBH_FMD5SUM, 1, NHPROXY_STRING(name),
      2, NHPROXY_INT_PTR(retval), NHPROXY_STRING_PTR(digest)))
	retval = -1;
    if (!retval) {
	if (nhproxy_clnt_log)
	    nhproxy_io_printf(nhproxy_clnt_log, "rpc [%u] dlbh_fmd5sum = %#s\n",
	      serial, digest);
	return digest;
    } else {
	free(digest);
	if (nhproxy_clnt_log)
	    nhproxy_io_printf(nhproxy_clnt_log,
	      "rpc [%u] dlbh_fmd5sum = NULL\n", serial);
	return (char *)0;
    }
}

void
nhproxy_cb_flush_screen()
{
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] flush_screen()\n", serial);
    }
    (void)nhproxy_rpc(NHPROXY_EXT_CID_FLUSH_SCREEN, 0, 0);
}

void
nhproxy_cb_doredraw()
{
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] doredraw()\n", serial);
    }
    (void)nhproxy_rpc(NHPROXY_EXT_CID_DOREDRAW, 0, 0);
}

void
nhproxy_cb_interface_mode(mode)
unsigned long mode;
{
    unsigned short serial;
    if (nhproxy_clnt_log) {
	unsigned long m = mode;
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] interface_mode(", serial);
	if (m & NHPROXY_EXT_IM_STATUS) {
	    nhproxy_io_printf(nhproxy_clnt_log, "EXT_IM_STATUS");
	    m &= ~NHPROXY_EXT_IM_STATUS;
	}
	if (m & NHPROXY_EXT_IM_DISPLAY_LAYERS) {
	    if (m != mode)
		nhproxy_io_printf(nhproxy_clnt_log, " | ");
	    nhproxy_io_printf(nhproxy_clnt_log, "EXT_IM_DISPLAY_LAYERS");
	    m &= ~NHPROXY_EXT_IM_DISPLAY_LAYERS;
	}
	if (m || m == mode) {
	    if (m != mode)
		nhproxy_io_printf(nhproxy_clnt_log, " | ");
	    nhproxy_io_printf(nhproxy_clnt_log, "%lu", m);
	}
	nhproxy_io_printf(nhproxy_clnt_log, ")\n");
    }
    (void)nhproxy_rpc(NHPROXY_EXT_CID_INTERFACE_MODE, 1, NHPROXY_LONG(mode), 0);
}

int
nhproxy_cb_parse_options(opts)
char *opts;
{
    int retval;
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] parse_options(%#s)\n", serial, opts);
    }
    if (!nhproxy_rpc(NHPROXY_EXT_CID_PARSE_OPTIONS,
      1, NHPROXY_STRING(opts), 1, NHPROXY_INT_PTR(retval)))
	retval = -1;
    if (nhproxy_clnt_log)
	nhproxy_io_printf(nhproxy_clnt_log, "rpc [%u] parse_options = %d\n",
	  serial, retval);
    return retval;
}

char *
nhproxy_cb_get_option(opt)
char *opt;
{
    char *retval = (char *)0;
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] get_option(%#s)\n", serial, opt);
    }
    if (!nhproxy_rpc(NHPROXY_EXT_CID_GET_OPTION,
      1, NHPROXY_STRING(opt), 1, NHPROXY_STRING_PTR(retval))) {
	free(retval);
	if (nhproxy_clnt_log)
	    nhproxy_io_printf(nhproxy_clnt_log, "rpc [%u] get_option = NULL\n",
	      serial);
	return (char *)0;
    }
    if (nhproxy_clnt_log)
	nhproxy_io_printf(nhproxy_clnt_log, "rpc [%u] get_option = %#s\n",
	  serial, retval);
    return retval;
}

struct nhproxy_cb_get_player_choices_res *
nhproxy_cb_get_player_choices()
{
    struct nhproxy_cb_get_player_choices_res *retval;
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] get_player_choices()\n", serial);
    }
    retval=(struct nhproxy_cb_get_player_choices_res *)malloc(sizeof(*retval));
    if (!retval)
	return (struct nhproxy_cb_get_player_choices_res *)0;
    MEMZERO(retval, sizeof(*retval));
    if (!nhproxy_rpc(NHPROXY_EXT_CID_GET_PLAYER_CHOICES, 0, 1,
      NHPROXY_XDRF(nhproxy_cb_xdr_get_player_choices_res, retval))) {
	free(retval);
	if (nhproxy_clnt_log)
	    nhproxy_io_printf(nhproxy_clnt_log,
	      "rpc [%u] get_player_choices = NULL\n", serial);
	return (struct nhproxy_cb_get_player_choices_res *)0;
    }
    if (nhproxy_clnt_log) {
	int i;
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] get_player_choices = <", serial);
	for(i = 0; i < retval->n_aligns; i++) {
	    if (i)
		nhproxy_io_printf(nhproxy_clnt_log, ", ");
	    nhproxy_io_printf(nhproxy_clnt_log, "%#s", retval->aligns[i]);
	}
	nhproxy_io_printf(nhproxy_clnt_log, ">, <");
	for(i = 0; i < retval->n_genders; i++) {
	    if (i)
		nhproxy_io_printf(nhproxy_clnt_log, ", ");
	    nhproxy_io_printf(nhproxy_clnt_log, "%#s", retval->genders[i]);
	}
	nhproxy_io_printf(nhproxy_clnt_log, ">, <");
	for(i = 0; i < retval->n_races; i++) {
	    if (i)
		nhproxy_io_printf(nhproxy_clnt_log, ", ");
	    nhproxy_io_printf(nhproxy_clnt_log, "%#s", retval->races[i]);
	}
	nhproxy_io_printf(nhproxy_clnt_log, ">, <");
	for(i = 0; i < retval->n_roles; i++) {
	    if (i)
		nhproxy_io_printf(nhproxy_clnt_log, ", ");
	    nhproxy_io_printf(nhproxy_clnt_log, "{%#s, %#s}",
	      retval->roles[i].male, retval->roles[i].female);
	}
	nhproxy_io_printf(nhproxy_clnt_log, ">\n");
    }
    return retval;
}

void
nhproxy_cb_free_player_choices(choices)
struct nhproxy_cb_get_player_choices_res *choices;
{
    nhproxy_xdr_free(nhproxy_cb_xdr_get_player_choices_res, (char *)choices);
    free(choices);
}

struct nhproxy_cb_get_valid_selections_res *
nhproxy_cb_get_valid_selections()
{
    struct nhproxy_cb_get_valid_selections_res *retval;
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] get_valid_selections()\n", serial);
    }
    retval=(struct nhproxy_cb_get_valid_selections_res *)
      malloc(sizeof(*retval));
    if (!retval)
	return (struct nhproxy_cb_get_valid_selections_res *)0;
    MEMZERO(retval, sizeof(*retval));
    if (!nhproxy_rpc(NHPROXY_EXT_CID_GET_VALID_SELECTIONS, 0, 1,
      NHPROXY_XDRF(nhproxy_cb_xdr_get_valid_selections_res, retval))) {
	free(retval);
	if (nhproxy_clnt_log)
	    nhproxy_io_printf(nhproxy_clnt_log,
	      "rpc [%u] get_valid_selections = NULL\n", serial);
	return (struct nhproxy_cb_get_valid_selections_res *)0;
    }
    if (nhproxy_clnt_log) {
	int i;
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] get_valid_selections = %d, %d, %d, %d, <", serial,
	  retval->no_roles, retval->no_races, retval->no_aligns,
	  retval->no_genders);
	for(i = 0; i < retval->n_masks; i++) {
	    if (i)
		nhproxy_io_printf(nhproxy_clnt_log, ", ");
	    nhproxy_io_printf(nhproxy_clnt_log, "0x%08lX", retval->masks[i]);
	}
	nhproxy_io_printf(nhproxy_clnt_log, ">\n");
    }
    return retval;
}

void
nhproxy_cb_free_valid_selections(vs)
struct nhproxy_cb_get_valid_selections_res *vs;
{
    nhproxy_xdr_free(nhproxy_cb_xdr_get_valid_selections_res, (char *)vs);
    free(vs);
}

static struct nhproxy_cb_get_valid_selections_res *valid_selections;

void
nhproxy_cb_valid_selection_open()
{
    if (!valid_selections)
	valid_selections = nhproxy_cb_get_valid_selections();
}

/*
 * If s is negative, then iterate i over 0 <= i < n, otherwise do just one
 * iteration at i == s. Early exit as soon as we find a valid combination.
 */

#define ITERATE(i, s, n) for((i) = (s) >= 0 ? (s) : 0; \
			    !valid && ((s) >= 0 ? (i) == (s) : (i) < (n)); \
			    (i)++)

int
nhproxy_cb_valid_selection_check(role, race, gend, align)
int role, race, gend, align;
{
    int valid = 0;
    int pack;			/* No. masks packed in each element */
    int i, k, n;
    int irole, irace, igend, ialign;
    pack = 32 / valid_selections->no_genders;
    ITERATE(irole, role, valid_selections->no_roles)
	ITERATE(irace, race, valid_selections->no_races)
	    ITERATE(ialign, align, valid_selections->no_aligns) {
		n = (irole * valid_selections->no_races + irace) *
		  valid_selections->no_aligns + ialign;
		i = n / pack;
		k = (n % pack) * valid_selections->no_genders;
		ITERATE(igend, gend, valid_selections->no_genders)
		    if (valid_selections->masks[i] & 1L << k + igend)
			valid = 1;
	    }
    return valid;
}

#undef ITERATE

void
nhproxy_cb_valid_selection_close()
{
    if (valid_selections) {
	nhproxy_cb_free_valid_selections(valid_selections);
	valid_selections = (struct nhproxy_cb_get_valid_selections_res *)0;
    }
}

void
nhproxy_cb_quit_game()
{
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] quit_game()\n", serial);
    }
    (void)nhproxy_rpc(NHPROXY_EXT_CID_QUIT_GAME, 0, 0);
}

void
nhproxy_cb_display_score()
{
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] display_score()\n", serial);
    }
    (void)nhproxy_rpc(NHPROXY_EXT_CID_DISPLAY_SCORE, 0, 0);
}

void
nhproxy_cb_doset()
{
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] doset()\n", serial);
    }
    (void)nhproxy_rpc(NHPROXY_EXT_CID_DOSET, 0, 0);
}

struct nhproxy_cb_get_extended_commands_res *
nhproxy_cb_get_extended_commands()
{
    struct nhproxy_cb_get_extended_commands_res *retval;
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] get_extended_commands()\n", serial);
    }
    retval =
      (struct nhproxy_cb_get_extended_commands_res *)malloc(sizeof(*retval));
    if (!retval)
	return (struct nhproxy_cb_get_extended_commands_res *)0;
    MEMZERO(retval, sizeof(*retval));
    if (!nhproxy_rpc(NHPROXY_EXT_CID_GET_EXTENDED_COMMANDS, 0, 1,
      NHPROXY_XDRF(nhproxy_cb_xdr_get_extended_commands_res, retval))) {
	free(retval);
	if (nhproxy_clnt_log)
	    nhproxy_io_printf(nhproxy_clnt_log,
	      "rpc [%u] get_extended_commands = NULL\n", serial);
	return (struct nhproxy_cb_get_extended_commands_res *)0;
    }
    if (nhproxy_clnt_log) {
	int i;
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] get_extended_commands = <", serial);
	for(i = 0; i < retval->n_commands; i++) {
	    if (i)
		nhproxy_io_printf(nhproxy_clnt_log, ", ");
	    nhproxy_io_printf(nhproxy_clnt_log, "%#s", retval->commands[i]);
	}
	nhproxy_io_printf(nhproxy_clnt_log, ">\n");
    }
    return retval;
}

void
nhproxy_cb_free_extended_commands(commands)
struct nhproxy_cb_get_extended_commands_res *commands;
{
    nhproxy_xdr_free(nhproxy_cb_xdr_get_extended_commands_res,
      (char *)commands);
    free(commands);
}

int
nhproxy_cb_map_menu_cmd(ch)
int ch;
{
    int retval;
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] map_menu_cmd(%d)\n", serial, ch);
    }
    if (!nhproxy_rpc(NHPROXY_EXT_CID_MAP_MENU_CMD, 1, NHPROXY_INT(ch), 1,
      NHPROXY_INT_PTR(retval)))
	retval = ch;
    if (nhproxy_clnt_log)
	nhproxy_io_printf(nhproxy_clnt_log, "rpc [%u] map_menu_cmd = %d\n",
	  serial, retval);
    return retval;
}

int
nhproxy_cb_get_standard_winid(window)
char *window;
{
    int retval;
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] get_standard_winid(%#s)\n", serial, window);
    }
    if (!nhproxy_rpc(NHPROXY_EXT_CID_GET_STANDARD_WINID,
      1, NHPROXY_STRING(window), 1, NHPROXY_INT_PTR(retval)))
	retval = -1;
    if (nhproxy_clnt_log)
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] get_standard_winid = %d\n", serial, retval);
    return retval;
}

struct nhproxy_cb_get_tilesets_res *
nhproxy_cb_get_tilesets()
{
    struct nhproxy_cb_get_tilesets_res *retval;
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] get_tilesets()\n", serial);
    }
    retval=(struct nhproxy_cb_get_tilesets_res *)malloc(sizeof(*retval));
    if (!retval)
	return (struct nhproxy_cb_get_tilesets_res *)0;
    MEMZERO(retval, sizeof(*retval));
    if (!nhproxy_rpc(NHPROXY_EXT_CID_GET_TILESETS, 0, 1,
      NHPROXY_XDRF(nhproxy_cb_xdr_get_tilesets_res, retval))) {
	free(retval);
	if (nhproxy_clnt_log)
	    nhproxy_io_printf(nhproxy_clnt_log,
	      "rpc [%u] get_tilesets = NULL\n", serial);
	return (struct nhproxy_cb_get_tilesets_res *)0;
    }
    if (nhproxy_clnt_log) {
	int i;
	nhproxy_io_printf(nhproxy_clnt_log, "rpc [%u] get_tilesets = <",
	  serial);
	for(i = 0; i < retval->n_tilesets; i++) {
	    if (i)
		nhproxy_io_printf(nhproxy_clnt_log, ", ");
	    nhproxy_io_printf(nhproxy_clnt_log, "{%#s,%#s,%#s,%ld}",
	      retval->tilesets[i].name, retval->tilesets[i].file,
	      retval->tilesets[i].mapfile, retval->tilesets[i].flags);
	}
	nhproxy_io_printf(nhproxy_clnt_log, ">\n");
    }
    return retval;
}

void
nhproxy_cb_free_tilesets(tilesets)
struct nhproxy_cb_get_tilesets_res *tilesets;
{
    nhproxy_xdr_free(nhproxy_cb_xdr_get_tilesets_res, (char *)tilesets);
    free(tilesets);
}

struct nhproxy_cb_get_glyph_mapping_res *
nhproxy_cb_get_glyph_mapping()
{
    struct nhproxy_cb_get_glyph_mapping_res *retval;
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] get_glyph_mapping()\n", serial);
    }
    retval=(struct nhproxy_cb_get_glyph_mapping_res *)malloc(sizeof(*retval));
    if (!retval)
	retval = (struct nhproxy_cb_get_glyph_mapping_res *)0;
    MEMZERO(retval, sizeof(*retval));
    if (!nhproxy_rpc(NHPROXY_EXT_CID_GET_GLYPH_MAPPING, 0, 1,
      NHPROXY_XDRF(nhproxy_cb_xdr_get_glyph_mapping_res, retval))) {
	free(retval);
	if (nhproxy_clnt_log)
	    nhproxy_io_printf(nhproxy_clnt_log,
	      "rpc [%u] get_glpyh_mapping = NULL\n", serial);
	retval = (struct nhproxy_cb_get_glyph_mapping_res *)0;
    }
    if (nhproxy_clnt_log) {
	int i, j, k;
	struct nhproxy_cb_get_glyph_mapping_res_submapping *sm;
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] get_glpyh_mapping = %d, 0x%08lX, <\n  ",
	  serial, retval->no_glyph, retval->transparent);
	for(i = 0; i < retval->n_mappings; i++) {
	    if (i)
		nhproxy_io_printf(nhproxy_clnt_log, ",\n  ");
	    nhproxy_io_printf(nhproxy_clnt_log,
	      "{%#s, %d, %d, {0x%08lX, %#s}, <",
	      retval->mappings[i].flags, retval->mappings[i].base_mapping,
	      retval->mappings[i].alt_glyph, retval->mappings[i].symdef.rgbsym,
	      retval->mappings[i].symdef.description);
	    for(j = 0; j < retval->mappings[i].n_submappings; j++) {
		if (j)
		    nhproxy_io_fputc(',', nhproxy_clnt_log);
		nhproxy_io_printf(nhproxy_clnt_log, "\n    ");
		sm = retval->mappings[i].submappings + j;
		nhproxy_io_printf(nhproxy_clnt_log, "{{0x%08lX, %#s}, <",
		  sm->symdef.rgbsym, sm->symdef.description);
		for(k = 0; k < sm->n_glyphs; k++) {
		    if (k)
			nhproxy_io_printf(nhproxy_clnt_log, ", ");
		    nhproxy_io_printf(nhproxy_clnt_log, "{0x%08lX, %#s}",
		      sm->glyphs[k].rgbsym, sm->glyphs[k].description);
		}
		nhproxy_io_printf(nhproxy_clnt_log, ">}");
	    }
	    nhproxy_io_printf(nhproxy_clnt_log, "\n    >}");
	}
	nhproxy_io_printf(nhproxy_clnt_log, "\n  >\n");
    }
    return retval;
}

void
nhproxy_cb_free_glyph_mapping(mapping)
struct nhproxy_cb_get_glyph_mapping_res *mapping;
{
    nhproxy_xdr_free(nhproxy_cb_xdr_get_glyph_mapping_res, (char *)mapping);
    free(mapping);
}

struct nhproxy_cb_get_extensions_res *
nhproxy_cb_get_extensions()
{
    struct nhproxy_cb_get_extensions_res *retval;
    unsigned short serial;
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] get_extentions()\n", serial);
    }
    retval=(struct nhproxy_cb_get_extensions_res *)malloc(sizeof(*retval));
    if (!retval)
	return (struct nhproxy_cb_get_extensions_res *)0;
    MEMZERO(retval, sizeof(*retval));
    if (!nhproxy_rpc(NHPROXY_EXT_CID_GET_EXTENSIONS, 0, 1,
      NHPROXY_XDRF(nhproxy_cb_xdr_get_extensions_res, retval))) {
	free(retval);
	if (nhproxy_clnt_log)
	    nhproxy_io_printf(nhproxy_clnt_log,
	      "rpc [%u] get_extentions = NULL\n", serial);
	return (struct nhproxy_cb_get_extensions_res *)0;
    }
    if (nhproxy_clnt_log) {
	int i;
	nhproxy_io_printf(nhproxy_clnt_log,
	  "rpc [%u] get_extentions = <\n", serial);
	for(i = 0; i < retval->n_extensions; i++) {
	    if (i)
		nhproxy_io_printf(nhproxy_clnt_log, ", ");
	    nhproxy_io_printf(nhproxy_clnt_log, "{%#s, %#s, %d}",
	      retval->extensions[i].name, retval->extensions[i].version,
	      retval->extensions[i].no_procedures);
	}
	nhproxy_io_printf(nhproxy_clnt_log, ">\n");
    }
    return retval;
}

void
nhproxy_cb_free_extensions(extensions)
struct nhproxy_cb_get_extensions_res *extensions;
{
    nhproxy_xdr_free(nhproxy_cb_xdr_get_extensions_res, (char *)extensions);
    free(extensions);
}

void
nhproxy_cb_set_option_mod_status(optnam, status)
const char *optnam;
{
    unsigned short serial;
    static const char *option_mod_flags[] = { "SET_IN_FILE", "SET_VIA_PROG",
      "DISP_IN_GAME", "SET_IN_GAME" };
    if (nhproxy_clnt_log) {
	serial = nhproxy_rpc_get_next_serial();
	if (status >= 0 && status < SIZE(option_mod_flags))
	    nhproxy_io_printf(nhproxy_clnt_log,
	      "rpc [%u] set_option_mod_status(%#s, %s)\n",
	      serial, optnam, option_mod_flags[status]);
	else
	    nhproxy_io_printf(nhproxy_clnt_log,
	      "rpc [%u] set_option_mod_status(%#s, %d)\n",
	      serial, optnam, status);
    }
    (void)nhproxy_rpc(NHPROXY_EXT_CID_SET_OPTION_MOD_STATUS, 2,
      NHPROXY_STRING(optnam), NHPROXY_INT(status), 0);
}
