/* $Id: nhext.c,v 1.19 2004/12/22 23:15:04 ali Exp $
 * Copyright (C) 2001, 2002, 2003, 2004  Slash'EM Development Team
 * Copyright (C) 2004  J. Ali Harlow
 *
 * This file is part of NetHack Proxy.
 *
 * NetHack Proxy is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of the
 * License, or (at your option) any later version.
 *
 * NetHack Proxy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with NetHack Proxy; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307   
 * USA
 *
 * Alternatively (at your option) you may instead choose to redistribute
 * and/or modify NetHack Proxy under the terms of the NetHack General
 * Public License.
 *
 * You should have receieved a copy of the NetHack General Public License
 * along with NetHack Proxy; if not, download a copy from
 * http://www.nethack.org/common/license.html
 */

/* #define DEBUG */

#include "config.h"
#include <stdio.h>
#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#endif
#include <ctype.h>
#include "compat.h"
#include <nhproxy/system.h>
#include <nhproxy/xdr.h>
#include <nhproxy/common.h>

/*
 * This module implements the low-level NhExt protocols.
 */

static struct nhproxy_connection {
    int length;
    /* In sub-protocol 2, the serial number to
     * which the last reply read relates
     */
    unsigned short serial;
    NhProxyXdr *in, *out;
    NhProxyIO *rd, *wr;
    struct nhproxy_rpc_services *callbacks;
    int protocol;
} nhproxy_connection;

nhproxy_bool_t
nhproxy_rpc_init(rd, wr, cb)
NhProxyIO *rd, *wr;
struct nhproxy_rpc_services *cb;
{
    nhproxy_io_setmode(rd, NHPROXY_IO_NOAUTOFILL);
    nhproxy_connection.rd = rd;
    nhproxy_connection.wr = wr;
    nhproxy_connection.in = (NhProxyXdr *)malloc(sizeof(NhProxyXdr));
    nhproxy_connection.out = (NhProxyXdr *)malloc(sizeof(NhProxyXdr));
    if (!nhproxy_connection.in || !nhproxy_connection.out) {
	free(nhproxy_connection.in);
	free(nhproxy_connection.out);
	return nhproxy_false;
    }
    nhproxy_xdrio_create(nhproxy_connection.in, rd, NHPROXY_XDR_DECODE);
    nhproxy_xdrio_create(nhproxy_connection.out, wr, NHPROXY_XDR_ENCODE);
    nhproxy_connection.callbacks = cb;
    nhproxy_connection.protocol = 0;
    return nhproxy_true;
}

nhproxy_bool_t
nhproxy_rpc_set_protocol(protocol)
int protocol;
{
    if (protocol < 1 || protocol > 2)
	return nhproxy_false;
    nhproxy_connection.protocol = protocol;
    return nhproxy_true;
}

nhproxy_bool_t
nhproxy_rpc_async_mode()
{
    return nhproxy_connection.protocol > 1;
}

void
nhproxy_rpc_end()
{
    nhproxy_xdr_destroy(nhproxy_connection.in);
    nhproxy_xdr_destroy(nhproxy_connection.out);
    free(nhproxy_connection.in);
    free(nhproxy_connection.out);
    nhproxy_connection.in = NULL;
    nhproxy_connection.out = NULL;
}

static char *
nhproxy_subprotocol0_encode_value(buf, value)
char *buf, *value;
{
    *buf++ = '"';
    while (*value) {
	if (*value == '\\' || *value == '"')
	    *buf++ = '\\';
	*buf++ = *value++;
    }
    *buf++ = '"';
    return buf;
}

nhproxy_bool_t
nhproxy_subprotocol0_write_line(line)
struct nhproxy_line *line;
{
    int i, len;
    nhproxy_bool_t retval;
    char *buf, *bp;
    len = strlen(line->type) + 1;
    for(i = 0; i < line->n; i++) {
	len += strlen(line->tags[i]) + 1;
	len += strlen(line->values[i]) * 2 + 2 + 1;
    }
    buf = (char *)malloc(len);
    if (!buf)
	return nhproxy_false;
    (void)strcpy(buf, line->type);
    bp = buf + strlen(buf);
    for(i = 0; i < line->n; i++) {
	*bp++ = ' ';
	(void)strcpy(bp, line->tags[i]);
	bp += strlen(bp);
	*bp++ = ' ';
	bp = nhproxy_subprotocol0_encode_value(bp, line->values[i]);
    }
    *bp++ = '\n';
    retval =
      nhproxy_io_write(nhproxy_connection.wr, buf, bp - buf) == bp - buf &&
      nhproxy_io_flush(nhproxy_connection.wr);
    free(buf);
    return retval;
}

#define NHPROXY_SP0_NORMAL_SIZE		64
#define NHPROXY_SP0_SPECIAL_SIZE	200

/*
 * Return the next token read or NULL on error. An empty string
 * will be returned if the line ends before a token is found.
 */

static char *
nhproxy_subprotocol0_read_token(io)
NhProxyIO *io;
{
    int i, ch;
    static char token[NHPROXY_SP0_NORMAL_SIZE+1];
    for(i = 0;; ) {
	ch = nhproxy_io_getc(io);
	if (ch < 0) {
#ifdef DEBUG
	    fprintf(stderr,
	      "[%d] EOF/ERROR while reading sub-protocol 0 token\n", getpid());
#endif
	    return NULL;
	}
	if (ch == '\r' || ch == '\n')
	    break;
	if (ch == ' ') {
	    if (i)
		break;
	    else
		continue;
	}
	if (ch != '_' && !isalnum(ch)) {
#ifdef DEBUG
	    fprintf(stderr,
	      "[%d] Illegal character (0x%02X) while reading sub-protocol 0 token\n",
	      getpid(), ch);
#endif
	    return NULL;
	}
	if (i == NHPROXY_SP0_NORMAL_SIZE) {
#ifdef DEBUG
	    fprintf(stderr,
	      "[%d] Too many characters while reading sub-protocol 0 token\n",
	      getpid());
#endif
	    return NULL;
	}
	token[i++] = ch;
    }
    token[i] = '\0';
#ifdef DEBUG
    fprintf(stderr, "nhproxy_subprotocol0_read_token: %s\n", token);
#endif
    return token;
}

/*
 * Return the next value read or NULL on error. A NULL
 * will be returned if the line ends before a value can be found.
 */

static char *
nhproxy_subprotocol0_read_value(io, isspecial)
NhProxyIO *io;
int isspecial;
{
    int i, ch, esc = FALSE, invalue = FALSE;
    const int maxlen =
      isspecial ? NHPROXY_SP0_SPECIAL_SIZE : NHPROXY_SP0_NORMAL_SIZE;
    static char value[NHPROXY_SP0_SPECIAL_SIZE+1];
    for(i = 0;; ) {
	ch = nhproxy_io_getc(io);
	if (ch < 0) {
#ifdef DEBUG
	    fprintf(stderr,
	      "[%d] EOF/ERROR while reading sub-protocol 0 value\n", getpid());
#endif
	    return NULL;
	}
	if (!invalue) {
	    if (ch == '"')
		invalue = TRUE;
	    else if (ch == ' ')
		continue;
	    else {
#ifdef DEBUG
		fprintf(stderr,
		  "[%d] Read 0x%02X while expecting sub-protocol 0 value\n",
		  getpid(), ch);
#endif
		return NULL;
	    }
	} else if (ch < ' ' || ch > '~') {
#ifdef DEBUG
	    fprintf(stderr,
	      "[%d] Illegal character (0x%02X) while reading sub-protocol 0 value\n",
	      getpid(), ch);
#endif
	    return NULL;
	} else if (!esc && ch == '\\')
	    esc = TRUE;
	else if (!esc && ch == '"')
	    break;
	else if (i == maxlen) {
#ifdef DEBUG
	    fprintf(stderr,
	      "[%d] Too many characters while reading sub-protocol 0 value\n",
	      getpid());
#endif
	    return NULL;
	} else {
	    value[i++] = ch;
	    esc = FALSE;
	}
    }
    value[i] = '\0';
#ifdef DEBUG
    fprintf(stderr, "nhproxy_subprotocol0_read_value: %s\n", value);
#endif
    return value;
}

void
nhproxy_subprotocol0_free_line(line)
struct nhproxy_line *line;
{
    int i;
    for(i = 0; i < line->n; i++) {
	free(line->tags[i]);
	free(line->values[i]);
    }
    free(line->type);
    free(line->tags);
    free(line->values);
    free(line);
}

struct nhproxy_line *
nhproxy_subprotocol0_read_line()
{
    int i, iserror;
    char *s;
    struct nhproxy_line *line;
#ifdef DEBUG
    if (nhproxy_io_getc(nhproxy_connection.rd) >= 0)
	fprintf(stderr,
	  "[%d] NhExt: Non-empty buffer in nhproxy_subprotocol0_read_line\n",
	  getpid());
#endif
    i = nhproxy_io_filbuf(nhproxy_connection.rd, TRUE);
    if (i <= 0) {
#ifdef DEBUG
	fprintf(stderr,
	  "[%d] %s while trying to read sub-protocol 0 packet\n",
	  getpid(), i == 0 ? "EOF" : i == -2 ? "EWOULDBLOCK" : "ERROR");
#endif
	return NULL;
    }
    line = (struct nhproxy_line *) malloc(sizeof(*line));
    if (!line)
	return NULL;
    s = nhproxy_subprotocol0_read_token(nhproxy_connection.rd);
    if (!s) {
	free(line);
	return NULL;
    }
    iserror = !strcmp(s, "Error");
    line->type = strdup(s);
    line->n = 0;
    line->tags = (char **)0;
    line->values = (char **)0;
    for(i = 0;; i++) {
	s = nhproxy_subprotocol0_read_token(nhproxy_connection.rd);
	if (!s || !*s)
	    break;
	if (line->n) {
	    line->tags =
	      (char **)realloc(line->tags, (line->n + 1)*sizeof(char **));
	    line->values =
	      (char **)realloc(line->values, (line->n + 1)*sizeof(char **));
	    if (!line->tags || !line->values) {
#ifdef DEBUG
		fprintf(stderr,
		  "[%d] NhExt: Memory allocation failure; cannot get %u tags",
		  getpid(), line->n + 1);
#endif
		s = NULL;
		break;
	    }
	} else {
	    line->tags = (char **)malloc(sizeof(char **));
	    line->values = (char **)malloc(sizeof(char **));
	    if (!line->tags || !line->values) {
#ifdef DEBUG
		fprintf(stderr,
		  "[%d] NhExt: Memory allocation failure; cannot get 1 tag",
		  getpid());
#endif
		s = NULL;
		break;
	    }
	}
	line->tags[line->n] = strdup(s);
	s = nhproxy_subprotocol0_read_value(nhproxy_connection.rd,
	  iserror && !strcmp(s, "mesg"));
	if (!s) {
	    free(line->tags[line->n]);
	    break;
	}
	line->values[line->n] = strdup(s);
	line->n++;
    }
    if (s && nhproxy_io_getc(nhproxy_connection.rd) >= 0) {
	for(i = 1; nhproxy_io_getc(nhproxy_connection.rd) >= 0; i++)
	    ;
#ifdef DEBUG
	fprintf(stderr,
	  "[%d] %d extra character%s after valid sub-protocol 0 packet\n",
	  getpid(), i, i == 1 ? "": "s");
#endif
	s = NULL;
    }
    if (!s) {
	nhproxy_subprotocol0_free_line(line);
	return NULL;
    } else
	return line;
}

/*
 * This function is available for callers to use if sub-protocol 0 fails.
 * It returns a pointer to the received packet which was being processed
 * at the time.
 */

char *
nhproxy_subprotocol0_get_failed_packet(nb)
int *nb;
{
    if (nhproxy_connection.in)
	return nhproxy_io_getpacket(nhproxy_connection.rd, nb);
    else
	return (char *)0;
}

static void
nhproxy_rpc_default_handler(class, error)
int class;
const char *error;
{
    nhproxy_error(error);
}

static nhproxy_rpc_errhandler
nhproxy_rpc_error_handler = nhproxy_rpc_default_handler;

nhproxy_rpc_errhandler
nhproxy_rpc_set_errhandler(new)
nhproxy_rpc_errhandler new;
{
    nhproxy_rpc_errhandler old = nhproxy_rpc_error_handler;
    nhproxy_rpc_error_handler = new;
    return old;
}

#if HAVE_STDARG_H
static void
nhproxy_rpc_error(int class, const char *fmt, ...)
#else
static void
nhproxy_rpc_error(class, fmt, va_alist)
int class;
const char *fmt;
va_dcl
#endif
{
    va_list ap;
    char buf[128];
    VA_START(ap, fmt);
    vsprintf(buf, fmt, ap);
    nhproxy_rpc_error_handler(class, buf);
    va_end(ap);
}

static int
nhproxy_rpc_vparams1(xdrs, no, app)
NhProxyXdr *xdrs;
int no;
va_list *app;
{
    int retval = TRUE;
    int param;
    va_list ap;
    long param_l;
    long *param_pl;
    int param_i;
    int *param_pi;
    char *param_s;
    char **param_ps, *param_pc;
    nhproxy_bool_t param_b, *param_pb;
    int (*param_codec)();
    nhproxy_genericptr_t param_addr;
    ap = *app;
    while(retval && no--) {
	param = va_arg(ap, int);
	switch(param) {
	    case NHPROXY_PARAM_INT:
		param_i = va_arg(ap, int);
		retval = nhproxy_xdr_int(xdrs, &param_i);
		break;
	    case NHPROXY_PARAM_LONG:
		param_l = va_arg(ap, long);
		retval = nhproxy_xdr_long(xdrs, &param_l);
		break;
	    case NHPROXY_PARAM_STRING:
		param_s = va_arg(ap, char *);
		retval = nhproxy_xdr_string(xdrs, &param_s, (unsigned int)-1);
		break;
	    case NHPROXY_PARAM_BYTES:
		param_s = va_arg(ap, char *);
		param_i = va_arg(ap, int);
		retval = nhproxy_xdr_bytes(xdrs, &param_s, &param_i,
			(unsigned int)-1);
		break;
	    case NHPROXY_PARAM_BOOLEAN:
		param_b = va_arg(ap, int);	/* boolean is promoted to int */
		retval = nhproxy_xdr_bool(xdrs, &param_b);
		break;
	    case NHPROXY_PARAM_CHAR:
		param_i = va_arg(ap, int);	/* char is promoted to int */
		retval = nhproxy_xdr_int(xdrs, &param_i);
		break;
	    case NHPROXY_PARAM_PTR | NHPROXY_PARAM_INT:
		param_pi = va_arg(ap, int *);
		retval = nhproxy_xdr_int(xdrs, param_pi);
		break;
	    case NHPROXY_PARAM_PTR | NHPROXY_PARAM_LONG:
		param_pl = va_arg(ap, long *);
		retval = nhproxy_xdr_long(xdrs, param_pl);
		break;
	    case NHPROXY_PARAM_PTR | NHPROXY_PARAM_STRING:
		param_ps = va_arg(ap, char **);
		*param_ps = NULL;
		retval = nhproxy_xdr_string(xdrs, param_ps, (unsigned int)-1);
		break;
	    case NHPROXY_PARAM_PTR | NHPROXY_PARAM_BYTES:
		param_ps = va_arg(ap, char **);
		*param_ps = NULL;
		param_pi = va_arg(ap, int *);
		nhproxy_xdr_bytes(xdrs, param_ps, param_pi, (unsigned int)-1);
		break;
	    case NHPROXY_PARAM_PTR | NHPROXY_PARAM_BOOLEAN:
		param_pb = va_arg(ap, nhproxy_bool_t *);
		retval = nhproxy_xdr_bool(xdrs, param_pb);
		break;
	    case NHPROXY_PARAM_PTR | NHPROXY_PARAM_CHAR:
		param_pc = va_arg(ap, char *);
		retval = nhproxy_xdr_char(xdrs, param_pc);
		break;
	    case NHPROXY_PARAM_XDRF:
		param_codec = (int (*)())va_arg(ap, nhproxy_genericptr_t);
		param_addr = va_arg(ap, nhproxy_genericptr_t);
		retval = (*param_codec)(xdrs, param_addr);
		break;
	    default:
		nhproxy_rpc_error(NHPROXY_ERROR_INTERNAL,
		  "Bad key in proxy rpc (%d)", param);
		retval = FALSE;
		break;
	}
    }
    *app = ap;
    return retval;
}

/*
 * Note: nhproxy_rpc_params() does different things depending on whether
 * the XDR stream is set to encode or decode. In decode mode, the
 * header is assumed to have already been read whereas, in encode mode,
 * nhproxy_rpc_params() writes the reply header itself.
 */

#if HAVE_STDARG_H
nhproxy_bool_t
nhproxy_rpc_params(NhProxyXdr *xdrs, int no, ...)
#else
nhproxy_bool_t
nhproxy_rpc_params(xdrs, no, va_alist)
NhProxyXdr *xdrs;
int no
va_dcl
#endif
{
    nhproxy_bool_t retval;
    va_list ap;
    unsigned long value;
    NhProxyXdr sink;
    if (nhproxy_connection.protocol < 1 || nhproxy_connection.protocol > 2) {
	nhproxy_rpc_error(NHPROXY_ERROR_INTERNAL,
	  "nhproxy_rpc_params: Unsupported protocol %d",
	  nhproxy_connection.protocol);
	return nhproxy_false;
    }
    if (xdrs->x_op == NHPROXY_XDR_ENCODE) {
	sink.x_op = NHPROXY_XDR_COUNT;
	sink.x_pos = 0;
	VA_START(ap, no);
	retval = nhproxy_rpc_vparams1(&sink, no, &ap);
	va_end(ap);
	if (!retval)
	    nhproxy_rpc_error(NHPROXY_ERROR_INTERNAL, "Codec failed in sink");
	else {
	    value = sink.x_pos >> 2;
	    if (nhproxy_connection.protocol > 1) {
		value |= 0x8000;		/* Mark packet as a reply */
		value |= nhproxy_rpc_svc_get_serial() << 16;
	    }
	    retval = nhproxy_xdr_u_long(xdrs, &value);
	    if (!retval)
		nhproxy_rpc_error(NHPROXY_ERROR_COMMS,
		  "Failed to write header");
	}
	if (!retval)
	    return retval;
    }
    VA_START(ap, no);
    retval = nhproxy_rpc_vparams1(xdrs, no, &ap);
    if (!retval)
	nhproxy_rpc_error(NHPROXY_ERROR_COMMS, "Codec failed");
    va_end(ap);
    return retval;
}

#define NHPROXY_FLAG_ASYNC		1
#define NHPROXY_FLAG_UNSUPPORTED	2

static int nhproxy_n_flags;
static int *nhproxy_flags;

static int
nhproxy_extend_flags(n_flags)
int n_flags;
{
    int *new;
    if (n_flags <= nhproxy_n_flags)
	return 0;
    if (!nhproxy_flags)
	new = (int *)calloc(n_flags, sizeof(int));
    else
	new = (int *)realloc(nhproxy_flags, n_flags * sizeof(int));
    if (new) {
	nhproxy_flags = new;
	nhproxy_n_flags = n_flags;
	return 0;
    } else
	return -1;
}

void
nhproxy_rpc_set_async_masks(n,masks)
int n;
unsigned long *masks;
{
    int i;
    if (nhproxy_n_flags)
	for(i = 0; i < nhproxy_n_flags; i++)
	    nhproxy_flags[i] &= ~NHPROXY_FLAG_ASYNC;
    if (n) {
	for(i = 31; i > 0; i--)
	    if (masks[n - 1] & 1L << i)
		break;
	if (nhproxy_extend_flags(1 + (n - 1) * 32 + i + 1))
	    nhproxy_error(
	      "Memory allocation failure in nhproxy_rpc_set_async_masks");
#ifdef DEBUG
	fprintf(stderr, "nhproxy_rpc: Async IDs:");
#endif
	for(i = 1; i < nhproxy_n_flags; i++) {
	    if (masks[(i - 1) / 32] & 1L << ((i - 1) & 31))
		nhproxy_flags[i] |= NHPROXY_FLAG_ASYNC;
#ifdef DEBUG
	    if (nhproxy_flags[i] & NHPROXY_FLAG_ASYNC)
		fprintf(stderr, " %x", i);
#endif
	}
#ifdef DEBUG
	fprintf(stderr, "\n");
#endif
    }
#ifdef DEBUG
    else
	fprintf(stderr, "No async IDs\n");
#endif
}

void
nhproxy_rpc_set_unsupported(id)
int id;
{
    if (!nhproxy_extend_flags(id + 1))
	nhproxy_flags[id] |= NHPROXY_FLAG_UNSUPPORTED;
}

struct nhproxy_frame {
    struct nhproxy_frame *prev_fp;
    unsigned short serial;
    unsigned short length;
#ifdef DEBUG
    unsigned short id;
#endif
    unsigned char async;
    nhproxy_genericptr_t data;
};

static unsigned short rpc_serial = 0;
static struct nhproxy_frame *nhproxy_rpc_fp, *nhproxy_svc_fp;

unsigned short
nhproxy_rpc_get_next_serial()
{
    return rpc_serial + 1;
}

unsigned short
nhproxy_rpc_svc_get_serial()
{
    return nhproxy_svc_fp ? nhproxy_svc_fp->serial : 0;
}

static nhproxy_bool_t
nhproxy_store_reply(f)
struct nhproxy_frame *f;
{
    int n, nb;
    nhproxy_genericptr_t data;
    f->length = nhproxy_connection.length;
    if (!f->length) {
	/* Avoid malloc(0) which may return NULL */
	f->data = (nhproxy_genericptr_t)malloc(1);
	return nhproxy_true;
    }
    f->data = (nhproxy_genericptr_t)malloc(f->length);
    if (!f->data)
	return nhproxy_false;
    nhproxy_io_setautofill_limit(nhproxy_connection.rd, f->length);
    /* Can't use nhproxy_io_fread() since nhproxy_connection.length
     * may be too large for use as a member size.
     */
    data = f->data;
    nb = f->length;
    while(nb) {
	n = nhproxy_io_read(nhproxy_connection.rd, data, nb);
	if (n > 0) {
	    nb -= n;
	    data += n;
	} else {
	    /*
	     * Issue a comms error here. We will probably
	     * issue a protocol error (short reply) when
	     * we come to process the reply, which isn't
	     * very helpful, but it doesn't seem worth
	     * the overhead to block it -- ALI.
	     */
	    nhproxy_rpc_error(NHPROXY_ERROR_COMMS,
	      "Read from proxy interface failed");
	    f->length -= nb;
	    break;
	}
    }
    return nhproxy_true;
}

/*
 * nhproxy_rpc() is a generic function to call a remote procedure and return
 * a result. It cannot deal with every possible type of request and result
 * packet, but it's very simple to use for those it can deal with.
 *
 * Arguments are:
 *	id		The ID of the procedure to call
 *	request		List of arguments describing the request packet
 *	response	List of arguments describing the response packet
 *
 * For both the request and response packets, the list of arguments consists
 * of:
 *
 *	no			The number of fields in the packet
 *	field[1] .. field[no]	The field to send/receive.
 */

#if HAVE_STDARG_H
nhproxy_bool_t
nhproxy_rpc(unsigned short id, ...)
#else
int
nhproxy_rpc(id, va_alist)
unsigned short id;
va_dcl
#endif
{
    NhProxyXdr sink;
    va_list ap;
    unsigned long value, pos;
    int i, retval;
    int no;		/* Number of fields */
    struct nhproxy_frame frame, *f;
    NhProxyXdr xdrmem, *in;
    if (nhproxy_connection.protocol < 1 || nhproxy_connection.protocol > 2) {
	nhproxy_rpc_error(NHPROXY_ERROR_INTERNAL,
	  "nhproxy_rpc: Unsupported protocol %d", nhproxy_connection.protocol);
	return nhproxy_false;
    }
    if (nhproxy_flags && id < nhproxy_n_flags &&
      nhproxy_flags[id] & NHPROXY_FLAG_UNSUPPORTED) {
	return nhproxy_false;
    }
    frame.prev_fp = nhproxy_rpc_fp;
    nhproxy_rpc_fp = &frame;
    frame.serial = ++rpc_serial;
    frame.data = NULL;
#ifdef DEBUG
    frame.id = id;
    fprintf(stderr, "[%d] nhproxy_rpc: [%u] call(%X)\n",
      getpid(), rpc_serial, id);
    if (frame.prev_fp) {
	fprintf(stderr, "[%d] nhproxy_rpc: call stack:\n", getpid());
	for(f = frame.prev_fp; f; f = f->prev_fp)
	    fprintf(stderr, "[%d]\t[%u] call(%X)%s\n",
	      getpid(), f->serial, f->id, f->data ? " (reply stored)" : "");
    }
#endif
    sink.x_op = NHPROXY_XDR_COUNT;
    sink.x_pos = 0;
    VA_START(ap, id);
    no = va_arg(ap, int);
    if (!nhproxy_rpc_vparams1(&sink, no, &ap)) {
	nhproxy_rpc_error(NHPROXY_ERROR_INTERNAL, "Codec failed in sink");
	va_end(ap);
	--rpc_serial;
	nhproxy_rpc_fp = frame.prev_fp;
	return nhproxy_false;
    }
    va_end(ap);
    VA_START(ap, id);
    (void) va_arg(ap, int);
    value = (id << 16) | (sink.x_pos >> 2);
    nhproxy_xdr_u_long(nhproxy_connection.out, &value);
    pos = nhproxy_xdr_getpos(nhproxy_connection.out);
    if (!nhproxy_rpc_vparams1(nhproxy_connection.out, no, &ap) ||
      !nhproxy_io_flush(nhproxy_connection.wr)) {
	nhproxy_rpc_error(NHPROXY_ERROR_COMMS,
	  "Write to proxy interface failed");
	va_end(ap);
	nhproxy_rpc_fp = frame.prev_fp;
	return nhproxy_false;
    }
    if (nhproxy_xdr_getpos(nhproxy_connection.out) - pos != sink.x_pos) {
	nhproxy_rpc_error(NHPROXY_ERROR_INTERNAL,
	  "Miscounted length in proxy rpc ID %d (counted %lu, wrote %lu)",
	  id, sink.x_pos, nhproxy_xdr_getpos(nhproxy_connection.out) - pos);
	va_end(ap);
	nhproxy_rpc_fp = frame.prev_fp;
	return nhproxy_false;
    }
    if (nhproxy_flags && id < nhproxy_n_flags &&
      nhproxy_flags[id] & NHPROXY_FLAG_ASYNC) {
	no = va_arg(ap, int);
	if (no) {
	    nhproxy_rpc_error(NHPROXY_ERROR_PROTOCOL,
	      "Expecting reply from asynchronous procedure %d", id);
	    va_end(ap);
	    nhproxy_rpc_fp = frame.prev_fp;
	    return nhproxy_false;
	}
#ifdef DEBUG
	fprintf(stderr, "[%d] nhproxy_rpc: [%lu] %X sent\n",
	  getpid(), frame.serial, id);
#endif
	frame.async = TRUE;
    } else
	frame.async = FALSE;
    for(;;)
    {
	/* By dealing with incoming packets here, we reduce the chance that
	 * the remote end will need to buffer replies. We cannot do this,
	 * however, unless we are certain that we won't block on the read;
	 * otherwise, we could easily deadlock.
	 */
	if (frame.async && nhproxy_io_willblock(nhproxy_connection.rd)) {
	    va_end(ap);
	    nhproxy_rpc_fp = frame.prev_fp;
	    return nhproxy_true;
	}
	retval = nhproxy_rpc_svc(nhproxy_connection.callbacks);
	if (retval < 0) {
	    va_end(ap);
	    nhproxy_rpc_fp = frame.prev_fp;
	    return FALSE;
	} else if (!retval) {
	    if (!frame.async && (nhproxy_connection.protocol <= 1 ||
	      nhproxy_connection.serial == frame.serial)) {
		frame.length = nhproxy_connection.length;
		break;
	    } else {
		for(f = frame.prev_fp; f; f = f->prev_fp) {
		    if (nhproxy_connection.serial == f->serial &&
		      !f->async && !f->data) {
			if (!nhproxy_store_reply(f)) {
			    nhproxy_rpc_error(NHPROXY_ERROR_EXT_RESOURCEFAILURE,
			      "Not enough memory to store reply");
			    return nhproxy_false;
			}
			break;
		    }
		}
		if (!f) {
		    nhproxy_rpc_error(NHPROXY_ERROR_PROTOCOL,
		      "Reply received for unexpected serial %u (%d bytes)",
		      nhproxy_connection.serial, nhproxy_connection.length);
		    /* Junk the reply and continue to wait */
		    nhproxy_io_setautofill_limit(nhproxy_connection.rd,
		      nhproxy_connection.length);
		    for(i = 0; i < nhproxy_connection.length; i++)
			(void)nhproxy_io_getc(nhproxy_connection.rd);
		}
#ifdef DEBUG
		else {
		    long total = 0;
		    for(f = frame.prev_fp; f; f = f->prev_fp)
			if (f->data)
			    total += f->length;
		    fprintf(stderr,
		      "[%d] nhproxy_rpc: %u bytes stored (total %ld)\n",
		      getpid(), nhproxy_connection.length, total);
		}
#endif
	    }
	} else if (frame.data)
	    break;
    }
    no = va_arg(ap, int);
    /* External window ports are always allowed to return an empty reply to
     * a request. This indicates that the request is not supported.
     */
    if (no && !frame.length) {
	nhproxy_rpc_error(NHPROXY_ERROR_NOTSUPPORTED,
	  "Procedure %d not supported by remote end", id);
	va_end(ap);
	nhproxy_rpc_set_unsupported(id);
	nhproxy_rpc_fp = frame.prev_fp;
	return nhproxy_false;
    }
    if (frame.data) {
	in = &xdrmem;
	nhproxy_xdrmem_create(in, frame.data, frame.length, NHPROXY_XDR_DECODE);
    } else {
	in = nhproxy_connection.in;
	nhproxy_io_setautofill_limit(nhproxy_connection.rd, frame.length);
    }
    if (!nhproxy_rpc_vparams1(in, no, &ap)) {
	/*
	 * There are two important causes of nhproxy_rpc_vparams1() failing.
	 * Either the packet sent by the remote end is shorter than we
	 * are expecting or a comms error caused us to fail to read the
	 * whole packet. We distinguish these by checking if NOAUTOFILL
	 * is set (in which case we have read the advertised length).
	 * Note: When processing a stored reply we assume that the error
	 * must be a protocol error (we have no way of detecting if the
	 * short reply is due to a previous comms error).
	 */
	if (in == &xdrmem ||
	  nhproxy_io_getmode(nhproxy_connection.rd) & NHPROXY_IO_NOAUTOFILL)
	    nhproxy_rpc_error(NHPROXY_ERROR_PROTOCOL,
	      "Short reply received for protocol %d (received %d)", id,
	      frame.length);
	else
	    nhproxy_rpc_error(NHPROXY_ERROR_COMMS,
	      "Read from proxy interface failed");
	va_end(ap);
	if (in == &xdrmem) {
	    nhproxy_xdr_destroy(in);
	    free(frame.data);
	}
	nhproxy_rpc_fp = frame.prev_fp;
	return nhproxy_false;
    }
    /* If we're in sub-protocol 2, there may well be more bytes available
     * which form part of the next packet, but nhproxy_rpc_vparams1 should have
     * read the whole of this packet. If we're in sub-protocol 1, the tests
     * can be even more strict - see below. If we're processing a stored
     * packet, we can check this by looking at the read pointer.
     * Otherwise, we can check this since NOAUTOFILL will be set if the
     * advertised length has been consumed.
     */
    if (in == &xdrmem ? nhproxy_xdr_getpos(in) < frame.length :
      !(nhproxy_io_getmode(nhproxy_connection.rd) & NHPROXY_IO_NOAUTOFILL)) {
	/* Output an error and read and throw away the excess data. */
	if (in == &xdrmem)
	    i = frame.length - nhproxy_xdr_getpos(in);
	else
	    for(i = 0;; i++) {
		(void)nhproxy_io_getc(nhproxy_connection.rd);
		if (nhproxy_io_getmode(nhproxy_connection.rd) &
		  NHPROXY_IO_NOAUTOFILL)
		    break;
	    }
	nhproxy_rpc_error(NHPROXY_ERROR_PROTOCOL,
	  "Mismatch in RPC ID %d reply length (%d of %d unused)",
	  id, i, nhproxy_connection.length);
	va_end(ap);
	if (in == &xdrmem) {
	    nhproxy_xdr_destroy(in);
	    free(frame.data);
	}
	nhproxy_rpc_fp = frame.prev_fp;
	return nhproxy_false;
    }
    /* In sub-protocol 1, there should not be any more bytes available. If
     * there are then either the remote end wrote more data than it advertised
     * in the header, or it wrote two or more packets.
     */
    if (nhproxy_connection.protocol <= 1 &&
      nhproxy_io_getc(nhproxy_connection.rd) >= 0) {
	/* One or more bytes are still available. We output an error and
	 * read and throw away the excess data.
	 */
	for(i = 1; nhproxy_io_getc(nhproxy_connection.rd) >= 0; i++)
	    ;
	nhproxy_rpc_error(NHPROXY_ERROR_PROTOCOL,
	  "Mismatch in RPC ID %d reply length (%d of %d unused)",
	  id, i, nhproxy_connection.length);
	va_end(ap);
	if (in == &xdrmem) {
	    nhproxy_xdr_destroy(in);
	    free(frame.data);
	}
	nhproxy_rpc_fp = frame.prev_fp;
	return nhproxy_false;
    }
    va_end(ap);
    if (in == &xdrmem) {
	nhproxy_xdr_destroy(in);
	free(frame.data);
    }
    nhproxy_rpc_fp = frame.prev_fp;
#ifdef DEBUG
    fprintf(stderr, "[%d] nhproxy_rpc: [%lu] %X returns\n",
      getpid(), frame.serial, id);
#endif
    return nhproxy_true;
}

/*
 * nhproxy_rpc_svc() is a function to service incoming packets. It reads a
 * packet header from the remote process (either a child or a parent). Service
 * packets (those which are not replies) are dispatched and replied to, reply
 * packets are left in the NhProxy buffer (except for the header). The length
 * of the packet (as advised in the header) will be stored in the length
 * variable. In sub-protocol 2, the serial number to which the reply relates
 * is stored in the serial variable. For service packets, the ID is returned.
 * For replies, 0 is returned. Special packets are handled and 0xffff returned.
 *
 * nhproxy_rpc_svc() is thus suitable for use in two occasions. In the first, it
 * can be used to dispatch callbacks while waiting for a reply. In this mode
 * nhproxy_rpc_svc() should be called repeatedly until it returns 0. In the
 * second case, nhproxy_rpc_svc() can be used as the main loop of a server in
 * which case it should be called repeatedly to service requests and a zero ID
 * should be treated as an error.
 */

int
nhproxy_rpc_svc(services)
struct nhproxy_rpc_services *services;
{
    struct nhproxy_frame frame;
    static unsigned short serial = 0;
    int i, j, is_reply, is_special, type;
    unsigned short id;
    unsigned long value, word2;
    if (nhproxy_connection.protocol < 1 || nhproxy_connection.protocol > 2) {
	nhproxy_rpc_error(NHPROXY_ERROR_INTERNAL,
	  "nhproxy_rpc_svc: Unsupported protocol %d",
	  nhproxy_connection.protocol);
	return -1;
    }
    if (nhproxy_connection.protocol <= 1 &&
      nhproxy_io_getc(nhproxy_connection.rd) >= 0) {
	/* One or more bytes are already available. This means that a
	 * previous packet was not wholly used. We output an error and
	 * read and throw away the excess data.
	 */
	for(j = 1; nhproxy_io_getc(nhproxy_connection.rd) >= 0; j++)
	    ;
	nhproxy_rpc_error(NHPROXY_ERROR_PROTOCOL,
	  "Mismatch in packet length (%d of %d unused)",
	  j, nhproxy_connection.length);
    }
    nhproxy_io_setautofill_limit(nhproxy_connection.rd, 4);
    if (!nhproxy_xdr_u_long(nhproxy_connection.in, &value)) {
	nhproxy_rpc_error(NHPROXY_ERROR_COMMS,
	  "Read from proxy interface failed");
	return -1;
    }
    id = value >> 16;
    if (nhproxy_connection.protocol <= 1) {
	is_reply = !id;
	is_special = FALSE;
	nhproxy_connection.length = (value & 0xffff) << 2;
    } else {
	is_reply = value & 0x8000;
	is_special = id == 0xffff && !is_reply;
	if (is_special) {
	    type = (value & 0x7f00) >> 8;
	    nhproxy_connection.length = (value & 0xff) << 2;
	} else
	    nhproxy_connection.length = (value & 0x7fff) << 2;
    }
    if (is_reply)
	nhproxy_connection.serial = id;
    else if (is_special) {
	nhproxy_io_setautofill_limit(nhproxy_connection.rd,
	  nhproxy_connection.length);
	if (type == NHPROXY_SPECIAL_ERROR && nhproxy_connection.length >= 8) {
	    unsigned short serial;
	    unsigned char code;
	    if (!nhproxy_xdr_u_long(nhproxy_connection.in, &value) |
	      !nhproxy_xdr_u_long(nhproxy_connection.in, &word2)) {
		nhproxy_rpc_error(NHPROXY_ERROR_COMMS,
		  "Read from proxy interface failed");
		return -1;
	    }
	    serial = value >> 16;
	    id = value & 0xffff;
	    code = word2 & 0xff;
	    /* Discard any unprocessed remainder of known types before
	     * calling nhproxy_rpc_error() so that we maintain our state.
	     */
	    for(j = 8; j < nhproxy_connection.length; j++)
		if (nhproxy_io_getc(nhproxy_connection.rd) < 0)
		    break;
	    switch(code) {
		case NHPROXY_ERROR_EXT_UNSUPPORTED:
#ifdef DEBUG
		    nhproxy_rpc_error(code,
		      "Unsupported function %X in RPC serial %X", id, serial);
#endif
		    nhproxy_rpc_set_unsupported(id);
		    break;
		case NHPROXY_ERROR_EXT_UNAVAILABLE:
		    nhproxy_rpc_error(code,
		      "Unavailable function %X in RPC serial %X", id, serial);
		    break;
		case NHPROXY_ERROR_EXT_INVALIDENCODING:
		    nhproxy_rpc_error(code,
		      "Decoding error in RPC serial %X (ID %X)", serial, id);
		    break;
		case NHPROXY_ERROR_EXT_INVALIDPARAMS:
		    nhproxy_rpc_error(code,
		      "Invalid parameter(s) in RPC serial %X (ID %X)",
		      serial, id);
		    break;
		case NHPROXY_ERROR_EXT_RESOURCEFAILURE:
		    nhproxy_rpc_error(code,
		      "Ran out of resources in RPC serial %X (ID %X)",
		      serial, id);
		    break;
		default:
		    nhproxy_rpc_error(NHPROXY_ERROR_GENERIC,
		      "Error %X in RPC serial %X (ID %X)", code, serial, id);
	    }
	} else {
	    /* Discard special packets of unknown types.
	     */
	    for(j = 0; j < nhproxy_connection.length; j++)
		if (nhproxy_io_getc(nhproxy_connection.rd) < 0)
		    break;
	}
	return 0xffff;
    } else {
	frame.prev_fp = nhproxy_svc_fp;
	nhproxy_svc_fp = &frame;
	frame.serial = ++serial;
	nhproxy_io_setautofill_limit(nhproxy_connection.rd,
	  nhproxy_connection.length);
	for(i = 0; services[i].id; i++) {
	    if (id == services[i].id) {
		(*services[i].handler)(id, nhproxy_connection.in,
		  nhproxy_connection.out);
		break;
	    }
	}
	if (nhproxy_connection.protocol <= 1 &&
	  nhproxy_io_getc(nhproxy_connection.rd) >= 0) {
	    /* One or more bytes are still available. This means that the
	     * whole packet was not used. We output an error if we called
	     * a handler (otherwise the error is not helpful) and, in any
	     * case, read and throw away the excess data.
	     */
	    for(j = 1; nhproxy_io_getc(nhproxy_connection.rd) >= 0; j++)
		;
	    if (services[i].id)
		/* Note: This can only occur if there was not a recursive
		 * call (otherwise any error would have been detected
		 * either just before the next packet was read if this
		 * callback request was partially unused or just after
		 * the packet was processed if a reply packet was partially
		 * unused). The conclusion of all this is that
		 * nhproxy_connection.length is always valid. -- ALI
		 */
		nhproxy_rpc_error(NHPROXY_ERROR_PROTOCOL,
		  "Mismatch in callback ID %d request length (%d of %d unused)",
		  id, j, nhproxy_connection.length);
	}
	if (!services[i].id) {
#ifdef DEBUG
	    fprintf(stderr,"[%d] Unsupported proxy callback ID %d (%d known)\n",
	      getpid(), id, i);
#endif
	    nhproxy_rpc_error(NHPROXY_ERROR_NOTSUPPORTED,
	      "Procedure %d not supported", id);
	    nhproxy_rpc_params(nhproxy_connection.out, 0);
	}
	nhproxy_svc_fp = frame.prev_fp;
	if (!nhproxy_io_flush(nhproxy_connection.wr)) {
	    nhproxy_rpc_error(NHPROXY_ERROR_COMMS,
	      "Write to proxy interface failed");
	    return -1;
	}
    }
    return is_reply ? 0 : id;
}

void
nhproxy_rpc_send_error(id, error_code)
unsigned short id;
unsigned char error_code;
{
    unsigned short serial = nhproxy_rpc_svc_get_serial();
    unsigned long hdr, word1, word2;
    hdr = 0xffff0000 | NHPROXY_SPECIAL_ERROR << 8 | 2;
    word1 = serial << 16 | id;
    word2 = error_code;
    if ((!nhproxy_xdr_u_long(nhproxy_connection.out, &hdr) |
      !nhproxy_xdr_u_long(nhproxy_connection.out, &word1) |
      !nhproxy_xdr_u_long(nhproxy_connection.out, &word2)) ||
      !nhproxy_io_flush(nhproxy_connection.wr)) {
	nhproxy_rpc_error(NHPROXY_ERROR_COMMS,
	  "Write to proxy interface failed");
    }
}

static void
nhproxy_default_handler(error)
const char *error;
{
    fprintf(stderr, "NetHack Proxy: %s\n", error);
}

static nhproxy_errhandler nhproxy_error_handler = nhproxy_default_handler;

nhproxy_errhandler
nhproxy_set_errhandler(new)
nhproxy_errhandler new;
{
    nhproxy_errhandler old = nhproxy_error_handler;
    nhproxy_error_handler = new;
    return old;
}

#if HAVE_STDARG_H
void
nhproxy_error(const char *fmt, ...)
#else
void
nhproxy_error(fmt, va_alist)
const char *fmt;
va_dcl
#endif
{
    va_list ap;
    char buf[128];
    VA_START(ap, fmt);
    vsprintf(buf, fmt, ap);
    nhproxy_error_handler(buf);
    va_end(ap);
}
