/* $Id: server.c,v 1.36 2004/12/22 23:15:04 ali Exp $
 * Copyright (C) 2001, 2002, 2003, 2004  Slash'EM Development Team
 * Copyright (C) 2004  J. Ali Harlow
 *
 * This file is part of NetHack Proxy.
 *
 * NetHack Proxy is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of the
 * License, or (at your option) any later version.
 *
 * NetHack Proxy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with NetHack Proxy; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307   
 * USA
 *
 * Alternatively (at your option) you may instead choose to redistribute
 * and/or modify NetHack Proxy under the terms of the NetHack General
 * Public License.
 *
 * You should have receieved a copy of the NetHack General Public License
 * along with NetHack Proxy; if not, download a copy from
 * http://www.nethack.org/common/license.html
 */

/* #define DEBUG */

#include "config.h"
#include "compat.h"
#include <sys/time.h>
#include <ctype.h>
#ifdef MSWIN_API
#include <windows.h>
#else
#include <sys/types.h>
#include <sys/wait.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif
#include <nhproxy/system.h>
#include <nhproxy/xdr.h>
#include <nhproxy/common.h>
#include <nhproxy/server.h>

struct gbuf_layer {
    char start, stop;
    struct gbuf_row {
	char start, stop;
	int *glyphs;
    } *rows;
};

static nhproxy_bool_t NHPROXY_FDECL(proxy_print_glyph_layered, (int window,
  int nl, struct gbuf_layer *layers));

extern struct nhproxy_serv_callbacks *nhproxy__serv;

static nhproxy_bool_t proxy_print_glyph_layered();

static int proxy_protocol = 0;

static int proxy_no_mapwins = 0;

static int no_rows = 0, no_cols = 0, no_layers = 0;

static struct proxy_mapwin {
    int id;
    int c_rows, c_cols, c_layers;	/* Configured dimensions */
    struct gbuf_layer *gbuf_layers;
} *proxy_mapwins;

/* Flag to advise raw print functions not to attempt to use proxy */

static int in_proxy_init = 0;

/* Flag to indicate that the remote interface is authorized */

int nhproxy__authorized = 0;

/* The number of glyphs */

int nhproxy__no_glyph = -1;

/* Extensions to the NhExt protocol */
struct nhproxy_extension *nhproxy__extents = NULL;

/*
 * The glue functions.
 */

static int proxy_capc;
static char **proxy_capv;

nhproxy_bool_t
nhproxy_serv_get_capabilities(capcp, capvp)
int *capcp;
char ***capvp;
{
    *capcp = proxy_capc;
    *capvp = proxy_capv;
    return nhproxy_true;
}

nhproxy_bool_t
nhproxy_proc_init_nhwindows(argcp, argv)
int *argcp;
char **argv;
{
    int i, j, retval;
    struct nhproxy_init_nhwindow_req req;
    struct nhproxy_init_nhwindow_res res = {0, 0, 0, 0, 0};
    req.argc = *argcp;
    req.argv = argv;
    retval = nhproxy_rpc(NHPROXY_EXT_FID_INIT_NHWINDOWS,
      1, NHPROXY_XDRF(nhproxy_xdr_init_nhwindow_req, &req),
      1, NHPROXY_XDRF(nhproxy_xdr_init_nhwindow_res, &res));
    if (retval)
    {
	if (res.argc > *argcp) {
	    nhproxy_error("Bad argument list from init_nhwindows");
	    return nhproxy_false;
	}
	for(i = 0; i < res.argc; i++)
	{
	    for(j = i; j < *argcp; j++)
		if (!strcmp(res.argv[i], argv[j]))
		{
		    argv[i] = argv[j];
		    break;
		}
	    if (j == *argcp) {
		nhproxy_error("Bad argument from init_nhwindows");
		return nhproxy_false;
	    }
	}
	*argcp = res.argc;
	proxy_capc = res.capc;
	proxy_capv = res.capv;
    }
    return retval && res.inited;
}

nhproxy_bool_t
nhproxy_proc_player_selection(role, race, gend, align)
int *role, *race, *gend, *align;
{
    nhproxy_bool_t retval, quit;
    retval = nhproxy_rpc(NHPROXY_EXT_FID_PLAYER_SELECTION,
      4, NHPROXY_INT(*role), NHPROXY_INT(*race),
         NHPROXY_INT(*gend), NHPROXY_INT(*align),
      5, NHPROXY_INT_PTR(*role), NHPROXY_INT_PTR(*race), NHPROXY_INT_PTR(*gend),
         NHPROXY_INT_PTR(*align), NHPROXY_BOOLEAN_PTR(quit)) && !quit;
    return retval;
}

char *
nhproxy_proc_askname()
{
    char *name = (char *)0;
    (void)nhproxy_rpc(NHPROXY_EXT_FID_ASKNAME, 0, 1, NHPROXY_STRING_PTR(name));
    return name;
}

void
(nhproxy_proc_get_nh_event)()
{
    nhproxy_proc_get_nh_event();
}

void
(nhproxy_proc_exit_nhwindows)(str)
const char *str;
{
    nhproxy_proc_exit_nhwindows(str);
}

void
(nhproxy_proc_suspend_nhwindows)(str)
const char *str;
{
    nhproxy_proc_suspend_nhwindows(str);
}

void
(nhproxy_proc_resume_nhwindows)()
{
    nhproxy_proc_resume_nhwindows();
}

static int
nhproxy__mapwin_alloc_gbuf(int w, int rows, int cols, int layers)
{
    int n, i, j;
    char *memchunk;
    if (layers) {
	n = layers * sizeof(struct gbuf_layer) +
	  layers * rows * sizeof(struct gbuf_row) +
	  layers * rows * cols * sizeof(int);
	memchunk = malloc(n);
	if (!memchunk)
	    return nhproxy_false;
	free(proxy_mapwins[w].gbuf_layers);
	proxy_mapwins[w].gbuf_layers = (struct gbuf_layer *)memchunk;
	n = layers * sizeof(struct gbuf_layer);
	for(i = 0; i < layers; i++) {
	    proxy_mapwins[w].gbuf_layers[i].rows =
	      (struct gbuf_row *)(memchunk + n);
	    n += rows * sizeof(struct gbuf_row);
	    for(j = 0; j < rows; j++) {
		proxy_mapwins[w].gbuf_layers[i].rows[j].glyphs =
		  (int *)(memchunk + n);
		n += cols * sizeof(int);
	    }
	}
    } else
	proxy_mapwins[w].gbuf_layers = (struct gbuf_layer *)0;
    proxy_mapwins[w].c_rows = rows;
    proxy_mapwins[w].c_cols = cols;
    proxy_mapwins[w].c_layers = layers;
    return nhproxy_true;
}

int
nhproxy_proc_create_nhwindow(type)
int type;
{
    int id, w;
    if (type == NHPROXY_EXT_NHW_MAP) {
	struct proxy_mapwin *new;
	for(w = 0; w < proxy_no_mapwins; w++)
	    if (proxy_mapwins[w].id < 0)
		break;
	if (w == proxy_no_mapwins) {
	    if (proxy_mapwins)
		new = (struct proxy_mapwin *) realloc(proxy_mapwins,
		  (proxy_no_mapwins + 1) * sizeof(struct proxy_mapwin));
	    else
		new = (struct proxy_mapwin *)
		  malloc(sizeof(struct proxy_mapwin));
	    if (!new)
		return -1;
	    proxy_mapwins = new;
	    proxy_no_mapwins++;
	    memset(proxy_mapwins + w, 0, sizeof(*proxy_mapwins));
	}
	if (!nhproxy__mapwin_alloc_gbuf(w, no_rows, no_cols, no_layers)) {
	    proxy_mapwins[w].id = -1;
	    return -1;
	}
    }
    if (!nhproxy_rpc(NHPROXY_EXT_FID_CREATE_NHWINDOW, 1, NHPROXY_INT(type),
      1, NHPROXY_INT_PTR(id)))
	id = -1;
    if (type == NHPROXY_EXT_NHW_MAP) {
	proxy_mapwins[w].id = id;
	if (id >= 0)
	    nhproxy_serv_flush_layers(w, nhproxy_true, nhproxy__no_glyph);
    }
    return id;
}

int
nhproxy_serv_get_mapwin(window)
int window;
{
    int w;
    for(w = 0; w < proxy_no_mapwins; w++)
	if (proxy_mapwins[w].id == window)
	    return w;
    return -1;
}

void
nhproxy_proc_clear_nhwindow(window, rows, cols, layers)
int window, rows, cols, layers;
{
    int w = nhproxy_serv_get_mapwin(window);
    no_rows = rows;
    no_cols = cols;
    no_layers = layers;
    if (w >= 0) {
	if (proxy_mapwins[w].c_rows != rows ||
	  proxy_mapwins[w].c_cols != cols ||
	  proxy_mapwins[w].c_layers != layers) {
	    if (!nhproxy__mapwin_alloc_gbuf(w, rows, cols, layers)) {
		nhproxy_error("Not enough memory to resize window");
		return;
	    }
	}
	nhproxy_serv_flush_layers(w, nhproxy_true, nhproxy__no_glyph);
	(void)nhproxy_rpc(NHPROXY_EXT_FID_CLEAR_NHWINDOW, 4,
	  NHPROXY_INT(window), NHPROXY_INT(rows), NHPROXY_INT(cols),
	  NHPROXY_INT(layers), 0);
    } else
	(void)nhproxy_rpc(NHPROXY_EXT_FID_CLEAR_NHWINDOW, 4,
	  NHPROXY_INT(window), NHPROXY_INT(0), NHPROXY_INT(0), NHPROXY_INT(0),
	  0);
}

void
nhproxy_proc_display_nhwindow(window, blocking)
int window;
nhproxy_bool_t blocking;
{
    int w = nhproxy_serv_get_mapwin(window);
    if (w >= 0)
	nhproxy_serv_flush_layers(w, nhproxy_false, nhproxy__no_glyph);
    (void)nhproxy_rpc(NHPROXY_EXT_FID_DISPLAY_NHWINDOW,
      2, NHPROXY_INT(window), NHPROXY_BOOLEAN((nhproxy_bool_t)blocking), 0);
}

void
nhproxy_proc_destroy_nhwindow(window)
int window;
{
    struct proxy_mapwin *new;
    int w = nhproxy_serv_get_mapwin(window);
    (void)nhproxy_rpc(NHPROXY_EXT_FID_DESTROY_NHWINDOW, 1, NHPROXY_INT(window),
      0);
    if (w >= 0) {
	proxy_mapwins[w].id = -1;
	free(proxy_mapwins[w].gbuf_layers);
	proxy_mapwins[w].gbuf_layers = (struct gbuf_layer *)0;
	if (w == proxy_no_mapwins - 1) {
	    if (proxy_no_mapwins == 1) {
		free(proxy_mapwins);
		proxy_mapwins = (struct proxy_mapwin *)0;
		proxy_no_mapwins = 0;
	    } else {
		new = (struct proxy_mapwin *) realloc(proxy_mapwins,
		  (proxy_no_mapwins - 1) * sizeof(struct proxy_mapwin));
		if (new) {
		    proxy_mapwins = new;
		    proxy_no_mapwins--;
		}
	    }
	}
    }
}

void
(nhproxy_proc_curs)(window, x, y)
int window;
int x, y;
{
    nhproxy_proc_curs(window, x, y);
}

void
(nhproxy_proc_putstr)(window, attr, str)
int window;
int attr;
const char *str;
{
    nhproxy_proc_putstr(window, attr, str);
}

void
(nhproxy_proc_display_file)(fh)
int fh;
{
    nhproxy_proc_display_file(fh);
}

void
(nhproxy_proc_start_menu)(window)
int window;
{
    nhproxy_proc_start_menu(window);
}

void
(nhproxy_proc_add_menu)(window, glyph, identifier, ch, gch, attr, str,
  preselected)
int window, glyph, identifier, attr;
char ch, gch;
const char *str;
nhproxy_bool_t preselected;
{
    nhproxy_proc_add_menu(window, glyph, identifier, ch, gch, attr, str,
      preselected);
}

void
(nhproxy_proc_end_menu)(window, prompt)
int window;
const char *prompt;
{
    nhproxy_proc_end_menu(window, prompt);
}

int
nhproxy_proc_select_menu(window, how, menu_list)
int window, how;
struct nhproxy_mi **menu_list;
{
    struct nhproxy_select_menu_res ret = {0, 0, (struct nhproxy_mi *)0};
    if (!nhproxy_rpc(NHPROXY_EXT_FID_SELECT_MENU,
      2, NHPROXY_INT(window), NHPROXY_INT(how),
      1, NHPROXY_XDRF(nhproxy_xdr_select_menu_res, &ret))) {
	if (ret.selected)
	    nhproxy_xdr_free(nhproxy_xdr_select_menu_res, (char *)&ret);
	*menu_list = (struct nhproxy_mi *)0;
	return 0;	/* Nothing selected */
    }
    *menu_list = ret.selected;
    return ret.retval;
}

char
nhproxy_proc_message_menu(let, how, mesg)
char let;
int how;
const char *mesg;
{
    int ret;
    if (!nhproxy_rpc(NHPROXY_EXT_FID_MESSAGE_MENU, 3, NHPROXY_INT(let),
      NHPROXY_INT(how), NHPROXY_STRING(mesg), 1, NHPROXY_INT_PTR(ret)))
	ret = -1;
    return (char)ret;
}

void
(nhproxy_proc_update_inventory)()
{
    nhproxy_proc_update_inventory();
}

void
(nhproxy_proc_mark_synch)()
{
    nhproxy_proc_mark_synch();
}

void
(nhproxy_proc_wait_synch)()
{
    nhproxy_proc_wait_synch();
}

void
(nhproxy_proc_cliparound)(x, y)
int x, y;
{
    nhproxy_proc_cliparound(x, y);
}

void
(nhproxy_proc_update_positionbar)(posbar)
char *posbar;
{
    nhproxy_proc_update_positionbar(posbar);
}

void
nhproxy_serv_print_layer(w, x, y, z, glyph)
int w, x, y, z, glyph;
{
    struct gbuf_layer *l = proxy_mapwins[w].gbuf_layers + z;
    if (l->rows[y].glyphs[x] != glyph) {
	l->rows[y].glyphs[x] = glyph;
	if (l->start > y)
	    l->start = y;
	if (l->stop < y)
	    l->stop = y;
	if (l->rows[y].start > x)
	    l->rows[y].start = x;
	if (l->rows[y].stop < x)
	    l->rows[y].stop = x;
    }
}

nhproxy_bool_t
nhproxy_serv_flush_layers(w, clearing, no_glyph)
int w, clearing, no_glyph;
{
    int x, y, z;
    struct gbuf_layer *l = proxy_mapwins[w].gbuf_layers;
    nhproxy__no_glyph = no_glyph;
    if (!clearing) {
	if (proxy_mapwins[w].c_rows != no_rows ||
	  proxy_mapwins[w].c_cols != no_cols ||
	  proxy_mapwins[w].c_layers != no_layers) {
	    if (!nhproxy__mapwin_alloc_gbuf(w, no_rows, no_cols, no_layers))
		return nhproxy_false;
	    (void)nhproxy_rpc(NHPROXY_EXT_FID_CLEAR_NHWINDOW, 4,
	      NHPROXY_INT(proxy_mapwins[w].id), NHPROXY_INT(no_rows),
	      NHPROXY_INT(no_cols), NHPROXY_INT(no_layers), 0);
	}
	if (!proxy_print_glyph_layered(proxy_mapwins[w].id,
	  no_layers, proxy_mapwins[w].gbuf_layers))
	    return nhproxy_false;
    }
    for(z = 0; z < no_layers; z++) {
	l->stop = 0;
	l->start = no_rows - 1;
	for(y = 0; y < no_rows; y++) {
	    l->rows[y].stop = 0;
	    l->rows[y].start = no_cols - 1;
	    if (clearing)
		for(x = 0; x < no_cols; x++)
		    l->rows[y].glyphs[x] = no_glyph;
	}
	l++;
    }
    return nhproxy_true;
}

void
(nhproxy_proc_print_glyph)(window, x, y, glyph)
int window, x, y, glyph;
{
    nhproxy_proc_print_glyph(window, x, y, glyph);
}

int
nhproxy_proc_raw_print(str)
const char *str;
{
    int retval;
    static int active = 0;
    if (active || in_proxy_init)
	return nhproxy_false;
    active++;
    retval = nhproxy_rpc(NHPROXY_EXT_FID_RAW_PRINT, 1, NHPROXY_STRING(str), 0);
    active--;
    return retval;
}

int
nhproxy_proc_raw_print_bold(str)
const char *str;
{
    int retval;
    static int active = 0;
    if (active || in_proxy_init)
	return nhproxy_false;
    active++;
    retval = nhproxy_rpc(NHPROXY_EXT_FID_RAW_PRINT_BOLD, 1,
      NHPROXY_STRING(str), 0);
    active--;
    return retval;
}

int
nhproxy_proc_nhgetch()
{
    int ret;
    if (!nhproxy_rpc(NHPROXY_EXT_FID_NHGETCH, 0, 1, NHPROXY_INT_PTR(ret)))
	ret = 0;
    return ret;
}

int
nhproxy_proc_nh_poskey(x, y, mod)
int *x, *y, *mod;
{
    int ret, lx, ly, lmod;
    if (!nhproxy_rpc(NHPROXY_EXT_FID_NH_POSKEY, 0, 4, NHPROXY_INT_PTR(ret),
      NHPROXY_INT_PTR(lx), NHPROXY_INT_PTR(ly), NHPROXY_INT_PTR(lmod)))
	return nhproxy_proc_nhgetch();
    *x = lx;
    *y = ly;
    *mod = lmod;
    return ret;
}

void
(nhproxy_proc_nhbell)()
{
    nhproxy_proc_nhbell();
}

int
nhproxy_proc_doprev_message()
{
    int ret;
    if (!nhproxy_rpc(NHPROXY_EXT_FID_DOPREV_MESSAGE, 0, 1,
      NHPROXY_INT_PTR(ret)))
	ret = 0;
    return ret;
}

char
nhproxy_proc_yn_function(query, resp, def, countp)
const char *query, *resp;
char def;
int *countp;
{
    int ret, count;
    if (!nhproxy_rpc(NHPROXY_EXT_FID_YN_FUNCTION,
      3, NHPROXY_STRING(query), NHPROXY_STRING(resp), NHPROXY_INT(def),
      2, NHPROXY_INT_PTR(ret), NHPROXY_INT_PTR(count)))
	ret = def;
    if (ret == '#')
	*countp = count;
    return ret;
}

char *
nhproxy_proc_getlin(query)
const char *query;
{
    char *reply = (char *)0;
    (void)nhproxy_rpc(NHPROXY_EXT_FID_GETLIN, 1, NHPROXY_STRING(query), 1,
      NHPROXY_STRING_PTR(reply));
    return reply;
}

int
nhproxy_proc_get_ext_cmd()
{
    int extcmd;
    if (!nhproxy_rpc(NHPROXY_EXT_FID_GET_EXT_CMD, 0, 1,
      NHPROXY_INT_PTR(extcmd)))
	extcmd = -1;
    return extcmd;
}

void
(nhproxy_proc_number_pad)(state)
int state;
{
    nhproxy_proc_number_pad(state);
}

void
(nhproxy_proc_delay_output)()
{
    nhproxy_proc_delay_output();
}

void
(nhproxy_proc_change_color)(color, rgb, reverse)
int color;
long rgb;
int reverse;
{
    nhproxy_proc_change_color(color, rgb, reverse);
}

void
(nhproxy_proc_change_background)(white_or_black)
int white_or_black;
{
    nhproxy_proc_change_background(white_or_black);
}

short
nhproxy_proc_set_font_name(window, font)
int window;
char *font;
{
    int ret;
    if (!nhproxy_rpc(NHPROXY_EXT_FID_SET_FONT_NAME, 2, NHPROXY_INT(window),
      NHPROXY_STRING(font), 1, NHPROXY_INT_PTR(ret)))
	ret = -1;
    return (short)ret;
}

char *
nhproxy_proc_get_color_string()
{
    char *ret = (char *)0;
    if (!nhproxy_rpc(NHPROXY_EXT_FID_GET_COLOR_STRING, 0, 1,
      NHPROXY_STRING_PTR(ret))) {
	free(ret);
	return "";
    }
    return ret;
}

void
(nhproxy_proc_start_screen)()
{
    nhproxy_proc_start_screen();
}

void
(nhproxy_proc_end_screen)()
{
    nhproxy_proc_end_screen();
}

nhproxy_bool_t
nhproxy_proc_outrip(window, killed_by)
int window;
const char *killed_by;
{
    nhproxy_bool_t handled;
    if (!nhproxy_rpc(NHPROXY_EXT_FID_OUTRIP, 2, NHPROXY_INT(window),
      NHPROXY_STRING(killed_by), 1, NHPROXY_BOOLEAN_PTR(handled)))
	handled = nhproxy_false;
    return handled;
}

void
(nhproxy_proc_preference_update)(pref, value)
const char *pref, *value;
{
    nhproxy_proc_preference_update(pref, value);
}

void
nhproxy_proc_status(reconfig, nv, values)
int reconfig, nv;
const char **values;
{
    struct nhproxy_status_req req;
    req.reconfig = reconfig;
    req.nv = nv;
    req.values = values;
    (void)nhproxy_rpc(NHPROXY_EXT_FID_STATUS, 1,
      NHPROXY_XDRF(nhproxy_xdr_status_req, &req), 0);
}

static nhproxy_bool_t
proxy_print_glyph_layered(window, nl, layers)
int window;
int nl;
struct gbuf_layer *layers;
{
    int i, j, k;
    struct nhproxy_print_glyph_layered_req req;
    req.window = window;
    req.nl = nl;
    req.layers = (struct nhproxy_glyph_layer *)malloc(nl *
      sizeof(struct nhproxy_glyph_layer));
    if (!req.layers)
	return nhproxy_false;
    for(i = 0; i < nl; i++) {
	if (layers[i].stop - layers[i].start >= 0) {
	    req.layers[i].start = layers[i].start;
	    req.layers[i].nr = layers[i].stop - layers[i].start + 1;
	    req.layers[i].rows = (struct nhproxy_glyph_row *)
	      malloc(req.layers[i].nr * sizeof(struct nhproxy_glyph_row));
	    if (!req.layers[i].rows) {
		for(i--; i >= 0; i--)
		    free(req.layers[i].rows);
		free(req.layers);
		return nhproxy_false;
	    }
	    for(j = 0, k = layers[i].start; j < req.layers[i].nr; j++, k++)
		if (layers[i].rows[k].stop - layers[i].rows[k].start >= 0) {
		    req.layers[i].rows[j].start = layers[i].rows[k].start;
		    req.layers[i].rows[j].ng =
		      layers[i].rows[k].stop - layers[i].rows[k].start + 1;
		    req.layers[i].rows[j].glyphs =
		      layers[i].rows[k].glyphs + layers[i].rows[k].start;
		} else {
		    req.layers[i].rows[j].start = 0;
		    req.layers[i].rows[j].ng = 0;
		    req.layers[i].rows[j].glyphs = (int *)0;
		}
	} else {
	    req.layers[i].start = 0;
	    req.layers[i].nr = 0;
	    req.layers[i].rows = (struct nhproxy_glyph_row *)0;
	}
    }
    (void)nhproxy_rpc(NHPROXY_EXT_FID_PRINT_GLYPH_LAYERED,
      1, NHPROXY_XDRF(nhproxy_xdr_print_glyph_layered_req, &req), 0);
    for(i = 0; i < nl; i++)
	if (req.layers[i].rows)
	    free(req.layers[i].rows);
    free(req.layers);
    return nhproxy_true;
}

FILE *nhproxy__config_fp = NULL;

FILE *
nhproxy_serv_config_file_open()
{
    nhproxy__config_fp = tmpfile();
    if (nhproxy__config_fp) {
	/* Since this is currently the only use of writable dlbh streams,
	 * we simply use a hard-coded handle of zero.
	 */
	(void)nhproxy_rpc(NHPROXY_EXT_FID_SEND_CONFIG_FILE, 1, NHPROXY_INT(0),
	  0);
	rewind(nhproxy__config_fp);		/* Ready to read */
    }
    return nhproxy__config_fp;
}

void
nhproxy_serv_config_file_close(FILE *fp)
{
    fclose(nhproxy__config_fp);
    nhproxy__config_fp = NULL;
}

extern struct nhproxy_rpc_services nhproxy__callbacks[];

#ifdef DEBUG
static int NHPROXY_FDECL(debug_read, (void *, void *, unsigned int));
static int NHPROXY_FDECL(debug_write, (void *, void *, unsigned int));

#define READ_F	debug_read
#define WRITE_F	debug_write
#else
#define READ_F	proxy_read
#define WRITE_F	proxy_write
#endif

#ifdef MSWIN_API
#define READ_H	((void *)_get_osfhandle(0))
#define WRITE_H	((void *)_get_osfhandle(1))
#else
#define READ_H	((void *)0)
#define WRITE_H	((void *)1)
#endif

#ifdef MSWIN_API
int proxy_read(handle, buf, len)
void *handle;
void *buf;
int len;
{
    DWORD d;
    if (!ReadFile((HANDLE)handle, buf, len, &d, NULL)) {
	d = GetLastError();
	return d == ERROR_HANDLE_EOF || d == ERROR_BROKEN_PIPE ? 0 : -1;
    } else
	return d;
}

int proxy_pipe_read_nb(handle, buf, len)
void *handle;
void *buf;
int len;
{
    DWORD nb;
    if (!PeekNamedPipe((HANDLE)handle, NULL, 0, NULL, &nb, NULL))
	return -1;
    if (!nb)
	return -2;
    return READ_F(handle, buf, len);
}

int proxy_write(handle, buf, len)
void *handle;
void *buf;
int len;
{
    DWORD nb;
    if (!WriteFile((HANDLE)handle, buf, len, &nb, NULL))
	return -1;
    else
	return nb;
}
#else	/* MSWIN_API */
static int
proxy_read(handle, buf, len)
void *handle;
void *buf;
unsigned int len;
{
    int fd = (int)handle, nb;
    nb = read(fd, buf, len);
    return nb >= 0 ? nb : -1;
}

#ifdef HAVE_SELECT
static int
proxy_read_nb(handle, buf, len)
void *handle;
void *buf;
int len;
{
    int fd = (int)handle, retval;
    fd_set readfds;
    struct timeval tv = {0, 0};
    FD_ZERO(&readfds);
    FD_SET(fd, &readfds);
    retval = select(fd + 1, &readfds, NULL, NULL, &tv);
    if (retval < 0)
	return -1;
    if (!retval)
	return -2;
    return READ_F(handle, buf, len);
}
#endif

static int
proxy_write(handle, buf, len)
void *handle;
void *buf;
unsigned int len;
{
    int fd = (int)handle, nb;
    fd_set fds;
    struct timeval tv;
    FD_ZERO(&fds);
    FD_SET(fd,&fds);
    tv.tv_sec = 30;	/* Don't block forever but allow for slow connections */
    tv.tv_usec = 0;
    if (select(fd+1, NULL, &fds, NULL, &tv)) {
	nb = write(fd, buf, len);
	return nb >= 0 ? nb : -1;
    } else
	return -1;
}
#endif	/* MSWIN_API */

#define BUFSZ	256

struct proxy_auth_connection {
    int sin, sout, serr, pid;
    char error[BUFSZ + 1];
};

static void
proxy_auth_emit_error(struct proxy_auth_connection *auth)
{
    /*
     * If auth->error is empty then proxy_auth_verify was never called,
     * which implies that the remote end probably doesn't support
     * an authentication method which we allow.
     */
    const char *message = *auth->error ?
      auth->error : "Please upgrade your client.";
    int w;
    int argc = 1;
    char *argv[2];
    /*
     * We are called before init_nhwindows,
     * but a window is much more user friendly.
     */
    argv[0] = "hack";
    argv[1] = NULL;
    nhproxy_proc_init_nhwindows(&argc, argv);
    w = nhproxy_proc_create_nhwindow(NHPROXY_EXT_NHW_MENU);
    nhproxy_proc_putstr(w, NHPROXY_EXT_ATR_BOLD, "Authentication failed.");
    nhproxy_proc_putstr(w, NHPROXY_EXT_ATR_NONE, "");
    nhproxy_proc_putstr(w, NHPROXY_EXT_ATR_NONE, message);
    nhproxy_proc_display_nhwindow(w, nhproxy_true);
    nhproxy_proc_destroy_nhwindow(w);
    nhproxy_proc_exit_nhwindows(NULL);
}

static struct authentication {
    char prog[BUFSZ];
    char args[BUFSZ];
} authentication = { "", "" };

nhproxy_bool_t
nhproxy_serv_may_change_name()
{
    return !!getenv("HACKAUTHENTICATION") || nhproxy__authorized == 2;
}

static nhproxy_bool_t
match_optname(opt, len, name)
const char *opt, *name;
int len;
{
    int nlen = strlen(name);
    return len >= nlen && !strncmp(opt, name, nlen) && (len == nlen ||
      opt[nlen] == ':' || opt[nlen] == '=' || isspace((int)opt[nlen]));
}

static nhproxy_bool_t
parseauthopt(opts, len)
const char *opts;
int len;
{
    int i;

    while (isspace((int)*opts))
	opts++, len--;
    while (len && isspace((int)opts[len - 1]))
	len--;

    if (match_optname(opts, len, "prog")) {
	for (i = 4; i < len && isspace((int)opts[i]); i++)
	    ;
	if (i < len && (opts[i] == ':' || opts[i] == '=')) {
	    if (len - i - 1 >= sizeof(authentication.prog)) {
		nhproxy_error("HACKAUTHENTICATION: prog value too long");
		return nhproxy_false;
	    }
	    strncpy(authentication.prog, opts + i + 1, len - i - 1);
	    authentication.prog[len - i - 1] = '\0';
	} else {
	    nhproxy_error("HACKAUTHENTICATION: prog requires a value");
	    return nhproxy_false;
	}
    } else if (match_optname(opts, len, "args")) {
	for (i = 4; i < len && isspace((int)opts[i]); i++)
	    ;
	if (i < len && (opts[i] == ':' || opts[i] == '=')) {
	    if (len - i - 1 >= sizeof(authentication.args)) {
		nhproxy_error("HACKAUTHENTICATION: args value too long");
		return nhproxy_false;
	    }
	    strncpy(authentication.args, opts + i + 1, len - i - 1);
	    authentication.args[len - i - 1] = '\0';
	} else {
	    nhproxy_error("HACKAUTHENTICATION: args requires a value");
	    return nhproxy_false;
	}
    } else {
	nhproxy_error("HACKAUTHENTICATION: unknown option: %s", opts);
	return nhproxy_false;
    }
    return nhproxy_true;
}

static void
parseauthentication(opts)
const char *opts;
{
    char *op;

    /* Initial values */
    authentication.prog[0] = '\0';
    authentication.args[0] = '\0';

    while ((op = strchr(opts, ','))) {
	if (!parseauthopt(opts, op - opts))
	    return;
	opts = op + 1;
    }
    if (!parseauthopt(opts))
	return;

    if (!authentication.prog[0] && authentication.args[0]) {
	nhproxy_error(
	  "HACKAUTHENTICATION: Arguments given but no program specified.");
	authentication.args[0] = 0;
    }
}

/*
 * Returns non-NULL if helper program opened.
 */
 
static struct proxy_auth_connection *
proxy_auth_open()
{
    struct proxy_auth_connection *pac;
    char *authopts = getenv("HACKAUTHENTICATION");
#ifdef MSWIN_API
    if (authopts) {
	pac = (struct proxy_auth_connection *)malloc(sizeof(*pac));
	if (pac) {
	    pac->sin = pac->sout = pac->serr = pac->pid = -1;
	    strcpy(pac->error, "Authentication not supported under win32.");
	}
	return pac;
    } else
	return NULL;
#else
    int s_in[2], s_out[2], s_err[2], r, status;
    char buf[BUFSZ + 1];
    if (authopts)
	parseauthentication(authopts);
    if (authentication.prog[0]) {
	pac = (struct proxy_auth_connection *)malloc(sizeof(*pac));
	if (!pac)
	    return (struct proxy_auth_connection *)0;
	pac->sin = pac->sout = pac->serr = pac->pid = -1;
	pac->error[0] = '\0';
	if (pipe(s_in) || pipe(s_out) || pipe(s_err)) {
	    strcpy(pac->error, "Can't create pipes to helper program.");
	    return pac;
	}
	if (!(pac->pid = fork())) {
	    int i, j, argc = 1;
	    char **argv;
	    char args[BUFSZ];
	    (void)setgid(getgid());
	    (void)setuid(getuid());
	    dup2(s_in[0], 0);
	    dup2(s_out[1], 1);
	    dup2(s_err[1], 2);
	    close(s_in[0]);
	    close(s_in[1]);
	    close(s_out[0]);
	    close(s_out[1]);
	    close(s_err[0]);
	    close(s_err[1]);
	    strcpy(args, authentication.args);
	    for(i = 0; args[i]; i++)
		if (args[i] == ' ' && i && args[i - 1] != ' ')
		    argc++;
	    argv = (char **) malloc((argc + 1) * sizeof(char *));
	    if (!argv)
	    {
		fprintf(stderr, "Resource failure\n");
		exit(1);
	    }
	    argv[0] = authentication.prog;
	    j = 1;
	    argv[j] = args;
	    for(i = 0; args[i]; i++)
		if (args[i] == ' ' && i) {
		    args[i] = '\0';
		    while(args[i + 1] == ' ')
			i++;
		    argv[++j] = args + i + 1;
		}
	    argv[++j] = NULL;
	    execvp(authentication.prog, argv);
	    perror(authentication.prog);
	    exit(1);
	}
	close(s_in[0]);
	close(s_out[1]);
	close(s_err[1]);
	pac->sin = s_in[1];
	pac->sout = s_out[0];
	pac->serr = s_err[0];
	r = read(pac->sout, buf, BUFSZ);
	if (r < 0) {
	    strcpy(pac->error, "Error reading from helper program.");
	    return pac;
	}
	if (r == 0) {
	    /*
	     * Authentication program has failed. If it has output an
	     * error on stardard error then report this.
	     */
	    r = read(pac->serr, buf, BUFSZ);
	    if (r > 0) {
		if (buf[r - 1] == '\n')
		    r--;
		buf[r] = '\0';
		strcpy(pac->error, buf);
	    } else {
		waitpid(pac->pid, &status, 0);
		if (!WIFEXITED(status)) {
		    if (WIFSIGNALED(status))
			sprintf(pac->error, "Helper program died (signal %d).",
			  WTERMSIG(status));
		    else
			strcpy(pac->error, "Helper program died.");
		} else if (WEXITSTATUS(status)) {
		    r = read(pac->serr, buf, BUFSZ);
		    if (r > 0) {
			if (buf[r - 1] == '\n')
			    r--;
			buf[r] = '\0';
			strcpy(pac->error, buf);
		    } else
			sprintf(pac->error, "Helper program exited (error %d).",
			  WEXITSTATUS(status));
		}
		else
		    sprintf(pac->error, "Helper program exited.");
	    }
	}
	/* TODO: Do something with the challenge string if not empty */
	return pac;
    }
    else
	return NULL;
#endif	/* MSWIN_API */
}

static int
proxy_auth_verify(struct proxy_auth_connection *pac,
  const char *name, const char *response)
{
#ifndef MSWIN_API
    int r, status, verified = 0;
    char buf[BUFSZ + 1];
    if (*pac->error)
	return 0;
    strcpy(buf, name);
    strcat(buf, "\n");
    write(pac->sin, buf, strlen(buf));
    strcpy(buf, response);
    strcat(buf, "\n");
    write(pac->sin, buf, strlen(buf));
    waitpid(pac->pid, &status, 0);
    pac->pid = -1;
    if (!WIFEXITED(status)) {
	if (WIFSIGNALED(status))
	    sprintf(pac->error, "Helper program died (signal %d).",
	      WTERMSIG(status));
	else
	    strcpy(pac->error, "Helper program died.");
    } else if (WEXITSTATUS(status)) {
	r = read(pac->serr, buf, BUFSZ);
	if (r > 0) {
	    if (buf[r - 1] == '\n')
		r--;
	    buf[r] = '\0';
	    strcpy(pac->error, buf);
	} else
	    strcpy(pac->error, "Helper program exited.");
    }
    else
	verified = 1;
    return verified;
#else
    return 0;
#endif	/* MSWIN_API */
}

static void
proxy_auth_close(struct proxy_auth_connection *pac)
{
#ifndef MSWIN_API
    int status;
    if (pac->sin >= 0)
	close(pac->sin);
    if (pac->sout >= 0)
	close(pac->sout);
    if (pac->serr >= 0)
	close(pac->serr);
    if (pac->pid >= 0) {
	if (!waitpid(pac->pid, &status, WNOHANG)) {
	    kill(pac->pid, 15);
	    sleep(1);
	    (void)waitpid(pac->pid, &status, WNOHANG);
	}
    }
#endif
    free(pac);
}

static char *nhproxy__windowtype = NULL;

char *
nhproxy_serv_get_windowtype()
{
    return nhproxy__windowtype;
}

static nhproxy_bool_t
proxy_init(game_name, version_string, auth)
char *game_name, *version_string;
struct proxy_auth_connection *auth;
{
    int i, j, k;
    static char *name = (char *)0;
    NhProxyIO *rd, *wr;
    struct nhproxy_rpc_services *services;
    struct nhproxy_line *lp = (struct nhproxy_line *)0, line;
    char standard[8];
#if !defined(MSWIN_API) && defined(HAVE_SELECT)
    rd = nhproxy_io_open(READ_F, READ_H, NHPROXY_IO_RDONLY);
    if (rd)
	nhproxy_io_setnbfunc(rd, proxy_read_nb);
#else
# ifdef MSWIN_API
    void *read_h = READ_H;
    if (GetFileType((HANDLE)read_h) == FILE_TYPE_PIPE) {
	rd = nhproxy_io_open(READ_F, read_h, NHPROXY_IO_RDONLY);
	if (rd)
	    nhproxy_io_setnbfunc(rd, proxy_pipe_read_nb);
    } else
# endif
    rd = nhproxy_io_open(READ_F, READ_H, NHPROXY_IO_RDONLY | NHPROXY_IO_NBLOCK);
#endif
    if (!rd) {
	nhproxy_error("Failed to open read NhProxyIO stream");
	return nhproxy_false;
    }
    wr = nhproxy_io_open(WRITE_F, WRITE_H, NHPROXY_IO_WRONLY);
    if (!wr) {
	nhproxy_error("Failed to open write NhProxyIO stream");
	nhproxy_io_close(rd);
	return nhproxy_false;
    }
    if (nhproxy__extents && nhproxy__extents[0].name) {
	for(i = j = 0; nhproxy__extents[i].name; i++)
	    j += nhproxy__extents[i].no_procedures;
	for(i = 0; nhproxy__callbacks[i].id; i++)
	    ;
	j += i;
	services = (struct nhproxy_rpc_services *)
	  malloc(j * sizeof(struct nhproxy_rpc_services));
	if (!services) {
	    nhproxy_error("Not enough memory");
	    nhproxy_io_close(wr);
	    return nhproxy_false;
	}
	for(i = j = 0; nhproxy__extents[i].name; i++) {
	    (*nhproxy__extents[i].init)(0x8000 + j);
	    for(k = 0; k < nhproxy__extents[i].no_procedures; k++, j++) {
		services[j].id = 0x8000 + j;
		services[j].handler = nhproxy__extents[i].handler;
	    }
	}
	for(i = 0; nhproxy__callbacks[i].id; i++)
	    services[j + i] = nhproxy__callbacks[i];
    } else
	services = nhproxy__callbacks;
    if (!nhproxy_rpc_init(rd, wr, services)) {
	nhproxy_error("Failed to initialize nhproxy");
	nhproxy_io_close(rd);
	nhproxy_io_close(wr);
	return nhproxy_false;
    }
    line.type = "NhExt";
    line.n = auth ? 5 : 4;
    line.tags = (char **)malloc(line.n * sizeof(char *));
    line.values = (char **)malloc(line.n * sizeof(char *));
    if (!line.tags || !line.values) {
	nhproxy_error("Not enough memory");
	free(line.tags);
	free(line.values);
	nhproxy_io_close(rd);
	nhproxy_io_close(wr);
	return nhproxy_false;
    }
    line.tags[0] = "standard";
    (void)sprintf(standard, "%d.%d", NHPROXY_EXT_STANDARD_MAJOR,
      NHPROXY_EXT_STANDARD_MINOR);
    line.values[0] = standard;
    line.tags[1] = "game";
    line.values[1] = game_name;
    line.tags[2] = "version";
    line.values[2] = version_string;
    line.tags[3] = "protocols";
    line.values[3] = "1,2";
    if (auth) {
	line.tags[4] = "authmethods";
	line.values[4] = "1";
    }
    i = nhproxy_subprotocol0_write_line(&line);
    free(line.tags);
    free(line.values);
    if (!i) {
	nhproxy_error("Failed to write NhExt greeting");
failed:
	if (lp)
	    nhproxy_subprotocol0_free_line(lp);
	nhproxy_rpc_end();
	nhproxy_io_close(rd);
	nhproxy_io_close(wr);
	return nhproxy_false;
    }
    lp = nhproxy_subprotocol0_read_line();
    if (!lp) {
	nhproxy_error("Failed to read reply to NhExt greeting");
	goto failed;
    }
    if (strcmp(lp->type,"Ack")) {
	nhproxy_error("NhExt greeting not acknowledged");
	goto failed;
    }
    for(i = 0; i < lp->n; i++)
	if (!strcmp(lp->tags[i], "protocol")) {
	    if (!strcmp(lp->values[i], "1"))
		proxy_protocol = 1;
	    else if (!strcmp(lp->values[i], "2"))
		proxy_protocol = 2;
	    else {
		nhproxy_error("Illegal sub-protocol \"%s\"", lp->values[i]);
		goto failed;
	    }
	    break;
	}
    if (i == lp->n) {
	nhproxy_error("Missing protocol in acknowledgment");
	goto failed;
    }
    if (auth) {
	int method = 0;
	char *user = NULL;
	char *passwd = NULL;
	for(i = 0; i < lp->n; i++) {
	    if (!strcmp(lp->tags[i], "authmethod")) {
		if (!strcmp(lp->values[i], "1"))
		    method = 1;
	    } else if (!strcmp(lp->tags[i], "username"))
		user = lp->values[i];
	    else if (!strcmp(lp->tags[i], "password"))
		passwd = lp->values[i];
	}
	if (method == 1) {
	    nhproxy__authorized = proxy_auth_verify(auth, user, passwd);
	    if (nhproxy__authorized) {
		char *opts = malloc(strlen(user) + 6);
		if (opts) {
		    sprintf(opts, "name:%s", user);
		    nhproxy__authorized = 2;
		    nhproxy__serv->parse_options(opts);
		    nhproxy__authorized = 1;
		    free(opts);
		} else {
		    nhproxy_error("Failed to set user name");
		    goto failed;
		}
	    }
	}
    } else
	nhproxy__authorized = 1;
    nhproxy__windowtype = NULL;
    for(i = 0; i < lp->n; i++)
	if (!strcmp(lp->tags[i], "windowtype"))
	{
	    nhproxy__windowtype = lp->values[i];
	    break;
	}
    if (name)
	free(name);
    nhproxy_rpc_set_protocol(proxy_protocol);
    return nhproxy_true;
} 

#undef READ_F
#undef WRITE_F
#undef READ_H
#undef WRITE_H

#ifdef DEBUG
static void
debug_dump(buf, len, arrow)
void *buf;
unsigned int len;
char *arrow;
{
    int i, j, nc;
    long l;
    char cbuf[17];
    unsigned char *bp = buf;
    for(i = 0; i < len; ) {
	if ((i & 15) == 0) {
	    if (!i)
		fputs(arrow, stderr);
	    else {
		cbuf[16] = '\0';
		while(nc++ < 40)
		    fputc(' ', stderr);
		fputs(cbuf, stderr);
		fputs("\n  ", stderr);
	    }
	    nc = 2;
	}
	if (len - i >= 4) {
	    l = (long)bp[i] << 24 | (long)bp[i + 1] << 16 |
	      (long)bp[i + 2] << 8 | bp[i + 3];
	    fprintf(stderr, " %08X", l);
	    nc += 9;
	    for(j = 0; j < 4; j++, i++)
		cbuf[i & 15] = isgraph(bp[i]) || bp[i] == ' ' ?  bp[i] : '.';
	}
	else {
	    fprintf(stderr, " %02X", bp[i]);
	    nc += 3;
	    cbuf[i & 15] = isgraph(bp[i]) || bp[i] == ' ' ? bp[i] : '.';
	    i++;
	}
    }
    if (len) {
	cbuf[i & 15 ? i & 15 : 16] = '\0';
	while(nc++ < 40)
	    fputc(' ', stderr);
	fputs(cbuf, stderr);
    }
    fputc('\n', stderr);
}

static int
debug_read(handle, buf, len)
void *handle;
void *buf;
unsigned int len;
{
    int retval;
    retval = proxy_read(handle, buf, len);
    if (retval == -2)
	fputs("<- PENDING\n", stderr);
    else if (retval < 0)
	fputs("<- ERROR\n", stderr);
    else if (!retval)
	fputs("<- EOF\n", stderr);
    else
	debug_dump(buf, retval, "<-");
    return retval;
}

static int
debug_write(handle, buf, len)
void *handle;
void *buf;
unsigned int len;
{
    int retval;
    retval = proxy_write(handle, buf, len);
    if (retval < 0)
	fputs("-> ERROR\n", stderr);
    else
	debug_dump(buf, retval, "->");
    return retval;
}
#endif	/* DEBUG */

static unsigned long async_callbacks[] = {
    1 << NHPROXY_EXT_CID_DISPLAY_INVENTORY - NHPROXY_EXT_CID_DISPLAY_INVENTORY |
      1 << NHPROXY_EXT_CID_DOREDRAW - NHPROXY_EXT_CID_DISPLAY_INVENTORY |
      1 << NHPROXY_EXT_CID_INTERFACE_MODE - NHPROXY_EXT_CID_DISPLAY_INVENTORY |
      1 << NHPROXY_EXT_CID_SET_OPTION_MOD_STATUS -
      NHPROXY_EXT_CID_DISPLAY_INVENTORY,
};

void
nhproxy_serv_set_extensions(struct nhproxy_extension *extents)
{
    nhproxy__extents = extents;
}

void
nhproxy_serv_set_callbacks(struct nhproxy_serv_callbacks *callbacks)
{
    nhproxy__serv = callbacks;
}

nhproxy_bool_t
nhproxy_serv_accept(game_name, version_string)
char *game_name, *version_string;
{
    int retval;
    struct nhproxy_cb_subprot2_init request, result;
    struct proxy_auth_connection *auth;
    in_proxy_init = TRUE;
    auth = proxy_auth_open();
    if (!proxy_init(game_name, version_string, auth)) {
	nhproxy_error("Failed to initialize");
	return nhproxy_false;
    }
    if (proxy_protocol == 1)
	retval = nhproxy_rpc(NHPROXY_EXT_FID_INIT, 0, 0);
    else {
	request.n_masks = SIZE(async_callbacks);
	request.masks = async_callbacks;
	result.masks = (unsigned long *)0;
	retval = nhproxy_rpc(NHPROXY_EXT_FID_INIT,
	  1, NHPROXY_XDRF(nhproxy_cb_xdr_subprot2_init, &request),
	  1, NHPROXY_XDRF(nhproxy_cb_xdr_subprot2_init, &result));
	if (retval) {
	    nhproxy_rpc_set_async_masks(result.n_masks, result.masks);
	    free(result.masks);
	}
    }
    if (retval) {
	/*
	 * raw_print and raw_print_bold are now useable
	 * and proxy_auth_emit_error() needs them.
	 */
	in_proxy_init = FALSE;
	if (auth) {
	    if (!nhproxy__authorized) {
		proxy_auth_emit_error(auth);
		retval = nhproxy_false;
	    }
	    proxy_auth_close(auth);
	}
    }
    return retval;
}
