/* $Id: stream.c,v 1.6 2004/12/22 23:15:04 ali Exp $
 * Copyright (C) 2002, 2003  Slash'EM Development Team
 * Copyright (C) 2004  J. Ali Harlow
 *
 * This file is part of NetHack Proxy.
 *
 * NetHack Proxy is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of the
 * License, or (at your option) any later version.
 *
 * NetHack Proxy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with NetHack Proxy; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307   
 * USA
 *
 * Alternatively (at your option) you may instead choose to redistribute
 * and/or modify NetHack Proxy under the terms of the NetHack General
 * Public License.
 *
 * You should have receieved a copy of the NetHack General Public License
 * along with NetHack Proxy; if not, download a copy from
 * http://www.nethack.org/common/license.html
 */

/* NhProxy: stream support */

#include "config.h"
#include "compat.h"
#if STDC_HEADERS
#include <stdarg.h>
#endif
#if HAVE_LIMITS_H
#include <limits.h>
#else
#define INT_MAX		(1UL << 8 * sizeof(int))	/* Upper limit */
#endif
#if ENABLE_PRINTF_FP
#if HAVE_FLOAT_H
#include <float.h>
#else
#define DEFAULT_EXP_WIDTH	3		/* A reasonable default */
#endif
#endif
#include <nhproxy/system.h>
#include <nhproxy/xdr.h>

struct NhProxyIO_ {
    unsigned int flags;
    unsigned int autofill_limit;	/* Zero for no limit */
    nhproxy_io_func func, nb_func;
    nhproxy_genericptr_t handle;
    unsigned char buffer[1025];		/* Capable of storing n-1 chars */
    unsigned char *rp, *wp;
    NhProxyNB *nb;
};

#define ADVANCE_PTR(io, ptr, inc)				\
	do {							\
	    (ptr) += (inc);					\
	    if ((ptr) >= (io)->buffer + sizeof((io)->buffer))	\
		(ptr) -= sizeof((io)->buffer);			\
	} while(0)

/* Flags changeable by nhproxy_io_setmode */

#define NHPROXY_IO__USERFLAGS	(NHPROXY_IO_NOAUTOFILL | NHPROXY_IO_LINEBUF)

/* Flags readable by nhproxy_io_getmode */

#define NHPROXY_IO__READFLAGS	(NHPROXY_IO__USERFLAGS | \
				NHPROXY_IO_NBLOCK | \
				NHPROXY_IO_RDONLY | \
				NHPROXY_IO_WRONLY)

NhProxyIO *
nhproxy_io_open(func, handle, flags)
nhproxy_io_func func;
nhproxy_genericptr_t handle;
unsigned int flags;
{
    NhProxyIO *io;
    io = malloc(sizeof(*io));
    if (!io)
	return NULL;
    io->flags = flags;
    io->autofill_limit = 0;
    io->func = func;
    io->nb_func = NULL;
    io->handle = handle;
    io->rp = io->wp = io->buffer;
    io->nb = NULL;
    return io;
}

nhproxy_bool_t
nhproxy_io_close(io)
NhProxyIO *io;
{
    nhproxy_bool_t retval;
    retval = !nhproxy_io_flush(io);
    if (io->nb)
	(void)nhproxy_nb_close(io->nb);
    free(io);
    return retval;
}

unsigned int
nhproxy_io_getmode(io)
NhProxyIO *io;
{
    return io->flags & NHPROXY_IO__READFLAGS;
}

void
nhproxy_io_setmode(io, flags)
NhProxyIO *io;
unsigned int flags;
{
    io->flags &= ~NHPROXY_IO__USERFLAGS;
    io->flags |= flags & NHPROXY_IO__USERFLAGS;
}

/* Warning: The numbering of free blocks is a little odd */ 

static void
nhproxy_io__getfree(io, nf1, nf2)
NhProxyIO *io;
int *nf1, *nf2;
{
    /*  R
     *  W
     * +--------------------+
     * |222222222222222222|G|    1 - free1, 2 - free2, D - data, G - guard
     * +--------------------+
     *
     *        R        W
     * +--------------------+
     * |111|G|DDDDDDDD|22222|
     * +--------------------+
     *
     *  R            W
     * +--------------------+
     * |DDDDDDDDDDDD|22222|G|
     * +--------------------+
     *
     *  W     R
     * +--------------------+
     * |111|G|DDDDDDDDDDDDDD|
     * +--------------------+
     *
     *      W R
     * +--------------------+
     * |DDD|G|DDDDDDDDDDDDDD|  (and variations with guard byte at start or end)
     * +--------------------+
     */
    *nf1 = io->wp >= io->rp ? io->rp - io->buffer: io->rp - io->wp;
    *nf2 = io->wp >= io->rp ? io->buffer + sizeof(io->buffer) - io->wp : 0;
    /* Guard byte goes in free1 if non-empty else in free2 */
    if (*nf1)
	(*nf1)--;
    else
	(*nf2)--;
}

/* If an autofill limit is set, then autofill will be enabled until
 * that many bytes have been read and then disabled. Set a zero limit
 * to allow an unlimited number of bytes to be read (the default).
 * Notes:
 * 1. When a limit is in place, all calls to filbuf will reduce
 *    the limit by the number of bytes read regardless of whether the
 *    call was made internally or by the application.
 * 2. If there are unread bytes in the buffer at the time this function
 *    is called then the limit will be adjusted as if they had been read
 *    after the given limit was set.
 */

void
nhproxy_io_setautofill_limit(io, limit)
NhProxyIO *io;
unsigned int limit;
{
    int nf1, nf2, nb;
    nhproxy_io__getfree(io, &nf1, &nf2);
    nb = sizeof(io->buffer) - nf1 - nf2 - 1;
    if (limit > nb) {
	io->flags &= ~NHPROXY_IO_NOAUTOFILL;
	io->autofill_limit = limit - nb;
    } else {
	io->flags |= NHPROXY_IO_NOAUTOFILL;
	io->autofill_limit = 0;
    }
}

void
nhproxy_io_setnbfunc(io, func)
NhProxyIO *io;
nhproxy_io_func func;
{
    /* If there is a read pending then delay the closure until it completes */
    if (io->nb && !(io->flags & NHPROXY_IO_PENDING)) {
	nhproxy_nb_close(io->nb);
	io->nb = NULL;
    }
    io->nb_func = func;
    io->flags |= NHPROXY_IO_NBLOCK;
}

/**
 * nhproxy_io_filbuf:
 * @io: The stream to act on.
 * @blocking: True if nhproxy_io_fulbuf() should wait for input before returning.
 *
 * Returns: >0 if buffer now non-empty,
 *          else 0 on EOF, -1 on error, or -2 on would block
 */

int
nhproxy_io_filbuf(io, blocking)
NhProxyIO *io;
nhproxy_bool_t blocking;
{
    int retval, nf1, nf2;
    if (!blocking && !(io->flags & NHPROXY_IO_NBLOCK))
	return -2;
    /*
     * NHPROXY_IO_NBLOCK is advisory only, so no error on failure
     */
    if ((io->flags & NHPROXY_IO_NBLOCK) && !io->nb && !io->nb_func) {
	io->nb = nhproxy_nb_open(io->func, io->handle);
	if (!io->nb)
	    io->flags &= ~NHPROXY_IO_NBLOCK;
    }
    if (io->rp == io->wp && !(io->flags & NHPROXY_IO_PENDING))
	io->rp = io->wp = io->buffer;
    nhproxy_io__getfree(io, &nf1, &nf2);
    if (nf2) {
	if (nf1 && !(io->flags & NHPROXY_IO_PENDING)) {
	    /* Combine free blocks so that we don't make unneccesary
	     * calls to I/O function (which may be expensive).
	     */
	    io->flags &= ~NHPROXY_IO_SIMPLEBUFFER;
	    memmove(io->buffer, io->rp, io->wp - io->rp);
	    io->wp += io->buffer - io->rp;
	    io->rp = io->buffer;
	}
	nf1 += nf2;
	/* nf2 = 0 */
    }
    if (!nf1)
	return 1;	/* Buffer is full */
    /*
     * A simple buffer is one that has the whole of the latest packet read
     * stored starting at io->buffer and ending at io->wp - 1. Such buffers
     * may be used to retrieve said packet by calling nhproxy_io_getpacket().
     */
    if (io->wp == io->buffer)
	io->flags |= NHPROXY_IO_SIMPLEBUFFER;
    else
	io->flags &= ~NHPROXY_IO_SIMPLEBUFFER;
    if (io->nb) {
	retval = nhproxy_nb_read(io->nb, io->wp, nf1, blocking);
	/*
	 * If we have a read call pending then we must remeber this so that
	 * we don't change io->wp or allow a direct read or else NhProxyNB's
	 * requirements (of keeping read call parameters constant until a
	 * read is complete) would be broken.
	 */
	if (retval == -2)
	    io->flags |= NHPROXY_IO_PENDING;
	else {
	    io->flags &= ~NHPROXY_IO_PENDING;
	    if (io->nb_func) {
		/* Closure delayed until completion of pending read */
		nhproxy_nb_close(io->nb);
		io->nb = NULL;
	    }
	}
    } else if (!blocking)
	retval = (*io->nb_func)(io->handle, io->wp, nf1);
    else {
	retval = (*io->func)(io->handle, io->wp, nf1);
	if (retval < 0)
	    retval = -1;
    }
    if (retval > 0) {
	ADVANCE_PTR(io, io->wp, retval);
	if (!(io->flags & NHPROXY_IO_NOAUTOFILL) && io->autofill_limit) {
	    if (io->autofill_limit > retval)
		io->autofill_limit -= retval;
	    else {
		io->autofill_limit = 0;
		io->flags |= NHPROXY_IO_NOAUTOFILL;
	    }
	}
    }
    return retval;
}

int
nhproxy_io_getc(io)
NhProxyIO *io;
{
    unsigned char c;
    if (io->rp == io->wp &&
      (io->flags & NHPROXY_IO_NOAUTOFILL || nhproxy_io_filbuf(io, TRUE) <= 0))
	return -1;		/* getc doesn't distinguish EOF and ERROR */
    c = *io->rp;
    ADVANCE_PTR(io, io->rp, 1);
    if (io->rp == io->wp && io->nb)
	nhproxy_io_filbuf(io, FALSE);
    return c;
}

/**
 * nhproxy_io_read:
 * @io: The stream to act on.
 * @buf: The buffer to write to.
 * @nb: The maximum number of bytes to read.
 *
 * reads between 1 and @nb bytes from the stream. A read will
 * only be requested on the underlying I/O system if there would otherwise
 * be no bytes to return to the caller (and not even then if
 * %NHPROXY_IO_NOAUTOFILL is set). This prevents blocking but means that using
 * nhproxy_io_read() to read fixed sized datums is not useful. For example,
 * <programlisting>
 *	nhproxy_io_read(io, &amp;datum, sizeof(datum));
 * </programlisting>
 * may easily only partially read datum even though the rest of datum is
 * available for reading. Where the caller knows in advance that an infinite
 * block will not occur when reading, nhproxy_io_fread() should be used instead:
 * <programlisting>
 *	nhproxy_io_fread(&amp;datum, sizeof(datum), 1, io);
 * </programlisting>
 *
 * Returns: The number of bytes read or -1 on error.
 */

int
nhproxy_io_read(io, buf, nb)
NhProxyIO *io;
char *buf;
int nb;
{
    int i;
    int retval = 0;
    /* Satisfy from read buffer (or 1st half if split) */
    if (nb && io->rp != io->wp) {
	if (io->wp > io->rp)
	    i = io->wp - io->rp;
	else
	    i = io->buffer + sizeof(io->buffer) - io->rp;
	if (i > nb)
	    i = nb;
	memcpy(buf, io->rp, i);
	ADVANCE_PTR(io, io->rp, i);
	buf += i;
	nb -= i;
	retval += i;
    }
    /* Satisfy from 2nd half of read buffer */
    if (nb && io->rp != io->wp) {
	i = io->wp - io->rp;	/* rp must point at buffer at this point */
	if (i > nb)
	    i = nb;
	memcpy(buf, io->rp, i);
	ADVANCE_PTR(io, io->rp, i);
	buf += i;
	nb -= i;
	retval += i;
    }
    /* The buffer is empty or the request is satisfied.
     * If we've read some data or we shouldn't auto-fill then we're done
     */
    if (retval || io->flags & NHPROXY_IO_NOAUTOFILL) {
	if (io->nb)
	    nhproxy_io_filbuf(io, FALSE);
	return retval;
    }
    if (nb >= sizeof(io->buffer) && !(io->flags & NHPROXY_IO_PENDING)) {
	/* If caller still wants more than we can buffer and we don't have
	 * a read call pending, read direct
	 */
	io->flags &= ~NHPROXY_IO_SIMPLEBUFFER;
	if (io->nb)
	    i = nhproxy_nb_read(io->nb, buf, nb, TRUE);
	else
	    i = (*io->func)(io->handle, buf, nb);
	if (i <= 0) {
	    /* If we have previously read some data correctly, then return
	     * this number and leave EOF/ERROR reporting until later.
	     * Otherwise report EOF as 0 and ERROR as -1
	     */
	    return retval ? retval : i < 0 ? -1 : 0;
	}
	if (io->autofill_limit) {
	    if (io->autofill_limit > i)
		io->autofill_limit -= i;
	    else {
		io->autofill_limit = 0;
		io->flags |= NHPROXY_IO_NOAUTOFILL;
	    }
	}
	buf += i;
	nb -= i;
	retval += i;
    } else if (nb) {
	/* Caller wants a fragment or we have a read call pending;
	 * fill our buffer and then satisfy from there.
	 */
	i = nhproxy_io_filbuf(io, TRUE);
	if (i <= 0)
	    return retval ? retval : i < 0 ? -1 : 0;	/* As above */
	i = io->wp - io->rp;	/* rp must point at buffer at this point */
	if (i > nb)
	    i = nb;
	memcpy(buf, io->rp, i);
	ADVANCE_PTR(io, io->rp, i);
	buf += i;
	nb -= i;
	retval += i;
    }
    return retval;
}

/**
 * nhproxy_io_fread:
 * @buffer: The buffer to read into.
 * @size: The size of each member.
 * @nmemb: The number of members to read.
 * @io: The stream to act on.
 *
 * reads up to nmemb members of the given size and returns
 * the number read. Where fewer members were read than requested then an EOF
 * or ERROR has occured. Any partially read member will still be available
 * for reading after nhproxy_io_fread() returns.
 *
 * Returns: The number of members read or a negative value if the member size
 * is too large.
 */

int
nhproxy_io_fread(buffer, size, nmemb, io)
nhproxy_genericptr_t buffer;
int size, nmemb;
NhProxyIO *io;
{
    int nb, bf;
    int nbp = 0;		/* Number of bytes read in partial read */
    int nm = nmemb;		/* Members still to read */
    if (size > sizeof(io->buffer)) {
	/*
	 * We need enough room to store a partial member so that we don't
	 * get an indeterminate result. This means that the buffer must
	 * be capable of storing (size - 1) bytes, which is equivalent to
	 * saying that the buffer size (including the guard byte) must be
	 * no smaller than the size of a member.
	 */
	return -1;
    }
    if (size <= 0)
	return -1;
    bf = INT_MAX / size;	/* Blocking factor */
    while(nm) {
	if (bf > nm)
	    bf = nm;
	nb = nhproxy_io_read(io, buffer, bf * size);
	if (nb <= 0)
	    return nmemb - nm;
	buffer += nb;
	nm -= nb / size;
	nbp = nb % size;
	while(nbp) {
	    nb = nhproxy_io_read(io, buffer, size - nbp);
	    if (nb <= 0) {
		/* EOF or ERROR after a partial read.
		 *
		 * The buffer must be empty so we can place the partially
		 * read member back into it.
		 */
		io->flags &= ~NHPROXY_IO_SIMPLEBUFFER;
		memcpy(io->buffer, buffer - nbp, nbp);
		io->rp = io->buffer;
		io->wp = io->buffer + nbp;
		return nmemb - nm;
	    }
	    nbp += nb;
	    buffer += nb;
	    if (nbp == size) {
		nm--;
		nbp = 0;
	    }
	}
    }
    return nmemb;
}

/**
 * nhproxy_io_getpacket:
 * @io: The stream to act on.
 * @nb: The address of an integer in which to write the number of bytes read.
 *
 * gets the last packet read under certain circumstances.
 * This will always work if the caller sets no-autofill mode and calls
 * nhproxy_io_fillbuf() on an empty buffer. Calls to nhproxy_io_fread() must
 * also be avoided since this can cause simple buffer mode to be cancelled
 * (nhproxy_io_fread() could be re-written to avoid this if it became
 * important).
 *
 * Returns: The packet read or NULL on error.
 */

char *
nhproxy_io_getpacket(io, nb)
NhProxyIO *io;
int *nb;
{
    if (io->flags & NHPROXY_IO_SIMPLEBUFFER) {
	*nb = io->wp - io->buffer;
	return io->buffer;
    } else
	return NULL;
}

/* Return TRUE if a read call would block */

int
nhproxy_io_willblock(io)
NhProxyIO *io;
{
    if (io->flags & NHPROXY_IO_WRONLY)
	return -1;
    if (io->rp != io->wp)
	return FALSE;				/* Buffer non-empty */
    if (io->flags & NHPROXY_IO_NBLOCK)
	nhproxy_io_filbuf(io, FALSE);
    return io->rp == io->wp;
}

nhproxy_bool_t
nhproxy_io_flush(io)
NhProxyIO *io;
{
    unsigned char *s;
    int retval, nb1, nb2;
    if (io->rp == io->wp)
	return nhproxy_true;		/* Nothing to do */
    if (!(io->flags & NHPROXY_IO_WRONLY))
	return nhproxy_true;		/* Read streams don't need flushing */
    /*
     *        W       R
     * +--------------------+
     * |11111|FFFFF|G|222222|    1 - block1, 2 - block2, F - free, G - guard
     * +--------------------+
     *
     *        R        W
     * +--------------------+
     * |FFF|G|11111111|FFFFF|
     * +--------------------+
     *
     *  R            W
     * +--------------------+
     * |111111111111|FFFFF|G|
     * +--------------------+
     *
     *  W     R
     * +--------------------+
     * |FFF|G|22222222222222|
     * +--------------------+
     *
     *      W R
     * +--------------------+
     * |111|G|22222222222222|  (and variations with guard byte at start or end)
     * +--------------------+
     */
    nb1 = io->wp >= io->rp ? io->wp - io->rp : io->wp - io->buffer;
    nb2 = io->wp >= io->rp ? 0 : io->buffer + sizeof(io->buffer) - io->rp;
    if (nb1 && nb2) {
	/* Try and combine split buffer so that we don't make unneccesary
	 * calls to I/O function (which may be expensive).
	 */
	s = malloc(nb2);	/* For copy of 2nd half */
	if (s) {
	    memcpy(s, io->buffer, nb2);
	    memmove(io->buffer, io->rp, nb1);
	    memcpy(io->buffer + nb1, s, nb2);
	    io->rp = io->buffer;
	    io->wp = io->buffer + nb1 + nb2;
	    nb1 += nb2;
	    nb2 = 0;
	    free(s);
	}
    }
    /* Write out buffer (or 1st half if split) */
    while (nb1) {
	retval = (*io->func)(io->handle, io->rp, nb1);
	if (retval > 0) {
	    ADVANCE_PTR(io, io->rp, retval);
	    nb1 -= retval;
	} else
	    return nhproxy_false;
    }
    /* Write out 2nd half */
    while (nb2) {
	retval = (*io->func)(io->handle, io->rp, nb2);
	if (retval > 0) {
	    io->rp += retval;	/* rp can't wrap while writing 2nd half */
	    nb2 -= retval;
	} else
	    return nhproxy_false;
    }
    io->rp = io->wp = io->buffer;	/* Buffer is now empty */
    return nhproxy_true;
}

int
nhproxy_io_fputc(c, io)
int c;
NhProxyIO *io;
{
    unsigned char ch, *wp = io->wp;
    ch = c;
    *wp = ch;
    ADVANCE_PTR(io, wp, 1);
    if (wp == io->rp) {
	/* Buffer was full; flush first */
	if (!nhproxy_io_flush(io))
	    return -1;
	*io->wp = ch;
	ADVANCE_PTR(io, io->wp, 1);
    }
    else
	io->wp = wp;
    if (io->flags & NHPROXY_IO_LINEBUF && ch == '\n' && !nhproxy_io_flush(io))
	return -1;
    return (int)ch;
}

int
nhproxy_io_write(io, buf, nb)
NhProxyIO *io;
char *buf;
int nb;
{
    int i;
    int retval = 0, nf1, nf2;
    /* If the buffer is not empty, then top it up first */
    if (nb && io->wp != io->rp) {
	nhproxy_io__getfree(io, &nf1, &nf2);
	/* If free2 is non-empty, then wp must point at it so store there first
	 */
	if (nb && nf2) {
	    i = nf2;
	    if (i > nb)
		i = nb;
	    memcpy(io->wp, buf, i);
	    ADVANCE_PTR(io, io->wp, i);
	    buf += i;
	    nb -= i;
	    retval += i;
	    /* nf2 -= i; */
	}
	/* If there is any excess then store in free1
	 * (which wp must now point to after filling free2)
	 */
	if (nb && nf1) {
	    i = nf1;
	    if (i > nb)
		i = nb;
	    memcpy(io->wp, buf, i);
	    ADVANCE_PTR(io, io->wp, i);
	    buf += i;
	    nb -= i;
	    retval += i;
	    /* nf1 -= i; */
	}
    }
    if (nb) {
	/* Then flush the buffer (which must be either empty (NOP) or full) */
	if (!nhproxy_io_flush(io))
	    return retval ? retval : -1;
	/* If we can't buffer the data then write it directly until we can.
	 */
	while (nb >= sizeof(io->buffer)) {
	    i = (*io->func)(io->handle, buf, nb);
	    if (i <= 0)
		return retval ? retval : i;
	    buf += i;
	    nb -= i;
	    retval += i;
	}
    }
    if (nb) {
	/* Finally, place the last fragment in the by now empty buffer */
	memcpy(io->buffer, buf, nb);
	io->wp += nb;
	retval += nb;
    }
    return retval;
}

/* A version of write that honours LINEBUF */

int
nhproxy_io_writet(io, buf, nb)
NhProxyIO *io;
char *buf;
int nb;
{
    int i, retval = 0;
    if (!(io->flags & NHPROXY_IO_LINEBUF))
	return nhproxy_io_write(io, buf, nb);
    for(i = nb - 1; i >= 0; i--)
	if (buf[i] == '\n') {
	    retval = nhproxy_io_write(io, buf, i + 1);
	    if (retval <= i || !nhproxy_io_flush(io) || retval == nb)
		return retval;
	    break;
	}
    i = nhproxy_io_write(io, buf + retval, nb - retval);
    if (i > 0)
	retval += i;
    else if (!retval)
	retval = i;
    return retval;
}

#define NHPROXY_IO_FMT_LJUST	0x00001		/* neg width */
#define NHPROXY_IO_FMT_SIGN	0x00002		/* + prefix */
#define NHPROXY_IO_FMT_ALT	0x00004		/* # prefix */
#define NHPROXY_IO_FMT_ZEROPAD	0x00008		/* 0 prefix */
#define NHPROXY_IO_FMT_WIDTH	0x00010		/* width present */
#define NHPROXY_IO_FMT_PRECISION	0x00020		/* precision present */
#define NHPROXY_IO_FMT_SHORT	0x00040		/* h prefix */
#define NHPROXY_IO_FMT_LONG	0x00080		/* l prefix */
#define NHPROXY_IO_FMT_UNSIGNED	0x00100		/* u, x, X, p or P */
#define NHPROXY_IO_FMT_HEX	0x00200		/* x, X, p or P */
#define NHPROXY_IO_FMT_PTR	0x00400		/* p or P */
#define NHPROXY_IO_FMT_CAPS	0x00800		/* Upper case variant */
#define NHPROXY_IO_FMT_NEG	0x01000		/* Value is negative */
#define NHPROXY_IO_FMT_EXP	0x02000		/* Output exponent suffix */
#define NHPROXY_IO_FMT_DONE	0x04000		/* %fmt finished */

static int
nhproxy_io_pad(io, flags, padding, pre)
NhProxyIO *io;
int flags, padding, pre;
{
    int i, nb = 0;
    int do_pad = (flags & (NHPROXY_IO_FMT_WIDTH | NHPROXY_IO_FMT_LJUST)) ==
      (pre ? NHPROXY_IO_FMT_WIDTH :
      NHPROXY_IO_FMT_WIDTH | NHPROXY_IO_FMT_LJUST);
    int zero_pad = pre && flags & NHPROXY_IO_FMT_ZEROPAD;
    if (flags & (NHPROXY_IO_FMT_SIGN | NHPROXY_IO_FMT_NEG))
	padding--;
    if ((flags & (NHPROXY_IO_FMT_HEX | NHPROXY_IO_FMT_ALT)) ==
      (NHPROXY_IO_FMT_HEX | NHPROXY_IO_FMT_ALT))
	padding -= 2;
    if (do_pad && !zero_pad) {
	for(i = 0; i < padding; i++)
	    if (nhproxy_io_fputc(' ', io) < 0)
		return -1;
	    else
		nb++;
    }
    if (pre) {
	if (flags & (NHPROXY_IO_FMT_SIGN | NHPROXY_IO_FMT_NEG)) {
	    if (nhproxy_io_fputc(flags & NHPROXY_IO_FMT_NEG ? '-' : '+', io) <
	      0)
		return -1;
	    else
		nb++;
	}
	if ((flags & (NHPROXY_IO_FMT_HEX | NHPROXY_IO_FMT_ALT)) ==
	  (NHPROXY_IO_FMT_HEX | NHPROXY_IO_FMT_ALT)) {
	    if (nhproxy_io_fputc('0', io) < 0)
		return -1;
	    if (nhproxy_io_fputc(flags & NHPROXY_IO_FMT_CAPS ? 'X' : 'x', io) <
	      0)
		return -1;
	    nb += 2;
	}
    }
    if (do_pad && zero_pad) {
	for(i = 0; i < padding; i++)
	    if (nhproxy_io_fputc('0', io) < 0)
		return -1;
	    else
		nb++;
    }
    return nb;
}

/* Warning: Always cast signed integers to long before passing to printi */

static int
nhproxy_io_printi(io, flags, width, v)
NhProxyIO *io;
int flags, width;
unsigned long v;
{
    int i, retval = 0, digit, w, padding;
    int radix = flags & NHPROXY_IO_FMT_HEX ? 16 : 10;
    static const char *lc_digits = "0123456789abcdef";
    static const char *uc_digits = "0123456789ABCDEF";
    const char *digits = flags & NHPROXY_IO_FMT_CAPS ? uc_digits : lc_digits;
    char buffer[8 * sizeof(v) / 3 + 1];		/* Recalculate if radix < 10 */
    char *p;
    if (!(flags & NHPROXY_IO_FMT_UNSIGNED) && (long)v < 0) {
	flags |= NHPROXY_IO_FMT_NEG;
	v = -(long)v;
    }
    p = buffer + sizeof(buffer);
    w = 0;
    do {
	digit = v % radix;
	v /= radix;
	*--p = digits[digit];
	w++;
    } while (v ||
      flags & NHPROXY_IO_FMT_PTR && w < sizeof(nhproxy_genericptr_t) * 2);
    padding = width - w;
    i = nhproxy_io_pad(io, flags, padding, TRUE);
    if (i < 0)
	return -1;
    else
	retval += i;
    if (nhproxy_io_write(io, p, w) != w)
	return -1;
    else
	retval += w;
    i = nhproxy_io_pad(io, flags, padding, FALSE);
    if (i < 0)
	return -1;
    else
	retval += i;
    return retval;
}

/*
 * A cut down version of printf. Not all ANSI C features are implemented.
 * Use with caution.
 */

int
nhproxy_io_vprintf(io, fmt, ap)
NhProxyIO *io;
char *fmt;
va_list ap;
{
    int i, j, nb = 0;
    unsigned long ul;
    char *s;
    int flags, width, precision;
#if ENABLE_PRINTF_FP
    double d;
    int padding, expon, dp, sgn;
#ifdef DEFAULT_EXP_WIDTH
    static int exp_width = DEFAULT_EXP_WIDTH;
#else
    static int exp_width = 0;
    if (!exp_width) {
	i = DBL_MAX_10_EXP;
	while (i) {
	    i /= 10;
	    exp_width++;
	}
	exp_width++;
    }
#endif
#endif	/* ENABLE_PRINTF_FP */
    while(*fmt) {
	for(i = 0; fmt[i] && fmt[i] != '%'; i++)
	    ;
	if (i) {
	    if (nhproxy_io_writet(io, fmt, i) != i)
		return -1;
	    else
		nb += i;
	    fmt += i;
	}
	if (!*fmt)
	    break;
	flags = 0;
	while (!(flags & NHPROXY_IO_FMT_DONE) && *++fmt) {
	    switch(*fmt) {
		case '-':
		    flags |= NHPROXY_IO_FMT_LJUST;
		    break;
		case '+':
		    flags |= NHPROXY_IO_FMT_SIGN;
		    break;
		case '#':
		    flags |= NHPROXY_IO_FMT_ALT;
		    break;
		case '.':
		    flags |= NHPROXY_IO_FMT_PRECISION;
		    precision = 0;
		    break;
		case '*':
		    if (flags & NHPROXY_IO_FMT_PRECISION)
			precision = va_arg(ap, int);
		    else {
			flags |= NHPROXY_IO_FMT_WIDTH;
			width = va_arg(ap, int);
			if (width < 0) {
			    flags |= NHPROXY_IO_FMT_LJUST;
			    width = -width;
			}
		    }
		    break;
		case '0':
		    if (flags & NHPROXY_IO_FMT_PRECISION)
			precision *= 10;
		    else if (flags & NHPROXY_IO_FMT_WIDTH)
			width *= 10;
		    else
			flags |= NHPROXY_IO_FMT_ZEROPAD;
		    break;
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
		    if (flags & NHPROXY_IO_FMT_PRECISION) {
			precision *= 10;
			precision += *fmt - '0';
		    } else if (flags & NHPROXY_IO_FMT_WIDTH) {
			width *= 10;
			width += *fmt - '0';
		    } else {
			flags |= NHPROXY_IO_FMT_WIDTH;
			width = *fmt - '0';
		    }
		    break;
		case 'h':
		    flags |= NHPROXY_IO_FMT_SHORT;
		    flags &= ~NHPROXY_IO_FMT_LONG;
		    break;
		case 'l':
		    flags |= NHPROXY_IO_FMT_LONG;
		    flags &= ~NHPROXY_IO_FMT_SHORT;
		    break;
		case 'c':
		    i = nhproxy_io_pad(io, flags, width - 1, TRUE);
		    if (i < 0)
			return -1;
		    else
			nb += i;
		    if (nhproxy_io_fputc((char)va_arg(ap, int), io) < 0)
			return -1;
		    else
			nb++;
		    i = nhproxy_io_pad(io, flags, width - 1, FALSE);
		    if (i < 0)
			return -1;
		    else
			nb += i;
		    flags |= NHPROXY_IO_FMT_DONE;
		    break;
		case 'P':
		case 'p':
		case 'X':
		case 'x':
		    if (*fmt == 'p' || *fmt == 'P')
			flags |= NHPROXY_IO_FMT_PTR;
		    if (*fmt == 'P' || *fmt == 'X')
			flags |= NHPROXY_IO_FMT_CAPS;
		    flags |= NHPROXY_IO_FMT_HEX;
		    /* Fall through */
		case 'u':
		    flags |= NHPROXY_IO_FMT_UNSIGNED;
		    flags &= ~NHPROXY_IO_FMT_SIGN;
		    /* Fall through */
		case 'd':
		case 'i':
		    if (flags & NHPROXY_IO_FMT_LONG)
			ul = va_arg(ap, long);
		    else if (flags & NHPROXY_IO_FMT_UNSIGNED)
			ul = va_arg(ap, unsigned int);
		    else
			ul = (long)va_arg(ap, int);
		    i = nhproxy_io_printi(io, flags, width, ul);
		    if (i < 0)
			return -1;
		    else
			nb += i;
		    flags |= NHPROXY_IO_FMT_DONE;
		    break;
#if ENABLE_PRINTF_FP
		case 'E':
		case 'F':
		case 'G':
		    flags |= NHPROXY_IO_FMT_CAPS;
		    /* Fall through */
		case 'e':
		case 'f':
		case 'g':
		    if (!(flags & NHPROXY_IO_FMT_PRECISION))
			precision = 6;
		    d = va_arg(ap, double);
		    s = fcvt(d, precision, &dp, &sgn);
		    if (*fmt == 'f')
			j = strlen(s);
		    else {
			j = precision;
			if (*fmt == 'e' || *fmt == 'E')
			    j++;
			s = fcvt(d, j - dp, &dp, &sgn);
			if (j > precision || dp > precision || dp < -3) {
			    flags |= NHPROXY_IO_FMT_EXP;
			    expon = dp - 1;
			    dp = 1;
			}
		    }
		    if (sgn)
			flags |= NHPROXY_IO_FMT_NEG;
		    padding = width - j;
		    if (dp <= 0)
			padding += dp - 2;
		    else if (dp < j)
			padding--;
		    if (flags & NHPROXY_IO_FMT_EXP)
			padding -= 5;
		    i = nhproxy_io_pad(io, flags, padding, TRUE);
		    if (i < 0)
			return -1;
		    else
			nb += i;
		    if (dp <= 0) {
			if (nhproxy_io_write(io, "0.", 2) != 2)
			    return -1;
			for(i = dp; i < 0; i++)
			    if (nhproxy_io_fputc('0', io) < 0)
				return -1;
			if (nhproxy_io_write(io, s, j) != j)
			    return -1;
			nb += 2 - dp;
		    } else if (dp < j) {
			if (nhproxy_io_write(io, s, dp) != dp)
			    return -1;
			if (nhproxy_io_fputc('.', io) < 0)
			    return -1;
			if (nhproxy_io_write(io, s + dp, j - dp) != j - dp)
			    return -1;
			nb++;
		    } else if (nhproxy_io_write(io, s, j) != j)
			return -1;
		    nb += j;
		    if (flags & NHPROXY_IO_FMT_EXP) {
			if (nhproxy_io_fputc(
			  flags & NHPROXY_IO_FMT_CAPS ? 'E' : 'e', io) < 0)
			    return -1;
			if (nhproxy_io_printi(io, NHPROXY_IO_FMT_ZEROPAD |
			  NHPROXY_IO_FMT_WIDTH | NHPROXY_IO_FMT_SIGN, exp_width,
			  (long)expon) < 0)
			    return -1;
		    }
		    i = nhproxy_io_pad(io, flags, padding, FALSE);
		    if (i < 0)
			return -1;
		    else
			nb += i;
		    flags |= NHPROXY_IO_FMT_DONE;
		    break;
#endif	/* ENABLE_PRINTF_FP */
		case 's':
		    s = va_arg(ap, char *);
		    if (flags & NHPROXY_IO_FMT_PRECISION)
			j = precision;
		    else if (s && flags & NHPROXY_IO_FMT_ALT)
			for (i = 0, j = 2; s[i]; i++, j++) {
			    if (strchr("\n\t\v\b\r\f\a\\\"", s[i]))
				j++;
			    else if (s[i] < ' ' || s[i] > '~')
				j += 3;
			}
		    else
			j = s ? strlen(s) : 6;
		    i = nhproxy_io_pad(io, flags, width - j, TRUE);
		    if (i < 0)
			return -1;
		    else
			nb += i;
		    if (s && flags & NHPROXY_IO_FMT_ALT) {
			char *esc;
			nhproxy_io_fputc('"', io);
			for (i = 0; s[i]; i++) {
			    if ((esc = strchr("\n\t\v\b\r\f\a\\\"", s[i]))) {
				nhproxy_io_fputc('\\', io);
				nhproxy_io_fputc(*esc, io);
			    } else if (s[i] < ' ' || s[i] > '~') {
				nhproxy_io_fputc('\\', io);
				nhproxy_io_fputc('x', io);
				nhproxy_io_printi(io,
				  NHPROXY_IO_FMT_HEX | NHPROXY_IO_FMT_CAPS |
				  NHPROXY_IO_FMT_WIDTH | NHPROXY_IO_FMT_ZEROPAD,
				  2, (unsigned char)s[i]);
			    } else
				nhproxy_io_fputc(s[i], io);
			}
			nhproxy_io_fputc('"', io);
		    } else if (nhproxy_io_writet(io, s ? s : "(null)", j) != j)
			return -1;
		    else
			nb += j;
		    i = nhproxy_io_pad(io, flags, width - j, FALSE);
		    if (i < 0)
			return -1;
		    else
			nb += i;
		    flags |= NHPROXY_IO_FMT_DONE;
		    break;
		case '%':
		default:
		    if (nhproxy_io_fputc(*fmt, io) < 0)
			return -1;
		    else
			nb++;
		    flags |= NHPROXY_IO_FMT_DONE;
		    break;
	    }
	}
	if (flags & NHPROXY_IO_FMT_DONE)
	    fmt++;
    }
    return nb;
}

#if HAVE_STDARG_H
int
nhproxy_io_printf(NhProxyIO *io, char *fmt, ...)
#else
int
nhproxy_io_printf(io, fmt, va_alist)
NhProxyIO *io;
char *fmt;
va_dcl
#endif
{
    int retval;
    va_list ap;
    VA_START(ap, fmt);
    retval = nhproxy_io_vprintf(io, fmt, ap);
    va_end(ap);
    return retval;
}
