/* $Id: xdr.c,v 1.7 2004/12/22 23:15:04 ali Exp $
 * Copyright (C) 2001, 2002, 2003  Slash'EM Development Team
 * Copyright (C) 2004  J. Ali Harlow
 *
 * This file is part of NetHack Proxy.
 *
 * NetHack Proxy is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of the
 * License, or (at your option) any later version.
 *
 * NetHack Proxy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with NetHack Proxy; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307   
 * USA
 *
 * Alternatively (at your option) you may instead choose to redistribute
 * and/or modify NetHack Proxy under the terms of the NetHack General
 * Public License.
 *
 * You should have receieved a copy of the NetHack General Public License
 * along with NetHack Proxy; if not, download a copy from
 * http://www.nethack.org/common/license.html
 */

/* #define DEBUG */

#include "config.h"
#ifdef DEBUG
#include <stdio.h>
#endif
#ifdef STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif
#include "compat.h"
#include <nhproxy/system.h>
#include <nhproxy/xdr.h>

/*
 * This module implements the NhExt version of the XDR support routines.
 * It is functionally equivalent to the code provided with rpcgen, and
 * except for the xdrio stream can be replaced by that on systems which
 * support it. Rpcgen is included in the freely available RPCSRC 4.0
 * from Sun Microsystems, which can be downloaded in 17 shar files from
 * ftp://bcm.tmc.edu/nfs/ (this is mainly useful in testing).
 *
 * The xdrio stream has no direct equivalent in RPCSRC. Sun's xdrrec
 * stream has the capabilities but this requires both client and server
 * to use it which is not acceptable. Sun's xdrstdio stream is very
 * similar in function to xdrio but would either require changes to
 * the source of RPCSRC or the use of local pipes to connect xdrstdio
 * to nhproxy_io.
 */

#ifdef DEBUG
static void
debug_dump(buf, len, arrow)
nhproxy_genericptr_t buf;
unsigned int len;
char *arrow;
{
    int i, j, nc, ni;
    long l;
    char cbuf[17];
    unsigned char *bp = buf;
    for(i = 0; i < len; ) {
	if ((i & 15) == 0) {
	    if (!i) {
		ni = fprintf(stderr, "[%d] ", getpid());
		ni += fputs(arrow, stderr);
	    }
	    else {
		cbuf[16] = '\0';
		while(nc++ < 40)
		    fputc(' ', stderr);
		fputs(cbuf, stderr);
		fputs("\n", stderr);
		for(j = 0; j <= ni; j++)
		    fputc(' ', stderr);
	    }
	    nc = 2;
	}
	if (len - i >= 4) {
	    l = (long)bp[i] << 24 | (long)bp[i + 1] << 16 |
	      (long)bp[i + 2] << 8 | bp[i + 3];
	    fprintf(stderr, " %08X", l);
	    nc += 9;
	    for(j = 0; j < 4; j++, i++)
		cbuf[i & 15] = isgraph(bp[i]) || bp[i] == ' ' ?  bp[i] : '.';
	} else {
	    fprintf(stderr, " %02X", bp[i]);
	    nc += 3;
	    cbuf[i & 15] = isgraph(bp[i]) || bp[i] == ' ' ? bp[i] : '.';
	    i++;
	}
    }
    if (len) {
	cbuf[i & 15 ? i & 15 : 16] = '\0';
	while(nc++ < 40)
	    fputc(' ', stderr);
	fputs(cbuf, stderr);
    }
    fputc('\n', stderr);
}
#endif	/* DEBUG */

unsigned int
nhproxy_xdr_getpos(xdrs)
NhProxyXdr *xdrs;
{
    return xdrs->x_pos;
}

nhproxy_bool_t
nhproxy_xdr_setpos(xdrs, pos)
NhProxyXdr *xdrs;
unsigned int pos;
{
    if (!xdrs->x_size || pos > xdrs->x_size)
	return FALSE;
    xdrs->x_pos = pos;
    return TRUE;
}

void
nhproxy_xdr_free(codec, addr)
char *addr;
nhproxy_bool_t (*codec)(NhProxyXdr *, nhproxy_genericptr_t);
{
    NhProxyXdr xdrs;
    xdrs.x_op = NHPROXY_XDR_FREE;
    (void)(*codec)(&xdrs, addr);
}

static nhproxy_bool_t
nhproxy_xdrmem_read(xdrs, addr, size)
NhProxyXdr *xdrs;
char *addr;
int size;
{
    if (size < 0)
	return FALSE;
    if (size > xdrs->x_size - xdrs->x_pos)
	size = xdrs->x_size - xdrs->x_pos;
    memcpy(addr, (unsigned char *)xdrs->x_data + xdrs->x_pos, size);
#ifdef DEBUG
    debug_dump(addr, size, "<<");
#endif
    xdrs->x_pos += size;
    return TRUE;
}

static nhproxy_bool_t
nhproxy_xdrmem_write(xdrs, addr, size)
NhProxyXdr *xdrs;
const char *addr;
int size;
{
    if (size < 0 || size > xdrs->x_size - xdrs->x_pos)
	return FALSE;
#ifdef DEBUG
    debug_dump(addr, size, ">>");
#endif
    memcpy((unsigned char *)xdrs->x_data + xdrs->x_pos, addr, size);
    xdrs->x_pos += size;
    return TRUE;
}

static void
nhproxy_xdrmem_destroy(xdrs)
NhProxyXdr *xdrs;
{
}

void
nhproxy_xdrmem_create(xdrs, addr, size, op)
NhProxyXdr *xdrs;
char *addr;
unsigned int size;
nhproxy_xdr_op_t op;
{
    xdrs->x_data = addr;
    xdrs->x_op = op;
    xdrs->x_size = size;
    xdrs->x_pos = 0;
    xdrs->x_read = nhproxy_xdrmem_read;
    xdrs->x_write = nhproxy_xdrmem_write;
    xdrs->x_destroy = nhproxy_xdrmem_destroy;
}

static nhproxy_bool_t
nhproxy_xdrio_read(xdrs, addr, size)
NhProxyXdr *xdrs;
char *addr;
int size;
{
    int retval;
    if (size <= 0)
	return !size;
    retval = nhproxy_io_fread(addr, size, 1, (NhProxyIO *)xdrs->x_data);
#ifdef DEBUG
    if (retval < 0)
	/*
	 * Only nhproxy_xdr_bytestring() calls us with anything larger
	 * than 4 bytes. If this is triggered then either NhProxyIO
	 * needs to be modified to increase the buffer size or the
	 * NhExt protocol needs to be modified to keep all strings
	 * below NhProxyIO's limit.
	 */
	fprintf(stderr, "nhproxy_xdrio: datum too large to read (%d)\n", size);
#endif
    if (retval == 1) {
#ifdef DEBUG
	debug_dump(addr, size, "<<");
#endif
	xdrs->x_pos += size;
    }
#ifdef DEBUG
    else
	fprintf(stderr, "[%d] << EOF\n", getpid());
#endif
    return retval == 1;
}

static nhproxy_bool_t
nhproxy_xdrio_write(xdrs, addr, size)
NhProxyXdr *xdrs;
const char *addr;
int size;
{
    int retval;
    if (size < 0)
	return FALSE;
#ifdef DEBUG
    debug_dump(addr, size, ">>");
#endif
    retval = nhproxy_io_write((NhProxyIO *)xdrs->x_data,
      (nhproxy_genericptr_t)addr, size);
    xdrs->x_pos += retval;
    return retval == size;
}

static void
nhproxy_xdrio_destroy(xdrs)
NhProxyXdr *xdrs;
{
    (void)nhproxy_io_flush((NhProxyIO *)xdrs->x_data);
}

void
nhproxy_xdrio_create(xdrs, io, op)
NhProxyXdr *xdrs;
NhProxyIO *io;
nhproxy_xdr_op_t op;
{
    xdrs->x_data = io;
    xdrs->x_op = op;
    xdrs->x_size = 0;		/* xdr_setpos() is not supported */
    xdrs->x_pos = 0;
    xdrs->x_read = nhproxy_xdrio_read;
    xdrs->x_write = nhproxy_xdrio_write;
    xdrs->x_destroy = nhproxy_xdrio_destroy;
}
