/* $Id: test_xdr.c,v 1.4 2004/12/22 23:15:06 ali Exp $
 * Copyright (C) 2001, 2002  Slash'EM Development Team
 * Copyright (C) 2004  J. Ali Harlow
 *
 * This file is part of NetHack Proxy.
 *
 * NetHack Proxy is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of the
 * License, or (at your option) any later version.
 *
 * NetHack Proxy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with NetHack Proxy; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307   
 * USA
 *
 * Alternatively (at your option) you may instead choose to redistribute
 * and/or modify NetHack Proxy under the terms of the NetHack General
 * Public License.
 *
 * You should have receieved a copy of the NetHack General Public License
 * along with NetHack Proxy; if not, download a copy from
 * http://www.nethack.org/common/license.html
 */

/*
 * This module tests the NetHack Proxy implementation of XDR by using it to
 * write and read data to and from an implementation using standard XDR.
 *
 * Note: You need an implementation of XDR to build this test program.
 *	 It should be built once with USE_STD_XDR defined to get a version which
 *	 uses Sun Microsystems's rpc library and once without to get a
 *	 version which uses our implementation. The two can then be tested
 *	 for compatibility.
 */

#include <stdio.h>

#ifdef USE_STD_XDR
#include <rpc/xdr.h>
#else
#include <nhproxy/nhproxy.h>
#endif	/* USE_STD_XDR */

#ifdef USE_STD_XDR
#define SIZE(array)		(sizeof(array)/sizeof(*(array)))
#define nhproxy_bool_t		bool_t
#define nhproxy_false		FALSE
#define nhproxy_true		TRUE

#define NHPROXY_XDR_DECODE	XDR_DECODE
#define NHPROXY_XDR_ENCODE	XDR_ENCODE
#define NHPROXY_XDR_FREE	XDR_FREE

typedef enum xdr_op nhproxy_xdr_op_t;
typedef XDR NhProxyXdr;
typedef xdrproc_t nhproxy_xdrproc_t;

#define nhproxy_xdrmem_create	xdrmem_create
#define nhproxy_xdr_getpos	xdr_getpos
#define nhproxy_xdr_setpos	xdr_setpos
#define nhproxy_xdr_destroy	xdr_destroy
#define nhproxy_xdr_short	xdr_short
#define nhproxy_xdr_long	xdr_long
#define nhproxy_xdr_int		xdr_int
#define nhproxy_xdr_char	xdr_char
#define nhproxy_xdr_u_short	xdr_u_short
#define nhproxy_xdr_u_long	xdr_u_long
#define nhproxy_xdr_u_int	xdr_u_int
#define nhproxy_xdr_u_char	xdr_u_char
#define nhproxy_xdr_enum	xdr_enum
#define nhproxy_xdr_bool	xdr_bool
#define nhproxy_xdr_vector	xdr_vector
#define nhproxy_xdr_array	xdr_array
#define nhproxy_xdr_string	xdr_string
#else
typedef nhproxy_bool_t (*nhproxy_xdrproc_t) ();
#endif	/* USE_STD_XDR */

#define SIZE(array)		(sizeof(array)/sizeof(*(array)))

nhproxy_bool_t
test_long(xdrs, op)
NhProxyXdr *xdrs;
nhproxy_xdr_op_t op;
{
    int i;
    long l;
    for(i = 0; i < 8; i++) {
	l = i;
	if (!nhproxy_xdr_long(xdrs, &l)) {
	    fprintf(stderr, "nhproxy_xdr_long failed\n");
	    return nhproxy_false;
	}
	if (op == NHPROXY_XDR_DECODE)
	    printf("%ld ", l);
    }
    if (op == NHPROXY_XDR_DECODE)
	putchar('\n');
    return nhproxy_true;
}

nhproxy_bool_t
test_vector(xdrs, op)
NhProxyXdr *xdrs;
nhproxy_xdr_op_t op;
{
    int i;
    short vector[8];
    for(i = 0; i < 8; i++)
	vector[i] = i;
    if (!nhproxy_xdr_vector(xdrs, (char *)vector, SIZE(vector), sizeof(short),
      (nhproxy_xdrproc_t)nhproxy_xdr_short)) {
	fprintf(stderr, "nhproxy_xdr_vector failed\n");
	return nhproxy_false;
    }
    if (op == NHPROXY_XDR_DECODE) {
	printf("[ ");
	for(i = 0; i < 8; i++)
	    printf("%d ", vector[i]);
	putchar(']');
	putchar('\n');
    }
    return nhproxy_true;
}

nhproxy_bool_t
test_array(xdrs, op)
NhProxyXdr *xdrs;
nhproxy_xdr_op_t op;
{
    int i;
    unsigned int length;
    unsigned char *array;
    if (op == NHPROXY_XDR_ENCODE)
    {
	array = malloc(5);
	if (!array) {
	    fprintf(stderr, "malloc failed\n");
	    return nhproxy_false;
	}
	array[0] = 'h';
	array[1] = 'e';
	array[2] = array[3] = 'l';
	array[4] = 'o';
	length = 5;
    }
    else
	array = NULL;
    if (!nhproxy_xdr_array(xdrs, (char **)&array, &length, (unsigned int)-1,
      sizeof(unsigned char), (nhproxy_xdrproc_t)nhproxy_xdr_u_char)) {
	fprintf(stderr, "nhproxy_xdr_array failed\n");
	return nhproxy_false;
    }
    if (op == NHPROXY_XDR_DECODE) {
	printf("< ");
	for(i = 0; i < length; i++)
	    printf("'%c' ", array[i]);
	putchar('>');
	putchar('\n');
    }
    free(array);
    array = NULL;
    length = 0;
    if (!nhproxy_xdr_array(xdrs, (char **)&array, &length, (unsigned int)-1,
      sizeof(unsigned char), (nhproxy_xdrproc_t)nhproxy_xdr_u_char)) {
	fprintf(stderr, "nhproxy_xdr_array failed\n");
	return nhproxy_false;
    }
    if (op == NHPROXY_XDR_DECODE) {
	printf("< ");
	for(i = 0; i < length; i++)
	    printf("'%c' ", array[i]);
	putchar('>');
	putchar('\n');
    }
    if (array) {
	fprintf(stderr, "nhproxy_xdr_array allocated on empty array\n");
	return nhproxy_false;
    }
    return nhproxy_true;
}

nhproxy_bool_t
test_string(xdrs, op)
NhProxyXdr *xdrs;
nhproxy_xdr_op_t op;
{
    int i;
    char *string;
    if (op == NHPROXY_XDR_ENCODE)
	string = strdup("Hello world!");	/* Multiple of 4 bytes */
    else
	string = NULL;
    if (!nhproxy_xdr_string(xdrs, &string, (unsigned int)-1)) {
	fprintf(stderr, "nhproxy_xdr_string failed\n");
	return nhproxy_false;
    }
    if (op == NHPROXY_XDR_DECODE)
	printf("\"%s\"\n", string);
    free(string);
    if (op == NHPROXY_XDR_ENCODE)
	string = strdup("Goodbye");	/* Not a multiple of 4 bytes */
    else
	string = NULL;
    if (!nhproxy_xdr_string(xdrs, &string, (unsigned int)-1)) {
	fprintf(stderr, "nhproxy_xdr_string failed\n");
	return nhproxy_false;
    }
    if (op == NHPROXY_XDR_DECODE)
	printf("\"%s\"\n", string);
    free(string);
    if (op == NHPROXY_XDR_ENCODE)
	string = strdup("");
    else
	string = NULL;
    if (!nhproxy_xdr_string(xdrs, &string, (unsigned int)-1)) {
	fprintf(stderr, "nhproxy_xdr_string failed\n");
	return nhproxy_false;
    }
    if (op == NHPROXY_XDR_DECODE)
	printf("\"%s\"\n", string);
    free(string);
    return nhproxy_true;
}

nhproxy_bool_t
test_bool(xdrs, op)
NhProxyXdr *xdrs;
nhproxy_xdr_op_t op;
{
    nhproxy_bool_t bool = nhproxy_true;
    if (!nhproxy_xdr_bool(xdrs, &bool)) {
	fprintf(stderr, "nhproxy_xdr_bool failed\n");
	return nhproxy_false;
    }
    if (op == NHPROXY_XDR_DECODE)
	printf("%s ", bool == nhproxy_true ? "TRUE" :
	  bool == nhproxy_false ? "FALSE" : "???");
    bool = nhproxy_false;
    if (!nhproxy_xdr_bool(xdrs, &bool)) {
	fprintf(stderr, "nhproxy_xdr_bool failed\n");
	return nhproxy_false;
    }
    if (op == NHPROXY_XDR_DECODE)
	printf("%s\n", bool == nhproxy_true ? "TRUE" :
	  bool == nhproxy_false ? "FALSE" : "???");
    return nhproxy_true;
}

nhproxy_bool_t
test_enum(xdrs, op)
NhProxyXdr *xdrs;
nhproxy_xdr_op_t op;
{
    nhproxy_xdr_op_t e;
    e = NHPROXY_XDR_DECODE;
    if (!nhproxy_xdr_enum(xdrs, (int *)&e)) {
	fprintf(stderr, "nhproxy_xdr_enum failed\n");
	return nhproxy_false;
    }
    if (op == NHPROXY_XDR_DECODE)
	switch(e)
	{
	    case NHPROXY_XDR_DECODE:	printf("NHPROXY_XDR_DECODE "); break;
	    case NHPROXY_XDR_ENCODE:	printf("NHPROXY_XDR_ENCODE "); break;
	    case NHPROXY_XDR_FREE:	printf("NHPROXY_XDR_FREE "); break;
	    default:			printf("??? "); break;
	}
    e = NHPROXY_XDR_ENCODE;
    if (!nhproxy_xdr_enum(xdrs, (int *)&e)) {
	fprintf(stderr, "nhproxy_xdr_enum failed\n");
	return nhproxy_false;
    }
    if (op == NHPROXY_XDR_DECODE)
	switch(e)
	{
	    case NHPROXY_XDR_DECODE:	printf("NHPROXY_XDR_DECODE\n"); break;
	    case NHPROXY_XDR_ENCODE:	printf("NHPROXY_XDR_ENCODE\n"); break;
	    case NHPROXY_XDR_FREE:	printf("NHPROXY_XDR_FREE\n"); break;
	    default:			printf("???\n"); break;
	}
    return nhproxy_true;
}

int
codec(buf, len, maxlen, op)
char *buf;
unsigned int *len;
unsigned int maxlen;
nhproxy_xdr_op_t op;
{
    int retval;
    NhProxyXdr xdrs;
    unsigned int pos;
    nhproxy_xdrmem_create(&xdrs, buf, maxlen, op);
    if (op == NHPROXY_XDR_ENCODE)
	pos = nhproxy_xdr_getpos(&xdrs);
    retval = test_long(&xdrs, op) &&
      test_vector(&xdrs, op) &&
      test_array(&xdrs, op) &&
      test_string(&xdrs, op) &&
      test_bool(&xdrs, op) &&
      test_enum(&xdrs, op);
    if (op == NHPROXY_XDR_ENCODE)
	*len = nhproxy_xdr_getpos(&xdrs) - pos;
    nhproxy_xdr_destroy(&xdrs);
    return retval;
}

nhproxy_bool_t
reader()
{
    unsigned int nb;
    char buf[200];
    nb = fread(buf, 1, sizeof(buf), stdin);
    return codec(buf, &nb, nb, NHPROXY_XDR_DECODE);
}

nhproxy_bool_t
writer()
{
    unsigned int nb;
    char buf[200];
    if (!codec(buf, &nb, sizeof(buf), NHPROXY_XDR_ENCODE))
	return nhproxy_false;
    fwrite(buf, 1, nb, stdout);
    return nhproxy_true;
}

main(argc, argv)
int argc;
char **argv;
{
    int retval;
    if (argc != 2 || argv[1][0]!='-' || argv[1][1]!='r' && argv[1][1]!='w' ||
      argv[1][2]) {
	fprintf(stderr, "Usage: %s -r | -w\n", argv[0]);
	exit(1);
    }
    if (argv[1][1]=='r')
	retval = reader();
    else
	retval = writer();
    exit(!retval);
}
