/***********************************************************************
    Copyright (C) 2005 Frdric HENRY <neryel@reveries.info>

    This file is part of NiNaR.

    NiNaR is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published 
    by the Free Software Foundation; either version 2 of the License,
    or (at your option) any later version.

    NiNaR is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of 
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License 
    along with NiNaR; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307  USA.

    File information: $Id: character.m,v 1.9 2005/01/16 22:57:51 neryel Exp $

***********************************************************************/

#include <glib.h>
#include <stdlib.h>
#include <string.h>

#include "character.h"
#include "tools.h"
#include "weapon.h"
#include "map.h"

/* Probabilities all attributes */ 
#define SPEED_PROBA 0.5
#define DODGE_PROBA 0.2
#define STAMINA_PROBA 2

/* Minima */
#define HP_MIN 10
#define MOVE_MIN 2

/* character list and current character */
static GList * _characters = NULL;
static GList * _current_character = NULL;

@implementation Character
/** 
 * Sets the turn to the next character, by sending him the new_turn
 * message.
 *
 * Return the characters whose it is turn.
 **/
+ (id) next_character
{
  g_return_val_if_fail (_characters, nil);
  id next;

  if (_current_character == NULL)
    {
      _current_character = g_list_first (_characters);
      next = _current_character->data;
      [next new_turn];
    }
  else
    {
      next = _current_character->data;
      [next new_turn];
    }
  _current_character = g_list_next (_current_character);
  
  /* \bug infinite loop if all characters are dead */
  if ([next get_hp] <= 0)
    {
      return [self next_character];
    }

  return next;
}

/**
 * Init: add the character to the list *
 **/
- init
{
  [super init];
  _characters = g_list_append (_characters, self);
  return self;
}

/**
 * Init from attributes and values of an XML element. 
 * A character is coded like this:
 * <character name = "john" picture = "guy.png" hit = "5" ... />
 **/
- init_from_attributes: (const char **) attributes values: (const char **) values
{
  /* [self init] already done in parent */
  [super init_from_attributes: attributes values: values];
  for (int i = 0; attributes[i] && values[i]; i++)
    {
      if (!strcmp (attributes[i], "name"))
        [self set_name: values[i]];
      else if (!strcmp (attributes[i], "hit"))
        _hit = strtol (values[i], NULL, 0);
      else if (!strcmp (attributes[i], "speed"))
        _speed = strtol (values[i], NULL, 0);
      else if (!strcmp (attributes[i], "dodge"))
        _dodge = strtol (values[i], NULL, 0);
      else if (!strcmp (attributes[i], "stamina"))
        _stamina = strtol (values[i], NULL, 0);
      else if (!strcmp (attributes[i], "weapon"))
        [self set_weapon_from_name: values[i]];
    }

  [self set_hp];
  
  return self;
}

/**
 * Free allocated ressources 
 **/
- free
{
  if (_name)
    {
      g_free (_name);
    }
  _characters = g_list_remove (_characters, self);
  return [super free];
}

/** 
 * Attack another character
 *
 * \param character: the character to attack
 * \return the number of dealt damages
 **/
- (int) attack: (id) character
{
  g_return_val_if_fail (_weapon, 0);

  if (character == self)
    {
      printf ("%s: \"Sorry, but I don't want to commit suicide.\"\n", _name);
      return ATTACK_SELF;
    }
  
  if (_fired_ammos >= [_weapon get_ammos])
    {
      printf ("No more ammos\n");
      return ATTACK_NO_MORE_AMMO;
    }
  _fired_ammos++;

  /* \todo change distance evaluation, either better approx either
     directly use d = sqrt (dx + dy) */
  int distance = abs ([character get_x] - [self get_x]) + abs ([character get_y] - [self get_y]);
  double proba = [_weapon get_hit_proba_at_distance: distance];

  int att = binomial (proba, _hit);
  int def = [character get_dodge];
  printf ("Attack power: %d\n", att);
  printf ("Defense power: %d\n", def);
  att -= def;
  
  if (att < 0)
    att = 0;

  att *= [_weapon get_power];
  printf ("%d damages\n", att);
  [character lose_hp: att];

  printf ("%d ammos left\n", [_weapon get_ammos] - _fired_ammos);


  return att;
}

/**
 * Loses hp, and die if hp < 0 
 **/
- (void) lose_hp: (int) hp
{
  _hp -= hp;
  printf ("%s has %d HP left\n", _name, _hp);

  if (_hp <= 0)
    {  
      printf ("%s died\n", _name);
      [_map remove_object: self];
      /* Note that the object is NOT destroyed */
    }
}

/**
 * Move the character 1 case in the specified direction
 *
 * \return neg number if move failed (obstacle or no more move), or number
 * of move left else.
 **/
- (int) move: (enum character_move) direction
{
  int x = _x + (direction == MOVE_RIGHT) - (direction == MOVE_LEFT);
  int y = _y + (direction == MOVE_DOWN) - (direction == MOVE_UP);

  /* Check that it is not occupied */
  if ([_map get_object_at_x: x y: y])
    {
      return MOVE_OBSTACLE;
    }
  else if (x < 0 || y < 0 || x > [_map get_x_max] || y > [_map get_y_max])
    {
      return MOVE_OFF_LIMIT;
    }
  else if (_moved >= _max_move)
    {
      printf ("No more move for this turn\n");
      return MOVE_NO_MORE_MOVE;
    }
  else 
    {
      _moved++;
      [self set_x: x y: y];
      [_map refresh];
      return _max_move - _moved;
    }
}

/**
 * Returns the character name 
 **/
- (const char *) get_name
{
  return _name;
}

/**
 * Sets the character name
 **/
- (void) set_name: (const char *) name
{
  _name = g_strdup (name);
}

/**
 * Returns the character HP
 **/
- (int) get_hp
{
  return _hp;
}

/**
 * Return a speed result
 **/
- (int) get_speed
{
  return binomial (SPEED_PROBA, _speed);
}

/**
 * Return a dodge result
 **/
- (int) get_dodge
{
  return binomial (DODGE_PROBA, _dodge);
}

/**
 * Return a stamina result
 **/
- (int) get_stamina
{
  return binomial (STAMINA_PROBA, _stamina);
}
      

/**
 * Sets the character HP, basing from its stamina
 **/
- (void) set_hp
{
  _hp = HP_MIN + [self get_stamina];
}

/**
 * Sets the character weapon
 **/
- (void) set_weapon: (id) weapon
{
  _weapon = weapon;
}

/**
 * Gets the weapon (can be nil)
 **/
- (id) get_weapon
{
  return _weapon;
}

/**
 * Sets the weapon from a name, useful for the XML
 **/
- (void) set_weapon_from_name: (const char *) name
{
  id weapon = [Weapon get_weapon_from_name: name];
  [self set_weapon: weapon];
}


/**
 * Message received when it is a new turn
 **/
- (void) new_turn
{
  printf ("%s's turn!\n", _name);

  _fired_ammos = 0;
  _moved = 0;

  /* \todo change the move equation or at least use #defines */
  _max_move = MOVE_MIN + [self get_speed];
  printf ("%s gets %d move this turn\n", _name, _max_move);
}


/**
 * \brief Overrides Field_object method
 *
 * Character pics are size (Width, Width) instead of (Width, Height),
 * so we must do a translation
 **/
- (void) reposition
{
  [super reposition];
  [self move_absolute_x: 0 y: - (CASE_WIDTH - CASE_HEIGHT)];
}
@end
