/***********************************************************************
    Copyright (C) 2005 Frdric HENRY <neryel@reveries.info>

    This file is part of NiNaR.

    NiNaR is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published 
    by the Free Software Foundation; either version 2 of the License,
    or (at your option) any later version.

    NiNaR is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of 
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License 
    along with NiNaR; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307  USA.

    File information: $Id: graphical_object.m,v 1.9 2005/01/15 18:01:48 neryel Exp $

***********************************************************************/

#include <stdio.h>
#include <string.h>

#include "graphical_object.h"
#include "map.h"

@implementation Graphical_object
/**
 * \brief Init a graphical object
 *
 * Sets x and y to -1 so we know the object is not usable.
 * Sets pointers to NULL.
 **/
- init
{
  _picture = NULL;
  _map = nil;
  
  return self;
}

/**
 * Init from attributes and values of an XML element. 
 **/
- init_from_attributes: (const char **) attributes values: (const char **) values
{
  [self init];

  for (int i = 0; attributes[i] && values[i]; i++)
    {
      if (!strcmp (attributes[i], "picture"))
        [self load_picture: values[i]];
    }

  return self;
}

/**
 * \brief Init a graphical object from file
 **/
- init_from_file: (const char *) filename
{
  [self init];
  [self load_picture: filename];

  return self;
}

/**
 * Free the graphical object's space
 **/
- free
{
  if (_picture)
    {
      gdk_pixbuf_unref (_picture);
      _picture = NULL;
    }

  return [super free];
}

/**
 * \brief Display the object
 **/
- (void) display
{
  /* Fails if it hasn't been properly initialised */
  g_return_if_fail (_picture);
  g_return_if_fail (_map);

  /* Not really necessary, but gcc prefers like this */
  GtkWidget * widget = [(Map *)_map get_widget];
  g_return_if_fail (widget);

  /* Move the object according to the map window position */
  int x = _x_absolute - [_map get_x_offset];
  int y = _y_absolute - [_map get_y_offset];
  
  gdk_draw_pixbuf (widget->window,
    widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
    _picture, 0, 0, x, y, -1, -1, 
    GDK_RGB_DITHER_NONE, 0, 0);    
}

/**
 * \brief Set the object position
 *
 * Note that x and y correspond to the pixels in the picture, *NOT* to
 * the position in cases in the isometric field !
 **/
- (void) set_absolute_x: (int) x y: (int) y
{ 
  _x_absolute = x;
  _y_absolute = y;
}

/**
 * \brief Move the object by x and y
 **/
- (void) move_absolute_x: (int) x y: (int) y
{
  _x_absolute += x;
  _y_absolute += y;
}

/** 
 * \brief Loads the picture to the pixbuf
 *
 * \param filename : name of the file. *NOT* the absolute path. It
 * searches the picture in DATADIR/images. 
 **/
- (void) load_picture: (const char *) filename
{
  gchar * absolute_filename = g_strdup_printf ("%s/images/%s", DATADIR, filename);
  GError * error = NULL;

  _picture = gdk_pixbuf_new_from_file (absolute_filename, &error);

  if (_picture == NULL)
    {
      g_warning ("Graphical_object::load_picture: could not load picture %s (%s)", filename, absolute_filename);
      if (error)
        {
          g_warning ("%s", error->message);
          g_error_free (error);
        }
    }

  g_free (absolute_filename);
}

/** 
 * \brief Sets the map where the object belong.
 *
 * You don't have to do this manually: it is done when the object is
 * added to the map with add_object.
 **/
- (void) set_map: (id) map;
{
  _map = map;
}

@end
