/***********************************************************************
    Copyright (C) 2005 Frdric HENRY <neryel@reveries.info>

    This file is part of NiNaR.

    NiNaR is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published 
    by the Free Software Foundation; either version 2 of the License,
    or (at your option) any later version.

    NiNaR is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of 
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License 
    along with NiNaR; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307  USA.

    File information: $Id: map.m,v 1.20 2005/01/17 00:33:38 neryel Exp $

***********************************************************************/

#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "map.h"
#include "graphical_object.h"
#include "field_object.h"
#include "obstacle.h"
#include "wall.h"
#include "tile.h"
#include "tools.h"
#include "character.h"
#include "map_user_interface.h"


/* Default width and height (in pixels) */
#define MAP_DEFAULT_WIDTH 100
#define MAP_DEFAULT_HEIGHT 100

/* Page increment for the scrollbar */
#define PAGE_INCREMENT 5

/* Buffer size */
#define BUFFER_SIZE 42

/* Margin to add/substract to the calculated min and max isometric X
   and Y */
#define MARGIN 3

/* Callback for when the x scrollbar moved */
static gboolean
x_move_callback (GtkAdjustment * adjustment, id map)
{
  int x = (int) adjustment->value;
  [map set_relative_offset_x: x];

  return TRUE;
}

/* Callback for when the y scrollbar moved */
static gboolean
y_move_callback (GtkAdjustment * adjustment, id map)
{
  int y = (int) adjustment->value;
  [map set_relative_offset_y: y];
  
  return TRUE;
}

/* Callback for expose event : refresh the map */
static gboolean
expose_callback (GtkWidget * w __attribute__((unused)), 
  GdkEventExpose * event __attribute__ ((unused)), 
  id map)
{
  [map refresh];

  return TRUE;
}

/* Sort function used to tell which of 2 field_objects is the nearer. */
static gint 
compare_func (id a, id b)
{
  int total_a = [a get_x] + [a get_y];
  int total_b = [b get_x] + [b get_y];

  return total_a - total_b;
}

/* Only used to communicate with load_file, since we can only pass one
   parameter... 
   \bug this is dirty */
const char * map_current_file_name = NULL;

/* Callback function used for parsing the XML file */
static void
map_load (GMarkupParseContext * context __attribute__((unused)),
  const gchar * element_name,
  const gchar ** attributes, const gchar ** values,
  gpointer data,
  GError ** error)
{
  if (*error)
    {
      g_warning ("map_load: Parse error occured");
      g_warning ("%s", (*error)->message);
    }

  id map = (id) data;

  if (!strcmp (element_name, "map"))
    {
      for (int i = 0; attributes[i] && values[i]; i++)
        {
          if (!strcmp (attributes[i], "file"))
            {
              char * true_path = path_from_neighbour (values[i], map_current_file_name);
              [map load_tiles_map: true_path];
              g_free (true_path);
            }
        }
    }
  else if (!strcmp (element_name, "character"))
    {
      id character = [[Character alloc] init_from_attributes: attributes values: values];
      [map add_object: character];
    }
  else if (!strcmp (element_name, "obstacle"))
    {
      id obstacle = [[Obstacle alloc] init_from_attributes: attributes values: values];
      [map add_object: obstacle];
    }
  else if (!strcmp (element_name, "wall"))
    {
      id wall = [[Wall alloc] init_from_attributes: attributes values: values];
      [map add_object: wall];
    }
  else if (!strcmp (element_name, "Obstacle"))
    {
      [Obstacle init_all_from_attributes: attributes values: values map: map];
    }
  else if (!strcmp (element_name, "Wall"))
    {
      [Wall init_all_from_attributes: attributes values: values map: map];
    }
  else
    {
      g_warning ("map_load: element '%s' not recognized, ignoring it", element_name);
    }
}


@implementation Map
/**
 * Inits the map 
 **/
- init
{
  /** Widgets initialization **/
  /* Create a drawing area */
  _widget = gtk_drawing_area_new ();
  gtk_widget_set_size_request (_widget, 
    MAP_DEFAULT_WIDTH, MAP_DEFAULT_HEIGHT);
  /** Connect the callbacks **/
  g_signal_connect (G_OBJECT (_widget), "expose_event",
    G_CALLBACK (expose_callback), self);

  /* Add keyboard and mouse events */
  GTK_WIDGET_SET_FLAGS (_widget, GTK_CAN_FOCUS);
  gtk_widget_add_events (_widget, GDK_BUTTON_PRESS_MASK | GDK_KEY_PRESS_MASK);
  g_signal_connect (G_OBJECT (_widget), "button_press_event",
    G_CALLBACK (map_ui_mouse_callback), self);
  g_signal_connect (G_OBJECT (_widget), "key_press_event",
    G_CALLBACK (map_ui_keyboard_callback), self);
  

  _tiles = NULL;
  _field_objects = NULL;
  _objects = NULL;

  _x_offset = 0;
  _y_offset = 0;

  _cases_x = 0;
  _cases_y = 0;

  return self;
}


/**
 * Init from file
 **/
- init_from_file: (const char *) filename
{
  [self init];
  [self load_file: filename];
  return self;
}

/**
 * Free map's ressources
 **/
- free 
{
  if (_widget)
    {
      gtk_widget_destroy (_widget);
      _widget = NULL;
    }
  
  if (_tiles)
    {
      for (unsigned x = 0; x < _cases_x; x++)
        {
          g_free (_tiles[x]);
        }
      g_free (_tiles);
    }

  for (GList * i = g_list_first (_field_objects); i; i = g_list_next (i))
    {
      id object = i->data;
      [object free];
    }
  g_list_free (_field_objects);

  for (GList * i = g_list_first (_objects); i; i = g_list_next (i))
    {
      id object = i->data;
      [object free];
    }
  g_list_free (_objects);
  
  return [super free];
}

/**
 * Load a file containing the map and obstacles and, and, and.
 **/
- (void) load_file: (const char *) filename
{
  FILE * file = fopen (filename, "r");
  if (!file)
    {
      g_warning ("map: load_file: could not open file %s, aborting", filename);
      return;
    }
  map_current_file_name = filename;

  char buffer[BUFFER_SIZE];
  gboolean next = TRUE;
  GMarkupParseContext * context;
  GMarkupParser parser = {map_load, NULL, NULL, NULL, NULL};

  context = g_markup_parse_context_new (&parser, 0, self, NULL);

  while (next && fgets (buffer, BUFFER_SIZE, file))
    {
      next = g_markup_parse_context_parse (context,
        buffer, BUFFER_SIZE-1, NULL);

      if (!next)
        {
          g_warning ("Error parsing map file");
        }
    }

  g_markup_parse_context_free (context);
  fclose (file);
}

/**
 * Loads the map for tiles from a file
 **/
- (void) load_tiles_map: (const char *) filename
{
  FILE * file = fopen (filename, "r");
  if (!file)
    {
      g_warning ("map: load_tiles_map: could not open file %s, aborting", filename);
      return;
    }
  
  if (!fscanf (file, "width:%u\n", &_cases_x))
    {
      g_warning ("map: load_tiles_map: could not get width value, aborting");
      fclose (file);
      return;
    }

  if (!fscanf (file, "height:%u\n", &_cases_y))
    {
      g_warning ("map: load_tiles_map: could not get height value, aborting");
      fclose (file);
      return;
    }

  _tiles = g_malloc (_cases_x * sizeof (char *));
  for (unsigned x = 0; x < _cases_x; x++)
    {
      _tiles[x] = g_malloc (_cases_y * sizeof (char));
    }

  /* Read the tiles map */
  for (unsigned y = 0; y < _cases_y; y++)
    {
      int c;
      for (unsigned x = 0; x < _cases_x; x++)
        {
          id tile;

          c = fgetc (file);
          if (c == EOF)
            {
              g_warning ("map: load_tiles_map: unexpected EOF in %s", filename);
              fclose (file);
              return;
            }
          /* Check whether the mapping leads to a tile */

          if ((tile = [Tile get_tile_from_key: (char) c]))
            {
              [tile set_map: self];
              _tiles[x][y] = (char) c;
            }
          else
            {
              _tiles[x][y] = 0;
              g_warning ("map: load_tiles_map: key '%c' unknown, ignoring it", (char) c);
            }
        }
      /* Go to end of line */
      while ((c = fgetc (file)) != '\n')
        {
          if (c == EOF)
            {
              g_warning ("map: load_tiles_map: unexpected EOF in %s", filename);
              fclose (file);
              return;
            }
        }
    }

  for (unsigned y = 0; y < _cases_y; y++)
    {
      for (unsigned x = 0; x < _cases_x; x++)
        {
          printf ("%c", _tiles[x][y]);
        }
      printf ("\n");
    }

  fclose (file);
  
}

/**
 * \brief Adds an object in the field
 *
 * The object is added to the good list according to its class type.
 **/
- (void) add_object: (id) object
{
  if ([object isKindOf: [Field_object class]])
    {
      _field_objects = g_list_append (_field_objects, object);
    }
  else if ([object isKindOf: [Graphical_object class]])
    {
      _objects = g_list_append (_objects, object);
    }
  else 
    {
      g_return_if_reached ();
    }
  
  [object set_map: self];
}

/** 
 * Remove an object from the field
 **/
- (void) remove_object: (id) object
{
    if ([object isKindOf: [Field_object class]])
    {
      _field_objects = g_list_remove (_field_objects, object);
      [object set_map: nil];
    }
  else if ([object isKindOf: [Graphical_object class]])
    {
      _objects = g_list_remove (_objects, object);
      [object set_map: nil];
    }
  else
    {
      g_warning ("Parameter is not a Graphical_object, but a %s", [object name]);
    }

    [self refresh];
}

/**
 * \brief Refresh the map
 *
 * Send the 'display' message to all objects of the map.
 **/
- (void) refresh
{
  GdkRectangle rect = {0, 0, [self get_width], [self get_height]};
  gdk_window_begin_paint_rect (_widget->window, &rect);
  /* First, erase all */
  gdk_draw_rectangle (_widget->window,
    _widget->style->fg_gc[GTK_WIDGET_STATE (_widget)],
    TRUE, 0, 0, [self get_width], [self get_height]);

  /* Estimate isometric position of the window's corners */
  struct couple corners[4] = {{_x_offset, _y_offset}, 
                              {_x_offset + [self get_width], _y_offset},
                              {_x_offset , _y_offset + [self get_height]},
                              {_x_offset + [self get_width] , _y_offset + [self get_height]}};
  struct couple max = {0, 0};
  struct couple min = {_cases_x, _cases_y};
  for (int i = 0; i < 4; i++)
    {
      corners[i] = absolute_to_isometric (corners[i]);
      if (corners[i].x > max.x)
        max.x = corners[i].x;
      if (corners[i].y > max.y)
        max.y = corners[i].y;
      if (corners[i].x < min.x)
        min.x = corners[i].x;
      if (corners[i].y < min.y)
        min.y = corners[i].y;
    }
  /* Adds a margin so that not entire tiles are displayed too, and so
     that big objects are too */
  min.x -= MARGIN;
  min.y -= MARGIN;
  max.x += MARGIN;
  max.y += MARGIN;

  if (min.x < 0)
    min.x = 0;
  if (min.y < 0)
    min.y = 0;
  if ((unsigned) max.x > _cases_x)
    max.x = _cases_x;
  if ((unsigned) max.y > _cases_y)
    max.y = _cases_y;

  /* Start with the tiles */
  for (int x = min.x; x < max.x ; x++)
    {
      for (int y = min.y ; y < max.y; y++)
        {
          id tile = [Tile get_tile_from_key: _tiles[x][y]];
          [tile display_at_x: x y: y];
        }
    }
  
  /* Then the field objects */
  /** \todo it isn't necessary to do it if no object move, we just
      lose time **/
  _field_objects = g_list_sort (_field_objects, (GCompareFunc) compare_func);
  for (GList * i = g_list_first (_field_objects); i ; i = g_list_next (i))
    {
      Field_object * object = (Field_object *) i->data;
      if ([object get_x] < max.x && [object get_y] < max.y && [object get_x] > min.x - MARGIN && [object get_y] > min.y - MARGIN)
        {
          [object display];
        }
    }

  /* Then the other Graphical_objects */
  for (GList * i = g_list_first (_objects); i ; i = g_list_next (i))
    {
      Graphical_object * object = (Graphical_object *) i->data;
      [object display];
    }
  gdk_window_end_paint (_widget->window);
}

/**
 * Sets X and Y offset 
 **/
- (void) set_offset_x: (int) x_offset y: (int) y_offset
{
  _x_offset = x_offset;
  _y_offset = y_offset;
  [self refresh];
}

/**
 * Sets X relative offset 
 **/
- (void) set_relative_offset_x: (int) x
{
  _relative_x_offset = x;
  struct couple offset = {_relative_x_offset - MARGIN, _relative_y_offset - MARGIN};
  offset = isometric_to_absolute (offset);

  [self set_offset_x: offset.x y: offset.y];
}

/**
 * Sets y relative offset 
 **/
- (void) set_relative_offset_y: (int) y
{
  _relative_y_offset = y;
  struct couple offset = {_relative_x_offset - MARGIN, _relative_y_offset - MARGIN};
  offset = isometric_to_absolute (offset);

  [self set_offset_x: offset.x y: offset.y];
}

/**
 * Move offset from X and Y pixels
 **/
- (void) move_offset_x: (int) x_offset y: (int) y_offset
{
  _x_offset += x_offset;
  _y_offset += y_offset;
  [self refresh];
}

/**
 * Return x offset 
 **/
- (int) get_x_offset
{
  return _x_offset;
}

/**
 * Return y offset 
 **/
- (int) get_y_offset
{
  return _y_offset;
}

/** 
 * Return map's width
 **/
- (int) get_width
{
  int width;
  gdk_drawable_get_size (_widget->window, &width, NULL);

  return width;
}

/** 
 * Return map's height
 **/
- (int) get_height
{
  int height;
  gdk_drawable_get_size (_widget->window, NULL, &height);

  return height;
}

/**
 * Width in cases
 **/
- (int) get_x_max
{
  return _cases_x;
}

/**
 * Height in cases
 **/
- (int) get_y_max
{
  return _cases_y;
}
 

/**
 * Return map's widget
 **/
- (GtkWidget *) get_widget
{
  return _widget;
}

/**
 * \brief Create a scrolled window containing the drawing area
 **/
- (GtkWidget *) create_scrolled_window
{
  /* Start calculating min and max for adjustments */
  GtkObject * hadjustment;
  GtkObject * vadjustment;
  GtkWidget * scrolled_window;
  
  /*  hadjustment = gtk_adjustment_new (_x_offset,
    - (CASE_WIDTH * _cases_y / 2), (CASE_WIDTH * (_cases_x)/2), 
    1, PAGE_INCREMENT, 100);*/
  hadjustment = gtk_adjustment_new (0,
    -5, _cases_x,
    1, PAGE_INCREMENT, 1);
  vadjustment = gtk_adjustment_new (_y_offset,
    -5, _cases_y,
    1, PAGE_INCREMENT, 1);
  g_signal_connect (G_OBJECT (hadjustment), "value_changed", 
    G_CALLBACK (x_move_callback), self);
  g_signal_connect (G_OBJECT (vadjustment), "value_changed", 
    G_CALLBACK (y_move_callback), self);
  
  scrolled_window = gtk_scrolled_window_new (GTK_ADJUSTMENT(hadjustment), 
    GTK_ADJUSTMENT (vadjustment));
  gtk_container_add (GTK_CONTAINER (scrolled_window), _widget);


  return scrolled_window;
}

/**
 * \brief Return the object at a given position
 *
 * \param x, y: the (isometric) position. 
 * \return the object positioned at this case, or nil if there isn't. 
 **/
- (id) get_object_at_x: (int) x y: (int) y
{
  /* Only search _field_objects */
  for (GList * i = g_list_first (_field_objects) ; i ; i = g_list_next (i))
    {
      Field_object * object = i->data;
      if ([object get_x] == x && [object get_y] == y)
        {
          return object;
        }
    }

  return nil;
}
@end

