/***********************************************************************
    Copyright (C) 2005 Frdric HENRY <neryel@reveries.info>

    This file is part of NiNaR.

    NiNaR is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published 
    by the Free Software Foundation; either version 2 of the License,
    or (at your option) any later version.

    NiNaR is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of 
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License 
    along with NiNaR; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307  USA.

    File information: $Id: obstacle.m,v 1.7 2005/01/17 00:33:38 neryel Exp $

***********************************************************************/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "obstacle.h"
#include "tools.h"
#include "map.h"

@implementation Obstacle
/**
 * Init all obstacles with some attributes... and a map file.
 *
 * Not that the map file actually isn't like the tile's one: here,
 * either there is this obstacle (any character) or there isn't
 * (space).
 **/
+ (void) init_all_from_attributes: (const char **) attributes values: (const char **) values map: (id) map
{
  int hit_coef = 0;
  const char * picture = NULL;
  char * filename = NULL;
  int width = 0;
  int height = 0;

  for (int i = 0; attributes[i] && values[i]; i++)
    {
      if (!strcmp (attributes[i], "hit_coef"))
        hit_coef = strtol (values[i], NULL, 0);
      if (!strcmp (attributes[i], "map"))
        {
          filename = path_from_neighbour (values[i], map_current_file_name);
        }

      if (!strcmp (attributes[i], "picture"))
        /* No copy, the instances'll do it... or not */
        picture = values[i];
      if (!strcmp (attributes[i], "width"))
        width = strtol (values[i], NULL, 0);
      if (!strcmp (attributes[i], "height"))
        height = strtol (values[i], NULL, 0);
    }
  if (!picture || !filename || !width || !height)
    {
      g_warning ("Obstacle: +init_all_from_(...): not enought attributes, aborting");
      if (filename)
        g_free (filename);
    }
  else
    {
      FILE * file = fopen (filename, "r");
      g_free (filename);
      
      if (! file)
        {
          g_warning ("Obstacle: +init_all_from_(...): can't open file, aborting");
          return;
        }
      /* Read the tiles map */
      for (int y = 0; y < width; y++)
        {
          int c;
          for (int x = 0; x < height; x++)
            {
              c = fgetc (file);
              if (c == EOF)
                {
                  g_warning ("map: load_tiles_map: unexpected EOF in %s", filename);
                  fclose (file);
                  return;
                }
              else if (c != ' ')
                {
                  id obstacle = [[self alloc] init];
                  [obstacle load_picture :picture];
                  [obstacle set_hit_coef :hit_coef];
                  [obstacle set_x: x y: y];
                  [map add_object: obstacle];
                }
            }
          /* Go to end of line */
          while ((c = fgetc (file)) != '\n')
            {
              if (c == EOF)
                {
                  g_warning ("obstacle: init_all_from_(...): unexpected EOF in %s", filename);
                  fclose (file);
                  return;
                }
            }
        }
    }
}

/**
 * Init from attributes and values of an XML element. 
 **/
- init_from_attributes: (const char **) attributes values: (const char **) values
{
  [super init_from_attributes: attributes values: values];
  for (int i = 0; attributes[i] && values[i]; i++)
    {
      if (!strcmp (attributes[i], "hit_coef"))
        [self set_hit_coef: strtol (values[i], NULL, 0)];
    }

  return self;
}

/**
 * Set hit coef 
 **/
- (void) set_hit_coef: (int) coef
{
  _hit_coef = coef;
}

/** 
 * Returns hit coef 
 **/
- (int) get_hit_coef
{
  return _hit_coef;
}

/**
 * \brief Overrides Field_object method
 *
 * Since Obstacles have format (Width, Width) instead of (Width,
 * Height), we have to do a translation
 **/
- (void) reposition
{
  [super reposition];
  [self move_absolute_x: 0 y: - (CASE_WIDTH - CASE_HEIGHT)];
}
@end
