/***********************************************************************
    Copyright (C) 2005 Frdric HENRY <neryel@reveries.info>

    This file is part of NiNaR.

    NiNaR is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published 
    by the Free Software Foundation; either version 2 of the License,
    or (at your option) any later version.

    NiNaR is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of 
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License 
    along with NiNaR; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307  USA.

    File information: $Id: tile.m,v 1.3 2005/01/15 12:18:32 neryel Exp $

***********************************************************************/

#include <glib.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "tile.h"

#define BUFFER_SIZE 42

/**
 * Structure used for the mapping between char and tile
 **/
struct mapping
{
  char key;
  id tile;
};

/* List of the char_tiles */
static GList * tiles_list = NULL;

/* Callback function used for parsing the xml file */
static void
tile_create (GMarkupParseContext * context __attribute__((unused)),
  const gchar * element_name, 
  const gchar ** attributes, const gchar ** values,
  gpointer data __attribute__((unused)),
  GError ** error)
{
  if (*error)
    {
      g_warning ("tile_create: Parse error occured");
      g_warning ("%s", (*error)->message);
    }

  if (strcmp (element_name, "tile"))
    {
      g_warning ("Only know <tile /> element, ignoring <%s>", element_name);
      return;
    }

  id tile = [[Tile alloc] init];
  char key;

  for (int i = 0; attributes[i] && values[i]; i++)
    {
      if (!strcmp (attributes[i], "picture"))
        {
          [tile load_picture: values[i]];
        }
      else if (!strcmp (attributes[i], "cost"))
        {
          [tile set_cost: strtol (values[i], NULL, 0)];
        }
      else if (!strcmp (attributes[i], "key"))
        {
          key = values[i][0];
        }
      else 
        {
          g_warning ("tile_create: attribute %s not recognized, ignoring.", attributes[i]);
        }
    }
  [Tile set_tile: tile to_key: key];
  
}

@implementation Tile
/**
 * Map a tile to a char
 **/
+ (void) set_tile: (id) tile to_key: (char) c
{
  struct mapping *m = g_malloc (sizeof (struct mapping));
  m->key = c;
  m->tile = tile;
  tiles_list = g_list_append (tiles_list, m);
}

/**
 * Get a tile knowing the char... or nil if there is no one
 **/
+ (id) get_tile_from_key: (char) c
{
  for (GList * i = g_list_first (tiles_list) ; i ; i = g_list_next (i))
    {
      struct mapping * m = i->data;
      if (m->key == c)
        {
          return m->tile;
        }
    }
  return nil;
}

/**
 * Free every tiles and mappings
 **/
+ (void) free_all
{
  if (tiles_list)
    {
      for (GList * i = g_list_first (tiles_list) ; i ; i = g_list_next (i))
        {
          struct mapping * m = i->data;
          [m->tile free];
          g_free (m);
        }
      g_list_free (tiles_list);
      tiles_list = NULL;
    }
}

/**
 * Load a file, creating the tiles it contains and preparing the
 * mapping between these tiles and chars
 **/
+ (void) load_tiles: (const char *) filename
{
  FILE * file = fopen (filename, "r");
  g_return_if_fail (filename);

  char buffer[BUFFER_SIZE];
  gboolean next = TRUE;
  GMarkupParseContext * context;
  GMarkupParser parser = {tile_create, NULL, NULL, NULL, NULL};

  context = g_markup_parse_context_new (&parser, 0, NULL, NULL);

  while (next && fgets (buffer, BUFFER_SIZE, file))
    {
      next = g_markup_parse_context_parse (context,
        buffer, BUFFER_SIZE-1, NULL);
      if (!next)
        {
          g_warning ("Error parsing tiles");
        }
    }

  g_markup_parse_context_free (context);
  fclose (file);
}

/**
 * Set tile cost 
 **/
- (void) set_cost: (unsigned) cost
{
  _cost = cost;
}

/**
 * Get tile cost
 **/
- (unsigned) get_cost
{
  return _cost;
}

/**
 * \brief Display the tile at the appropriate position
 *
 * This is used so it is not necessary to have a tile by case
 **/
- (void) display_at_x: (int) x y: (int) y
{
  [self set_x: x y: y];
  [self display];
}
@end
