/***********************************************************************
    Copyright (C) 2005 Frdric HENRY <neryel@reveries.info>

    This file is part of NiNaR.

    NiNaR is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published 
    by the Free Software Foundation; either version 2 of the License,
    or (at your option) any later version.

    NiNaR is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of 
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License 
    along with NiNaR; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307  USA.

    File information: $Id: tools.m,v 1.4 2005/01/17 00:33:38 neryel Exp $

***********************************************************************/

#include <glib.h>
#include <string.h>

#include "tools.h"

/**
    Functions used for changing referential
    Note that the 'absolute' positions are NOT the pixels in the
    drawable, but the position in the map, which can be negative. The
    map's offsets must still be applied to these values.
    Also note that the absolute point given correspond to the relative
    point of the top-left corner, which gives the absolute top-center
    point. In order to have the absolute top-left point, a translation
    of -CASE_WIDTH/2 on the x axis is required.
**/
/* Relative (isometric field) to absolute (the drawable) */
struct couple
isometric_to_absolute (struct couple isometric)
{
  struct couple absolute;
  absolute.x = (isometric.x - isometric.y) * (CASE_WIDTH / 2);
  absolute.y = (isometric.x + isometric.y) * (CASE_HEIGHT / 2);

  return absolute;
}

/* The inverse :-) */
struct couple
absolute_to_isometric (struct couple absolute)
{
  struct couple isometric;
  isometric.x = (int) ((double) absolute.x / CASE_WIDTH + (double) absolute.y / CASE_HEIGHT);
  isometric.y = (int) ((double) absolute.y / CASE_HEIGHT - (double) absolute.x / CASE_WIDTH);
  
  return isometric;
}


/**
 * Returns the result of a binomial law.
 *
 * \param proba: the probability of one attempt. If it is > 1, it
 * becomes a multiplicator instead o a probability.
 * be > 1 or < 0, but it will be considered as 1 or 0.
 * \param repetitions: number of repetitions: similarly, there can be
 * less than 0 repetitons...
 **/
int
binomial (double proba, int repetitions)
{
  int result = 0;
  if (proba <= 1)
    {
      for (int i = 0; i < repetitions; i++)
        {
          if (g_random_double () < proba)
            {
              result++;
            }
        }
    }
  else
    {
      result = proba * repetitions;
    }
  
  return result;
}

/**
 * Convert a local path pointed by neighbour to an absolute (or
 * relative, but working) path. E.g., path_from_neighbour ("foo",
 * "/usr/plop/bar") will return "/usr/plop/foo".
 *
 * \return the working path, who must be freed with g_free ()
 **/
char *
path_from_neighbour (const char * filename, const char * neighbour)
{
  char * last_slash = strrchr (neighbour, '/');
  if (last_slash == NULL)
    {
      return g_strdup (filename);
    }
  else
    {
      int len_to_slash = last_slash - neighbour;

      /* Allocate new filename - the + 2 is for the \0 and the / */
      char * new_filename = g_malloc (len_to_slash + strlen(filename) + 2);
      strncpy (new_filename, neighbour, len_to_slash+1);
      strcpy ((new_filename + len_to_slash + 1), filename);
      return new_filename;
    }
}
