/***********************************************************************
    Copyright (C) 2005 Frdric HENRY <neryel@reveries.info>

    This file is part of NiNaR.

    NiNaR is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published 
    by the Free Software Foundation; either version 2 of the License,
    or (at your option) any later version.

    NiNaR is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of 
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License 
    along with NiNaR; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307  USA.

    File information: $Id: wall.m,v 1.8 2005/01/16 18:56:07 neryel Exp $

***********************************************************************/

#include <glib.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "map.h"
#include "wall.h"
#include "tools.h"

/* Extension for south/north and east/west */
#define END_SOUTH_NORTH "_ns.png"
#define END_EAST_WEST "_ew.png"

@implementation Wall
/**
 * Free allocated memory
 **/
- free
{
  if (_base_filename)
    {
      g_free (_base_filename);
    }
  return [super free];
}

/**
 * \brief Overrides Graphical_object display method.
 *
 * Since we don't load the picture in load_picture, we have to do it
 * here the first time the message is received.
 **/
- (void) display
{
  /* Simplest case : the picture has already been loaded, great, we
     suppose x and y are correct since usually wall don't move. */
  if (_picture)
    {
      [super display];
    }
  /* If the picture has not been loaded... it is a bit more
     complicated. First, we check if we need to load it at all,
     then we see if there is some adjacent wall and choose the
     correct picture. */
  else if (_do_not_load)
    {
      /* OK, nothing to display */
    }
  else 
    {
      id object;

      /* Checks if it is an E-W wall */
      object = [_map get_object_at_x: (_x - 1) y: _y];
      if (object && [object isKindOf: [Wall class]])
        {
          gchar * full_filename = g_strdup_printf ("%s%s", 
            _base_filename, END_EAST_WEST);
          [super load_picture: full_filename];
          if (!_picture)
            {
              g_warning ("Problem opening image file");
              return;
            }
          /* The move isn't on this case, but between this one and
             this precedent, so there is an offset */
          [self move_absolute_x: (- CASE_WIDTH / 4) y: (- CASE_HEIGHT / 4)];
          [self display];
          g_free (full_filename);
        }

      /* Checks if it is a S-N wall */
      object = [_map get_object_at_x: _x y: (_y - 1)];
      if (object && [object isKindOf: [Wall class]])
        {
          gchar * full_filename = g_strdup_printf ("%s%s", 
            _base_filename, END_SOUTH_NORTH);
          if (!_picture)
            {
              /* OK, just a S-N wall */
              [super load_picture: full_filename];
              if (!_picture)
                {
                  g_warning ("Problem opening image file");
                  return;
                }
              [self move_absolute_x: (CASE_WIDTH / 4) y: (- CASE_HEIGHT / 4)];
              [self display];
            }
          else
            {
              /* Uh-oh. A S-N wall AND a E-W wall. Let's compose the
                 two pictures. */
              /** \todo the result is quite ugly, isn't there another
                  way to do it ? Or maybe it comes from the pic ? **/
              GdkPixbuf * picture_ew;
              GdkPixbuf * picture_ns;

              picture_ew = _picture;
              [super load_picture: full_filename];
              picture_ns = _picture;

              _picture = gdk_pixbuf_copy (picture_ew);
              /* Fill with transparent black */
              gdk_pixbuf_fill (_picture, 0x0);
              gdk_pixbuf_copy_area (picture_ns,
                0, 0,
                CASE_WIDTH - CASE_WIDTH/4, CASE_WIDTH - CASE_HEIGHT/4,
                _picture,
                CASE_WIDTH/4, CASE_HEIGHT/4);
              gdk_pixbuf_copy_area (picture_ew,
                CASE_WIDTH/4, 0,
                CASE_WIDTH/2, CASE_WIDTH-CASE_HEIGHT/4,
                _picture,
                0, CASE_HEIGHT/4);

              gdk_pixbuf_unref (picture_ew);
              gdk_pixbuf_unref (picture_ns);

              [self move_absolute_x: CASE_WIDTH/4 y: -CASE_HEIGHT/4]; /** ??? */
              [self display];
            }                
          g_free (full_filename);
        }
    }
}

/**
 * \brief Overrides Graphical_object method
 *
 * Since we still don't know if it is a north/south wall or an
 * east/west wall, we can't load the picture yet, so we just store the
 * base filename, and we will had "_ns.png" or "_ew.png". 
 * 
 * \param The base filename, ie without "_ns.png" or "_ew.png". For
 * example "wall_brick". 
 **/
- (void) load_picture: (const char *) filename
{
  _base_filename = g_strdup (filename);
}

/**
 * \brief Overrides Field_object method
 *
 * It's little probable, but if a wall move, we can't assume the
 * calculs done to see if it is a S-N or E-W be true, so we must
 * show it must be recomputed.
 **/
- (void) set_x: (int) x y: (int) y
{
  /* This also influence neighbour walls... or maybe we receive this
     message from a neighbour wall so it isn't necessary */
  if (_x != x || _y == y)
    {
      id wall;
      
      wall = [_map get_object_at_x: (x+1) y: y];
      if ([wall isKindOf: [Wall class]])
        {
          [wall set_x: (x+1) y: y];
        }
      wall = [_map get_object_at_x: x y: (y+1)];
      if ([wall isKindOf: [Wall class]])
        {
          [wall set_x: (x+1) y: y];
        }
    }

  /* Now for us */
  [super set_x: x y: y];

  _do_not_load = FALSE;
  if (_picture)
    {
      gdk_pixbuf_unref (_picture);
      _picture = NULL;
    }
}
@end
