/***********************************************************************
    Copyright (C) 2005 Frdric HENRY <neryel@reveries.info>

    This file is part of NiNaR.

    NiNaR is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published 
    by the Free Software Foundation; either version 2 of the License,
    or (at your option) any later version.

    NiNaR is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of 
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License 
    along with NiNaR; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307  USA.

    File information: $Id: weapon.m,v 1.5 2005/01/17 00:33:38 neryel Exp $

***********************************************************************/

#include <glib.h>
#include <string.h>
#include <stdlib.h>

#include "weapon.h"

/* Max range (where hit probability reaches zero) for a weapon (time
   the range) */
#define MAX_RANGE 2

/* Size for the buffer between the parser and the file */
#define BUFFER_SIZE 42

/* Weapon list */
static GList * _weapons = NULL;

/* Callback function used for parsing the xml file */
static void
weapon_create (GMarkupParseContext * context __attribute__((unused)),
  const gchar * element_name,
  const gchar ** attributes, const gchar ** values,
  gpointer data __attribute__((unused)),
  GError ** error)
{
  if (*error)
    {
      g_warning ("weapon_create: Parse error occured");
      g_warning ("%s", (*error)->message);
    }

  if (!strcmp ("weapon", element_name))
    {
      id weapon = [[Weapon alloc] init_from_attributes: attributes values: values];
      _weapons = g_list_append (_weapons, weapon);
    }
}



@implementation Weapon
/**
 * Loads all weapons from an XML file.
 * This file can contain other elements, but only the weapons
 * will be loaded.
 **/
+ (void) add_all_from_file: (const char *) filename
{
  FILE * file = fopen (filename, "r");
  g_return_if_fail (filename);

  char buffer[BUFFER_SIZE];
  gboolean next = TRUE;
  GMarkupParseContext * context;
  GMarkupParser parser = {weapon_create, NULL, NULL, NULL, NULL};

  context = g_markup_parse_context_new (&parser, 0, NULL, NULL);

  while (next && fgets (buffer, BUFFER_SIZE, file))
    {
      next = g_markup_parse_context_parse (context,
        buffer, BUFFER_SIZE-1, NULL);
      if (!next)
        {
          g_warning ("Weapon:add_all_from_file: '%s': error parsing weapons", filename);
        }
    }

  g_markup_parse_context_free (context);
  fclose (file);
}

/**
 * Free all weapons
 **/
+ (void) free_all
{
  for (GList * i = g_list_first (_weapons); i; i = g_list_next (i))
    {
      id weapon = i->data;
      [weapon free];
    }
  g_list_free (_weapons);
}

/**
 * Get a weapon knowing its name, or nil if weapon doesn't exist.
 **/
+ (id) get_weapon_from_name: (const char *) name
{
  for (GList * i = g_list_first (_weapons); i; i = g_list_next (i))
    {
      id weapon = i->data;
      if (!strcmp (name, [weapon get_name]))
        {
          return weapon;
        }
    }

  return nil;
}

/**
 * Free the weapon
 **/
- free
{
  if (_name)
    {
      g_free (_name);
      _name = NULL;
    }
  return [super free];
}

/**
 * Basic init
 **/
- init
{
  _name = NULL;
  _hit_proba = 0;
  _range = 0;
  _ammos = 0;
  _power = 0;

  return self;
}

/**
 * \brief Init the weapon from attributes of an XML element
 *
 * Beware: this method is only called by add_all_from_file not by the
 * map loader. So you can't have a weapon in just one map. Or you can,
 * but you'll have to add_all_from_file: this map too. 
 *
 * We don't check is everything has been set or not, either 
 **/
- init_from_attributes: (const char **) attributes values: (const char **) values
{
  [self init];
  
  for (int i = 0; attributes[i] && values[i]; i++)
    {
      if (!strcmp ("name", attributes[i]))
        [self set_name: values[i]];
      else if (!strcmp ("hit_proba", attributes[i]))
        _hit_proba = strtod (values[i], NULL);
      else if (!strcmp ("range", attributes[i]))
        _range = strtol (values[i], NULL, 0);
      else if (!strcmp ("ammos", attributes[i]))
        _ammos = strtol (values[i], NULL, 0);
      else if (!strcmp ("power", attributes[i]))
        _power = strtol (values[i], NULL, 0);
    }
  return self;
}

/**
 * The weapon name
 *
 * Name should not have uppercase letter
 **/
- (const char *) get_name
{
  return _name;
}

/**
 * Sets the weapon name
 **/
- (void) set_name: (const char *) name
{
  if (_name)
    {
      g_free (_name);
    }
  _name = g_strdup (name);
}

/**
 * Hit probability
 *
 * Probability of the weapon to hit.
 * 
 * In battle, use get_hit_proba_at_distance instead.
 **/
- (double) get_hit_proba
{
  return _hit_proba;
}

/**
 * Hit probability at a certain distance
 *
 * Depends of the distance, _hit_proba, and _range, like this :
 *
 * \verbatim
 *
 *       /|\ probability
 * _hit_  |
 * proba -|---------------
 *        |                `,
 *        |                  `,
 *        |                    `,  
 *      0 |_____________________________________________\ distance
 *        |0       _range |       | MAX_RANGE * _range   /
 * 
 * \endverbatimg
 **/
- (double) get_hit_proba_at_distance: (int) distance
{
  if (distance <= _range)
    {
      return _hit_proba;
    }
  else if (distance > MAX_RANGE * _range)
    {
      return 0;
    }
  else
    {
      return (_hit_proba - ((distance - _range) * (_hit_proba/_range)));
    }
}

/**
 * Range of the weapon
 *
 * If there is more distance between source and target than the range,
 * the weapon won't have full hit chance.
 *
 * For battle computations, use get_hit_proba_at_distance instead.
 **/
- (int) get_range
{
  return _range;
}

/**
 * Gets the number of ammos
 *
 * The ammos correspond to the number of time a weapon can fire each
 * turn, not to its actual capacity, whose we don't care.
 **/
- (int) get_ammos
{
  return _ammos;
}

/**
 * Gets the weapon power 
 **/
- (int) get_power
{
  return _power;
}
@end
