/* Copyright (C) 1999--2001 Chris Vaill
   This file is part of normalize.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#define _POSIX_C_SOURCE 2

#include "config.h"

#if STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#else
# if HAVE_STDLIB_H
#  include <stdlib.h>
# endif
# if HAVE_STRING_H
#  include <string.h>
# else
#  ifndef HAVE_STRCHR
#   define strchr index
#   define strrchr rindex
#  endif
#  ifndef HAVE_MEMCPY
#   define memcpy(d,s,n) bcopy((s),(d),(n))
#   define memmove(d,s,n) bcopy((s),(d),(n))
#  endif
# endif
#endif
#if HAVE_MATH_H
# include <math.h>
#endif
#if HAVE_ERRNO_H
# include <errno.h>
#endif

#ifdef ENABLE_NLS
# define _(msgid) gettext (msgid)
# include <libintl.h>
#else
# define _(msgid) (msgid)
#endif
#define N_(msgid) (msgid)

#include "riff.h"
#include "common.h"

/* warn about clipping if we clip more than this fraction of the samples */
#define CLIPPING_WARN_THRESH 0.001

extern void progress_callback(char *prefix, float fraction_completed);
extern void *xmalloc(size_t size);

/* FIXME: remove, use audiofile-style interface */
extern riff_chunk_t *get_wav_data(riff_t *riff, struct wavfmt *fmt);

extern char *progname;
extern int verbose;
extern int use_limiter;
extern double lmtr_lvl;

static __inline__ long
get_sample(unsigned char *pdata, int bytes_per_sample)
{
  long sample;

  switch(bytes_per_sample) {
  case 1:
    sample = *pdata - 128;
    break;
  case 2:
#ifdef WORDS_BIGENDIAN
    sample = *((int8_t *)pdata + 1) << 8;
    sample |= *((int8_t *)pdata) & 0xFF;
#else
    sample = *((int16_t *)pdata);
#endif
    break;
  case 3:
    sample = *((int8_t *)pdata + 2) << 16;
    sample |= (*((int8_t *)pdata + 1) << 8) & 0xFF00;
    sample |= *((int8_t *)pdata) & 0xFF;
    break;
  case 4:
    sample = *((int32_t *)pdata);
#ifdef WORDS_BIGENDIAN
    sample = bswap_32(sample);
#endif
    break;
  default:
    /* shouldn't happen */
    fprintf(stderr,
	    _("%s: I don't know what to do with %d bytes per sample\n"),
	    progname, bytes_per_sample);
    sample = 0;
  }

  return sample;
}


static __inline__ void
put_sample(long sample, unsigned char *pdata, int bytes_per_sample)
{
  switch(bytes_per_sample) {
  case 1:
    *pdata = sample + 128;
    break;
  case 2:
#ifdef WORDS_BIGENDIAN
    sample = bswap_16(sample);
#endif
    *((int16_t *)pdata) = (int16_t)sample;
    break;
  case 3:
    *pdata = (unsigned char)sample;
    *(pdata + 1) = (unsigned char)(sample >> 8);
    *(pdata + 2) = (unsigned char)(sample >> 16);
    break;
  case 4:
#ifdef WORDS_BIGENDIAN
    sample = bswap_32(sample);
#endif
    *((int32_t *)pdata) = (int32_t)sample;
    break;
  default:
    /* shouldn't happen */
    fprintf(stderr,
	    _("%s: I don't know what to do with %d bytes per sample\n"),
	    progname, bytes_per_sample);
  }
}


/*
 * Limiter function:
 *
 *        / tanh((x + lev) / (1-lev)) * (1-lev) - lev        (for x < -lev)
 *        |
 *   x' = | x                                                (for |x| <= lev)
 *        |
 *        \ tanh((x - lev) / (1-lev)) * (1-lev) + lev        (for x > lev)
 *
 * With limiter level = 0, this is equivalent to a tanh() function;
 * with limiter level = 1, this is equivalent to clipping.
 */
static double
limiter(double x)
{
  double xp;

  if (x < -lmtr_lvl)
    xp = tanh((x + lmtr_lvl) / (1-lmtr_lvl)) * (1-lmtr_lvl) - lmtr_lvl;
  else if (x <= lmtr_lvl)
    xp = x;
  else
    xp = tanh((x - lmtr_lvl) / (1-lmtr_lvl)) * (1-lmtr_lvl) + lmtr_lvl;

  return xp;
}


/*
 * input is read from read_fd and output is written to write_fd:
 * filename is used only for messages.
 *
 * The psi pointer gives the peaks so we know if limiting is needed
 * or not.  It may be specified as NULL if this information is not
 * available.
 */
int
apply_gain(int read_fd, int write_fd, char *filename, double gain,
	   struct signal_info *psi)
{
  riff_t *riff;
  riff_chunk_t *chnk;
  struct wavfmt fmt;
  unsigned int nsamples, samples_done, nclippings;
  int bytes_per_sample, i;
  long sample, samplemax, samplemin;
  float clip_loss;
  FILE *rd_stream, *wr_stream;

  float last_progress = 0, progress;
  char prefix_buf[18];

  unsigned char *data_buf = NULL;
  int samples_in_buf, samples_recvd;
  int use_limiter_this_file;
#if USE_LOOKUPTABLE
  int min_pos_clipped; /* the minimum positive sample that gets clipped */
  int max_neg_clipped; /* the maximum negative sample that gets clipped */
  int16_t *lut = NULL;
#endif

  riff = riff_new(read_fd, RIFF_RDONLY);
  if (riff == NULL) {
    fprintf(stderr, _("%s: error making riff object\n"), progname);
    goto error1;
  }

  chnk = get_wav_data(riff, &fmt);
  if (chnk == NULL) {
    fprintf(stderr, _("%s: error getting wav data\n"), progname);
    goto error2;
  }

  bytes_per_sample = (fmt.bits_per_sample - 1) / 8 + 1;
  samplemax = (1 << (bytes_per_sample * 8 - 1)) - 1;
  samplemin = -samplemax - 1;

  /* ignore different channels, apply gain to all samples */
  nsamples = chnk->size / bytes_per_sample;

  /* set up sample buffer to hold 1/100 of a second worth of samples */
  /* (make sure it can hold at least the wav header, though) */
  samples_in_buf = (fmt.samples_per_sec / 100) * fmt.channels;
  if (chnk->offset + 8 > samples_in_buf * bytes_per_sample)
    data_buf = (unsigned char *)xmalloc(chnk->offset + 8);
  else
    data_buf = (unsigned char *)xmalloc(samples_in_buf * bytes_per_sample);

  /* open streams for reading and writing */
  rd_stream = fdopen(read_fd, "rb");
  wr_stream = fdopen(write_fd, "wb");
  if (rd_stream == NULL || wr_stream == NULL) {
    fprintf(stderr, _("%s: failed fdopen: %s\n"), progname, strerror(errno));
    goto error4;
  }
  /* copy the wav header */
  rewind(rd_stream);
  rewind(wr_stream);
  if (fread(data_buf, chnk->offset + 8, 1, rd_stream) < 1) {
    fprintf(stderr, _("%s: read failed: %s\n"), progname, strerror(errno));
    goto error4;
  }
  if (fwrite(data_buf, chnk->offset + 8, 1, wr_stream) < 1) {
    fprintf(stderr, _("%s: write failed: %s\n"), progname, strerror(errno));
    goto error4;
  }

  /*
   * Check if we actually need to do limiting on this file:
   * we don't if gain <= 1 or if the peaks wouldn't clip anyway.
   */
  use_limiter_this_file = use_limiter && gain > 1.0;
  if (use_limiter_this_file && psi) {
    if (psi->max_sample * gain <= samplemax
	&& psi->min_sample * gain >= samplemin)
      use_limiter_this_file = FALSE;
  }

#if USE_LOOKUPTABLE
  /*
   * If samples are 16 bits or less, build a lookup table for fast
   * adjustment.  This table is 128k, look out!
   */
  if (bytes_per_sample <= 2) {
    lut = (int16_t *)xmalloc((samplemax - samplemin + 1) * sizeof(int16_t));
    lut -= samplemin; /* so indices don't have to be offset */
    min_pos_clipped = samplemax + 1;
    max_neg_clipped = samplemin - 1;
    if (gain > 1.0) {
      if (use_limiter_this_file) {
	/* apply gain, and apply limiter to avoid clipping */
	for (i = samplemin; i < 0; i++)
	  lut[i] = ROUND(-samplemin * limiter(i * gain / (double)-samplemin));
	for (; i <= samplemax; i++)
	  lut[i] = ROUND(samplemax * limiter(i * gain / (double)samplemax));
#if DEBUG
	{
	  /* write the lookup table function for display in gnuplot */
	  FILE *tblout = fopen("lut.dat", "w");
	  for (i = samplemin; i <= samplemax; i++)
	    fprintf(tblout, "%d %d\n", i, lut[i]);
	  fclose(tblout);
	}
#endif
      } else {
	/* apply gain, and do clipping */
	for (i = samplemin; i <= samplemax; i++) {
	  sample = i * gain;
	  if (sample > samplemax) {
	    sample = samplemax;
	    if (i < min_pos_clipped)
	      min_pos_clipped = i;
	  } else if (sample < samplemin) {
	    sample = samplemin;
	    if (i > max_neg_clipped)
	      max_neg_clipped = i;
	  }
	  lut[i] = sample; /* negative indices are okay, see above */
	}
      }
    } else {
      /* just apply gain if it's less than 1 */
      for (i = samplemin; i <= samplemax; i++)
	lut[i] = i * gain;
    }
  }
#endif

  /* initialize progress meter */
  if (verbose >= VERBOSE_PROGRESS) {
    if (strrchr(filename, '/') != NULL) {
      filename = strrchr(filename, '/');
      filename++;
    }
    strncpy(prefix_buf, filename, 17);
    prefix_buf[17] = 0;
    progress_callback(prefix_buf, 0.0);
    last_progress = 0.0;
  }

  /* read, apply gain, and write, one chunk at time */
  nclippings = samples_done = 0;
  while ((samples_recvd = fread(data_buf, bytes_per_sample,
				samples_in_buf, rd_stream)) > 0) {
#if USE_LOOKUPTABLE
    if (lut) {
      /* use the lookup table if we built one */

      for (i = 0; i < samples_recvd; i++) {
	sample = get_sample(data_buf + (i * bytes_per_sample), bytes_per_sample);

	if (sample >= min_pos_clipped || sample <= max_neg_clipped)
	  nclippings++;
	sample = lut[sample];

	put_sample(sample, data_buf + (i * bytes_per_sample), bytes_per_sample);
      }

    } else {
#endif
      /* no lookup table, do it by hand */

      for (i = 0; i < samples_recvd; i++) {
	sample = get_sample(data_buf + (i * bytes_per_sample), bytes_per_sample);

	/* apply the gain to the sample */
	sample *= gain;

	if (gain > 1.0) {
	  if (use_limiter_this_file) {
	    /* do tanh limiting instead of clipping */
	    sample = samplemax * limiter(sample / (double)samplemax);
	  } else {
	    /* perform clipping */
	    if (sample > samplemax) {
	      sample = samplemax;
	      nclippings++;
	    } else if (sample < samplemin) {
	      sample = samplemin;
	      nclippings++;
	    }
	  }
	}

	put_sample(sample, data_buf + (i * bytes_per_sample), bytes_per_sample);
      }
#if USE_LOOKUPTABLE
    }
#endif

    if (fwrite(data_buf, bytes_per_sample,
	       samples_recvd, wr_stream) == 0) {
      fprintf(stderr, _("%s: failed fwrite: %s\n"), progname, strerror(errno));
    }

    samples_done += samples_recvd;

    /* update progress meter */
    if (verbose >= VERBOSE_PROGRESS) {
      progress = samples_done / (float)nsamples;
      if (progress >= last_progress + 0.01) {
	progress_callback(prefix_buf, progress);
	last_progress += 0.01;
      }
    }
  }

  /* make sure progress meter is finished */
  if (verbose >= VERBOSE_PROGRESS)
    progress_callback(prefix_buf, 1.0);

  if (fflush(wr_stream) == -1)
    fprintf(stderr, _("%s: failed fflush: %s\n"), progname, strerror(errno));

  if (!use_limiter_this_file) {
    clip_loss = (float)nclippings / (float)nsamples;

    if (verbose >= VERBOSE_INFO) {
      if (nclippings) {
	fprintf(stderr, "\n");
	fprintf(stderr, _("%s: %d clippings performed, %.4f%% loss\n"),
		progname, nclippings, clip_loss * 100);
      }
    } else if (verbose >= VERBOSE_PROGRESS) {
      if (clip_loss > CLIPPING_WARN_THRESH)
	fprintf(stderr,
        _("%s: Warning: lost %0.2f%% of data due to clipping              \n"),
		progname, clip_loss * 100);
    }
  }

#if USE_LOOKUPTABLE
  if (lut) {
    /* readjust the pointer to the beginning of the array */
    lut += samplemin;
    free(lut);
  }
#endif
  free(data_buf);
  riff_chunk_unref(chnk);
  riff_unref(riff);
  return 0;


  /* error handling stuff */
 error4:
  free(data_buf);
  /*error3:*/
  riff_chunk_unref(chnk);
 error2:
  riff_unref(riff);
 error1:
  return -1;
}
