/* Copyright (C) 1999--2001 Chris Vaill
   This file is part of normalize.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#define _POSIX_C_SOURCE 2

#include "config.h"

#include <stdio.h>
#include <time.h>
#if STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#else
# if HAVE_STDLIB_H
#  include <stdlib.h>
# endif
# if HAVE_STRING_H
#  include <string.h>
# else
#  ifndef HAVE_STRCHR
#   define strchr index
#   define strrchr rindex
#  endif
#  ifndef HAVE_MEMCPY
#   define memcpy(d,s,n) bcopy((s),(d),(n))
#   define memmove(d,s,n) bcopy((s),(d),(n))
#  endif
# endif
#endif
#if HAVE_MATH_H
# include <math.h>
#endif
#if HAVE_CTYPE_H
# include <ctype.h>
#endif
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#if HAVE_ERRNO_H
# include <errno.h>
#endif
#if HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#if HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#if HAVE_FCNTL_H
# include <fcntl.h>
#endif
#if HAVE_SYS_MMAN_H
# include <sys/mman.h>
#endif

#ifdef ENABLE_NLS
# define _(msgid) gettext (msgid)
# include <libintl.h>
# if HAVE_LOCALE_H
#  include <locale.h>
# endif
#else
# define _(msgid) (msgid)
#endif
#define N_(msgid) (msgid)

#include "getopt.h"
#include "common.h"

#define USE_TEMPFILE 0

#define AMPTODBFS(x) (20 * log10(x))
#define FRACTODB(x) (20 * log10(x))
#define DBFSTOAMP(x) pow(10,(x)/20.0)
#define DBTOFRAC(x) pow(10,(x)/20.0)

extern double signal_max_power(int fd, char *filename,
			       struct signal_info *psi);
extern double signal_max_power_stream(FILE *in, char *filename,
				      struct signal_info *psi);
extern int apply_gain(int read_fd, int write_fd, char *filename, double gain,
		      struct signal_info *psi);

void compute_levels(struct signal_info *sis, char **fnames, int nfiles);
double average_levels(struct signal_info *sis, int nfiles, double threshold);
#if 0
double amp_to_dBFS(double x);
double frac_to_dB(double x);
double dBFS_to_amp(double x);
double dB_to_frac(double x);
#endif
#if USE_TEMPFILE
int xmkstemp(char *template);
int xrename(const char *oldpath, const char *newpath);
#endif
int strncaseeq(const char *s1, const char *s2, size_t n);
void *xmalloc(size_t size);

extern char version[];
char *progname;
struct progress_struct progress_info;


void
usage()
{
  fprintf(stderr, _("\
Usage: %s [OPTION]... [FILE]...\n\
Normalize volume of multiple WAV files\n\
\n\
  -a, --amplitude=AMP          normalize the volume to the target amplitude\n\
                                 AMP [default 0.25 or -12dBFS]\n\
  -b, --batch                  batch mode: get average of all levels, and\n\
                                 use one adjustment, based on the average\n\
                                 level, for all files\n\
      --clipping               turn off limiter; do clipping instead\n\
      --fractions              display levels as fractions of maximum\n\
                                 amplitude instead of decibels\n\
  -g, --gain=ADJ               don't compute levels, just apply adjustment\n\
                                 ADJ to the files.  Use the suffix \"dB\"\n\
                                 to indicate a gain in decibels.\n\
  -l, --limiter=LEV            limit all samples above LEV [default -6dBFS]\n\
  -m, --mix                    mix mode: get average of all levels, and\n\
                                 normalize volume of each file to the\n\
                                 average\n\
  -n, --no-adjust              compute and display the volume adjustment,\n\
                                 but don't apply it to any of the files\n\
      --peak                   adjust by peak level instead of using\n\
                                 loudness analysis\n\
  -q, --quiet                  quiet (decrease verbosity to zero)\n\
  -t, --average-threshold=TH   when computing average level, ignore any\n\
                                 levels more than TH decibels from average\n\
  -T, --adjust-threshold=TH    don't bother applying any adjustment smaller\n\
                                 than TH decibels\n\
  -v, --verbose                increase verbosity\n\
\n\
  -V, --version                display version information and exit\n\
  -h, --help                   display this help and exit\n\
\n\
Report bugs to <cvaill@cs.columbia.edu>.\n"), progname);
}

enum {
  OPT_CLIPPING     = 1,
  OPT_PEAK         = 2,
  OPT_FRACTIONS    = 3
};

/* arguments */
int verbose = VERBOSE_PROGRESS;
int do_print_only = FALSE;
int do_apply_gain = TRUE;
double target = 0.25;
double threshold = -1.0; /* in decibels */
int do_compute_levels = TRUE;
int gain_in_decibels = FALSE;
int batch_mode = FALSE;
int mix_mode = FALSE;
int use_limiter = TRUE;
int use_peak = FALSE;
int use_fractions = FALSE;
double lmtr_lvl = 0.5;
double adjust_thresh = 0.125; /* don't adjust less than this many dB */

int
main(int argc, char *argv[])
{
  int fd, fd2, c, i, nfiles;
  struct signal_info *sis, *psi;
  double level, gain = 1.0, dBdiff;
  char **fnames, *p;
  struct stat st;
  int files_changed = FALSE;

  struct option longopts[] = {
    {"help", 0, NULL, 'h'},
    {"version", 0, NULL, 'V'},
    {"no-adjust", 0, NULL, 'n'},
    {"quiet", 0, NULL, 'q'},
    {"verbose", 0, NULL, 'v'},
    {"batch", 0, NULL, 'b'},
    {"amplitude", 1, NULL, 'a'},
    {"average-threshold", 1, NULL, 't'},
    {"threshold", 1, NULL, 't'}, /* deprecate */
    {"gain", 1, NULL, 'g'},
    {"limiter", 1, NULL, 'l'},
    {"adjust-threshold", 1, NULL, 'T'},
    {"mix", 0, NULL, 'm'},
    {"compression", 0, NULL, 'c'}, /* deprecate */
    {"limit", 0, NULL, 'c'}, /* deprecate */
    {"clipping", 0, NULL, OPT_CLIPPING},
    {"peak", 0, NULL, OPT_PEAK},
    {"fractions", 0, NULL, OPT_FRACTIONS},
    {NULL, 0, NULL, 0}
  };

  /* get program name */
  if ((progname = strrchr(argv[0], '/')) == NULL)
    progname = argv[0];
  else
    progname++;
  if (strlen(progname) > 16)
    progname[16] = '\0';

#if ENABLE_NLS
  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);
  textdomain(PACKAGE);
#endif

  /* get args */
  while ((c = getopt_long(argc, argv, "hVnvqbmcT:l:g:a:t:", longopts, NULL)) != EOF) {
    switch(c) {
    case 'a':
      target = strtod(optarg, &p);

      /* check if "dB" or "dBFS" is given after number */
      while(isspace(*p))
	p++;
      if (strncaseeq(p, "db", 2)) {

	/* amplitude given as dBFS */
	if (target > 0) {
	  target = -target;
	  fprintf(stderr, _("%s: normalizing to %f dBFS\n"), progname, target);
	}

	/* translate to fraction */
	target = DBFSTOAMP(target);

      } else {

	/* amplitude given as fraction */
	if (target < 0 || target > 1.0) {
	  fprintf(stderr, _("%s: error: bad target amplitude %f\n"),
		  progname, target);
	  exit(1);
	}
      }
      break;
    case 't':
      /* a negative threshold means don't use threshold (use 2*stddev) */
      threshold = strtod(optarg, NULL);
      break;
    case 'g':
      gain = strtod(optarg, &p);

      /* check if "dB" is given after number */
      while(isspace(*p))
	p++;
      if (strncaseeq(p, "db", 2)) {
	dBdiff = gain;
	gain = DBTOFRAC(dBdiff);
	gain_in_decibels = TRUE;
      }

      do_compute_levels = FALSE;
      batch_mode = TRUE;
      if (gain < 0) {
	usage();
	exit(1);
      }
      break;
    case 'n':
      do_print_only = TRUE;
      do_apply_gain = FALSE;
      break;
    case 'b':
      batch_mode = TRUE;
      break;
    case 'm':
      mix_mode = TRUE;
      break;
    case 'c':
      fprintf(stderr, _("%s: Warning: the -c option is deprecated, and may be removed in v1.0\n"),
	      progname);
      break;
    case 'l':
      lmtr_lvl = strtod(optarg, &p);
      /* check if "dB" is given after number */
      while(isspace(*p))
	p++;
      fprintf(stderr, _("%s: limiting samples greater than "), progname);
      if (strncaseeq(p, "db", 2)) {
	if (lmtr_lvl > 0)
	  lmtr_lvl = -lmtr_lvl;
	fprintf(stderr, "%f dB\n", lmtr_lvl);
	lmtr_lvl = DBFSTOAMP(lmtr_lvl);
      } else {
	if (lmtr_lvl < 0)
	  lmtr_lvl = -lmtr_lvl;
	fprintf(stderr, "%f\n", lmtr_lvl);
      }

      use_limiter = TRUE;
      break;
    case 'T':
      adjust_thresh = strtod(optarg, &p);
      if (adjust_thresh < 0)
	adjust_thresh = -adjust_thresh;
      fprintf(stderr, _("%s: ignoring adjustments less than %fdB\n"),
	      progname, adjust_thresh);
      break;
    case OPT_CLIPPING:
      use_limiter = FALSE;
      break;
    case OPT_PEAK:
      use_peak = TRUE;
      use_limiter = FALSE;
      break;
    case OPT_FRACTIONS:
      use_fractions = TRUE;
      break;
    case 'v':
      verbose++;
      break;
    case 'q':
      verbose = VERBOSE_QUIET;
      break;
    case 'V':
      fprintf(stderr, "normalize %s\n", version);
      exit(0);
    case 'h':
      usage();
      exit(0);
    default:
      usage();
      exit(1);
    }
  }
  if (mix_mode && batch_mode) {
    fprintf(stderr,
	    _("%s: error: the -m and -b options are mutually exclusive\n"),
	    progname);
    exit(1);
  }
  if (use_peak && (mix_mode || batch_mode)) {
    fprintf(stderr,
	    _("%s: error: -m and -b can't be used with the --peak option\n"),
	    progname);
    exit(1);
  }
  if (optind >= argc) {
    fprintf(stderr, _("Usage: %s [OPTION]... [FILE]...\n"), progname);
    fprintf(stderr, _("Try `%s --help' for more information.\n"), progname);
    exit(1);
  }


  /*
   * get sizes of all files, for progress calculation
   */
  nfiles = 0;
  progress_info.batch_size = 0;
  fnames = (char **)xmalloc((argc - optind) * sizeof(char *));
  progress_info.file_sizes = (off_t *)xmalloc((argc - optind) * sizeof(off_t));
  for (i = optind; i < argc; i++) {
    if (strcmp(argv[i], "-") == 0) {
      if (do_apply_gain) {
	fprintf(stderr, _("%s: Warning: stdin specified on command line, not adjusting files\n"), progname);
	do_apply_gain = FALSE;
	do_print_only = TRUE;
      }
      fnames[nfiles++] = argv[i];
    } else if (stat(argv[i], &st) == -1) {
      fprintf(stderr, _("%s: file %s: %s\n"),
	      progname, argv[i], strerror(errno));
    } else {
      /* we want the size of the data chunk in kilobytes, so subtract
         the size of the wav header and divide by size of kb */
      progress_info.file_sizes[nfiles] = (st.st_size - 36) / 1024;
      /* add the size of the file, in kb */
      progress_info.batch_size += progress_info.file_sizes[nfiles];
      fnames[nfiles] = argv[i];
      nfiles++;
    }
  }
  if (nfiles == 0) {
    fprintf(stderr, _("%s: no files!\n"), progname);
    return 1;
  }

  /* allocate space to store levels and peaks */
  sis = (struct signal_info *)xmalloc(nfiles * sizeof(struct signal_info));


  /*
   * Compute the levels
   */
  if (do_compute_levels) {
    compute_levels(sis, fnames, nfiles);

    /* anything that came back with a level of -1 was bad, so remove it */
    for (i = 0; i < nfiles; i++) {
      if (sis[i].level < 0) {
	nfiles--;
	memmove(sis + i, sis + i + 1,
		(nfiles - i) * sizeof(struct signal_info));
	memmove(fnames + i, fnames + i + 1,
		(nfiles - i) * sizeof(char *));
	memmove(progress_info.file_sizes + i, progress_info.file_sizes + i + 1,
		(nfiles - i) * sizeof(off_t));
      }
    }

    if (batch_mode || mix_mode) {
      level = average_levels(sis, nfiles, threshold);

      /* For mix mode, we set the target to the average level */
      if (mix_mode)
	target = level;

      /* For batch mode, we use one gain for all files */
      if (batch_mode)
	gain = target / level;

      if (do_print_only) {
	if (use_fractions)
	  printf(_("%-12.6f average level\n"), level);
	else
	  printf(_("%-8.4fdBFS average level\n"), AMPTODBFS(level));
      } else if (verbose >= VERBOSE_INFO) {
	if (use_fractions)
	  fprintf(stderr, _("Average level: %0.4f\n"), level);
	else
	  fprintf(stderr, _("Average level: %0.4fdBFS\n"), AMPTODBFS(level));
      }
    }

  } /* end of if (do_compute_levels) */


  /*
   * Check if we need to apply the gain --
   *
   *   If a file would be adjusted by an unnoticeable amount, we don't
   *   want to bother doing the adjustment.  The smallest noticeable
   *   difference varies with the absolute intensity and the pitch,
   *   but I don't think it's a stretch to say that a 0.25 dB
   *   difference is unnoticeable for most pairs of signals.
   *
   *   By default then, we allow amplitudes that are +/-0.125 dB from
   *   the target to pass without adjustment (mainly so that
   *   normalizing files for the second time has no effect).
   *
   *   Why 0.125 dB?  If we allow amplitudes that are 0.125 dB above
   *   and below the target, the total possible range is 0.25 dB,
   *   which shouldn't be noticeable.
   */
  if (do_apply_gain && batch_mode) {
    if (!gain_in_decibels)
      dBdiff = FRACTODB(gain);

    /* the do_compute_levels check makes sure we always apply gain if
       it was specified with -g */
    if (do_compute_levels && fabs(dBdiff) < adjust_thresh) {
      if (verbose >= VERBOSE_PROGRESS)
	fprintf(stderr,
		_("Files are already normalized, not adjusting...\n"));
      do_apply_gain = FALSE;
    } else if (verbose >= VERBOSE_PROGRESS) {
      if (!do_compute_levels) { /* if -g */
	if (gain_in_decibels)
	  printf(_("Applying adjustment of %fdB...\n"), dBdiff);
	else
	  printf(_("Applying adjustment of %f...\n"), gain);
      } else {
	printf(_("Applying adjustment of %0.2fdB...\n"), dBdiff);
      }
    }
  }


  /*
   * Apply the gain
   */
  if (do_apply_gain) {
#if USE_TEMPFILE
    struct stat stbuf;
    char *tmpfile, *p;
#endif

    progress_info.batch_start = time(NULL);
    progress_info.finished_size = 0;

    for (i = 0; i < nfiles; i++) {

      fd = open(fnames[i], O_RDONLY | O_BINARY);
      if (fd == -1) {
	fprintf(stderr, _("%s: error opening %s: %s\n"), progname, fnames[i],
		strerror(errno));
	continue;
      }

      if (!batch_mode) {
	if (use_peak)
	  gain = 1.0 / sis[i].peak;
	else
	  gain = target / sis[i].level;
	dBdiff = FRACTODB(gain);

	/* don't bother applying very small adjustments -- see above */
	if (fabs(dBdiff) < adjust_thresh) {
	  if (verbose >= VERBOSE_PROGRESS)
	    fprintf(stderr, _("%s already normalized, not adjusting...\n"),
		    fnames[i]);
	  continue;
	}

	if (verbose >= VERBOSE_PROGRESS)
	  fprintf(stderr, _("Applying adjustment of %0.2fdB to %s...\n"),
		  dBdiff, fnames[i]);
      }

#if USE_TEMPFILE
      /* Create temporary file name, and open it for writing.  We want
       * it to be in the same directory (and therefore, in the same
       * filesystem, for a fast rename). */
      tmpfile = (char *)xmalloc(strlen(fnames[i]) + 16);
      strcpy(tmpfile, fnames[i]);
      if ((p = strrchr(tmpfile, '/')) == NULL)
	p = tmpfile;
      else
	p++;
      strcpy(p, "_normXXXXXX");
      fd2 = xmkstemp(tmpfile);
      if (fd2 == -1) {
	fprintf(stderr, _("%s: error opening temp file: %s\n"), progname,
		strerror(errno));
	close(fd);
	free(tmpfile);
	continue;
      }

      /* preserve original permissions */
      fstat(fd, &stbuf);
      fchmod(fd2, stbuf.st_mode);
#else
      fd2 = open(fnames[i], O_WRONLY | O_BINARY);
      if (fd2 == -1) {
	fprintf(stderr, _("%s: error opening %s: %s\n"), progname, fnames[i],
		strerror(errno));
	close(fd);
	continue;
      }
#endif

      progress_info.file_start = time(NULL);
      progress_info.on_file = i;

      psi = do_compute_levels ? &sis[i] : NULL;
      if (apply_gain(fd, fd2, fnames[i], gain, psi) == -1) {
	fprintf(stderr, _("%s: error applying adjustment to %s: %s\n"),
		progname, fnames[i], strerror(errno));
      }
      files_changed = TRUE;

      progress_info.finished_size += progress_info.file_sizes[i];

      close(fd);
      close(fd2);

#if USE_TEMPFILE
      /* move the temporary file back to the original file */
      if (xrename(tmpfile, fnames[i]) == -1) {
	fprintf(stderr, _("%s: error moving %s to %s: %s\n"), progname,
		tmpfile, fnames[i], strerror(errno));
	exit(1);
      }
      free(tmpfile);
#endif

      if (verbose >= VERBOSE_PROGRESS && !batch_mode)
	fprintf(stderr, "\n");
    }

    /* we're done with the second progress meter, so go to next line */
    if (verbose >= VERBOSE_PROGRESS && batch_mode)
      fputc('\n', stderr);

  } else if (batch_mode && do_print_only) {

    /* if we're not applying the gain, just print it out, and we're done */
    if (use_fractions) {
      printf(_("%-12f volume adjustment\n"), gain);
    } else {
      char cbuf[32];
      sprintf(cbuf, "%fdB", FRACTODB(gain));
      printf(_("%-12s volume adjustment\n"), cbuf);
    }

  } /* end of if (do_apply_gain) */

  free(sis);
  free(progress_info.file_sizes);
  free(fnames);

  /* if the -n option was given but we didn't adjust any files, return
   * exit status 2 to let scripts know nothing was changed */
  if (!files_changed && !do_print_only)
    return 2;

  return 0;
}

/*
 * Compute the RMS levels of the files.
 */
void
compute_levels(struct signal_info *sis, char **fnames, int nfiles)
{
  double power;
  int i, fd;
  char cbuf[32];
  struct wavfmt fmt = { 1, 2, 44100, 176400, 0, 16 };

  if (verbose >= VERBOSE_PROGRESS) {
    fprintf(stderr, _("Computing levels...\n"));

    if (do_print_only) {
      if (batch_mode)
	fprintf(stderr, _("  level        peak\n"));
      else
	fprintf(stderr, _("  level        peak         gain\n"));
    }
  }

  progress_info.batch_start = time(NULL);
  progress_info.finished_size = 0;

  for (i = 0; i < nfiles; i++) {

    sis[i].level = 0;

    if (strcmp(fnames[i], "-") == 0) {
      progress_info.file_start = time(NULL);
      progress_info.on_file = i;
      errno = 0;

      /* for a stream, format info is passed through sis[i].fmt */
      memcpy(&sis[i].fmt, &fmt, sizeof(struct wavfmt));
      power = signal_max_power_stream(stdin, NULL, &sis[i]);
      fnames[i] = "STDIN";

    } else {

      fd = open(fnames[i], O_RDONLY | O_BINARY);
      if (fd == -1) {
	fprintf(stderr, _("%s: error opening %s: %s\n"), progname, fnames[i],
		strerror(errno));
	sis[i].level = -1;
	goto error_update_progress;
      }

      progress_info.file_start = time(NULL);
      progress_info.on_file = i;
      errno = 0;

      power = signal_max_power(fd, fnames[i], &sis[i]);
    }

    if (power < 0) {
      fprintf(stderr, _("%s: error reading %s: %s\n"), progname, fnames[i],
	      strerror(errno));
      sis[i].level = -1;
      goto error_close_fd;
    }
    if (power < EPSILON) {
      if (verbose >= VERBOSE_PROGRESS) {
	fprintf(stderr,
		"\r                                     "
		"                                     \r");
	fprintf(stderr,
		_("File %s has zero power, ignoring...\n"), fnames[i]);
      }
      sis[i].level = -1;
      goto error_close_fd;
    }

    if (do_print_only) {

      /* clear the progress meter first */
      if (verbose >= VERBOSE_PROGRESS)
	fprintf(stderr,
		"\r                                     "
		"                                     \r");

      if (use_fractions)
	sprintf(cbuf, "%0.6f", sis[i].level);
      else
	sprintf(cbuf, "%0.4fdBFS", AMPTODBFS(sis[i].level));
      printf("%-12s ", cbuf);
      if (use_fractions)
	sprintf(cbuf, "%0.6f", sis[i].peak);
      else
	sprintf(cbuf, "%0.4fdBFS", AMPTODBFS(sis[i].peak));
      printf("%-12s ", cbuf);
      if (!batch_mode) {
	if (use_fractions)
	  sprintf(cbuf, "%0.6f", target / sis[i].level);
	else
	  sprintf(cbuf, "%0.4fdB", AMPTODBFS(target / sis[i].level));
	printf("%-10s ", cbuf);
      }
      printf("%s\n", fnames[i]);

    } else if (verbose >= VERBOSE_INFO) {
      fprintf(stderr,
	      "\r                                     "
	      "                                     \r");
      if (use_fractions)
	fprintf(stderr, _("Level for %s: %0.4f (%0.4f peak)\n"),
		fnames[i], sis[i].level, sis[i].peak);
      else
	fprintf(stderr, _("Level for %s: %0.4fdBFS (%0.4fdBFS peak)\n"),
		fnames[i], AMPTODBFS(sis[i].level), AMPTODBFS(sis[i].peak));
    }

  error_close_fd:
    if (strcmp(fnames[i], "STDIN") != 0)
      close(fd);

  error_update_progress:
    progress_info.finished_size += progress_info.file_sizes[i];
  }

  /* we're done with the level calculation progress meter, so go to
     next line */
  if (verbose == VERBOSE_PROGRESS && !do_print_only)
    fputc('\n', stderr);
}

/*
 * For batch mode, we take the levels for all the input files, throw
 * out any that appear to be statistical aberrations, and average the
 * rest together to get one level and one gain for the whole batch.
 */
double
average_levels(struct signal_info *sis, int nlevels, double threshold)
{
  int i, files_to_avg;
  double sum, level_difference, std_dev, variance;
  double level, mean_level;
  char *badlevels;

  /* badlevels is a boolean array marking the level values to be thrown out */
  badlevels = (char *)xmalloc(nlevels * sizeof(char));
  memset(badlevels, 0, nlevels * sizeof(char));

  /* get mean level */
  sum = 0;
  for (i = 0; i < nlevels; i++)
    sum += sis[i].level;
  mean_level = sum / nlevels;

  /* if no threshold is specified, use 2 * standard dev */
  if (threshold < 0.0) {

    /*
     * We want the standard dev of the levels, but we need it in decibels.
     * Therefore, if u is the mean, the variance is
     *                  (1/N)summation((20*log10(x/u))^2)
     *       instead of (1/N)summation((x-u)^2),
     * which it would be if we needed straight variance "by the numbers".
     */

    /* get variance */
    sum = 0;
    for (i = 0; i < nlevels; i++) {
      double tmp = FRACTODB(sis[i].level / mean_level);
      sum += tmp * tmp;
    }
    variance = sum / nlevels;

    /* get standard deviation */
    if (variance < EPSILON)
      std_dev = 0.0;
    else
      std_dev = sqrt(variance);
    if (verbose >= VERBOSE_INFO)
      fprintf(stderr, _("Standard deviation is %0.2f dB\n"), std_dev);

    threshold = 2 * std_dev;
  }

  /*
   * Throw out level values that seem to be aberrations
   * (so that one "quiet song" doesn't throw off the average)
   * We define an aberration as a level that is > 2*stddev dB from the mean.
   */
  if (threshold > EPSILON && nlevels > 1) {
    for (i = 0; i < nlevels; i++) {

      /* Find how different from average the i'th file's level is.
       * The "level" here is actually the signal's maximum amplitude,
       * from which we can compute the difference in decibels. */
      level_difference = fabs(FRACTODB(mean_level / sis[i].level));

      /* mark as bad any level that is > threshold different than the mean */
      if (level_difference > threshold) {
	if (verbose >= VERBOSE_INFO) {
	  if (use_fractions) {
	    fprintf(stderr,
		_("Throwing out level of %0.4f (different by %0.2fdB)\n"),
		    sis[i].level, level_difference);
	  } else {
	    fprintf(stderr,
		_("Throwing out level of %0.4fdBFS (different by %0.2fdB)\n"),
		    AMPTODBFS(sis[i].level), level_difference);
	  }
	}
	badlevels[i] = TRUE;
      }
    }
  }

  /* throw out the levels marked as bad */
  files_to_avg = 0;
  sum = 0;
  for (i = 0; i < nlevels; i++)
    if (!badlevels[i]) {
      sum += sis[i].level;
      files_to_avg++;
    }

  if (files_to_avg == 0) {
    fprintf(stderr, _("%s: all files ignored, try using -t 100\n"), progname);
    exit(1);
  }

  free(badlevels);

  level = sum / files_to_avg;

  return level;
}

#define LINE_LENGTH 79
void
progress_callback(char *prefix, float fraction_completed)
{
  char buf[LINE_LENGTH + 32]; /* need +1, but +32 in case of huge ETA's */
  time_t now, time_spent;
  unsigned int file_eta_hr, file_eta_min, file_eta_sec;
  off_t kb_done;
  float batch_fraction;
  unsigned int batch_eta_hr, batch_eta_min, batch_eta_sec;

  if (fraction_completed <= 0.0) {
    if (progress_info.batch_size == 0) {
      fprintf(stderr,
        _(" %s  --%% done, ETA --:--:-- (batch  --%% done, ETA --:--:--)"),
	      prefix);
    } else {
      batch_fraction = (progress_info.finished_size
			/ (float)progress_info.batch_size);
      fprintf(stderr,
        _(" %s  --%% done, ETA --:--:-- (batch %3.0f%% done, ETA --:--:--)"),
	      prefix, batch_fraction * 100);
    }
    fprintf(stderr, "\r");
    return;
  }

  now = time(NULL);
  if (fraction_completed > 1.0)
    fraction_completed = 1.0;

  /* figure out the ETA for this file */
  file_eta_hr = file_eta_sec = file_eta_min = 0;
  time_spent = now - progress_info.file_start;
  if (fraction_completed == 0.0)
    file_eta_sec = 0;
  else
    file_eta_sec = floor((float)time_spent / fraction_completed
			 - (float)time_spent + 0.5);
  file_eta_min = file_eta_sec / 60;
  file_eta_sec = file_eta_sec % 60;
  file_eta_hr = file_eta_min / 60;
  file_eta_min = file_eta_min % 60;
  if (file_eta_hr > 99)
    file_eta_hr = 99;

  /* figure out the ETA for the whole batch */
  kb_done = progress_info.finished_size
    + fraction_completed * progress_info.file_sizes[progress_info.on_file];
  batch_fraction = (float)kb_done / (float)progress_info.batch_size;
  batch_eta_hr = batch_eta_min = batch_eta_sec = 0;
  time_spent = now - progress_info.batch_start;
  batch_eta_sec = floor((float)time_spent / batch_fraction
			- (float)time_spent + 0.5);

  batch_eta_min = batch_eta_sec / 60;
  batch_eta_sec = batch_eta_sec % 60;
  batch_eta_hr = batch_eta_min / 60;
  batch_eta_min = batch_eta_min % 60;
  if (batch_eta_hr > 99)
    batch_eta_hr = 99;

  sprintf(buf, _(" %s %3.0f%% done, ETA %02d:%02d:%02d (batch %3.0f%% done, ETA %02d:%02d:%02d)"),
	  prefix, fraction_completed * 100,
	  file_eta_hr, file_eta_min, file_eta_sec,
	  batch_fraction * 100,
	  batch_eta_hr, batch_eta_min, batch_eta_sec);

  fprintf(stderr, "%s\r", buf);
}


#if 0
/*
 * decibel conversion routines
 */
__inline__ double
amp_to_dBFS(double x)
{
  return 20 * log10(x);
}

__inline__ double
frac_to_dB(double x)
{
  return 20 * log10(x);
}

__inline__ double
dBFS_to_amp(double x)
{
  return pow(10, x / 20.0);
}

__inline__ double
dB_to_frac(double x)
{
  return pow(10, x / 20.0);
}
#endif

#if USE_TEMPFILE
/*
 * This works like the BSD mkstemp, except that we don't unlink the
 * file, since we end up renaming it to something else.
 */
int
xmkstemp(char *template)
{
  static char sfx[7] = "AAAAAA";
  char *p;
  int fd, i, done;

  p = template + strlen(template) - 6;
  if (strcmp(p, "XXXXXX") != 0) {
    errno = EINVAL;
    return -1;
  }

  do {
    strcpy(p, sfx);

    /* increment the suffix */
    done = 0; i = 5;
    while (!done && i >= 0) {
      sfx[i]++;
      if (sfx[i] > 'Z') {
	sfx[i] = 'A';
	i--;
      } else {
	done = 1;
      }
    }
    if (!done) {
      errno = EEXIST;
      return -1;
    }

    /* attempt to open the file */
    fd = open(template, O_RDWR | O_CREAT | O_EXCL | O_BINARY, 0600);

  } while (fd == -1 && errno == EEXIST);

  return fd;
}


/*
 * Move the file "oldpath" to "newpath", or copy and delete if they
 * are on different filesystems.
 */
int
xrename(const char *oldpath, const char *newpath)
{
  if (rename(oldpath, newpath) == -1) {
    if (errno == EXDEV) {
      /* files are on different filesystems, so we have to copy */
      FILE *in, *out;
      char buf[4096];
      size_t sz;

      in = fopen(oldpath, "rb");
      if (in == NULL)
	return -1;
      out = fopen(newpath, "wb");
      if (out == NULL) {
	fclose(in);
	return -1;
      }

      while ((sz = fread(buf, 1, 4096, in)) > 0)
	fwrite(buf, 1, sz, out);

      if (ferror(in) || ferror(out)) {
	fclose(in);
	fclose(out);
	return -1;
      }
      if (fclose(in) == EOF) {
	fclose(out);
	return -1;
      }
      if (fclose(out) == EOF)
	return -1;
      if (unlink(oldpath) == -1)
	return -1;
    } else {
      return -1;
    }
  }

  return 0;
}
#endif

/*
 * Return nonzero if the two strings are equal, ignoring case, up to
 * the first n characters
 */
int
strncaseeq(const char *s1, const char *s2, size_t n)
{
  for ( ; n > 0; n--) {
    if (tolower(*s1++) != tolower(*s2++))
      return 0;
  }

  return 1;
}

void *
xmalloc(size_t size)
{
  void *ptr = malloc(size);
  if (ptr == NULL) {
    fprintf(stderr, _("%s: unable to malloc\n"), progname);
    exit(1);
  }
  return ptr;
}
