/* novag.h -- main header
   Copyright (C) 2007 Maximiliano Pin

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#ifndef NOVAG_H
#define NOVAG_H

#include <stdexcept>
#include <iostream>
#include <string>
#include <vector>
#include <cstdlib>
#include <cstdio>
#include <termios.h>
#include <unistd.h>

typedef const std::string& cstr;
typedef bool player_t;
#define WHITE false
#define BLACK true

class Serial
{
public:
	Serial () : fd (-1) {}
	~Serial ();
	void open (cstr device, speed_t speed);
	void close ();
	void send (cstr line);
	void recv (std::string* line);
	void wait_readable ();
	int get_fd () const { return fd; }
private:
	int fd;
};

struct Move
{
	int ox;
	int oy;
	int dx;
	int dy;
	bool short_castle;
	bool long_castle;
	bool take;
	bool ep;
	char piece;
	char change;
	Move () : ox (-1), oy (-1), dx (-1), dy (-1),
	          short_castle (false), long_castle (false), take (false),
	          ep (false), piece (' '), change (' ') {}
	std::string toString (bool print_piece = false);
};

class Position
{
public:
	Position ();
	Position (const Position& prev, cstr move);
	cstr get_last_move () const { return last_move; }
	int get_move_num () const { return move_num; }
	player_t get_move_player () const { return move_player; }
	void print () const;
private:
	typedef std::vector<std::pair<int,int> > slot_set_t;
	void decode_move (cstr move, Move* smove);
	void deduct_move (Move* smove);
	void gen_origs (char piece, char rpiece, int x, int y,
	                slot_set_t* origs);
	void find (char piece, int x, int y, int ix, int iy,
	           int limit, slot_set_t* slots);
	char slot[8][8];
	std::string last_move;
	int move_num;
	player_t move_player;
};

class Game
{
public:
	Game () { reset (); }
	bool add_move (int num, player_t player, cstr move);
	void add_move (cstr move);
	void take_back ();
	void reset ();
	const Position& pos () const { return positions.back(); }
	cstr get_last_move () const { return pos().get_last_move(); }
	int get_move_num () const { return pos().get_move_num(); }
	player_t get_move_player () const { return pos().get_move_player(); }
private:
	std::vector<Position> positions;
};

class Novag
{
public:
	Novag (Serial& s) : serial (s), reference_mode (false),
	                    state (NORMAL) {}
	void start ();
	void set_reference_mode (bool on) { reference_mode = on; }
	void set_castle_convert (bool on) { castle_conversion = on; }
	void received_line (cstr line);
	void send_move (cstr move);
private:
	void query_game ();
	void received_move (cstr line);
	void game_line (cstr line);
	bool add_move (int num, player_t player, std::string move);
	void castle_convert (std::string* move, player_t player);
	Serial& serial;
	Game game;
	bool reference_mode;
	bool castle_conversion;
	typedef enum { NORMAL, RECV_GAME_HDR, RECV_GAME } state_t;
	state_t state;
	int expected_moves;
};

void dbg_out (cstr msg);
void err_out (cstr msg);
void tokenize (cstr s, std::vector<std::string>* tkns);

#endif
