/* position.cc -- position tracking
   Copyright (C) 2007 Maximiliano Pin

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "novag.h"
#include <cstring>
#include <cctype>

using namespace std;

static char ini_pos[8][8] =
	{ { 'R', 'P', ' ', ' ', ' ', ' ', 'p', 'r' },
	  { 'N', 'P', ' ', ' ', ' ', ' ', 'p', 'n' },
	  { 'B', 'P', ' ', ' ', ' ', ' ', 'p', 'b' },
	  { 'Q', 'P', ' ', ' ', ' ', ' ', 'p', 'q' },
	  { 'K', 'P', ' ', ' ', ' ', ' ', 'p', 'k' },
	  { 'B', 'P', ' ', ' ', ' ', ' ', 'p', 'b' },
	  { 'N', 'P', ' ', ' ', ' ', ' ', 'p', 'n' },
	  { 'R', 'P', ' ', ' ', ' ', ' ', 'p', 'r' } };

Position::Position ()
{
	memcpy (slot, ini_pos, sizeof(slot));
	move_num = 0;
	move_player = BLACK;
}

Position::Position (const Position& prev, cstr move)
{
	memcpy (slot, prev.slot, sizeof(slot));
	move_num = prev.move_num;
	move_player = !prev.move_player;
	if (move_player == WHITE)
		move_num++;

	Move smove;
	decode_move (move, &smove);
	dbg_out ("DECODED:  " + smove.toString(true));
	deduct_move (&smove);
	dbg_out ("DEDUCTED: " + smove.toString(true));

	if (smove.short_castle) {
		slot[4][smove.oy] = slot[7][smove.oy] = ' ';
		slot[5][smove.oy] = (move_player == BLACK) ? 'r' : 'R';
		slot[6][smove.oy] = (move_player == BLACK) ? 'k' : 'K';
	}
	else if (smove.long_castle) {
		slot[4][smove.oy] = slot[0][smove.oy] = ' ';
		slot[3][smove.oy] = (move_player == BLACK) ? 'r' : 'R';
		slot[2][smove.oy] = (move_player == BLACK) ? 'k' : 'K';
	}
	else if (smove.ox >= 0 && smove.oy >= 0) {
		if (smove.change == ' ')
			slot[smove.dx][smove.dy] = slot[smove.ox][smove.oy];
		else
			slot[smove.dx][smove.dy] = smove.change;
		slot[smove.ox][smove.oy] = ' ';
		if (smove.ep) {
			slot[smove.dx][smove.oy] = ' ';
		}
	}
	else {
		err_out ("I don't understand: " + move);
	}

	last_move = smove.toString();

	dbg_out ("--- " + get_last_move());
	print();
	dbg_out ("---");
}

void Position::print () const
{
	for (int y = 7; y >= 0; y--) {
		string l;
		for (int x = 0; x < 8; x++) {
			l += ' ';
			l += slot[x][y];
		}
		dbg_out (l);
	}
}

void Position::decode_move (cstr move, Move* smove)
{
	const char *p = move.c_str();
	if (move == "O-O" || move == "o-o") {
		smove->short_castle = true;
		smove->piece = 'K';
		smove->oy = (move_player == BLACK) ? 7 : 0;
		return;
	}
	if (move == "O-O-O" || move == "o-o-o") {
		smove->long_castle = true;
		smove->piece = 'K';
		smove->oy = (move_player == BLACK) ? 7 : 0;
		return;
	}
	if (isupper (*p)) {
		smove->piece = *p++;
	}
	if (*p != '-' && *p != 'x') {
		int col = -1, row = -1;
		if (isalpha (*p)) {
			col = *p++ - 'a';
		}
		if (isdigit (*p)) {
			row = *p++ - '1';
		}
		if (*p == '-' || *p == 'x' || isalpha(*p) && !isupper(*p)) {
			smove->ox = col;
			smove->oy = row;
		}
		else {
			smove->dx = col;
			smove->dy = row;
		}
	}
	if (*p == '-' || *p == 'x') {
		p++;
	}
	if (isalpha(*p) && !isupper(*p)) {
		smove->dx = *p++ - 'a';
		smove->dy = *p++ - '1';
	}
	if (*p == '=' || *p == '/') {
		p++;
	}
	if (isupper (*p)) {
		if (smove->piece == ' ')
			smove->piece = 'P';
		smove->change = *p;
	}
	if (smove->piece == ' ' && smove->oy < 0)
		smove->piece = 'P';
}

void Position::deduct_move (Move* smove)
{
	if (smove->ox < 0 || smove->oy < 0) {
		slot_set_t origs;
		gen_origs (smove->piece, smove->piece, smove->dx, smove->dy,
		           &origs);

		slot_set_t::iterator it;
		for (it = origs.begin(); it != origs.end(); it++) {
			if (!(smove->ox >= 0 && it->first != smove->ox ||
			      smove->oy >= 0 && it->second != smove->oy)) {
				break;
			}
		}
		if (it != origs.end()) {
			smove->ox = it->first;
			smove->oy = it->second;
		}
	}
	if (smove->ox >= 0 && smove->oy >= 0) {
		smove->piece = toupper (slot[smove->ox][smove->oy]);
		smove->take = (slot[smove->dx][smove->dy] != ' ');
		if (!smove->take && smove->piece == 'P' &&
		    smove->ox != smove->dx) {
			smove->ep = true;
			smove->take = true;
		}
	}
	if (smove->piece == 'P' && (smove->dy == 0 || smove->dy == 7) &&
	    smove->change == ' ') {
		smove->change = 'Q';
	}
}

void Position::gen_origs (char piece, char rpiece, int x, int y,
                          slot_set_t* origs)
{
	if (piece == 'Q' || piece == 'K') {
		gen_origs ('R', piece, x, y, origs);
		gen_origs ('B', piece, x, y, origs);
		return;
	}

	if (move_player == BLACK) rpiece = tolower (rpiece);

	int i;
	switch (piece) {
	case 'R':
		find (rpiece, x, y, -1, 0, 8, origs);
		find (rpiece, x, y, 1, 0, 8, origs);
		find (rpiece, x, y, 0, -1, 8, origs);
		find (rpiece, x, y, 0, 1, 8, origs);
		break;
	case 'B':
		find (rpiece, x, y, -1, -1, 8, origs);
		find (rpiece, x, y, -1, 1, 8, origs);
		find (rpiece, x, y, 1, -1, 8, origs);
		find (rpiece, x, y, 1, 1, 8, origs);
		break;
	case 'N':
		find (rpiece, x, y, 1, 2, 1, origs);
		find (rpiece, x, y, 2, 1, 1, origs);
		find (rpiece, x, y, 2, -1, 1, origs);
		find (rpiece, x, y, 1, -2, 1, origs);
		find (rpiece, x, y, -1, -2, 1, origs);
		find (rpiece, x, y, -2, -1, 1, origs);
		find (rpiece, x, y, -2, 1, 1, origs);
		find (rpiece, x, y, -1, 2, 1, origs);
		break;
	case 'P':
		int dir = (move_player == BLACK) ? 1 : -1;
		find (rpiece, x, y, 0, dir, 2, origs);
		find (rpiece, x, y, -1, dir, 1, origs);
		find (rpiece, x, y, 1, dir, 1, origs);
		break;
	}
}

void Position::find (char piece, int x, int y, int ix, int iy,
                     int limit, slot_set_t* slots)
{
	while (limit--) {
		x += ix;
		y += iy;
		if (x < 0 || x > 7 || y < 0 || y > 7) {
			break;
		}
		if (slot[x][y] == piece) {
			slots->push_back (make_pair (x, y));
			break;
		}
		if (slot[x][y] != ' ') break;
	}
}

string Move::toString (bool print_piece)
{
	string s;
	if (short_castle) {
		s = "O-O";
		goto end;
	}
	if (long_castle) {
		s = "O-O-O";
		goto end;
	}
	if (print_piece) s = piece;
	if (ox >= 0) s += ('a' + ox);
	if (oy >= 0) s += ('1' + oy);
	s += (take ? 'x' : '-');
	s += ('a' + dx);
	s += ('1' + dy);
	if (change != ' ') {
		s += '/';
		s += change;
	}
	if (ep) {
		s += "ep";
	}
end:
	return s;
}
