/***************************************************************************
 *            nc_distance.c
 *
 *  Tue May  8 11:05:35 2007
 *  Copyright  2007  Sandro Dias Pinto Vitenti
 *  <sandro@isoftware.com.br>
 ****************************************************************************/
/*
 * numcosmo
 * Copyright (C) Sandro Dias Pinto Vitenti 2012 <sandro@lapsandro>
 * numcosmo is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * numcosmo is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * SECTION:nc_distance
 * @title: Cosmological Distances and Times
 * @short_description: Calculate cosmological distances and related quantities.
 *
 * This object implements several distances used in cosmology, here we have
 * the following definitions.
 * 
 * The Hubble scale is simply defined as the inverse of the Hubble function $H(z)$ [nc_hicosmo_H()],
 * \begin{equation}\label{eq:def:DH}
 * D_H(z) = \frac{c}{H(z)}, \qquad D_{H0} = \frac{c}{H_0}. 
 * \end{equation}
 * where $c$ is the speed of light [ncm_c_c()], $z$ is the redshift
 * and $H_0 \equiv H(0)$ is the Hubble parameter [nc_hicosmo_H0()]. The comoving distance $D_c$ is 
 * defined as
 * \begin{equation}\label{eq:def:Dc}
 * D_c(z) = \int_0^z \frac{dz^\prime}{E (z^\prime)},
 * \end{equation}
 * where $E(z)$ is the normalized Hubble function [nc_hicosmo_E()], i.e., 
 * \begin{equation}\label{eq:def:Ez} 
 * E(z) \equiv \frac{H(z)}{H_0}.
 * \end{equation}  
 * Note that both quantities are 
 * adimensional, in other words, one can think them as in units of the Hubble 
 * scale today.
 * 
 * The transverse comoving distance $D_t$ and its derivative with respect to $z$ are 
 * given by
 * \begin{equation}\label{eq:def:Dt}
 * D_t(z) = \frac{\sinh\left(\sqrt{\Omega_{k0}}D_c(z)\right)}{\sqrt{\Omega_{k0}}}, 
 * \qquad \frac{dD_t}{dz}(z) = \frac{\cosh\left(\sqrt{\Omega_{k0}}D_c(z)\right)}{E(z)},
 * \end{equation}
 * where $\Omega_{k0}$ is the value of the curvature today [nc_hicosmo_Omega_k()].
 * Using the definition above we have that the luminosity distance is
 * \begin{equation}\label{eq:def:Dl}
 * D_l = (1+z)D_t(z),
 * \end{equation}
 * and the distance modulus is given by
 * \begin{equation}\label{eq:def:mu}
 * \mu(z) = 5\log_{10}(D_l(z)) + 25,
 * \end{equation}
 * where $\log_{10}$ represents the logarithm in the decimal base. Note that 
 * the distance modulus is usually defined as 
 * $$5\log_{10}(D_{H0}D_l(z)/\text{pc}) - 5,$$
 * where $\text{pc}$ is parsec [ncm_c_pc()]. Thus, this differs from our 
 * definition by a factor of $5\log_{10}(D_{H0}/\text{Mpc})$, where $\text{Mpc}$ 
 * is megaparsec [ncm_c_Mpc()].
 * 
 * 
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif /* HAVE_CONFIG_H */
#include "build_cfg.h"

#include "nc_distance.h"
#include "math/integral.h"
#include "math/ncm_c.h"
#include "math/ncm_cfg.h"
#include "math/ncm_spline_cubic_notaknot.h"

typedef struct _ComovingDistanceArgument{
  NcHICosmo *cosmo;
  gint diff;
} ComovingDistanceArgument;

enum
{
  PROP_0,
  PROP_ZF
};

G_DEFINE_TYPE (NcDistance, nc_distance, G_TYPE_OBJECT);

static void
nc_distance_constructed (GObject *object)
{
  /* Chain up : start */
  G_OBJECT_CLASS (nc_distance_parent_class)->constructed (object);
  {

  }
}

static void
nc_distance_dispose (GObject *object)
{
  NcDistance *dist = NC_DISTANCE (object);

  ncm_function_cache_clear (&dist->comoving_distance_cache);
  ncm_function_cache_clear (&dist->time_cache);
  ncm_function_cache_clear (&dist->lookback_time_cache);
  ncm_function_cache_clear (&dist->conformal_time_cache);
  ncm_function_cache_clear (&dist->sound_horizon_cache);

  ncm_ode_spline_clear (&dist->comoving_distance_spline);

  ncm_model_ctrl_clear (&dist->ctrl);

  /* Chain up : end */
  G_OBJECT_CLASS (nc_distance_parent_class)->dispose (object);
}

static void
nc_distance_finalize (GObject *object)
{

  /* Chain up : end */
  G_OBJECT_CLASS (nc_distance_parent_class)->finalize (object);
}

static void
nc_distance_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  NcDistance *dist = NC_DISTANCE (object);
  g_return_if_fail (NC_IS_DISTANCE (object));

  switch (prop_id)
  {
    case PROP_ZF:
      dist->z_f = g_value_get_double (value);
      ncm_model_ctrl_force_update (dist->ctrl);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
nc_distance_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  NcDistance *dist = NC_DISTANCE (object);
  g_return_if_fail (NC_IS_DISTANCE (object));

  switch (prop_id)
  {
    case PROP_ZF:
      g_value_set_double (value, dist->z_f);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
nc_distance_class_init (NcDistanceClass *klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS (klass);

  object_class->constructed  = nc_distance_constructed;
  object_class->set_property = nc_distance_set_property;
  object_class->get_property = nc_distance_get_property;
  object_class->dispose = nc_distance_dispose;
  object_class->finalize = nc_distance_finalize;

  g_object_class_install_property (object_class,
                                   PROP_ZF,
                                   g_param_spec_double ("zf",
                                                        NULL,
                                                        "Final cached redshift",
                                                        0.0,
                                                        G_MAXDOUBLE,
                                                        10.0,
                                                        G_PARAM_READWRITE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB));
}

/**
 * nc_distance_new:
 * @z_f: final redshift $z_f$.
 *
 * Creates a new #NcDistance object optimized to perform distance calculations
 * to redshift up to $z_f$.
 *
 * Returns: a new #NcDistance.
 */
NcDistance *
nc_distance_new (gdouble z_f)
{
  return g_object_new (NC_TYPE_DISTANCE, "zf", z_f, NULL);
}

/**
 * nc_distance_ref:
 * @dist: a #NcDistance.
 *
 * FIXME
 *
 * Returns: (transfer full): FIXME
 */
NcDistance *
nc_distance_ref (NcDistance *dist)
{
  return g_object_ref (dist);
}

/**
 * nc_distance_free:
 * @dist: a #NcDistance.
 *
 * FIXME
 *
 */
void
nc_distance_free (NcDistance *dist)
{
  g_object_unref (dist);
}

/**
 * nc_distance_clear:
 * @dist: a #NcDistance.
 *
 * FIXME
 *
 */
void
nc_distance_clear (NcDistance **dist)
{
  g_clear_object (dist);
}

static gdouble dcddz (gdouble y, gdouble x, gpointer userdata);

/**
 * nc_distance_prepare:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 *
 * FIXME
 *
 */
void
nc_distance_prepare (NcDistance *dist, NcHICosmo *cosmo)
{
  dist->comoving_distance_cache->clear = TRUE;
  dist->time_cache->clear = TRUE;
  dist->lookback_time_cache->clear = TRUE;
  dist->conformal_time_cache->clear = TRUE;

  dist->sound_horizon_cache->clear = TRUE;

  if (dist->comoving_distance_spline == NULL)
  {
    NcmSpline *s = ncm_spline_cubic_notaknot_new ();
    dist->comoving_distance_spline =
      ncm_ode_spline_new_full (s, dcddz, 0.0, 0.0, dist->z_f);
    
    ncm_spline_free (s);
  }

  ncm_ode_spline_prepare (dist->comoving_distance_spline, cosmo);

  ncm_model_ctrl_update (dist->ctrl, NCM_MODEL (cosmo));

  return;
}

/**
 * nc_distance_prepare_if_needed:
 * @dist: FIXME,
 * @cosmo: FIXME
 *
 * FIXME
 * 
 */
static void
nc_distance_init (NcDistance *dist)
{
  dist->use_cache = TRUE;

  dist->comoving_distance_cache = ncm_function_cache_new (1, NCM_INTEGRAL_ABS_ERROR, NCM_INTEGRAL_ERROR);

  dist->time_cache = ncm_function_cache_new (1, NCM_INTEGRAL_ABS_ERROR, NCM_INTEGRAL_ERROR);
  dist->lookback_time_cache = ncm_function_cache_new (1, NCM_INTEGRAL_ABS_ERROR, NCM_INTEGRAL_ERROR);
  dist->conformal_time_cache = ncm_function_cache_new (1, NCM_INTEGRAL_ABS_ERROR, NCM_INTEGRAL_ERROR);

  dist->sound_horizon_cache = ncm_function_cache_new (1, NCM_INTEGRAL_ABS_ERROR, NCM_INTEGRAL_ERROR);

  dist->comoving_distance_spline = NULL;

  dist->ctrl = ncm_model_ctrl_new (NULL);
}

/**
 * nc_distance_hubble:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 *
 * Calculate the curvature scale today as defined in Eq $\eqref{eq:def:DH}$ in
 * units of megaparsec (Mpc) [ncm_c_Mpc()].
 *
 * Returns: $D_{H0}$.
 */
gdouble
nc_distance_hubble (NcDistance *dist, NcHICosmo *cosmo)
{
  NCM_UNUSED (dist);
  return ncm_c_c () / (nc_hicosmo_H0 (cosmo) * 1.0e3);
}

static gdouble comoving_distance_integral_argument(gdouble z, gpointer p);

/**
 * nc_distance_comoving:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 * @z: redshift $z$.
 * 
 * Calculate the comoving distance $D_c (z)$ as defined in Eq. $\eqref{eq:def:Dc}$.
 *
 * Returns: $D_c(z)$.
 */
gdouble
nc_distance_comoving (NcDistance *dist, NcHICosmo *cosmo, gdouble z)
{
  gdouble result, error;
  gsl_function F;

  nc_distance_prepare_if_needed (dist, cosmo);

  if (ncm_model_impl (NCM_MODEL (cosmo)) & NC_HICOSMO_IMPL_cd)
    return nc_hicosmo_cd (cosmo, z);

  if (z <= dist->z_f)
    return ncm_spline_eval (dist->comoving_distance_spline->s, z);

  F.function = &comoving_distance_integral_argument;
  F.params = cosmo;

  if (dist->use_cache)
    ncm_integral_cached_0_x (dist->comoving_distance_cache, &F, z, &result, &error);
  else
    ncm_integral_locked_a_b (&F, 0.0, z, 0.0, NCM_INTEGRAL_ERROR, &result, &error);

  return result;
}

static gdouble
comoving_distance_integral_argument(gdouble z, gpointer p)
{
  NcHICosmo *cosmo = NC_HICOSMO (p);
  const gdouble E2 = nc_hicosmo_E2 (cosmo, z);
  if (GSL_SIGN(E2) == -1.0)
    return GSL_POSINF;
  return 1.0 / sqrt (E2);
}

static gdouble
dcddz (gdouble cd, gdouble z, gpointer userdata)
{
  NcHICosmo *cosmo = NC_HICOSMO (userdata);
  const gdouble E2 = nc_hicosmo_E2 (cosmo, z);
  NCM_UNUSED (cd);
  return 1.0 / sqrt (E2);
}

/**
 * nc_distance_transverse:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 * @z: redshift $z$.
 *
 * Compute the transverse comoving distance $D_t (z)$ defined in Eq. $\eqref{eq:def:Dt}$.
 *
 * Returns: $D_t(z)$.
 */
gdouble
nc_distance_transverse (NcDistance *dist, NcHICosmo *cosmo, gdouble z)
{
  const gdouble Omega_k = nc_hicosmo_Omega_k (cosmo);
  const gdouble sqrt_Omega_k = sqrt (fabs (Omega_k));
  const gdouble comoving_dist = nc_distance_comoving (dist, cosmo, z);
  const gint k = fabs (Omega_k) < NCM_ZERO_LIMIT ? 0 : (Omega_k > 0.0 ? -1 : 1);
  gdouble Dt;

  if (gsl_isinf (comoving_dist)) 
    return comoving_dist;

  switch (k)
  {
    case 0:
      Dt = comoving_dist;
      break;
    case -1:
      Dt = sinh (sqrt_Omega_k * comoving_dist) / sqrt_Omega_k;
      break;
    case 1:
      Dt = sin (sqrt_Omega_k * comoving_dist) / sqrt_Omega_k;
      break;
    default:
      g_assert_not_reached();
      Dt = 0.0;
      break;
  }

  g_assert (Dt >= 0);
  
  return Dt;
}

/**
 * nc_distance_dtransverse_dz:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 * @z: redshift $z$.
 *
 * Compute the derivative of $D_t(z)$ with respect to $z$ defined in 
 * Eq. $\eqref{eq:def:Dt}$.
 *
 * Returns: $\frac{dD_t(z)}{dz}$.
 */
gdouble
nc_distance_dtransverse_dz (NcDistance *dist, NcHICosmo *cosmo, gdouble z)
{
  gdouble Omega_k = nc_hicosmo_Omega_k (cosmo);
  gdouble sqrt_Omega_k = sqrt (fabs (Omega_k));
  gdouble E = sqrt (nc_hicosmo_E2(cosmo, z));
  gint k = fabs (Omega_k) < NCM_ZERO_LIMIT ? 0 : (Omega_k > 0.0 ? -1 : 1);

  switch (k)
  {
    case 0:
      return 1.0 / E;
      break;
    case -1:
    {
      gdouble comoving_dist = nc_distance_comoving (dist, cosmo, z);
      return cosh (sqrt_Omega_k * comoving_dist) / E;
      break;
    }
    case 1:
    {
      gdouble comoving_dist = nc_distance_comoving (dist, cosmo, z);
      return ncm_c_sign_sin (sqrt_Omega_k * comoving_dist) * cos (sqrt_Omega_k * comoving_dist) / E; // LOOK
      break;
    }
    default:
      g_assert_not_reached();
      return 0.0;
      break;
  }
}

/**
 * nc_distance_luminosity:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 * @z: redshift $z$.
 *
 * Compute the luminosity distance $D_l(z)$  defined in Eq. $\eqref{eq:def:Dl}$.
 *
 * Returns: $D_l(z)$.
 */
gdouble
nc_distance_luminosity (NcDistance *dist, NcHICosmo *cosmo, gdouble z)
{
  const gdouble Dt = nc_distance_transverse (dist, cosmo, z);
  const gdouble Dl = (1.0 + z) * Dt;

  return Dl;
}

/**
 * nc_distance_modulus:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 * @z: redshift $z$.
 *
 * Compute the distance modulus $\mu(z)$ defined in Eq. $\eqref{eq:def:mu}$.
 *
 * Returns: $\mu(z)$.
 */
gdouble
nc_distance_modulus (NcDistance *dist, NcHICosmo *cosmo, gdouble z)
{
  const gdouble Dl = nc_distance_luminosity (dist, cosmo, z);
  if (!gsl_finite (Dl))
    return Dl;
  return (5.0 * log10 (Dl) + 25.0);
}

/**
 * nc_distance_luminosity_hef:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 * @z_he: redshift $z_{he}$ in our local frame.
 * @z_cmb: redshift $z_{CMB}$ in the CMB frame.
 *
 * Calculate the luminosity distance $D_l$ corrected to our local frame.
 *
 * Returns: $D_l(z_{hef},z_{CMB})$.
 */
gdouble 
nc_distance_luminosity_hef (NcDistance *dist, NcHICosmo *cosmo, gdouble z_he, gdouble z_cmb)
{
  const gdouble Dt = nc_distance_transverse (dist, cosmo, z_cmb);
  const gdouble Dl = (1.0 + z_he) * Dt;

  return Dl;
}

/**
 * nc_distance_modulus_hef:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 * @z_he: redshift $z_{he}$ in our local frame.
 * @z_cmb: redshift $z_{CMB}$ in the CMB frame.
 *
 * Calculate the distance modulus using the frame corrected luminosity distance
 * [nc_distance_luminosity_hef()].
 *
 * Returns: $\mu(z_{hef},z_{CMB})$.
 */
gdouble 
nc_distance_modulus_hef (NcDistance *dist, NcHICosmo *cosmo, gdouble z_he, gdouble z_cmb)
{
  const gdouble Dl = nc_distance_luminosity_hef (dist, cosmo, z_he, z_cmb);
  if (!gsl_finite (Dl))
    return Dl;
  return (5.0 * log10 (Dl) + 25.0);
}

/**
 * nc_distance_angular_diameter_curvature_scale:
 * @dist: a #NcDistance
 * @cosmo: a #NcHICosmo
 *
 * We define the angular diameter curvature scale $D_a(z_\star)$ as
 * $$D_a(z_\star) = \frac{E(z_\star)}{1 + z_\star} D_t(z_\star),$$ 
 * where $z_\star$ is the decoupling redshift, given by [nc_distance_decoupling_redshift()], 
 * $E(z_\star)$ is the normalized Hubble function [Eq. $\eqref{eq:def:Ez}$] and 
 * $D_t(z_\star)$ is the transverse comoving distance [Eq. $\eqref{eq:def:Dt}$] both computed at $z_\star$.
 *
 * Returns: $D_a(z_\star)$.
 */
gdouble
nc_distance_angular_diameter_curvature_scale (NcDistance *dist, NcHICosmo *cosmo)
{
  const gdouble z_star = nc_distance_decoupling_redshift (dist, cosmo);
  if (gsl_finite (z_star))
  {
    return sqrt (nc_hicosmo_E2 (cosmo, z_star)) *
      nc_distance_transverse (dist, cosmo, z_star) / (1.0 + z_star);
  }
  else
    return GSL_NAN;
}

/**
 * nc_distance_shift_parameter:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 * @z: redshift $z$.
 *
 * The shift parameter $R(z)$ is defined as
 * \begin{align}
 * R(z) &=& \frac{\sqrt{\Omega_m H_0^2}}{c} (1 + z) D_A(z) \\
 * &=& \sqrt{\Omega_m} D_t(z),
 * \end{align}
 * where $\Omega_m$ is the matter density paremeter [nc_hicosmo_Omega_m()],
 * $D_A(z) = D_{H_0} D_t(z) / (1 + z)$ is the angular diameter distance and 
 * $D_t(z)$ is the tranverse comoving distance [Eq. $\eqref{eq:def:Dt}$].
 *
 * Returns: $R(z)$.
 */
gdouble
nc_distance_shift_parameter (NcDistance *dist, NcHICosmo *cosmo, gdouble z)
{
  const gdouble sqrt_mod_Omega_m = sqrt (fabs (nc_hicosmo_Omega_m (cosmo)));
  const gdouble transverse = nc_distance_transverse (dist, cosmo, z);
  return sqrt_mod_Omega_m * transverse;
}

/**
 * nc_distance_shift_parameter_lss:
 * @dist: a #NcDistance
 * @cosmo: a #NcHICosmo
 *
 * Compute the shift parameter $R(z)$ [nc_distance_shift_parameter()] at the
 * decoupling redshift $z_\star$ [nc_distance_decoupling_redshift()].
 *
 * Returns: $R(z_\star)$.
 */
gdouble
nc_distance_shift_parameter_lss (NcDistance *dist, NcHICosmo *cosmo)
{
  const gdouble sqrt_mod_Omega_m = sqrt(fabs(nc_hicosmo_Omega_m (cosmo)));
  const gdouble z_star = nc_distance_decoupling_redshift (dist, cosmo);
  if (gsl_finite (z_star))
  {
    const gdouble transverse = nc_distance_transverse (dist, cosmo, z_star);
    return sqrt_mod_Omega_m * transverse;
  }
  else
    return GSL_NAN;
}

/**
 * nc_distance_comoving_lss:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 *
 * Compute the comoving distance $D_c(z)$ [Eq. \eqref{eq:def:Dc}] at the 
 * decoupling redshift $z_\star$ [nc_distance_decoupling_redshift()].
 *
 * Returns: $D_c(z_\star)$.
 */
gdouble
nc_distance_comoving_lss (NcDistance *dist, NcHICosmo *cosmo)
{
  const gdouble z_star = nc_distance_decoupling_redshift (dist, cosmo);
  if (gsl_finite (z_star))
    return nc_distance_comoving (dist, cosmo, z_star);
  else
    return GSL_NAN;
}

/**
 * nc_distance_decoupling_redshift:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 *
 * The decoupling redshift $z_\star$ corresponds to the epoch of 
 * the last scattering surface of the cosmic microwave background photons. 
 *
 * This function computes $z_\star$ using [nc_hicosmo_z_lss()], if @cosmo implements 
 * it, or using Hu & Sugiyama fitting formula <link linkend="XHu1996">Hu (1996)</link>, 
 * $$ z_\star = 1048 \left(1 + 1.24 \times 10^{-3}  (\Omega_b h^2)^{-0.738}\right) \left(1 + g_1 (\Omega_m h^2)^{g_2}\right),$$
 * where $\Omega_b h^2$ [nc_hicosmo_Omega_bh2()] and $\Omega_m h^2$ [nc_hicosmo_Omega_mh2()] 
 * are, respectively, the baryonic and matter density parameters times the square 
 * of the dimensionless Hubble parameter $h$,  $H_0 = 100 \, h \, \text{km/s} \, \text{Mpc}^{-1}$. 
 * The parameters $g_1$ and $g_2$ are given by
 * $$g_1 = \frac{0.0783 (\Omega_b h^2)^{-0.238}}{(1 + 39.5 (\Omega_b h^2)^{0.763})} 
 * \; \text{and} \; g_2 = \frac{0.56}{\left(1 + 21.1 (\Omega_b h^2)^{1.81}\right)}.$$
 * 
 * Returns: $z_\star$
 */
gdouble
nc_distance_decoupling_redshift (NcDistance *dist, NcHICosmo *cosmo)
{
  NCM_UNUSED (dist);
  if (ncm_model_impl (NCM_MODEL (cosmo)) & NC_HICOSMO_IMPL_z_lss)
    return nc_hicosmo_z_lss (cosmo);
  else
  {
    gdouble omega_b_h2 = nc_hicosmo_Omega_bh2 (cosmo);
    gdouble omega_m_h2 = nc_hicosmo_Omega_mh2 (cosmo); 
    gdouble g1 = 0.0783 * pow (omega_b_h2, -0.238) / (1.0 + 39.5 * pow (omega_b_h2, 0.763));
    gdouble g2 = 0.560 / (1.0 + 21.1 * pow (omega_b_h2, 1.81));
    return 1048.0 * (1.0 + 1.24e-3 * pow (omega_b_h2, -0.738)) * (1.0 + g1 * pow (omega_m_h2, g2));
  }
}

static gdouble sound_horizon_integral_argument(gdouble z, gpointer p);

/**
 * nc_distance_sound_horizon:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 * @z: redshift $z$.
 *
 * Compute the sound horizon $r_s$, 
 * \begin{equation}
 * r_s (z) = \int_{z}^\infty \frac{c_s(z^\prime)}{E(z^\prime)} dz^\prime,
 * \end{equation}
 * where $c_s(z)$ is the speed of sound wave and $E(z)$ is the normalized Hubble function [nc_hicosmo_E()].
 * 
 * The integrand is given by
 * \begin{equation}\label{eq:def:rs:integrand}
 * \frac{c_s(z^\prime)}{H(z^\prime)} = \frac{1}{\sqrt{E(z^\prime)^2 (3 + \frac{9}{4} (1 + 0.2271 n_{eff}) \frac{\Omega_b}{\Omega_r (1 + z^\prime)})}},
 * \end{equation}
 * where $n_{eff}$ is the effective number of neutrinos [ncm_c_neutrino_n_eff()], 
 * $\Omega_b$ [nc_hicosmo_Omega_b()] and $\Omega_r$ [nc_hicosmo_Omega_r()] are the baryonic and radiation density parameter, respectively. 
 * If $\Omega_r = 0$, the integrand returns 0.0.
 *
 * Returns: $r_s(z)$.
 */
gdouble
nc_distance_sound_horizon (NcDistance *dist, NcHICosmo *cosmo, gdouble z)
{
  gdouble result, error;
  gsl_function F;

  F.function = &sound_horizon_integral_argument;
  F.params = cosmo;

  if (dist->use_cache)
    ncm_integral_cached_x_inf (dist->sound_horizon_cache, &F, z, &result, &error);
  else
    ncm_integral_locked_a_inf (&F, z, NCM_INTEGRAL_ABS_ERROR, NCM_INTEGRAL_ERROR, &result, &error);

  return result;
}

static gdouble
sound_horizon_integral_argument (gdouble z, gpointer p)
{
  NcHICosmo *cosmo = NC_HICOSMO (p);
  gdouble E2 = nc_hicosmo_E2 (cosmo, z);
  gdouble omega_b = nc_hicosmo_Omega_b (cosmo);
  gdouble omega_r = nc_hicosmo_Omega_r (cosmo);
  if (omega_r == 0.0)
    return 0.0;
  if (GSL_SIGN(E2) == -1.0)
    return GSL_POSINF;
  return
    1.0 / sqrt (E2 * (3.0 + 9.0 / 4.0 * (1.0 + ncm_c_neutrino_n_eff () * 0.2271) * omega_b / (omega_r * (1.0 + z))));
}

/**
 * nc_distance_dsound_horizon_dz:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 * @z: redshift $z$.
 *
 * Calculate the sound horizon [nc_distance_sound_horizon()] derivative with respect to $z$, 
 * $$\frac{d r_s(z)}{dz} = - \frac{c_s(z)}{E(z)},$$
 * where $c_s(z) / E(z)$ is given by Eq. \eqref{eq:def:rs:integrand}.
 *
 * Returns: $\frac{d r_s(z)}{dz}$.
 */
gdouble
nc_distance_dsound_horizon_dz (NcDistance *dist, NcHICosmo *cosmo, gdouble z)
{
  NCM_UNUSED (dist);
  return -sound_horizon_integral_argument (z, cosmo);
}

/**
 * nc_distance_acoustic_scale:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 *
 * Compute the acoustic scale $l_A (z_\star)$ at $z_\star$ [nc_distance_decoupling_redshift()],
 * \begin{equation}
 * l_A(z_\star) = \pi \frac{D_t (z_\star)}{r_s (z_\star)},
 * \end{equation}
 * where $D_t(z_\star)$ is the comoving transverse distance [nc_distance_transverse()] 
 * and $r_s(z_\star)$ is the sound horizon [nc_distance_sound_horizon()] both 
 * both computed at $z_\star$.
 *
 * Returns: $l_A(z_\star)$.
 */
gdouble
nc_distance_acoustic_scale (NcDistance *dist, NcHICosmo *cosmo)
{
  gdouble z = nc_distance_decoupling_redshift (dist, cosmo);
  if (gsl_finite (z))
    return M_PI * nc_distance_transverse (dist, cosmo, z) / nc_distance_sound_horizon (dist, cosmo, z);
  else
    return GSL_NAN;
}

/**
 * nc_distance_drag_redshift:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 *
 * Drag redshift is the epoch at which baryons were released from photons. 
 * 
 * This function computes $z_d$ using the fitting formula given in
 * <link linkend="XEisenstein1998">Eisenstein & Hu (1998)</link>,
 * $$z_d = \frac{1291 (\Omega_m h^2)^{0.251}}{(1 + 0.659 (\Omega_m h^2)^{0.828})}
 * \left(1 + b_1 (\Omega_b h^2)^{b_2}\right),$$
 * where $\Omega_b h^2$ [nc_hicosmo_Omega_bh2()] and $\Omega_m h^2$ [nc_hicosmo_Omega_mh2()] 
 * are, respectively, the baryonic and matter density parameters times the square 
 * of the dimensionless Hubble parameter $h$,  $H_0 = 100 \, h \, \text{km/s} \, \text{Mpc}^{-1}$. 
 * The parameters $b_1$ and $b_2$ are given by
 * $$b_1 = 0.313 (\Omega_m h^2)^{-0.419} \left(1 + 0.607 (\Omega_m h^2)^{0.674}\right) \; 
 * \text{and} \; b_2 = 0.238 (\Omega_m h^2)^{0.223}.$$
 * 
 * Returns: $z_d$.
 */
gdouble
nc_distance_drag_redshift (NcDistance *dist, NcHICosmo *cosmo)
{
  gdouble omega_m_h2 = nc_hicosmo_Omega_mh2 (cosmo);
  gdouble omega_b_h2 = nc_hicosmo_Omega_bh2 (cosmo);
  gdouble b1 = 0.313 * pow (omega_m_h2, -0.419) * (1.0 + 0.607 * pow (omega_m_h2, 0.674));
  gdouble b2 = 0.238 * pow (omega_m_h2, 0.223);

  NCM_UNUSED (dist);
  
  return 1291.0 * pow (omega_m_h2, 0.251) / (1.0 + 0.659 * pow (omega_m_h2, 0.828)) *
    (1.0 + b1 * pow (omega_b_h2, b2));
}

/**
 * nc_distance_dilation_scale:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 * @z: redshift $z$.
 *
 * The dilation scale is the cube root of the product of the radial dilation
 * times the square of the transverse dilation -- (arXiv:astro-ph/0501171)
 *
 * Returns: $D_V(z)$.
 */
gdouble
nc_distance_dilation_scale (NcDistance *dist, NcHICosmo *cosmo, gdouble z)
{
  gdouble Dt = nc_distance_transverse (dist, cosmo, z);
  gdouble E = sqrt (nc_hicosmo_E2 (cosmo, z));
  gdouble Dv = cbrt (Dt * Dt * z / E);
  return Dv;
}

/**
 * nc_distance_bao_A_scale:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 * @z: the redshift $z$.
 *
 * Bao 'A' scale D_v(z) sqrt(Omega_m) / z -- (arXiv:astro-ph/0501171)
 *
 * Returns: FIXME
 */
gdouble
nc_distance_bao_A_scale (NcDistance *dist, NcHICosmo *cosmo, gdouble z)
{
  gdouble Dv = nc_distance_dilation_scale (dist, cosmo, z);
  gdouble sqrt_Omega_m = sqrt (nc_hicosmo_Omega_m (cosmo));
  return sqrt_Omega_m * Dv / z;
}

/**
 * nc_distance_bao_r_Dv:
 * @dist: a #NcDistance.
 * @cosmo: a #NcHICosmo.
 * @z: the redshift $z$.
 *
 * r(z_d) / D_v(z) -- (arXiv:0705.3323).
 *
 * Returns: FIXME
 */
gdouble
nc_distance_bao_r_Dv (NcDistance *dist, NcHICosmo *cosmo, gdouble z)
{
  gdouble r_zd;
  gdouble Dv;

  if (ncm_model_impl (NCM_MODEL (cosmo)) & NC_HICOSMO_IMPL_as_drag)
    r_zd = nc_hicosmo_as_drag (cosmo);
  else
  {
    gdouble zd = nc_distance_drag_redshift (dist, cosmo);
    if (!gsl_finite (zd))
      return GSL_NAN;
    r_zd = nc_distance_sound_horizon (dist, cosmo, zd);
  }

  Dv = nc_distance_dilation_scale (dist, cosmo, z);
  return r_zd / Dv;
}

/***************************************************************************
 *            cosmic_time.c
 *
 *  Wed Nov 12 17:06:27 2008
 *  Copyright  2008  Sandro Dias Pinto Vitenti
 *  <sandro@isoftware.com.br>
 ****************************************************************************/

static gdouble
_nc_time_integrand (gdouble z, gpointer p)
{
  NcHICosmo *cosmo = NC_HICOSMO (p);
  const gdouble x = 1.0 + z;
  const gdouble E = sqrt (nc_hicosmo_E2 (cosmo, z));
  return 1.0 / (x * E);
}

gdouble
nc_distance_cosmic_time (NcDistance *dist, NcHICosmo *cosmo, gdouble z)
{
  gdouble result, error;
  gsl_function F;
  F.function = &_nc_time_integrand;
  F.params = cosmo;

  if (dist->use_cache)
    ncm_integral_cached_x_inf (dist->time_cache, &F, z, &result, &error);
  else
    ncm_integral_locked_a_inf (&F, z, NCM_INTEGRAL_ABS_ERROR, NCM_INTEGRAL_ERROR, &result, &error);

  return result;
}

gdouble
nc_distance_lookback_time (NcDistance *dist, NcHICosmo *cosmo, gdouble z)
{
  gdouble result, error;
  gsl_function F;
  F.function = &_nc_time_integrand;
  F.params = cosmo;

  if (dist->use_cache)
    ncm_integral_cached_0_x (dist->lookback_time_cache, &F, z, &result, &error);
  else
    ncm_integral_locked_a_b (&F, 0.0, z, 0.0, NCM_INTEGRAL_ERROR, &result, &error);

  return result;
}

gdouble
nc_distance_conformal_lookback_time (NcDistance *dist, NcHICosmo *cosmo, gdouble z)
{
  return nc_distance_comoving (dist, cosmo, z);
}

static gdouble
nc_conformal_time_integrand (gdouble logx, gpointer p)
{
  if (logx > GSL_LOG_DBL_MAX)
    return 0.0;
  else
  {
    NcHICosmo *cosmo = NC_HICOSMO (p);
    const gdouble z = expm1 (logx);
    const gdouble x = 1.0 + z;
    const gdouble E = sqrt (nc_hicosmo_E2 (cosmo, z));

    if (!gsl_finite (E))
      return 0.0;
    else
      return x / E;
  }
}

gdouble
nc_distance_conformal_time (NcDistance *dist, NcHICosmo *cosmo, gdouble z)
{
  gdouble result, error;
  gsl_function F;

  F.function = &nc_conformal_time_integrand;
  F.params = cosmo;

  if (dist->use_cache)
    ncm_integral_cached_x_inf (dist->conformal_time_cache, &F, log1p(z), &result, &error);
  else
    ncm_integral_locked_a_inf (&F, log1p(z), NCM_INTEGRAL_ABS_ERROR, NCM_INTEGRAL_ERROR, &result, &error);

  return result;
}

typedef struct _NcDistanceFuncData
{
  NcDistance *dist;
  NcDistanceFunc0 f0;
  NcDistanceFunc1 f1;
} NcDistanceFuncData;

static void
_nc_distance_func0 (NcmMSet *mset, gpointer obj, const gdouble *x, gdouble *f)
{
  NcDistanceFuncData *dist_data = (NcDistanceFuncData *)obj;
  NCM_UNUSED (x);
  f[0] = dist_data->f0 (dist_data->dist, NC_HICOSMO (ncm_mset_peek (mset, nc_hicosmo_id ())));
}

static void
_nc_distance_func1 (NcmMSet *mset, gpointer obj, const gdouble *x, gdouble *f)
{
  NcDistanceFuncData *dist_data = (NcDistanceFuncData *)obj;
  f[0] = dist_data->f1 (dist_data->dist, NC_HICOSMO (ncm_mset_peek (mset, nc_hicosmo_id ())), x[0]);
}

static void
_nc_distance_free (gpointer obj)
{
  NcDistanceFuncData *dist_data = (NcDistanceFuncData *)obj;
  nc_distance_free (dist_data->dist);
  g_slice_free (NcDistanceFuncData, dist_data);
}

/**
 * nc_distance_func0_new:
 * @dist: FIXME
 * @f0: (scope notified): FIXME
 *
 * Returns: (transfer full): FIXME
 */
NcmMSetFunc *
nc_distance_func0_new (NcDistance *dist, NcDistanceFunc0 f0)
{
  NcDistanceFuncData *dist_data = g_slice_new (NcDistanceFuncData);
  dist_data->dist = nc_distance_ref (dist);
  dist_data->f0 = f0;
  dist_data->f1 = NULL;
  return ncm_mset_func_new (&_nc_distance_func0, 0, 1, dist_data, &_nc_distance_free);
}

/**
 * nc_distance_func1_new:
 * @dist: FIXME
 * @f1: (scope notified): FIXME
   *
 * Returns: (transfer full): FIXME
   */
NcmMSetFunc *
nc_distance_func1_new (NcDistance *dist, NcDistanceFunc1 f1)
{
  NcDistanceFuncData *dist_data = g_slice_new (NcDistanceFuncData);
  dist_data->dist = nc_distance_ref (dist);
  dist_data->f0 = NULL;
  dist_data->f1 = f1;
  return ncm_mset_func_new (&_nc_distance_func1, 1, 1, dist_data, &_nc_distance_free);
}
