/***************************************************************************
 *            test_ncm_model.c
 *
 *  Mon May 7 16:39:21 2012
 *  Copyright  2012  Mariana Penna Lima & Sandro Dias Pinto Vitenti
 *  <pennalima@gmail.com>
 ****************************************************************************/
/*
 * numcosmo
 * Copyright (C) Mariana Penna Lima & Sandro Dias Pinto Vitenti 2012 <pennalima@gmail.com>, <sandro@isoftware.com.br>
 * numcosmo is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * numcosmo is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#undef GSL_RANGE_CHECK_OFF
#endif /* HAVE_CONFIG_H */
#include <numcosmo/numcosmo.h>
#include "ncm_model_test.h"

typedef struct _TestNcmModel
{
  GType type;
  guint sparam_len, vparam_len;
  gchar *name, *nick;
  NcmModelTest *tm;
  NcmReparam *reparam;
} TestNcmModel;

void test_ncm_model_new (TestNcmModel *test, gconstpointer pdata);
void test_ncm_model_child_new (TestNcmModel *test, gconstpointer pdata);
void test_ncm_model_child_child_new (TestNcmModel *test, gconstpointer pdata);
void test_ncm_model_reparam_new (TestNcmModel *test, gconstpointer pdata);
void test_ncm_model_free (TestNcmModel *test, gconstpointer pdata);

void test_ncm_model_test_new (TestNcmModel *test, gconstpointer pdata);
void test_ncm_model_test_length (TestNcmModel *test, gconstpointer pdata);
void test_ncm_model_test_defval (TestNcmModel *test, gconstpointer pdata);
void test_ncm_model_test_name_symbol (TestNcmModel *test, gconstpointer pdata);
void test_ncm_model_test_setget (TestNcmModel *test, gconstpointer pdata);
void test_ncm_model_test_setget_prop (TestNcmModel *test, gconstpointer pdata);
void test_ncm_model_test_setget_vector (TestNcmModel *test, gconstpointer pdata);
void test_ncm_model_test_setget_model (TestNcmModel *test, gconstpointer pdata);
void test_ncm_model_test_name_index (TestNcmModel *test, gconstpointer pdata);
void test_ncm_model_test_dup (TestNcmModel *test, gconstpointer pdata);

#define TEST_NCM_MODEL_NTYPES 4

gint
main (gint argc, gchar *argv[])
{
  gint i;

  g_test_init (&argc, &argv, NULL);
  ncm_cfg_init ();
  ncm_cfg_enable_gsl_err_handler ();
  gpointer ccc[TEST_NCM_MODEL_NTYPES][3] = {
    {"model",            &test_ncm_model_new,             &test_ncm_model_free},
    {"model/child",      &test_ncm_model_child_new,       &test_ncm_model_free},
    {"model/grandchild", &test_ncm_model_child_child_new, &test_ncm_model_free},
    {"model/reparam",    &test_ncm_model_reparam_new,     &test_ncm_model_free},
  };

  for (i = 0; i < TEST_NCM_MODEL_NTYPES; i++)
  {
    gchar *d;

    d = g_strdup_printf ("/ncm/%s/new", (gchar *) ccc[i][0]);
    g_test_add (d, TestNcmModel, NULL, ccc[i][1], &test_ncm_model_test_new, ccc[i][2]);
    g_free (d);

    d = g_strdup_printf ("/ncm/%s/length", (gchar *) ccc[i][0]);
    g_test_add (d, TestNcmModel, NULL, ccc[i][1], &test_ncm_model_test_length, ccc[i][2]);
    g_free (d);

    d = g_strdup_printf ("/ncm/%s/defval", (gchar *) ccc[i][0]);
    g_test_add (d, TestNcmModel, NULL, ccc[i][1], &test_ncm_model_test_defval, ccc[i][2]);
    g_free (d);

    d = g_strdup_printf ("/ncm/%s/name_symbol", (gchar *) ccc[i][0]);
    g_test_add (d, TestNcmModel, NULL, ccc[i][1], &test_ncm_model_test_name_symbol, ccc[i][2]);
    g_free (d);

    d = g_strdup_printf ("/ncm/%s/setget", (gchar *) ccc[i][0]);
    g_test_add (d, TestNcmModel, NULL, ccc[i][1], &test_ncm_model_test_setget, ccc[i][2]);
    g_free (d);

    d = g_strdup_printf ("/ncm/%s/setget/prop", (gchar *) ccc[i][0]);
    g_test_add (d, TestNcmModel, NULL, ccc[i][1], &test_ncm_model_test_setget_prop, ccc[i][2]);
    g_free (d);

    d = g_strdup_printf ("/ncm/%s/setget/vector", (gchar *) ccc[i][0]);
    g_test_add (d, TestNcmModel, NULL, ccc[i][1], &test_ncm_model_test_setget_vector, ccc[i][2]);
    g_free (d);

    d = g_strdup_printf ("/ncm/%s/setget/model", (gchar *) ccc[i][0]);
    g_test_add (d, TestNcmModel, NULL, ccc[i][1], &test_ncm_model_test_setget_model, ccc[i][2]);
    g_free (d);

    d = g_strdup_printf ("/ncm/%s/name_index", (gchar *) ccc[i][0]);
    g_test_add (d, TestNcmModel, NULL, ccc[i][1], &test_ncm_model_test_name_index, ccc[i][2]);
    g_free (d);

    d = g_strdup_printf ("/ncm/%s/dup", (gchar *) ccc[i][0]);
    g_test_add (d, TestNcmModel, NULL, ccc[i][1], &test_ncm_model_test_dup, ccc[i][2]);
    g_free (d);
  }

  g_test_run ();
}

void
test_ncm_model_new (TestNcmModel *test, gconstpointer pdata)
{
  test->type       = NCM_TYPE_MODEL_TEST;
  test->tm         = g_object_new (test->type, NULL);
  test->sparam_len = SPARAM_LEN1;
  test->vparam_len = VPARAM_LEN1;
  test->name       = name_tot[0];
  test->nick       = nick_tot[0];
  test->reparam    = NULL;

  g_assert (test->type != 0);
}

void
test_ncm_model_child_new (TestNcmModel *test, gconstpointer pdata)
{
  test->type       = NCM_TYPE_MODEL_TEST_CHILD;
  test->tm         = g_object_new (test->type, NULL);
  test->sparam_len = SPARAM_LEN1 + SPARAM_LEN2;
  test->vparam_len = VPARAM_LEN1 + VPARAM_LEN2;
  test->name       = name_tot[1];
  test->nick       = nick_tot[1];
  test->reparam    = NULL;

  g_assert (test->type != 0);
}

void
test_ncm_model_child_child_new (TestNcmModel *test, gconstpointer pdata)
{
  test->type       = NCM_TYPE_MODEL_TEST_CHILD_CHILD;
  test->tm         = g_object_new (test->type, NULL);
  test->sparam_len = SPARAM_LEN1 + SPARAM_LEN2 + SPARAM_LEN3;
  test->vparam_len = VPARAM_LEN1 + VPARAM_LEN2 + VPARAM_LEN3;
  test->name       = name_tot[2];
  test->nick       = nick_tot[2];
  test->reparam    = NULL;

  g_assert (test->type != 0);
}

NcmReparam *
_test_ncm_model_create_reparam (TestNcmModel *test)
{
  guint size = ncm_model_len (NCM_MODEL (test->tm));
  NcmMatrix *T = ncm_matrix_new (size, size);
  NcmVector *v = ncm_vector_new (size);
  NcmReparamLinear *relin;
  NcmBootstrap *bstrap = ncm_bootstrap_sized_new (size);
  NcmRNG *rng = ncm_rng_new (NULL);
  guint i;
  guint cdesc_n = size / 10;
  if (cdesc_n == 0) cdesc_n = 1;

  ncm_bootstrap_remix (bstrap, rng);

  ncm_matrix_set_zero (T);
  ncm_vector_set_zero (v);

  for (i = 0; i < size; i++)
  {
    ncm_matrix_set (T, i, ncm_bootstrap_get (bstrap, i), g_test_rand_double_range (1.0, 10.0));
    ncm_vector_set (v, i, g_test_rand_double ());
  }

  relin = ncm_reparam_linear_new (size, T, v);
  ncm_reparam_linear_set_compat_type (relin, NCM_TYPE_MODEL_TEST);

  for (i = 0; i < cdesc_n; i++)
  {
    guint j = g_test_rand_int_range (0, size - 1);
    gchar *new_param = g_strdup_printf ("new_param_%u", j);
    gchar *new_param_symbol = g_strdup_printf ("NP_%u", j);
    ncm_reparam_set_param_desc_full (NCM_REPARAM (relin),
                                     j,
                                     new_param,
                                     new_param_symbol,
                                     -10.0,
                                     10.0,
                                     1.0,
                                     0.0,
                                     1.0,
                                     NCM_PARAM_TYPE_FIXED);
    g_free (new_param);
    g_free (new_param_symbol);
  }

  ncm_model_set_reparam (NCM_MODEL (test->tm), NCM_REPARAM (relin));
  ncm_reparam_free (NCM_REPARAM (relin));

  ncm_vector_free (v);
  ncm_matrix_free (T);
  ncm_rng_free (rng);

  return NCM_REPARAM (relin);
}

void
test_ncm_model_reparam_new (TestNcmModel *test, gconstpointer pdata)
{
  test->type       = NCM_TYPE_MODEL_TEST;
  test->tm         = g_object_new (test->type, NULL);
  test->sparam_len = SPARAM_LEN1;
  test->vparam_len = VPARAM_LEN1;
  test->name       = name_tot[0];
  test->nick       = nick_tot[0];
  test->reparam    = _test_ncm_model_create_reparam (test);

  g_assert (test->type != 0);
}

void
test_ncm_model_free (TestNcmModel *test, gconstpointer pdata)
{
  NCM_TEST_FREE (g_object_unref, test->tm);
}

void
test_ncm_model_test_new (TestNcmModel *test, gconstpointer pdata)
{
  NcmModelTest *tm = test->tm;
  NcmModel *model;
  g_assert (NCM_IS_MODEL (tm));
  g_assert (NCM_IS_MODEL_TEST (tm));
  model = NCM_MODEL (tm);

  g_assert (ncm_model_name (model) != test->name);
  g_assert (ncm_model_nick (model) != test->nick);

  g_assert_cmpstr (ncm_model_name (model), ==, test->name);
  g_assert_cmpstr (ncm_model_nick (model), ==, test->nick);
  g_assert (ncm_model_impl (model) == 0);
  g_assert (ncm_model_peek_reparam (model) == test->reparam);
}

void
test_ncm_model_test_length (TestNcmModel *test, gconstpointer pdata)
{
  NcmModelTest *tm = test->tm;
  NcmModel *model = NCM_MODEL (tm);
  guint model_len = 0;
  guint i;

  model_len = test->sparam_len;
  for (i = 0; i < test->vparam_len; i++)
  {
    g_assert_cmpint (ncm_model_vparam_len (model, i), ==, v_len_tot[i]);
    model_len += v_len_tot[i];
  }

  g_assert_cmpint (ncm_model_len (model), ==, model_len);
  g_assert_cmpint (ncm_model_sparam_len (model), ==, test->sparam_len);
  g_assert_cmpint (ncm_model_vparam_array_len (model), ==, test->vparam_len);
}

void
test_ncm_model_test_defval (TestNcmModel *test, gconstpointer pdata)
{
  NcmModelTest *tm = test->tm;
  NcmModel *model = NCM_MODEL (tm);
  guint i;

  for (i = 0; i < test->sparam_len; i++)
  {
    gdouble p = ncm_model_orig_param_get (model, i);
    ncm_assert_cmpdouble (p, ==, s_defval_tot[i]);
  }

  for (i = 0; i < test->vparam_len; i++)
  {
    guint j;
    for (j = 0; j < v_len_tot[i]; j++)
    {
      gdouble p = ncm_model_orig_vparam_get (model, i, j);
      ncm_assert_cmpdouble (p, ==, v_defval_tot[i]);
    }
  }
}

void
test_ncm_model_test_name_symbol (TestNcmModel *test, gconstpointer pdata)
{
  NcmModelTest *tm = test->tm;
  NcmModel *model = NCM_MODEL (tm);
  guint i;

  for (i = 0; i < test->sparam_len; i++)
  {
    const gchar *pname = ncm_model_orig_param_name (model, i);
    const gchar *psymbol = ncm_model_orig_param_symbol (model, i);
    g_assert (pname != s_name_tot[i]);
    g_assert (psymbol != s_symbol_tot[i]);
    g_assert_cmpstr (pname, ==, s_name_tot[i]);
    g_assert_cmpstr (psymbol, ==, s_symbol_tot[i]);
  }

  for (i = 0; i < test->vparam_len; i++)
  {
    guint j;
    for (j = 0; j < v_len_tot[i]; j++)
    {
      gchar *vp_name = g_strdup_printf ("%s_%u", v_name_tot[i], j);
      gchar *vp_symbol = g_strdup_printf ("%s_%u", v_symbol_tot[i], j);
      const gchar *pname = ncm_model_orig_param_name (model, ncm_model_vparam_index (model, i, j));
      const gchar *psymbol = ncm_model_orig_param_symbol (model, ncm_model_vparam_index (model, i, j));

      g_assert (pname != vp_name);
      g_assert (psymbol != vp_symbol);
      g_assert_cmpstr (pname, ==, vp_name);
      g_assert_cmpstr (psymbol, ==, vp_symbol);

      g_free (vp_name);
      g_free (vp_symbol);
    }
  }
}

void
test_ncm_model_test_finite (TestNcmModel *test, gconstpointer pdata)
{
  NcmModelTest *tm = test->tm;
  NcmModel *model = NCM_MODEL (tm);
  guint model_len = ncm_model_len (model);
  guint i;

  for (i = 0; i < model_len; i++)
  {
    g_assert (ncm_model_param_finite (model, i));
    g_assert (ncm_model_params_finite (model));

    ncm_model_param_set (model, i, GSL_NAN);
    g_assert (!ncm_model_param_finite (model, i));
    g_assert (!ncm_model_params_finite (model));
    ncm_model_param_set_default (model, i);
    g_assert (ncm_model_param_finite (model, i));
    g_assert (ncm_model_params_finite (model));

    ncm_model_param_set (model, i, GSL_POSINF);
    g_assert (!ncm_model_param_finite (model, i));
    g_assert (!ncm_model_params_finite (model));
    ncm_model_param_set_default (model, i);
    g_assert (ncm_model_param_finite (model, i));
    g_assert (ncm_model_params_finite (model));

    ncm_model_param_set (model, i, GSL_NEGINF);
    g_assert (!ncm_model_param_finite (model, i));
    g_assert (!ncm_model_params_finite (model));
    ncm_model_param_set_default (model, i);
    g_assert (ncm_model_param_finite (model, i));
    g_assert (ncm_model_params_finite (model));
  }
}

void
test_ncm_model_test_setget (TestNcmModel *test, gconstpointer pdata)
{
  NcmModelTest *tm = test->tm;
  NcmModel *model = NCM_MODEL (tm);
  guint model_len = ncm_model_len (model);
  NcmVector *tmp = ncm_vector_new (model_len);
  guint i;

  /* ncm_model_param_get / ncm_model_param_set */

  for (i = 0; i < model_len; i++)
  {
    gdouble lb = ncm_model_param_get_lower_bound (model, i);
    gdouble ub = ncm_model_param_get_upper_bound (model, i);
    gdouble val;
    while ((val = g_test_rand_double_range (lb, ub)))
    {
      if (val != ncm_model_param_get (model, i))
        break;
    }

    ncm_model_param_set (model, i, val);
    ncm_assert_cmpdouble (ncm_model_param_get (model, i), ==, val);

    ncm_model_param_set_default (model, i);
    ncm_assert_cmpdouble (ncm_model_param_get (model, i), !=, val);

    ncm_model_param_set (model, i, val);
    ncm_vector_set (tmp, i, val);
  }

  /* ncm_model_params_save_as_default */
  ncm_model_params_save_as_default (model);

  for (i = 0; i < model_len; i++)
  {
    ncm_assert_cmpdouble (ncm_model_param_get (model, i), ==, ncm_vector_get (tmp, i));
  }

  for (i = 0; i < test->sparam_len; i++)
  {
    ncm_model_param_set (model, i, s_defval_tot[i]);
  }

  for (i = 0; i < test->vparam_len; i++)
  {
    guint j;
    for (j = 0; j < v_len_tot[i]; j++)
    {
      ncm_model_param_set (model, ncm_model_vparam_index (model, i, j), v_defval_tot[i]);
    }
  }

  ncm_model_params_set_default (model);
  for (i = 0; i < model_len; i++)
  {
    ncm_assert_cmpdouble (ncm_model_param_get (model, i), ==, ncm_vector_get (tmp, i));
  }

  for (i = 0; i < test->sparam_len; i++)
  {
    ncm_model_param_set (model, i, s_defval_tot[i]);
  }
  for (i = 0; i < test->vparam_len; i++)
  {
    guint j;
    for (j = 0; j < v_len_tot[i]; j++)
    {
      ncm_model_param_set (model, ncm_model_vparam_index (model, i, j), v_defval_tot[i]);
    }
  }
  ncm_model_params_save_as_default (model);

  /* ncm_model_params_set_all_data */
  ncm_model_params_set_all_data (model, ncm_vector_data (tmp));
  for (i = 0; i < model_len; i++)
  {
    ncm_assert_cmpdouble (ncm_model_param_get (model, i), ==, ncm_vector_get (tmp, i));
  }

  ncm_vector_free (tmp);
}

void
test_ncm_model_test_setget_prop (TestNcmModel *test, gconstpointer pdata)
{
  NcmModelTest *tm = test->tm;
  NcmModel *model = NCM_MODEL (tm);
  guint i;

  /* ncm_model_param_get / ncm_model_param_set */

  for (i = 0; i < test->sparam_len; i++)
  {
    gdouble lb = ncm_model_orig_param_get_lower_bound (model, i);
    gdouble ub = ncm_model_orig_param_get_upper_bound (model, i);
    gdouble val, val_out;
    while ((val = g_test_rand_double_range (lb, ub)))
    {
      if (val != ncm_model_orig_param_get (model, i))
        break;
    }

    g_object_set (model, s_name_tot[i], val, NULL);
    g_object_get (model, s_name_tot[i], &val_out, NULL);
    ncm_assert_cmpdouble (val_out, ==, val);
  }

  for (i = 0; i < test->vparam_len; i++)
  {
    NcmVector *tmp = ncm_vector_new (v_len_tot[i]);
    NcmVector *tmp_out = NULL;
    guint j;
    for (j = 0; j < v_len_tot[i]; j++)
    {
      guint n = ncm_model_vparam_index (model, i, j);
      gdouble lb = ncm_model_orig_param_get_lower_bound (model, n);
      gdouble ub = ncm_model_orig_param_get_upper_bound (model, n);
      gdouble val;

      while ((val = g_test_rand_double_range (lb, ub)))
      {
        if (val != ncm_model_orig_param_get (model, n))
          break;
      }
      ncm_vector_set (tmp, j, val);
    }

    tmp_out = NULL;
    g_object_set (model, v_name_tot[i], tmp, NULL);
    g_object_get (model, v_name_tot[i], &tmp_out, NULL);

    for (j = 0; j < v_len_tot[i]; j++)
    {
      guint n = ncm_model_vparam_index (model, i, j);
      ncm_assert_cmpdouble (ncm_vector_get (tmp, j), ==, ncm_model_orig_param_get (model, n));
      ncm_assert_cmpdouble (ncm_vector_get (tmp, j), ==, ncm_vector_get (tmp_out, j));
    }

    NCM_TEST_FREE (ncm_vector_free, tmp);
    NCM_TEST_FREE (ncm_vector_free, tmp_out);
  }
}

void
test_ncm_model_test_setget_vector (TestNcmModel *test, gconstpointer pdata)
{
  NcmModelTest *tm = test->tm;
  NcmModel *model = NCM_MODEL (tm);
  guint model_len = ncm_model_len (model);

  NCM_TEST_FAIL (G_STMT_START {
    NcmVector *tmp2 = ncm_vector_new (model_len + 1);
    ncm_model_params_set_vector (model, tmp2);
    ncm_vector_free (tmp2);
  } G_STMT_END);
}

void
test_ncm_model_test_setget_model (TestNcmModel *test, gconstpointer pdata)
{
  NcmSerialize *ser = ncm_serialize_global ();
  NcmModelTest *tm1 = test->tm;
  NcmModel *model1 = NCM_MODEL (tm1);
  NcmModel *model2 = ncm_model_dup (model1, ser);
  NcmModelTest *tm3 = g_object_new (test->type,
                                    "VPBase0-length", ncm_model_vparam_len (model1, 0) + 1,
                                    "VPBase1-length", ncm_model_vparam_len (model1, 1) + 1,
                                    NULL);
  NcmModel *model3 = NCM_MODEL (tm3);
  guint model_len = ncm_model_len (model1);
  guint i;

  ncm_serialize_free (ser);

  g_assert (ncm_model_is_equal (model1, model2));
  g_assert (!ncm_model_is_equal (model1, model3));

  for (i = 0; i < model_len; i++)
  {
    gdouble lb = ncm_model_param_get_lower_bound (model2, i);
    gdouble ub = ncm_model_param_get_upper_bound (model2, i);
    gdouble val;
    while ((val = g_test_rand_double_range (lb, ub)))
    {
      if (val != ncm_model_param_get (model2, i))
        break;
    }

    ncm_model_param_set (model2, i, val);
  }

  ncm_model_params_set_model (model1, model2);

  for (i = 0; i < model_len; i++)
  {
    ncm_assert_cmpdouble (ncm_model_param_get (model1, i), ==, ncm_model_param_get (model2, i));
  }

  g_assert (!ncm_model_is_equal (model1, model3));
  g_assert (!ncm_model_is_equal (model3, model1));

  NCM_TEST_FREE (ncm_model_free, model2);
  NCM_TEST_FREE (ncm_model_free, model3);
}

void
test_ncm_model_test_name_index (TestNcmModel *test, gconstpointer pdata)
{
  NcmModelTest *tm = test->tm;
  NcmModel *model = NCM_MODEL (tm);
  guint i;

  for (i = 0; i < test->sparam_len; i++)
  {
    guint n;
    gboolean found = ncm_model_orig_param_index_from_name (model, s_name_tot[i], &n);
    const gchar *s_name_n = ncm_model_orig_param_name (model, n);
    const gchar *s_symbol_n = ncm_model_orig_param_symbol (model, n);

    g_assert (found);
    g_assert_cmpuint (n, ==, i);
    g_assert_cmpstr (s_name_n, ==, s_name_tot[n]);
    g_assert_cmpstr (s_symbol_n, ==, s_symbol_tot[n]);
  }

  for (i = 0; i < test->vparam_len; i++)
  {
    guint j;

    for (j = 0; j < v_len_tot[i]; j++)
    {
      gchar *v_name_ij = g_strdup_printf ("%s_%u", v_name_tot[i], j);
      gchar *v_symbol_ij = g_strdup_printf ("%s_%u", v_symbol_tot[i], j);
      guint n;
      gboolean found = ncm_model_orig_param_index_from_name (model, v_name_ij, &n);
      const gchar *v_name_n = ncm_model_orig_param_name (model, n);
      const gchar *v_symbol_n = ncm_model_orig_param_symbol (model, n);

      g_assert (found);
      g_assert_cmpstr (v_name_n, ==, v_name_ij);
      g_assert_cmpstr (v_symbol_n, ==, v_symbol_ij);

      g_free (v_name_ij);
      g_free (v_symbol_ij);
    }
  }
}

void
test_ncm_model_test_dup (TestNcmModel *test, gconstpointer pdata)
{
  NcmSerialize *ser = ncm_serialize_global ();
  NcmModelTest *tm = test->tm;
  NcmModel *model = NCM_MODEL (tm);
  NcmModel *model_dup = ncm_model_dup (model, ser);
  guint model_len = ncm_model_len (model);
  guint i;

  ncm_serialize_free (ser);
  g_assert (ncm_model_is_equal (model, model_dup));

  for (i = 0; i < model_len; i++)
  {
    ncm_assert_cmpdouble (ncm_model_param_get (model, i), ==, ncm_model_param_get (model_dup, i));
  }
}
