/*
    Numdiff - compare putatively similar files, 
    ignoring small numeric differences
    Copyright (C) 2005,2006  Ivano Primi  <ivprimi@libero.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<ctype.h>
#include"numdiff.h"

/* See io.c */
extern char* read_line (FILE* pf, int* errcode); 
extern void print_head (const char* str, size_t length);
extern void print_ws (unsigned n);
#ifdef __DEBUG__
extern void Eprint_head (const char* str, size_t length);
extern void Eprint_nonl (const char* str);
#endif /* __DEBUG__ */


#ifdef _USE_MPA

static
void field2cx (const char* field, size_t length, char** tail, 
	       int iscale, const struct numfmt* pnf, Complex* pz)
{
  char* ptr = (char*) field;
  char ch;

  ch = ptr[length];
  ptr[length] = '\0';
  str2C (ptr, tail, iscale, pnf, pz);
  ptr[length] = ch;
}

#else

static
Complex field2cx (const char* field, size_t length, char** tail)
{
  char* ptr = (char*) field;
  char ch;
  Complex z;

  ch = ptr[length];
  ptr[length] = '\0';
  z = str2C (ptr, tail);
  ptr[length] = ch;
  return z;
}

#endif /* _USE_MPA */

/*
  Be careful ! strNcmp() and strNcasecmp() are only used
  when n <= strlen (s) == strlen (t).
*/

static int strNcmp (const char* s, const char* t, size_t n)
{
  const char *p, *q;

  for (p = s, q = t; p < s + n && *p == *q; p++, q++);
  return p < s + n;
}

static int strNcasecmp (const char* s, const char* t, size_t n)
{
  const char *p, *q;

  for (p = s, q = t; p < s + n && tolower(*p) == tolower(*q); p++, q++);
  return p < s + n;
}

int cmp_fields (const char* field1, const char* field2,
	       size_t l1, size_t l2, argslist* argl,
	       Real* abserr, Real* relerr)
{
  char *tail1, *tail2;
  Complex z1, z2;
  
#ifdef _USE_MPA
  field2cx (field1, l1, &tail1, argl->iscale, &argl->nf1, &z1);
  field2cx (field2, l2, &tail2, argl->iscale, &argl->nf2, &z2);
#else
  z1 = field2cx (field1, l1, &tail1);
  z2 = field2cx (field2, l2, &tail2);
#endif

#ifdef __DEBUG__
  fprintf (stderr, "l1 = %zu, tail1 - field1 = %zu\n", l1, tail1 - field1);
  fprintf (stderr, "l2 = %zu, tail2 - field2 = %zu\n", l2, tail2 - field2);
#endif  

  if (tail1 - field1 == l1 && tail2 - field2 == l2)
    {
      /*
	This second test manages the options -P
	and -N . If neither of them has been set,
	then the condition is always TRUE.
      */
      if ( (smart_cmp (&z2, &z1, argl->flag)) )
	{
	  int exit_code;

	  /* Numeric comparison */
#ifdef _USE_MPA
	  Complex w;
	  Real x1, x2;
	  int iscale = argl->iscale;

	  initC (&w);
	  initR (&x1);
	  initR (&x2);
	  Csub (z1, z2, &w, iscale);
	  Cabs (w, abserr, iscale);
#ifdef _MPA_DEBUG
	  fputs ("*** MPA Debug output\n", stderr);
	  fputs ("1st number= ( ", stderr);
	  debug_printno (z1.re, 20);
	  fputs (", ", stderr);
	  debug_printno (z1.im, 20);
	  fputs (" )\n", stderr);
	  fputs ("2nd number= ( ", stderr);
	  debug_printno (z2.re, 20);
	  fputs (", ", stderr);
	  debug_printno (z2.im, 20);
	  fputs (" )\n", stderr);
	  fputs ("abs. err= ", stderr);
	  debug_printno (*abserr, 20);
	  fputs ("\n***     ***     ***\n", stderr);
#endif
	  Cabs (z1, &x1, iscale);
	  Cabs (z2, &x2, iscale);
	  if ( cmp (&x1, &x2) > 0 )
	    x1 = copyR (x2);
	  if ( (is0(&x1)) )
	    *relerr = (is0(abserr)) ? copyR(Zero) : copyR(Inf);
	  else
	    divide (*abserr, x1, relerr, iscale);
	  delR (&x2);
	  delR (&x1);
	  delC (&w);
	  delC (&z2);
	  delC (&z1);
#else
	  Real x1, x2;

	  *abserr = Cabs (Csub (z1, z2));
	  x1 = Cabs (z1);
	  x2 = Cabs (z2);
	  if ( cmp (&x1, &x2) > 0 )
	    x1 = x2;
	  if ( (is0(&x1)) )
	    *relerr = (is0(abserr)) ? Zero : Inf;
	  else
	    *relerr = divide (*abserr, x1);
#endif /* _USE_MPA */

	  if (!(argl->optmask & _2_MASK))
	    exit_code = 
	      cmp (relerr, &argl->maxrelerr) > 0 && cmp (abserr, &argl->maxabserr) > 0 ? 2:0;
	  else
	    exit_code =
	      cmp (relerr, &argl->maxrelerr) > 0 || cmp (abserr, &argl->maxabserr) > 0 ? 2:0;
	  if ((argl->optmask & _SS_MASK) && (exit_code))
	    {
	      int test;

	      if ((test = cmp (abserr, &argl->Labserr)) > 0)
		{
#ifdef _USE_MPA
		  argl->Labserr = copyR (*abserr);
		  argl->Crelerr = copyR (*relerr);
#else
		  argl->Labserr = *abserr;
		  argl->Crelerr = *relerr;
#endif /* _USE_MPA */
		}
	      else if (test == 0 && cmp (relerr, &argl->Crelerr) > 0)
		{
#ifdef _USE_MPA
		  argl->Crelerr = copyR (*relerr);
#else
		  argl->Crelerr = *relerr;
#endif /* _USE_MPA */		  
		}
	      if ((test = cmp (relerr, &argl->Lrelerr)) > 0)
		{
#ifdef _USE_MPA
		  argl->Lrelerr = copyR (*relerr);
		  argl->Cabserr = copyR (*abserr);
#else
		  argl->Lrelerr = *relerr;
		  argl->Cabserr = *abserr;
#endif /* _USE_MPA */		  
		}
	      else if (test == 0 && cmp (abserr, &argl->Cabserr) > 0)
		{
#ifdef _USE_MPA
		  argl->Cabserr = copyR (*abserr);
#else
		  argl->Cabserr = *abserr;
#endif /* _USE_MPA */		  		  
		}
	    }
	  return exit_code;
	}
      else
	return 0;
    }
  else
    {
#ifdef _USE_MPA
      delC (&z2);
      delC (&z1);
#endif
      /* Byte by byte comparison */
      if ((argl->optmask & _SI_MASK))
	return (l1 != l2 || strNcasecmp (field1, field2, l1) != 0 ? 1 : 0);
      else
	return (l1 != l2 || strNcmp (field1, field2, l1) != 0 ? 1 : 0);
    }
}

int cmp_lines (const char* line1, const char* line2, 
	       unsigned long lineno, argslist* argl)
{
  int quiet_mode = argl->optmask & (_B_MASK | _Q_MASK);
  char* ifs;

  if (!argl->ifs)
    ifs = IFS;
  else
    ifs = argl->ifs;
  if (!line1 && !line2)
    return 0;
  else if (!line1)
    {
      if (!quiet_mode)
	{
	  printf ("\n##%-7lu\n", lineno);
	  fputs ("<==\n==> ", stdout);
	  puts (line2);
	}
#ifdef __DEBUG__
      fprintf (stderr, _("\nLine %lu:\n"), lineno);
      fputs ("<\n", stderr);
      fprintf (stderr, "> %s\n", line2);
#endif /* __DEBUG__ */
      return 1;
    }
  else if (!line2)
    {
      if (!quiet_mode)
	{
	  printf ("\n##%-7lu\n", lineno);
	  fputs ("<== ", stdout);
	  puts (line1);
	  puts ("==>");
	}
#ifdef __DEBUG__
      fprintf (stderr, _("\nLine %lu:\n"), lineno);
      fprintf (stderr, "< %s\n", line1);
      fputs (">\n", stderr);
#endif /* __DEBUG__ */
      return 1;
    }
  else
    {
      const char *field1, *field2;
      size_t l1, l2;
      unsigned long fieldno;
      Real abserr, relerr;
      int rv, lines_differ = 0, _1sttime = 1;


#ifdef __DEBUG__
      fprintf (stderr, _("\nLine %lu:\n"), lineno);
      fprintf (stderr, "< %s\n", line1);
      fprintf (stderr, "> %s\n", line2);
#endif /* __DEBUG__ */

#ifdef _USE_MPA
      initR (&abserr);
      initR (&relerr);
#endif
      field1 = line1 + strspn (line1, ifs);
      field2 = line2 + strspn (line2, ifs);
      for (fieldno = 0;
	   *field1 != '\0' && *field2 != '\0';
	   field1 += l1, field2 += l2, fieldno++)
	{
	  l1 = strcspn (field1, ifs);
	  l2 = strcspn (field2, ifs);
#ifdef __DEBUG__
	  fprintf (stderr, "[%lu]< \'", fieldno + 1);
	  Eprint_head (field1, l1);
	  fputs ("\'   > \'", stderr);
	  Eprint_head (field2, l2);
	  fputs ("\'\n", stderr); 
#endif /* __DEBUG__ */

	  /*
	    'field1' and 'field2' are compared if and only if
	    the bit whose position in the bitmask 'argl->fieldmask'
	    is given by 'fieldno' is on (== 1). If not changed
	    by the command line option '-F', 'argl->fieldmask'
	    has all its bits on.
	  */
          if( (argl->fieldmask[fieldno >> 3] & 0x80 >> (fieldno & 0x7)) )
	    {
	      rv = cmp_fields (field1, field2, l1, l2, argl, &abserr, &relerr);
	      if ( rv >= 1 )
		{
		  if (!(quiet_mode & _Q_MASK) && 
		      !(rv == 1 && argl->optmask & _SE_MASK) &&
		      !(rv == 2 && argl->optmask & _SD_MASK))
		    {
		      if ((_1sttime))
			{
			  puts (LINE_SEP);
			  _1sttime = 0;
			}
		      printf ("##%-7lu #:%-3lu ", lineno, fieldno +1);
		      fputs ("<== ", stdout);
		      print_head (field1, l1);
		      putchar ('\n');
		      print_ws (6+10);
		      /* 6 = size("## #: "), 10 = 7+3 */
		      fputs ("==> ", stdout);
		      print_head (field2, l2);
		      putchar ('\n');
		      if ( rv == 2 )
			{
			  fputs (_("@ Absolute error = "), stdout);
			  printno (abserr, DEF_LIM);
			  fputs (_(", Relative error = "), stdout);
			  printno (relerr, DEF_LIM); 
			  putchar ('\n');
			}
		      else
			{
			  putchar ('@');
			  print_ws (53);
			  puts ("@@");
			}
		    }
		  lines_differ = 1;
		}
	    }
	  l1 += strspn (field1 + l1, ifs);
	  l2 += strspn (field2 + l2, ifs);
	}
#ifdef _USE_MPA
      delR (&abserr);
      delR (&relerr);
#endif
      if (*field1 != '\0')
	{
	  if (!quiet_mode)
	    {
	      printf (_("\nLine %lu in file \"%s\" is shorter:\n"), 
		      lineno, argl->file2);
	      fputs ("<== ", stdout);
	      fputs (field1, stdout);
	      puts ("==>");
	    }
#ifdef __DEBUG__
	  fprintf (stderr, "[%lu]< \'", fieldno+1);
	  Eprint_nonl (field1);
	  fputs ("\'   > \'\'\n", stderr);
#endif /* __DEBUG__ */
	  return 1;
	}
      else if (*field2 != '\0')
	{
	  if (!quiet_mode)
	    {
	      printf (_("\nLine %lu in file \"%s\" is shorter:\n"), 
		      lineno, argl->file1);
	      fputs ("<==\n==> ", stdout);
	      fputs (field2, stdout);
	    }
#ifdef __DEBUG__
	  fprintf (stderr, "[%lu]< \'\'   > \'", fieldno+1);
	  Eprint_nonl (field2);
	  fputs ("\'\n", stderr); 
#endif /* __DEBUG__ */
	  return 1;
	}
      else
	return lines_differ;
    }
}

int cmp_files (FILE* pf1, FILE* pf2, argslist* argl)
{
  char *line1, *line2;
  int err1, err2, files_differ = 0;
  unsigned long lineno = 1;

  do
    {
      line1 = read_line (pf1, &err1);
      line2 = read_line (pf2, &err2);
      /*
	line1 and line2 are compared if and only if
	their position falls within the range specified by
	argl->firstln and argl->lastln. If not explicitly
	set on the command line, then argl->firstln == 1
	and argl->lastln == INTEGER_MAX (see numdiff.h)
      */
      if ( argl->firstln <= lineno && lineno <= argl->lastln ) 
	{
	  if ( (cmp_lines (line1, line2, lineno, argl)) )
	    files_differ = 1;
	}
      if ((line1))
	free ((void*)line1);
      if ((line2))
	free ((void*)line2);
      lineno++;
    } while (err1 == OK && err2 == OK);
  /* When we arrive here either err1 or err2 */
  /* is different from OK.                   */
  fflush (stdout);
  if (err1 == OK)
    {
      switch (err2)
	{
	case READING_ERROR:
	  fprintf (stderr, _("\n***  Error in reading from file \"%s\"\n"), argl->file2);
	  return READING_ERROR;
	case OUT_OF_MEMORY:
	  fprintf (stderr, _("\n***  Out of memory while reading from file \"%s\"\n"), argl->file2);
	  return OUT_OF_MEMORY;
	default:
	  fprintf (stderr, _("\n***  End of file \"%s\" reached\n"), argl->file2);
	  fprintf (stderr, _("     Likely the files \"%s\" and \"%s\" do not have the same number of lines !\n"),
		   argl->file1, argl->file2);
	}
      return files_differ;
    }
  else if (err1 == EOF_REACHED)
    {
      switch (err2)
	{
	case READING_ERROR:
	  fprintf (stderr, _("\n***  Error in reading from file \"%s\"\n"), argl->file2);
	  return READING_ERROR;
	case OUT_OF_MEMORY:
	  fprintf (stderr, _("\n***  Out of memory while reading from file \"%s\"\n"), argl->file2);
	  return OUT_OF_MEMORY;
	case OK:
	  fprintf (stderr, _("\n***  End of file \"%s\" reached\n"), argl->file1);
	  fprintf (stderr, _("     Likely the files \"%s\" and \"%s\" do not have the same number of lines !\n"),
		   argl->file1, argl->file2);
	}
      return files_differ;
    }
  else if (err1 == READING_ERROR)
    {
      fprintf (stderr, _("\n***  Error in reading from file \"%s\"\n"), argl->file1);
      return READING_ERROR;
    }
  else /* err1 == OUT_OF_MEMORY */
    {
      fprintf (stderr, _("\n***  Out of memory while reading from file \"%s\"\n"), argl->file1);
      return OUT_OF_MEMORY;
    }
}

