/*
    Copyright (C) 2005,2006,2007,2008,2009,2010,2011,2012,2013,2014,2015,2016  Ivano Primi  <ivprimi@libero.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include<stdio.h>
#include<stdlib.h>
#include<math.h>

#define START   1
#define END     1000
#define NCYCLES 10
#define EPSILON 1.0e-6
#define WARNING_LEVEL 2.5

typedef double (*Mathfunc) (double);

double Newton (Mathfunc pf, double y, double x_0)
{
  double fdev, ldev, rdev, h = EPSILON;

  rdev = ((*pf)(x_0+h) - (*pf)(x_0))/h;
  ldev = ((*pf)(x_0) - (*pf)(x_0-h))/h;
#ifdef _DEBUG_
  printf ("x_0 = %g, rdev = %g, ldev = %g\n", x_0, rdev, ldev);
#endif
  if (fabs(rdev-ldev) > WARNING_LEVEL)
    {
      fprintf (stderr,
	       "*** Newton(): Maybe the given function is not derivable in x = %g\n", 
	       x_0);
      exit (EXIT_FAILURE);
    }
  fdev = 0.5 * (ldev + rdev);
  if (fabs (fdev) < EPSILON)
    {
      fprintf (stderr,
	       "*** Newton(): We have met a critical point in x = %g\n", 
	       x_0);
      exit (EXIT_FAILURE);
    }
  else
    return x_0 - ((*pf)(x_0)-y)/fdev;
}

double sqr (double x)
{
  return x*x;
}

double pow_2_5 (double x)
{
  return x*x*sqrt(x);
}

double cube (double x)
{
  return x*x*x;
}

double pow_3_5 (double x)
{
  return x*x*x*sqrt(x);
}

double func (double x)
{
  return 2*x-sin(x);
}

int main (int argc, const char** argv)
{
  FILE *fp1, *fp2;
  int n;

  if (argc != 3)
    {
      fputs ("*** Usage: newton file1 file2\n", stderr);
      return 1;
    }
  else if ( !(fp1 = fopen(argv[1], "w")) )
    {
      fprintf (stderr, "*** Cannot open file \"%s\"\n", argv[1]);
      return 1;
    }
  else if ( !(fp2 = fopen(argv[2], "w")) )
    {
      fprintf (stderr, "*** Cannot open file \"%s\"\n", argv[2]);
      return 1;
    }
  else
    {
      Mathfunc f[5] = {sqr, pow_2_5, cube, pow_3_5, func};
      double x[5];

      fprintf (fp1, 
	       "         N   \t\t sqrt(N) \t\t 5throot(N^2) \t\t cbrt(N) \t\t 7throot(N^2) \t\t\t ~N\n");
      fprintf (fp2, 
	       "         N   \t\t sqrt(N) \t\t 5throot(N^2) \t\t cbrt(N) \t\t 7throot(N^2) \t\t\t ~N\n");
      for (n=START; n <= END; n++)
	{
	  int k;

	  fprintf (fp1, "%10d \t\t %10.6f \t\t %10.6f \t\t %10.6f \t\t %10.6f \t\t %10d\n", n, sqrt(n), pow(n, 0.4), cbrt(n), pow(n, 2.0/7), n);
	  x[0] = 10;
	  x[1] = 8;
	  x[2] = 5;
	  x[3] = 1;
	  x[4] = n;
	  for (k = 1; k <= NCYCLES; k++)
	    {
	      x[0] = Newton (f[0], n, x[0]);
	      x[1] = Newton (f[1], n, x[1]);
	      x[2] = Newton (f[2], n, x[2]);
	      x[3] = Newton (f[3], n, x[3]);
	      x[4] = Newton (f[4], n, x[4]);
	    }
	  fprintf (fp2, "%10d \t\t %10.6f \t\t %10.6f \t\t %10.6f \t\t %10.6f \t\t %11.6f \t\t %10.6f\n", 
		   n, x[0], x[1], x[2], x[3], func(x[4]), x[4]);
	}
      fclose(fp1);
      fclose(fp2);
      return 0;
    }
}

