/*             This file is part of the New World OS project
--                Copyright (C) 2005, 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: computer.c,v $
-- Revision 1.17  2006/11/30 00:51:12  jsedwards
-- Changed to use strlcpy instead of strcpy to stop warnings from linker.
--
-- Revision 1.16  2006/11/11 12:01:01  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.15  2006/10/26 01:51:26  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.14.2.3  2006/10/03 12:53:07  jsedwards
-- Changed so that instead of calling a separate routine after initialize to
-- change the already opened storage, you call it now with a type of storage
-- parameter and a path to the storage.  The problem with the other way was
-- if you tried reading a compressed file on another machine it tried to open
-- the default file which didn't exist.
--
-- Revision 1.14.2.2  2006/09/29 12:09:24  jsedwards
-- Added "list_persons" command.
--
-- Revision 1.14.2.1  2006/09/02 15:09:41  jsedwards
-- Add call to terminate objectify so reference list cache gets flushed back
-- to disk, etc.
--
-- Revision 1.14  2006/01/12 03:42:37  jsedwards
-- Changed the "what is number" routine to accept "phone number", "mobile
-- number", and "cell number" as well.
--
-- Revision 1.13  2006/01/12 03:35:22  jsedwards
-- Added help for "what is so and so's number".
--
-- Revision 1.12  2006/01/10 13:57:20  jsedwards
-- Added a phone number query command.
--
-- Revision 1.11  2006/01/09 03:27:13  jsedwards
-- Changed "describe" command to deal with persons or states.
--
-- Revision 1.10  2006/01/08 15:25:26  jsedwards
-- Added some more information to the help print out.
--
-- Revision 1.9  2006/01/08 15:12:46  jsedwards
-- Added "help" information.
--
-- Revision 1.8  2006/01/04 19:00:08  jsedwards
-- Made a separate subroutine to execute commands, instead of doing it in main
-- routine.  Changed main routine to check for parameters passed on command
-- line and reassemble them into a command string, which is passed to the
-- execute command subroutine.  This way a command can be passed to it via
-- the command line.
--
-- Revision 1.7  2006/01/04 15:15:41  jsedwards
-- Added "add city" and "list cities in state" commands.
--
-- Revision 1.6  2006/01/02 01:57:52  jsedwards
-- Fixed bug in get_input routine that was using a fixed size for the buffer
-- instead of the size parameter.
--
-- Revision 1.5  2006/01/01 21:19:30  jsedwards
-- Removed debugging print statement.
--
-- Revision 1.4  2006/01/01 19:50:50  jsedwards
-- Moved all of the tests for quitting (stop, exit, etc.) into a function
-- (is quit command) in objectify.c.
-- Added code to add mobile phone number to database.
--
-- Revision 1.3  2006/01/01 00:45:42  jsedwards
-- Added a command loop where it asks what to do and then tries to interpret
-- your request.
--
-- Revision 1.2  2005/12/31 19:03:33  jsedwards
-- Moved code to create a person object to the "person.c" file.
--
-- Revision 1.1  2005/12/31 14:49:08  jsedwards
-- Copied "create_person.c" into "computer.c" and "person.c" and then removed
-- "create_person.c".
--
-- Revision 1.15  2005/12/30 14:13:39  jsedwards
-- Removed Social Security Number stuff, not ready yet.
--
-- Revision 1.14  2005/12/30 05:12:57  jsedwards
-- Tweaked the birth date stuff and added social security number.
--
-- Revision 1.13  2005/12/29 18:40:38  jsedwards
-- Added code for the new long key and encryption stuff.
--
-- Revision 1.12  2005/12/27 19:52:21  jsedwards
-- Added code to generate the root object from a password.  Changed to find
-- class definition instead of hardcoded file name and to generate a random
-- person object id instead of generating it based on contents.
--
-- Revision 1.11  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.10  2005/12/21 23:28:15  jsedwards
-- Fixed so that it adds the references to the name and date objects.
--
-- Revision 1.9  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.8  2005/12/05 19:06:31  jsedwards
-- Moved calculation of header checksum down just before writing object to
-- disk to ensure no changes were made after the checksum was calculated.
-- Also added the "nickname" field.
--
-- Revision 1.7  2005/12/04 14:14:46  jsedwards
-- Changed to actually get input and create a person (finally)!
--
-- Revision 1.6  2005/12/04 04:13:02  jsedwards
-- Added 'nwos' prefix to create_xxxx function names and eliminated the
-- 'referring' object parameter from all of them.
--
-- Revision 1.5  2005/12/04 00:35:49  jsedwards
-- Changed to just create a date.
--
-- Revision 1.4  2005/12/04 00:23:46  jsedwards
-- Removed create year, date, and name stuff and put it in the date.c and
-- the name.c files.
--
-- Revision 1.3  2005/12/02 20:33:11  jsedwards
-- Changed so that name objects have "NAME" in the filename and added new
-- parameter to the create_reference_list calls.
--
-- Revision 1.2  2005/12/02 19:30:32  jsedwards
-- Changed to create Year object filename from the year itself.
--
-- Revision 1.1  2005/12/02 13:05:17  jsedwards
-- Initial version that only creates a name and a year.
--
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify.h"
#include "objectify_private.h"
#include "crc32.h"

#define MAX_NAME_SIZE 64
#define MAX_DATE_SIZE 16


static void get_input(char* descr, char* buffer, size_t size)
{
    char *ptr;

    while (1)
    {
	printf("%s: ", descr);
	fflush(stdout);
	fgets(buffer, size, stdin);
	ptr = strchr(buffer, '\n');
	if (ptr != NULL)
	{
	    *ptr = '\0';
	    break;
	}
	do { fgets(buffer, size, stdin); } while (strchr(buffer, '\n') == NULL);
	printf("input too long - try again!\n");
    }
}


static bool is_what_is_number(char* command)
{
    static char* what_is = "what is ";
    char* ptr;
    bool result = false;

    if (strncasecmp(command, what_is, strlen(what_is)) == 0)
    {
	ptr = strchr(command, '\'');
	if (ptr != NULL)
	{
	    ptr++;
	    if (*ptr == 's') ptr++;
	    result = (strcasecmp(ptr, " number") == 0 || strcasecmp(ptr, " phone number") == 0
		      || strcasecmp(ptr, " cell number") == 0 || strcasecmp(ptr, " mobile number") == 0);
	}
    }

    return result;
}


void execute_command(char *command)
{
    char* ptr1;
    char* ptr2;
    char* name;
    ObjRef ref;

    if (strcasecmp(command, "add person") == 0)
    {
	nwos_add_person();
    }
    else if (strcasecmp(command, "list persons") == 0)
    {
	nwos_list_persons();
    }
    else if (strncasecmp(command, "describe ", strlen("describe ")) == 0 && strlen(command) > strlen("describe "))
    {
	/* copy just the name part into name buffer */
	name = malloc(strlen(command));   /* allocate enough room for entire command even though we won't use it all */
	assert(strlcpy(name, command + strlen("describe "), strlen(command)) < strlen(command));

	if (!nwos_find_name(name, &ref))
	{
	    printf("I don't know of anyone or anything by the name of: %s\n", name);
	}
	else
	{
	    if (nwos_any_persons_named(name))
	    {
		nwos_describe_person(name);
	    }

	    if (nwos_any_states_named(name))
	    {
		nwos_describe_state(name);
	    }
	}
    }
    else if (strncasecmp(command, "list cities in ", strlen("list cities in ")) == 0 && strlen(command) > strlen("list cities in "))
    {
	nwos_list_cities_in_state(command + strlen("list cities in "));
    }
    else if (strcasecmp(command, "add city") == 0)
    {
	ObjRef  city_ref;
	nwos_add_city(&city_ref);
    }
    else if (strstr(command, " mobile phone is ") != NULL)
    {
	ptr1 = strstr(command, " mobile phone is ");
	ptr2 = ptr1 + strlen(" mobile phone is ");      /* point to the start of the phone number */
	if (ptr1[-2] == '\'' && ptr1[-1] == 's')
	{
	    ptr1 -= 2;
	}
	else if (ptr1[-1] == '\'')
	{
	    ptr1 -= 1;
	}
	*ptr1 = '\0';    /* terminate the name here */

	nwos_add_mobile_phone(command, ptr2);
    }
    else if (is_what_is_number(command))
    {
	*strchr(command, '\'') = '\0';
	nwos_find_phone_for_person(command + strlen("what is "));
    }
    else
    {
	printf("I'm sorry I don't understand\n");
    }
}


int main(int argc, char* argv[])
{
    ObjRef root_object_ref;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    char command[80];
    int i;

    printf("\n");

    nwos_get_key_from_password(big_key, sizeof(big_key));

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_initialize_objectify(bf_key, linear, serial, DEFAULT_TYPE, DEFAULT_FILE);

    nwos_set_root_object(&root_object_ref);

    if (argc == 1)   /* nothing on the command line */
    {
	while (1)
	{
	    printf("\n");

	    get_input("How may I help you?", command, sizeof(command));

	    printf("\n");

	    if (nwos_is_quit_command(command)) break;

	    if (strcasecmp(command, "help") == 0 || strcasecmp(command, "commands") == 0 || strcasecmp(command, "?") == 0)
	    {
		printf("The only commands that are available in this version are (without quotes):\n\n");

		printf("   \"add person\" - adds a new person to the database.\n");
		printf("   \"list persons\" - list persons to the database.\n");
		printf("   \"add city\" - adds a new city to the database (not of much use in this version).\n");
		printf("   \"describe <name>\" - describes any person(s) with the name <name> in the database.\n");
		printf("   \"list cities in <state>\" - lists the cities in the database for state <state>.\n");
		printf("   \"<name>'s mobile phone is <number>\" - adds the phone number <number> to the person <name>.\n");
		printf("   \"what is <name>'s number\" - prints out the phone number for the person <name>.\n");

		printf("\n   \"commands\" or \"help\" - print this information.\n");
		printf("   \"done\", \"exit\", \"finish\", \"quit\", \"stop\", or \"terminate\" - exit the program.\n");

		printf("\nPlease note that this version is picky about the spacing, you need to have exactly one space\n");
		printf("as shown in the commands above.\n");
	    }
	    else
	    {
		execute_command(command);
	    }
	}

	printf("Goodbye!\n\n");
    }
    else
    {
	command[0] = '\0';
	for (i = 1; i < argc; i++)
	{
	    if (strlen(command) + strlen(argv[i]) + 1 > sizeof(command))
	    {
		fprintf(stderr, "The word %s exceeds size of input buffer\n", argv[i]);
		exit(1);
	    }
	    if (command[0] != '\0')     /* there are already words in command so add a space in front of this word */
	    {
		strncat(command, " ", sizeof(command));
	    }
	    strncat(command, argv[i], sizeof(command));
	}

	execute_command(command);
    }

    nwos_terminate_objectify();

    return 0;
}

