/*             This file is part of the New World OS project
--                Copyright (C) 2005, 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: date.c,v $
-- Revision 1.18  2006/11/11 12:01:03  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.17  2006/10/26 01:51:26  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.16.2.3  2006/10/25 12:22:27  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.16.2.2  2006/09/10 20:43:30  jsedwards
-- Changed to use sizeof instead of hard coded 4 (not sure 4 was correct).
--
-- Revision 1.16.2.1  2006/09/02 01:11:19  jsedwards
-- Changed "nwos_object_size" to "nwos_reference_list_size" and added the
-- object reference to "nwos_fill_in_common_header" so it can put the "id"
-- in the header now.
--
-- Revision 1.16  2006/01/09 03:19:12  jsedwards
-- Changed the upgrade routine to search through the month class reference
-- list itself, to find all of the months instead of using the "get month ref"
-- routine.  The "get month ref" routine reads the month objects to find the
-- right one and since they are of the wrong size the checksum fails.
--
-- Revision 1.15  2006/01/06 14:08:52  jsedwards
-- Added code to upgrade a 0004 month object to a 0005 month object.
--
-- Revision 1.14  2006/01/01 21:49:01  jsedwards
-- Moved date, phone, us_state, and word class creations out of "big bang"
-- and into the respective files.
--
-- Revision 1.13  2006/01/01 00:41:23  jsedwards
-- Removed count from month object.  Since it was always 2, why have it?
-- Added date, month, and year to string routines.
--
-- Revision 1.12  2005/12/30 14:07:03  jsedwards
-- Added is_leap_year function.
--
-- Revision 1.11  2005/12/29 17:50:53  jsedwards
-- Commented out printf debugging statements, that aren't useful now.
--
-- Revision 1.10  2005/12/29 17:47:58  jsedwards
-- Changed to make Year object variable sized (although in reality at this
-- time it is fixed at 4 digits in the year).  This allows the write object
-- to disk routine to fill in the remainder of the 512 bytes with random data.
--
-- Revision 1.9  2005/12/27 18:32:36  jsedwards
-- Changed to look up class definition instead of using a fixed file name.
-- Also changed so that object id is random instead of based upon contents.
--
-- Revision 1.8  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.7  2005/12/21 03:52:27  jsedwards
-- Moved US state objects creation from 'create_class_def.c' into 'date.c'.
--
-- Revision 1.6  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.5  2005/12/05 05:22:46  jsedwards
-- Change to call read_reference_list_from_disk for reference lists instead
-- of read_object_from_disk.  Also moved calculating checksums down to just
-- before writing object, so that no changes occur after computing them.
--
-- Revision 1.4  2005/12/04 14:09:10  jsedwards
-- Added checks when creating a date that it is a valid date.
--
-- Revision 1.3  2005/12/04 04:13:02  jsedwards
-- Added 'nwos' prefix to create_xxxx function names and eliminated the
-- 'referring' object parameter from all of them.
--
-- Revision 1.2  2005/12/04 00:37:46  jsedwards
-- Added get_month function and made create_date function work.
--
-- Revision 1.1  2005/12/03 23:22:09  jsedwards
-- Initial version extracted from create_person.c.
--
*/


#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"
#include "objectify_private.h"


/*---------------------------------*/
/* Create classes related to dates */
/*---------------------------------*/

void nwos_setup_date()
{
    printf ("Creating Year class definition.\n");
    nwos_create_class_definition("YEAR");

    printf ("Creating Month class definition.\n");
    nwos_create_class_definition("MONTH");
    printf ("Creating 12 Month objects\n");
    nwos_create_all_twelve_months();

    printf ("Creating Date class definition.\n");
    nwos_create_class_definition("DATE");
}


typedef struct {         /* this class should inherit from thing, if we had inheritance */
    EveryObject header;
    ObjRef      definition;
    uint8       count;       /* always 2 */
    char        storage[2];  /* 1-12 - note names should be found from references for different languages */
    uint8       minimum_days;
    uint8       maximum_days;
} Old_C_struct_Month;


#if 0
/* this is needed to remove the "count" from the structure */
void nwos_upgrade_months_from_0004_to_0005()
{
    int i;
    EveryObject header;
    ObjRef month_class_ref;
    C_struct_Class_Definition class_def_obj;
    Old_C_struct_Month old_month_obj;
    C_struct_Month new_month_obj;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_months;


    assert(nwos_find_class_definition("MONTH", &month_class_ref));

    nwos_read_class_definition(&month_class_ref, &class_def_obj);

    ref_list_size = nwos_reference_list_size(&class_def_obj.header.object.references);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(&class_def_obj.header.object.references, ref_list, ref_list_size);

    num_months = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    assert(num_months == 12);

    for (i = 0; i < num_months; i++)
    {
	nwos_read_object_headers_from_disk(&ref_list->references[i], &header);

	if (memcmp(header.common.version, "0004", 4) != 0)
	{
	    printf("    Month %08lx is not version 0004, skipping it!\n", nwos_ref_to_word(&ref_list->references[i]));
	    continue;
	}

	nwos_read_object_from_disk(&ref_list->references[i], &old_month_obj, sizeof(old_month_obj));

	memcpy(&new_month_obj.header, &old_month_obj.header, sizeof(new_month_obj.header));
	memcpy(new_month_obj.header.common.version, HEADER_VERSION, sizeof(new_month_obj.header.common.version));
	new_month_obj.storage[0] = old_month_obj.storage[0];
	new_month_obj.storage[1] = old_month_obj.storage[1];
	new_month_obj.minimum_days = old_month_obj.minimum_days;
	new_month_obj.maximum_days = old_month_obj.maximum_days;

	nwos_crc32_calculate((uint8*) &new_month_obj.definition, sizeof(new_month_obj) - sizeof(EveryObject), new_month_obj.header.common.data_chksum);

	nwos_remove_object(&ref_list->references[i]);   /* remove the old object from disk */

	nwos_write_object_to_disk(&ref_list->references[i], &new_month_obj, sizeof(new_month_obj));
    }

    free(ref_list);
    ref_list = NULL;

}
#endif

ObjCreateResult nwos_create_year(uint16 year, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[sizeof(C_struct_Year) + 4];    /* year object is now really variable sized so do this kludge */
    C_struct_Year* ptr_year_obj = (C_struct_Year*)kludge;
    ObjRef class_ref;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_years;
    int i;
    char year_ascii[4];
    ObjCreateResult result = CREATED_NEW;

    assert(1000 <= year && year <= 9999);   /* for now limit it to 4 digit years */

    year_ascii[0] = '0' +  (year / 1000);
    year_ascii[1] = '0' + ((year / 100) % 10);
    year_ascii[2] = '0' + ((year / 10)  % 10);
    year_ascii[3] = '0' +  (year        % 10);

    /* first find out if we already have this year */

    assert(nwos_find_class_definition("YEAR", &class_ref));

    nwos_read_class_definition(&class_ref, &class_def_obj);

    ref_list_size = nwos_reference_list_size(&class_def_obj.header.object.references);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(&class_def_obj.header.object.references, ref_list, ref_list_size);

    num_years = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    /* printf("num_years: %d\n", num_years); */

    for (i = 0; i < num_years; i++)
    {
	nwos_read_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge));

	/* remember the ptr_year_obj points to the kludge buffer */

	if (ptr_year_obj->count == 4 && memcmp(ptr_year_obj->storage, year_ascii, 4) == 0)   /* found a match */
	{
	    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
	    result = FOUND_EXISTING;
	    break;
	}
    }

    free(ref_list);
    ref_list = NULL;

    if (result != FOUND_EXISTING)   /* didn't find it */
    {
	memset(kludge, 0, sizeof(kludge));  /* zero it out */

	/* remember the ptr_year_obj points to the kludge buffer */

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&ptr_year_obj->header.common, ref, &class_ref);

	ptr_year_obj->count = 4;
	memcpy(ptr_year_obj->storage, year_ascii, 4);

	nwos_create_reference_list(ref, &ptr_year_obj->header.object.references);

	nwos_crc32_calculate((uint8*) &ptr_year_obj->header.object, sizeof(ObjectHeader), ptr_year_obj->header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &ptr_year_obj->count, sizeof(kludge) - sizeof(EveryObject), ptr_year_obj->header.common.data_chksum);

	nwos_write_object_to_disk(ref, kludge, sizeof(kludge));

        nwos_add_to_reference_list(ref, &class_def_obj.header.object.references);
    }

    return result;
}


static void create_month(uint8 month, uint8 min, uint8 max, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Month month_obj;
    ObjRef class_ref;
    char month_ascii[2];

    assert(1 <= month && month <= 12);

    month_ascii[0] = '0' + (month / 10);
    month_ascii[1] = '0' + (month % 10);

    /* get the class definition object so we have the reference list */

    assert(nwos_find_class_definition("MONTH", &class_ref));

    nwos_read_class_definition(&class_ref, &class_def_obj);

    memset(&month_obj, 0, sizeof(month_obj));  /* zero it out */

    nwos_generate_new_id(ref);

    nwos_fill_in_common_header(&month_obj.header.common, ref, &class_ref);

    memcpy(month_obj.storage, month_ascii, sizeof(month_obj.storage));
    month_obj.minimum_days = min;
    month_obj.maximum_days = max;

    nwos_create_reference_list(ref, &month_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &month_obj.header.object, sizeof(ObjectHeader), month_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &month_obj.definition, sizeof(month_obj) - sizeof(EveryObject), month_obj.header.common.data_chksum);

    nwos_write_object_to_disk(ref, &month_obj, sizeof(month_obj));

    nwos_add_to_reference_list(ref, &class_def_obj.header.object.references);
}


void nwos_create_all_twelve_months()
{
    ObjRef reference;
    ObjRef english_lang_ref;
    ObjRef month_ref;

    nwos_find_language("English", &english_lang_ref);

    create_month(1,  31, 31, &reference);
    nwos_create_word("January", &reference, &english_lang_ref, &month_ref);

    create_month(2,  28, 29, &reference);
    nwos_create_word("February", &reference, &english_lang_ref, &month_ref);

    create_month(3,  31, 31, &reference);
    nwos_create_word("March", &reference, &english_lang_ref, &month_ref);

    create_month(4,  30, 30, &reference);
    nwos_create_word("April", &reference, &english_lang_ref, &month_ref);

    create_month(5,  31, 31, &reference);
    nwos_create_word("May", &reference, &english_lang_ref, &month_ref);

    create_month(6,  30, 30, &reference);
    nwos_create_word("June", &reference, &english_lang_ref, &month_ref);

    create_month(7,  31, 31, &reference);
    nwos_create_word("July", &reference, &english_lang_ref, &month_ref);

    create_month(8,  31, 31, &reference);
    nwos_create_word("August", &reference, &english_lang_ref, &month_ref);

    create_month(9,  30, 30, &reference);
    nwos_create_word("September", &reference, &english_lang_ref, &month_ref);

    create_month(10, 31, 31, &reference);
    nwos_create_word("October", &reference, &english_lang_ref, &month_ref);

    create_month(11, 30, 30, &reference);
    nwos_create_word("November", &reference, &english_lang_ref, &month_ref);

    create_month(12, 31, 31, &reference);
    nwos_create_word("December", &reference, &english_lang_ref, &month_ref);
}


/* The months should always exist (all twelve of them) */

void nwos_get_month_ref(uint8 month, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Month month_obj;
    ObjRef reference;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_months;
    int i;
    char month_ascii[2];
    ObjCreateResult result = CREATED_NEW;

    assert(1 <= month && month <= 12);

    month_ascii[0] = '0' + (month / 10);
    month_ascii[1] = '0' + (month % 10);

    assert(nwos_find_class_definition("MONTH", &reference));

    nwos_read_class_definition(&reference, &class_def_obj);

    ref_list_size = nwos_reference_list_size(&class_def_obj.header.object.references);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(&class_def_obj.header.object.references, ref_list, ref_list_size);

    num_months = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    /* printf("num_months: %d\n", num_months); */

    for (i = 0; i < num_months; i++)
    {
	nwos_read_object_from_disk(&ref_list->references[i], &month_obj, sizeof(month_obj));

	if (memcmp(month_obj.storage, month_ascii, 2) == 0)   /* found a match */
	{
	    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
	    result = FOUND_EXISTING;
	    break;
	}
    }

    free(ref_list);
    ref_list = NULL;

    assert(result == FOUND_EXISTING);   /* we had to have found it */
}


bool nwos_is_leap_year(int year)
{
    return ((year % 4 == 0) && (year % 100 != 0)) || (year % 400 == 0);
}
    


ObjCreateResult nwos_create_date(uint16 year, uint8 month, uint8 day, ObjRef* ref)
{
    C_struct_Date date_obj;
    EveryObject year_header;
    C_struct_Month month_obj;

    ObjRef date_class_ref;
    ObjRef year_ref;
    ObjRef month_ref;
    ObjRef object_class;

    ReferenceList* year_ref_list;
    ReferenceList* month_ref_list;
    size_t ref_list_size;
    int num_years;
    int num_months;
    int i;
    int j;
    uint8 days_per_month[12] = { 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

    ObjCreateResult year_result;
    ObjCreateResult result = CREATED_NEW;

    assert(1 <= month && month <= 12);
    assert(1 <= day && month <= days_per_month[month-1]);
    assert(1000 <= year && year <= 9999);
    if (month == 2 && day == 29)  /* more checking required */
    {
	assert(nwos_is_leap_year(year));
    }

    assert(nwos_find_class_definition("DATE", &date_class_ref));

    /* first get the month and year objects */

    year_result = nwos_create_year(year, &year_ref);
    nwos_get_month_ref(month, &month_ref);

    /* if both of them already existed see if our date already existed */

    if (year_result == FOUND_EXISTING)
    {
	/* get the year reference list */

	nwos_read_object_headers_from_disk(&year_ref, &year_header);

	ref_list_size = nwos_reference_list_size(&year_header.object.references);

	year_ref_list = malloc(ref_list_size);

	if (year_ref_list == NULL) 
	{
	    perror("reading year reference list");
	    exit(1);
	}

	nwos_read_reference_list_from_disk(&year_header.object.references, year_ref_list, ref_list_size);

	num_years = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

	/* get the month reference list */

	nwos_read_object_from_disk(&month_ref, &month_obj, sizeof(month_obj));

	ref_list_size = nwos_reference_list_size(&month_obj.header.object.references);

	month_ref_list = malloc(ref_list_size);

	if (month_ref_list == NULL) 
	{
	    perror("reading month reference list");
	    exit(1);
	}

	nwos_read_reference_list_from_disk(&month_obj.header.object.references, month_ref_list, ref_list_size);

	num_months = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

	/* printf("num_years: %d - num_months: %d\n", num_years, num_months); */

	for (i = 0; i < num_years; i++)
	{
	    for (j = 0; j < num_months; j++)
	    {
		if (is_same_object(&year_ref_list->references[i], &month_ref_list->references[j]))
		{
		    nwos_get_object_class(&year_ref_list->references[i], &object_class);   /* find out what kind of object it is */

		    if (is_same_object(&object_class, &date_class_ref))   /* it is a date object */
		    {
			nwos_read_object_from_disk(&year_ref_list->references[i], &date_obj, sizeof(date_obj));

			if (date_obj.day_of_month == day)
			{
			    memcpy(ref, &year_ref_list->references[i], sizeof(ObjRef));
			    result = FOUND_EXISTING;
			    break;
			}
		    }
		}
	    }
	}

	free(year_ref_list);
	year_ref_list = NULL;
	free(month_ref_list);
	month_ref_list = NULL;
    }

    if (result != FOUND_EXISTING)   /* didn't find it */
    {
	memset(&date_obj, 0, sizeof(date_obj));  /* zero it out */

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&date_obj.header.common, ref, &date_class_ref);

	memcpy(&date_obj.year, &year_ref, sizeof(ObjRef));
	nwos_add_to_references(ref, &year_ref);

	memcpy(&date_obj.month, &month_ref, sizeof(ObjRef));
	nwos_add_to_references(ref, &month_ref);

	date_obj.day_of_month = day;

	nwos_create_reference_list(ref, &date_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &date_obj.header.object, sizeof(ObjectHeader), date_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &date_obj.year, sizeof(date_obj) - sizeof(EveryObject), date_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &date_obj, sizeof(date_obj));

        nwos_add_to_references(ref, &date_class_ref);
    }

    return result;
}


bool nwos_year_to_string(ObjRef* ref, char* string, size_t size)
{
    uint8 kludge[sizeof(C_struct_Year) + 4];    /* year object is now really variable sized so do this kludge */
    C_struct_Year* ptr_year_obj = (C_struct_Year*)kludge;

    assert(size >= 5);

    nwos_read_object_from_disk(ref, kludge, sizeof(kludge));

    /* remember ptr_year_obj points to the kludge buffer */

    /* this is going to break if the size of a year isn't 4 digits!! */

    assert(ptr_year_obj->count = 4);

    memcpy(string, ptr_year_obj->storage, 4);

    string[4] = '\0';

    return true;
}


bool nwos_month_number_to_string(ObjRef* ref, char* string, size_t size)
{
    C_struct_Month month_obj;

    assert(size >= 3);

    nwos_read_object_from_disk(ref, &month_obj, sizeof(month_obj));

    memcpy(string, month_obj.storage, 2);

    string[2] = '\0';

    return true;
}


bool nwos_date_to_string(ObjRef* ref, char* string, size_t size)
{
    C_struct_Date date_obj;

    nwos_read_object_from_disk(ref, &date_obj, sizeof(date_obj));

    assert(size >= 11);

    nwos_month_number_to_string(&date_obj.month, string, size);

    assert(string[2] == '\0');

    string[2] = '/';
    string[3] = '0' + (date_obj.day_of_month / 10);
    string[4] = '0' + (date_obj.day_of_month % 10);
    string[5] = '/';
    
    nwos_year_to_string(&date_obj.year, &string[6], size);

    return true;
}



