/*             This file is part of the New World OS project
--                   Copyright (C) 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: expand_sparse.c,v $
-- Revision 1.4  2006/11/11 12:01:03  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.3  2006/11/02 11:49:28  jsedwards
-- Fixed all cases where 'z' was used as a format for 'off64_t' values because
-- the older compiler complains.
--
-- Revision 1.2  2006/10/26 01:51:26  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.9  2006/10/24 03:11:43  jsedwards
-- Change to use disk size stored on disk instead of defined constant.
--
-- Revision 1.1.2.8  2006/10/19 01:54:00  jsedwards
-- Fixed format specifiers for off64_t and uint32 which is now an unsigned
-- instead of unsigned long.
--
-- Revision 1.1.2.7  2006/10/11 13:19:04  jsedwards
-- Change so that first block is not written over the first block on the disk
-- because it over writes the disk block size (and version).
--
-- Revision 1.1.2.6  2006/10/10 07:32:36  jsedwards
-- Move bit map defines to objectify_private.h instead of spread all over.
--
-- Revision 1.1.2.5  2006/10/09 13:06:06  jsedwards
-- Change bit map size from 256 to 8192 (to improve speed, disk_usage program
-- took over an hour to run with 256 byte bit maps).
--
-- Revision 1.1.2.4  2006/10/08 13:46:38  jsedwards
-- Fix order of bits within the byte (calling 0 the most significant bit).
--
-- Revision 1.1.2.3  2006/10/07 23:35:58  jsedwards
-- Added code to fill in the block bit maps.
--
-- Revision 1.1.2.2  2006/10/07 22:30:18  jsedwards
-- Changed to convert the reference id to a block number so it can be expanded
-- into a disk drive or partition.
--
-- Revision 1.1.2.1  2006/09/17 13:53:44  jsedwards
-- Program to un-compress a non-sparse file into the regular sparse file.
--
*/


#define _LARGEFILE64_SOURCE

#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "objectify_private.h"


#define NUM_CHUNKS_CACHE 1024

struct {
    uint32 chunk;
    uint8* map;
} chunk_map[NUM_CHUNKS_CACHE];


static uint32 reserved_public_blocks;
static uint32 blocks_on_disk;
static uint32 private_blocks_on_disk;


static void print_usage(char *program)
{
    fprintf(stderr, "usage: %s input-file\n", program);
}


static inline uint32 hash_ref(unsigned char ref[4])
{
    uint32 result;

    result = ((uint32)(ref[0]) << 24) | ((uint32)(ref[1]) << 16) | 
             ((uint32)(ref[2]) << 8) | ((uint32)(ref[3]));

    return (result - NUM_PUBLIC_IDS) % private_blocks_on_disk + reserved_public_blocks;
}


void flush_maps_to_disk(int obj_file_desc)
{
    int i;
    off64_t block;
    char ref_str[64];
    int j;

    for (i = 0; i < NUM_CHUNKS_CACHE; i++)
    {
	if (chunk_map[i].chunk == 0) break;   /* it's Miller time! */

	block = chunk_map[i].chunk;

	printf("flush %08x", (uint32)block);

	for (j = 0; j < BIT_MAP_BYTES; j++)
	{
	    if (chunk_map[i].map[j] != 0) printf(" %d:%02x", j, chunk_map[i].map[j]);
	}

	printf("\n");

	if (lseek64(obj_file_desc, block << 8, SEEK_SET) < 0)
	{
	    sprintf(ref_str, "lseek64 block map:%08x", (uint32)block);
	    perror(ref_str);
	    exit(1);
	}

	if (write(obj_file_desc, chunk_map[i].map, BIT_MAP_BYTES) != BIT_MAP_BYTES)
	{
	    sprintf(ref_str, "%s block map:%08x", DEFAULT_FILE, (uint32)block);
	    perror(ref_str);
	}
    }
}


int main(int argc, char* argv[])
{
    int obj_file_desc;
    off64_t ref;
    char ref_str[128];
    uint8 block[FILE_BLOCK_SIZE];
    size_t bytes_read;
    FILE* ifp;
    uint32 chunk;
    int num_blocks = 0;
    int num_maps = 0;
    int byte_num;
    int bit_num;
    int i;
    uint32 old_disk_size;
    uint8 header[16];

    if (argc != 2)
    {
	print_usage(argv[0]);
	exit(1);
    }

    if (*argv[1] == '-')
    {
	fprintf(stderr, "error: this program doesn't have any options\n");
	print_usage(argv[0]);
	exit(1);
    }


    /* Read the disk header */

    obj_file_desc = open(DEFAULT_FILE, O_RDONLY);

    if (obj_file_desc < 0)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }

    if (read(obj_file_desc, header, sizeof(header)) != sizeof(header))
    {
	sprintf(ref_str, "reading header of %s", DEFAULT_FILE);
	perror(ref_str);
	exit(1);
    }

    if (close(obj_file_desc) != 0)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }

    if (memcmp(&header[0], MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header\n");
	exit(1);
    }

    if (memcmp(&header[4], VERSION_STRING, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header\n");
	exit(1);
    }

    reserved_public_blocks = ((uint32)header[8] << 24) | ((uint32)header[9]<< 16) | ((uint32)header[10] << 8) | ((uint32)header[11]);

    blocks_on_disk = ((uint32)header[12] << 24) | ((uint32)header[13]<< 16) | ((uint32)header[14] << 8) | ((uint32)header[15]);

    private_blocks_on_disk = blocks_on_disk - reserved_public_blocks;


    /* Open the disk for writing */

    obj_file_desc = open(DEFAULT_FILE, O_CREAT | O_WRONLY | O_LARGEFILE);

    if (obj_file_desc < 0)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }


    if (fchmod(obj_file_desc, 0600) != 0)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }

    ifp = fopen(argv[1], "r");

    if (ifp == NULL)
    {
	perror(argv[1]);
	exit(1);
    }

    bytes_read = fread(block, 1, FILE_BLOCK_SIZE, ifp);

    old_disk_size = ((uint32)block[12] << 24) | ((uint32)block[13] << 16) | ((uint32)block[14] << 8) | ((uint32)block[15]);

    printf("compressed file version: %c%c%c%c  disk_size: %08x\n",
	   block[4], block[5], block[6], block[7], old_disk_size);

    if (memcmp(&block[0], MAGIC_NUMBER, 4) != 0)
    {
	printf("Compressed file missing magic number: %02x%02x%02x%02x\n",
	       block[0], block[1], block[2], block[3]);
	fclose(ifp);
	close(obj_file_desc);
	exit(1);
    }

    if (memcmp(&block[4], &header[4], 4) != 0)
    {
	printf("Version mismatch - disk: %c%c%c%c   file: %c%c%c%c\n",
	       block[4], block[5], block[6], block[7],
	       header[4], header[5], header[6], header[7]);
	fclose(ifp);
	close(obj_file_desc);
	exit(1);
    }


    if (blocks_on_disk != old_disk_size)
    {
	printf("Disk size mismatch - disk: %u   file: %u\n", blocks_on_disk, old_disk_size);
	printf("This version of %s cannot deal with different sized disks\n", argv[0]);
	fclose(ifp);
	close(obj_file_desc);
	exit(1);
    }


    bytes_read = fread(block, 1, FILE_BLOCK_SIZE, ifp);

    while (bytes_read == FILE_BLOCK_SIZE)
    {
	ref = hash_ref(&block[4]);

	assert(BLOCKS_IN_CHUNK == 65536);  /* if this breaks need to fix the hard coded junk below */

	if ((ref & 0xffe0) == 0)     /* special hack to get around blocks that overwrite the bit map */
	{
	    ref = (ref + 32) & 0xffffffe0;
	}

	chunk = (uint32)ref & 0xFFFF0000;   /* each chunk is 2048 bits */

	if (num_maps == 0 || chunk_map[(num_maps - 1) % NUM_CHUNKS_CACHE].chunk != chunk)   /* have a new one */
	{
	    if (num_maps > 0 && (num_maps % NUM_CHUNKS_CACHE) == 0)    /* cache is full */
	    {
		flush_maps_to_disk(obj_file_desc);

		for (i = 0; i < NUM_CHUNKS_CACHE; i++)
		{
		    chunk_map[i].chunk = 0;
		}
	    }

	    chunk_map[num_maps % NUM_CHUNKS_CACHE].chunk = chunk;

	    if (chunk_map[num_maps % NUM_CHUNKS_CACHE].map == NULL)
	    {
		chunk_map[num_maps % NUM_CHUNKS_CACHE].map = malloc(BIT_MAP_BYTES);
	    }

	    assert(chunk_map[num_maps % NUM_CHUNKS_CACHE].map != NULL);

	    memset(chunk_map[num_maps % NUM_CHUNKS_CACHE].map, 0, BIT_MAP_BYTES);

	    chunk_map[num_maps % NUM_CHUNKS_CACHE].map[0] = 0xff;
	    chunk_map[num_maps % NUM_CHUNKS_CACHE].map[1] = 0xff;
	    chunk_map[num_maps % NUM_CHUNKS_CACHE].map[2] = 0xff;
	    chunk_map[num_maps % NUM_CHUNKS_CACHE].map[3] = 0xff;

	    num_maps++;
	}

	byte_num = ((uint32)ref % BLOCKS_IN_CHUNK) / 8;
	assert(byte_num < BIT_MAP_BYTES);
	bit_num = (uint32)ref % 8;
	chunk_map[(num_maps - 1) % NUM_CHUNKS_CACHE].map[byte_num] |= (0x80 >> bit_num);

	printf("id: %02x%02x%02x%02x  %08x  map_byte: %4d.%d-%02x\n",
	       block[4], block[5], block[6], block[7], (uint32)ref, byte_num, bit_num, chunk_map[(num_maps - 1) % NUM_CHUNKS_CACHE].map[byte_num]);
	fflush(stdout);

	if (lseek64(obj_file_desc, ref << 8, SEEK_SET) < 0)
	{
	    sprintf(ref_str, "lseek64 ref:%08x", (uint32)ref);
	    perror(ref_str);
	    exit(1);
	}

	if (write(obj_file_desc, block, sizeof(block)) != sizeof(block))
	{
	    sprintf(ref_str, "%s ref:%08x", DEFAULT_FILE, (uint32)ref);
	    perror(ref_str);
	    exit(1);
	}

	num_blocks++;

	bytes_read = fread(block, 1, FILE_BLOCK_SIZE, ifp);
    }

    if (ferror(ifp))
    {
	perror(argv[1]);
	exit(1);
    }

    chunk_map[num_maps % NUM_CHUNKS_CACHE].chunk = 0;
    flush_maps_to_disk(obj_file_desc);

    if (fclose(ifp) != 0)
    {
	perror(argv[1]);
	exit(1);
    }

    if (close(obj_file_desc) != 0)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }

    printf("Number of blocks written: %d  number of maps: %d\n", num_blocks, num_maps);

    return 0;
}

