/*             This file is part of the New World OS project
--                   Copyright (C) 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation
--      51 Franklin Street, Fifth Floor
--      Boston, MA 02110-1301
--      USA
--      http://www.fsf.org/licenses
--
-- $Log: file.c,v $
-- Revision 1.16  2006/11/29 18:49:49  jsedwards
-- Change so both path and file object reference are passed to restore_file.
-- Also changed file error handling to exit on errors (this is probably bad).
--
-- Revision 1.15  2006/11/28 13:13:29  jsedwards
-- Changed to use BSD name for modification time in stat struct and added a
-- define to map it to the Linux name when compiling on Linux.
--
-- Revision 1.14  2006/11/27 13:48:08  jsedwards
-- Changed to use variable sized counts for disc lists (so number of files can
-- be larger than 127).
--
-- Revision 1.13  2006/11/19 16:38:18  jsedwards
-- Added find_matching_disc_list function.
--
-- Revision 1.12  2006/11/19 15:28:20  jsedwards
-- Made get_disc_list_object_size a global function.
--
-- Revision 1.11  2006/11/19 14:48:47  jsedwards
-- Change check_file_md5sum function to return one of three possible results
-- (file not found, md5sum match, or md5sum mismatch) instead of just a
-- boolean saying file found or not found.
--
-- Revision 1.10  2006/11/18 15:09:09  jsedwards
-- Added "max_size" parameter to read_variable_sized_object_from_disk because
-- objects are no longer limited to one file block.
--
-- Revision 1.9  2006/11/18 14:33:03  jsedwards
-- Change size of kludge buffer for reading and writing disc lists to max size
-- of disc list instead of file block size, because now disc list objects can
-- be larger than one file block.
--
-- Revision 1.8  2006/11/07 14:07:36  jsedwards
-- Completely rearranged create_file_without_storing_data function to deal
-- with files that have duplicate md5sums (and are possibly exact duplicates).
-- Fixed bug where all of disc id was not being compared.
--
-- Revision 1.7  2006/11/06 13:51:24  jsedwards
-- Change so create_file_without_storing_data returns the reference for the
-- file_path instead of the file.  Also fixed bug in class name.
--
-- Revision 1.6  2006/11/05 21:29:51  jsedwards
-- Add functions for finding and creating disc_copy and storage_location
-- objects.
--
-- Revision 1.5  2006/11/04 18:59:21  jsedwards
-- Added routines to find or create a Disc_List.
--
-- Revision 1.4  2006/11/02 11:45:59  jsedwards
-- Change maximum file size from 4GB - 1 to 2GB - 1 because the old compiler
-- isn't happy about the 32 bit constant.  It needs to be addressed sometime
-- in the future anyway.
--
-- Revision 1.3  2006/10/29 13:28:40  jsedwards
-- Added routines to just record a files md5sum without storing the file and
-- to check the md5sum of a file.
--
-- Revision 1.2  2006/10/26 01:51:27  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.24  2006/10/25 12:22:28  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.1.2.23  2006/10/22 13:11:55  jsedwards
-- Changed to pass approximate number of blocks needed for a file to
-- set_sequential_blocks instead of spacing.  It now computes the
-- spacing from the blocks needed because the blocks on disk is variable.
--
-- Revision 1.1.2.22  2006/10/18 13:10:32  jsedwards
-- Changed printf formats for uint32 that is an integer instead of long.
--
-- Revision 1.1.2.21  2006/10/07 22:25:21  jsedwards
-- Changed so that sequential blocks are handled within objectify when the
-- 'set_sequential_blocks' function is called first.
--
-- Revision 1.1.2.20  2006/10/06 04:42:49  jsedwards
-- Changed to use new id generator that generates ids that scan across the
-- disk drive instead of totally random so that files don't cause so much
-- repositioning of the heads and hopefully will be faster.
--
-- Revision 1.1.2.19  2006/09/18 11:38:12  jsedwards
-- Fix so block sequence number wraps correctly.
--
-- Revision 1.1.2.18  2006/09/18 01:34:23  jsedwards
-- Changed for 256 byte blocks and use a "reference_list" to point to them.
--
-- Revision 1.1.2.17  2006/09/01 13:27:20  jsedwards
-- Changed "nwos_object_size" to "nwos_reference_list_size" and added the
-- object reference to "nwos_fill_in_common_header" so it can put the "id"
-- in the header now.
--
-- Revision 1.1.2.16  2006/08/26 15:30:11  jsedwards
-- Added routines to find and create MD5 objects and modified file routines
-- to use them instead of having the md5sum embedded in the file object.
-- Also removed the "tolower" for path names, it was a mistake.
--
-- Revision 1.1.2.15  2006/08/25 01:37:47  jsedwards
-- Added code to test md5sum on restoring the file.
--
-- Revision 1.1.2.14  2006/08/24 13:01:18  jsedwards
-- Added code to compute the MD5 checksum while reading the file.
--
-- Revision 1.1.2.13  2006/08/24 12:58:08  jsedwards
-- Changed the <utime.h> include to <sys/time.h> because although the man page
-- says "utimes()" is defined in <utime.h>, gcc still gives a warning.  I only
-- see "utimes()" in <sys/time.h>.
--
-- Revision 1.1.2.12  2006/08/24 11:57:21  jsedwards
-- Added code to restore the modification time of the file when it is restored.
--
-- Revision 1.1.2.11  2006/08/23 13:23:09  jsedwards
-- Added code to save the files time stamp.
--
-- Revision 1.1.2.10  2006/08/23 12:41:56  jsedwards
-- Changed to use fread instead of fgetc for every byte.
--
-- Revision 1.1.2.9  2006/08/20 15:35:27  jsedwards
-- Removed many of the debugging printf statements.
--
-- Revision 1.1.2.8  2006/08/20 15:13:43  jsedwards
-- Fix MAX_BLOCKS define.
--
-- Revision 1.1.2.7  2006/08/20 14:54:01  jsedwards
-- Fixed bug in num_blocks calculation and reading small (< 28 bytes) files.
--
-- Revision 1.1.2.6  2006/08/20 02:17:53  jsedwards
-- Fixed several bugs in the create_file function and added the restore_file
-- function.
--
-- Revision 1.1.2.5  2006/08/19 20:02:55  jsedwards
-- Fix comment about create_file routine.
--
-- Revision 1.1.2.4  2006/08/19 18:58:18  jsedwards
-- Added code to store a file in objects.
--
-- Revision 1.1.2.3  2006/08/19 18:51:59  jsedwards
-- Changed so checksum uses new "file" element instead of "count" for it's base.
--
-- Revision 1.1.2.2  2006/08/19 14:34:29  jsedwards
-- Added creation of MD5sum class (moved here from big_bang.c).
--
-- Revision 1.1.2.1  2006/08/18 13:04:03  jsedwards
-- Initial version.
--
*/


#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "crc32.h"
#include "md5/md5.h"
#include "objectify.h"
#include "objectify_private.h"
#include "time_stamp.h"

/* Map linux file modification time name in stat struct */
#ifdef linux
#define st_mtimespec st_mtim
#endif

#define DATA_STORAGE_SIZE (FILE_BLOCK_SIZE - 12)   /* 4 bytes each for flags, id, checksum */

typedef struct file_block {
    uint8 flags[4];
    ObjRef id;
    uint8 checksum[4];
    uint8 storage[DATA_STORAGE_SIZE];
} DataObject;



/* create the file and file id (path) classes */

void nwos_setup_file()
{
    ObjRef class_ref;

    if (nwos_find_class_definition("FILE", &class_ref))
    {
	printf("FILE class already exists: %02x%02x%02x%02x\n",
	       class_ref.id[0], class_ref.id[1], class_ref.id[2], class_ref.id[3]);
    }
    else
    {
	printf ("Creating File class definition.\n");
	nwos_create_class_definition("FILE");
    }

    if (nwos_find_class_definition("FILE PATH", &class_ref))
    {
	printf("FILE PATH class already exists: %02x%02x%02x%02x\n",
	       class_ref.id[0], class_ref.id[1], class_ref.id[2], class_ref.id[3]);
    }
    else
    {
	printf ("Creating Social Security Number class definition.\n");
	nwos_create_class_definition("FILE PATH");
    }

    if (nwos_find_class_definition("MD5SUM", &class_ref))
    {
	printf("MD5 SUM class already exists: %02x%02x%02x%02x\n",
	       class_ref.id[0], class_ref.id[1], class_ref.id[2], class_ref.id[3]);
    }
    else
    {
	printf ("Creating MD5 Sum class definition.\n");
	nwos_create_class_definition("MD5SUM");
    }

    if (nwos_find_class_definition("DISC LIST", &class_ref))
    {
	printf("DISC LIST class already exists: %02x%02x%02x%02x\n",
	       class_ref.id[0], class_ref.id[1], class_ref.id[2], class_ref.id[3]);
    }
    else
    {
	printf ("Creating Disc List class definition.\n");
	nwos_create_class_definition("DISC LIST");
    }

    if (nwos_find_class_definition("DISC COPY", &class_ref))
    {
	printf("DISC COPY class already exists: %02x%02x%02x%02x\n",
	       class_ref.id[0], class_ref.id[1], class_ref.id[2], class_ref.id[3]);
    }
    else
    {
	printf ("Creating Disc Copy class definition.\n");
	nwos_create_class_definition("DISC COPY");
    }

    if (nwos_find_class_definition("STORAGE LOCATION", &class_ref))
    {
	printf("STORAGE LOCATION class already exists: %02x%02x%02x%02x\n",
	       class_ref.id[0], class_ref.id[1], class_ref.id[2], class_ref.id[3]);
    }
    else
    {
	printf ("Creating Location class definition.\n");
	nwos_create_class_definition("STORAGE LOCATION");
    }
}


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* File path (name) object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

bool nwos_find_file_path(char* path, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    ObjRef path_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;
    size_t length;
    int i;

    length = strlen(path);

    assert(length <= 255);
    
    assert(nwos_find_class_definition("FILE PATH", &path_class_ref));

    nwos_read_class_definition(&path_class_ref, &class_def_obj);

    ref_list_size = nwos_reference_list_size(&class_def_obj.header.object.references);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(&class_def_obj.header.object.references, ref_list, ref_list_size);

    num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &path_class_ref))
	{
	    nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_path_object_size);

	    /* remember ptr_path_obj points to the kludge buffer */

	    if (ptr_path_obj->count == length && strncasecmp((char*)ptr_path_obj->storage, path, length) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    free(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}




/* Find existing path or create new */

ObjCreateResult nwos_create_file_path(char* path, ObjRef* file_ref, ObjRef* ref)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    size_t length;
    ObjRef path_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    int i;

    length = strlen(path);

    assert(length <= 255);
    
    /* first find out if we already have this path */

    if (!nwos_find_file_path(path, ref))   /* didn't find it */
    {
	memset(kludge, 0, sizeof(kludge));  /* zero it out */

	/* remember ptr_path_obj points to the kludge buffer */

	assert(nwos_find_class_definition("FILE PATH", &path_class_ref));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&ptr_path_obj->header.common, ref, &path_class_ref);

	memcpy(&ptr_path_obj->file, file_ref, sizeof(ptr_path_obj->file));

	ptr_path_obj->count = length;
	for (i = 0; i < length; i++) ptr_path_obj->storage[i] = path[i];

	nwos_create_reference_list(ref, &ptr_path_obj->header.object.references);

	nwos_crc32_calculate((uint8*) &ptr_path_obj->header.object, sizeof(ObjectHeader), ptr_path_obj->header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &ptr_path_obj->file, sizeof(C_struct_File_Path) + length - sizeof(EveryObject), ptr_path_obj->header.common.data_chksum);

	nwos_write_object_to_disk(ref, kludge, sizeof(C_struct_File_Path) + length);

	nwos_add_to_references(ref, &path_class_ref);

	ptr_path_obj = malloc(sizeof(C_struct_File_Path) + length);
	nwos_read_object_from_disk(ref, ptr_path_obj, sizeof(C_struct_File_Path) + length);
	assert(memcmp(kludge, ptr_path_obj, sizeof(C_struct_File_Path) + length) == 0);

	memset(kludge, 0, sizeof(kludge));  /* clear it */
	nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_path_object_size);  /* read the other way */
	assert(memcmp(ptr_path_obj, kludge, sizeof(C_struct_File_Path) + length) == 0);

	free(ptr_path_obj);
	ptr_path_obj = NULL;

	result = CREATED_NEW;
    }

    return result;
}



bool nwos_file_path_to_string(ObjRef* ref, char* string, size_t size)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj;
    int i;
    size_t path_obj_size;

    /* read the path object into the kludge buffer and then after we know what size it is malloc space and copy it there */
    nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_path_object_size);

    /* point to the kludge buffer temporarily to get the size of the object */
    ptr_path_obj = (C_struct_File_Path*)kludge;

    path_obj_size = sizeof(C_struct_File_Path) + ptr_path_obj->count;
    /* printf("path_obj_size: %d\n", path_obj_size); */
    assert(path_obj_size > sizeof(C_struct_File_Path));

    /* now we have the size allocate memory for it and copy it there */
    ptr_path_obj = malloc(path_obj_size);
    memcpy(ptr_path_obj, kludge, path_obj_size);

    for (i = 0; i < ptr_path_obj->count; i++) 
    {
	assert(i < size);
	string[i] = ptr_path_obj->storage[i];
    }
    string[i] = '\0';

    free(ptr_path_obj);

    return true;
}


bool nwos_file_path_to_file(ObjRef* ref, ObjRef* file_ref)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj;
    size_t path_obj_size;

    /* read the path object into the kludge buffer and then after we know what size it is malloc space and copy it there */
    nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_path_object_size);

    /* point to the kludge buffer temporarily to get the size of the object */
    ptr_path_obj = (C_struct_File_Path*)kludge;

    path_obj_size = sizeof(C_struct_File_Path) + ptr_path_obj->count;
    /* printf("path_obj_size: %d\n", path_obj_size); */
    assert(path_obj_size > sizeof(C_struct_File_Path));

    /* now we have the size allocate memory for it and copy it there */
    ptr_path_obj = malloc(path_obj_size);
    memcpy(ptr_path_obj, kludge, path_obj_size);

    memcpy(file_ref, &ptr_path_obj->file, sizeof(*file_ref));

    free(ptr_path_obj);

    return true;
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* MD5 object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

bool nwos_find_md5(uint8 md5sum[16], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    ObjRef md5_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;
    C_struct_MD5sum md5_object;
    int i;

    
    assert(nwos_find_class_definition("MD5SUM", &md5_class_ref));

    nwos_read_class_definition(&md5_class_ref, &class_def_obj);

    ref_list_size = nwos_reference_list_size(&class_def_obj.header.object.references);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(&class_def_obj.header.object.references, ref_list, ref_list_size);

    num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &md5_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &md5_object, sizeof(md5_object));

	    if (memcmp(md5_object.md5sum, md5sum, sizeof(md5sum)) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    free(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}




/* Find existing MD5sum or create new */

ObjCreateResult nwos_create_md5(uint8 md5sum[16], ObjRef* ref)
{
    ObjRef md5_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    C_struct_MD5sum md5_object;
    C_struct_MD5sum* ptr_md5_obj;


    /* first find out if we already have this md5 */

    if (!nwos_find_md5(md5sum, ref))   /* didn't find it */
    {
	assert(nwos_find_class_definition("MD5SUM", &md5_class_ref));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&md5_object.header.common, ref, &md5_class_ref);

	memcpy(md5_object.md5sum, md5sum, sizeof(md5_object.md5sum));

	nwos_create_reference_list(ref, &md5_object.header.object.references);

	nwos_crc32_calculate((uint8*) &md5_object.header.object, sizeof(ObjectHeader), md5_object.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &md5_object.md5sum, sizeof(C_struct_MD5sum) - sizeof(EveryObject), md5_object.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &md5_object, sizeof(md5_object));

	nwos_add_to_references(ref, &md5_class_ref);

	ptr_md5_obj = malloc(sizeof(C_struct_MD5sum));
	nwos_read_object_from_disk(ref, ptr_md5_obj, sizeof(C_struct_MD5sum));
	assert(memcmp(&md5_object, ptr_md5_obj, sizeof(C_struct_MD5sum)) == 0);
	free(ptr_md5_obj);
	ptr_md5_obj = NULL;

	result = CREATED_NEW;
    }

    return result;
}




/*-------------------------------------------------------------------------------------------------------------------*/
/* File object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/


/* Create a file record (doesn't store data of file, just time, md5, etc.) */

ObjCreateResult nwos_create_file_without_storing_data(char* path, ObjRef* path_ref)
{
    size_t length;
    struct stat stat_struct;
    ObjCreateResult result = CREATED_NEW;   /* for now we can't create objects with the same name so it always creates */
    FILE* fp;
    size_t bytes_read;
    ObjRef file_class_ref;
    ObjRef file_ref;
    ObjRef md5sum_ref;
    ObjRef object_class;
    uint8 buffer[4096];
    uint32 file_length;
    uint32 size;
    int num_blocks;
    C_struct_File file_obj;
    C_struct_MD5sum md5_obj;
    TimeStamp mod_time;
    MD5_CTX context;    /* MD5 checksum context */
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;
    uint8 digest[16];
    int i;

    /* for now we can't create a file with a duplicate file name */
    assert(!nwos_find_file_path(path, path_ref));

    if (stat(path, &stat_struct) != 0)
    {
	perror(path);
	return ERROR_OCCURRED;       /* FIX THIS to handle this more gracefully */
    }

    if (stat_struct.st_size == 0)
    {
	fprintf(stderr, "Cannot handle empty files at this time\n");
	return ERROR_OCCURRED;
    }

    if (stat_struct.st_size > 2147483647)
    {
	fprintf(stderr, "Cannot handle files larger than 2GB at this time\n");
	return ERROR_OCCURRED;
    }


    length = strlen(path);

    assert(length <= 255);
    
    printf("creating file record: %s ", path);
    fflush(stdout);

    assert(nwos_find_class_definition("FILE", &file_class_ref));

    /* now open the file and compute the md5sum */

    fp = fopen(path, "rb");

    file_length = 0;
    num_blocks = 0;

    MD5Init(&context);   /* initialize the MD5 checksum context */

    bytes_read = sizeof(buffer);   /* go through loop at least once */

    while (bytes_read == sizeof(buffer))
    {
	bytes_read = fread(buffer, sizeof(uint8), sizeof(buffer), fp);

	if (bytes_read < sizeof(buffer))    /* eof or error */
	{
	    if (ferror(fp))
	    {
	      perror(path);
	      exit(1);            /* FIX THIS - needs to clean up and exit gracefully!! */
	    }
	}

	file_length = file_length + bytes_read;

	if (bytes_read > 0)   /* we have data to store */
	{
		MD5Update(&context, buffer, (unsigned)bytes_read);    /* include this data in the md5 checksum */
	}
    }

    fclose(fp);

    printf("length: %u\n", file_length);

    MD5Final(digest, &context);   /* finish computing the md5 sum */

    nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, mod_time);

    printf("MD5 sum: ");
    for (i = 0; i < 16; i++) printf("%02x", digest[i]);
    printf("\n");

    fflush(stdout);

    void_reference(&file_ref);   /* signal no existing file object found that matches everything */

    if (nwos_create_md5(digest, &md5sum_ref) == FOUND_EXISTING)
    {
	printf("Found existing MD5: %02x%02x%02x%02x\n", 
	       md5sum_ref.id[0], md5sum_ref.id[1], md5sum_ref.id[2], md5sum_ref.id[3]);

	nwos_read_object_from_disk(&md5sum_ref, &md5_obj, sizeof(md5_obj));

	ref_list_size = nwos_reference_list_size(&md5_obj.header.object.references);

	ref_list = malloc(ref_list_size);

	if (ref_list == NULL) 
	{
	    perror("reading reference list");
	    exit(1);
	}

	nwos_read_reference_list_from_disk(&md5_obj.header.object.references, ref_list, ref_list_size);

	num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);

	    if (is_same_object(&object_class, &file_class_ref))
	    {
		nwos_read_object_from_disk(&ref_list->references[i], &file_obj, sizeof(file_obj));

		assert(is_same_object(&md5sum_ref, &file_obj.md5sum));

		size = ((uint32)file_obj.size[0] << 24) | ((uint32)file_obj.size[1] << 16)
		                      | ((uint32)file_obj.size[2] << 8) | (uint32)file_obj.size[3];

		printf("    size: %u\n", size);

		if (size == file_length && memcmp(file_obj.modification_time, mod_time, sizeof(mod_time)) == 0)
		{
		    copy_reference(&file_ref, &ref_list->references[i]);   /* use existing object instead */
		    break;
		}
	    }
	}

	free(ref_list);
	ref_list = NULL;
    }


    if (is_void_reference(&file_ref))   /* didn't find a file object that matched exactly */
    {
	nwos_generate_new_id(&file_ref);

	memset(&file_obj, 0, sizeof(file_obj));  /* zero it out */

	nwos_fill_in_common_header(&file_obj.header.common, &file_ref, &file_class_ref);

	/* store file size as 4 bytes in big endian order */
	file_obj.size[0] = file_length >> 24;
	file_obj.size[1] = file_length >> 16;
	file_obj.size[2] = file_length >> 8;
	file_obj.size[3] = file_length;

	memcpy(&file_obj.modification_time, &mod_time, sizeof(file_obj.modification_time));

	copy_reference(&file_obj.md5sum, &md5sum_ref);

	nwos_create_reference_list(&file_ref, &file_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &file_obj.header.object, sizeof(ObjectHeader), file_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &file_obj.size[0], sizeof(C_struct_File) + (num_blocks * sizeof(ObjRef)) - sizeof(EveryObject), file_obj.header.common.data_chksum);

	nwos_write_object_to_disk(&file_ref, &file_obj, sizeof(file_obj));

	nwos_add_to_references(&file_ref, &md5sum_ref);   /* add to this md5sum object's reference list */

	nwos_add_to_references(&file_ref, &file_class_ref);
    }

    assert(nwos_create_file_path(path, &file_ref, path_ref) == CREATED_NEW); /* verified above that this didn't already exist */

    nwos_add_to_reference_list(path_ref, &file_obj.header.object.references);

    return result;
}



/* Create new file - returns error if file already exists */

ObjCreateResult nwos_create_file(char* path, ObjRef* ref)
{
    size_t length;
    struct stat stat_struct;
    ObjCreateResult result;
    FILE* fp;
    size_t bytes_read;
    ObjRef file_class_ref;
    ObjRef path_ref;
    ObjRef data_block_ref;
    DataObject data_obj;
    uint32 file_length;
    int num_blocks;
    C_struct_File file_obj;
    MD5_CTX context;    /* MD5 checksum context */
    uint8 digest[16];
    uint8 ivec[IVEC_SIZE];
    int seq = 0;

    if (stat(path, &stat_struct) != 0)
    {
	perror(path);
	return ERROR_OCCURRED;       /* FIX THIS to handle this more gracefully */
    }

    if (stat_struct.st_size == 0)
    {
	fprintf(stderr, "Cannot handle empty files at this time\n");
	return ERROR_OCCURRED;
    }

    if (stat_struct.st_size > 1073741824)
    {
	fprintf(stderr, "Cannot handle files larger than 1GB at this time\n");
	return ERROR_OCCURRED;
    }

    memset(ivec, 0, sizeof(ivec));

    length = strlen(path);

    assert(length <= 255);
    
    nwos_generate_new_id(ref);

    /* leave room for 10 blocks for misc. objects */
    nwos_set_sequential_blocks((stat_struct.st_size + FILE_BLOCK_SIZE - 1) / FILE_BLOCK_SIZE + 10);

    /* first find out if we already have this path */

    result = nwos_create_file_path(path, ref, &path_ref);

    printf("creating file: %s ", path);
    fflush(stdout);

    if (result == CREATED_NEW)   /* it's ok, doesn't exist already */
    {
	memset(&file_obj, 0, sizeof(file_obj));  /* zero it out */

	assert(nwos_find_class_definition("FILE", &file_class_ref));

	nwos_fill_in_common_header(&file_obj.header.common, ref, &file_class_ref);

	/* create a reference list that is a list of the data blocks for the file */
	nwos_create_reference_list(NULL, &file_obj.block_list);

	printf("block_list: %02x%02x%02x%02x\n",
	       file_obj.block_list.id[0], file_obj.block_list.id[1], file_obj.block_list.id[2], file_obj.block_list.id[3]);

	/* now open the file and start creating the data block objects */

	fp = fopen(path, "rb");

	file_length = 0;
	num_blocks = 0;

	MD5Init(&context);   /* initialize the MD5 checksum context */

	memset(data_obj.flags, 0, sizeof(data_obj.flags));

	bytes_read = DATA_STORAGE_SIZE;   /* go through loop at least once */

	while (bytes_read == DATA_STORAGE_SIZE)
	{
	    bytes_read = fread(data_obj.storage, sizeof(uint8), DATA_STORAGE_SIZE, fp);

	    if (bytes_read < DATA_STORAGE_SIZE)    /* eof or error */
	    {
		if (ferror(fp))
		{
		    perror(path);
		    exit(1);            /* FIX THIS - needs to clean up and exit gracefully!! */
		}
	    }

	    file_length = file_length + bytes_read;

	    if (bytes_read > 0)   /* we have data to store */
	    {
		MD5Update(&context, data_obj.storage, (unsigned)bytes_read);    /* include this data in the md5 checksum */

		nwos_generate_new_id(&data_block_ref);

		memcpy(&data_obj.id, &data_block_ref, sizeof(ObjRef));

#if 0
		printf("data block seq: %d ref: %02x%02x%02x%02x\n", seq, data_block_ref.id[0], data_block_ref.id[1], data_block_ref.id[2], data_block_ref.id[3]);
#endif

		nwos_crc32_calculate((uint8*) &data_obj.storage, bytes_read, data_obj.checksum);

		/* fill in the remainder with random data */
		while (bytes_read < DATA_STORAGE_SIZE) data_obj.storage[bytes_read++] = random();

		nwos_write_object_to_disk_and_encrypt(&data_block_ref, &data_obj, sizeof(data_obj), ivec, nwos_random_sequence[seq]);

		nwos_add_to_reference_list(&data_block_ref, &file_obj.block_list);

		seq = (seq + 1) % NUM_STORED_SEQ;    /* wrap around if we've used all of them */
	    }
	}

	fclose(fp);

	printf("length: %u  num_blocks:%d  object size:%zd\n", file_length, num_blocks, sizeof(C_struct_File) + (num_blocks * sizeof(ObjRef)));

	MD5Final(digest, &context);   /* finish computing the md5 sum */

	nwos_create_md5(digest, &file_obj.md5sum);

	nwos_add_to_references(ref, &file_obj.md5sum);   /* add to this md5sum object's reference list */

	{
	  int i;
	  printf("MD5 sum: ");
	  for (i = 0; i < 16; i++) printf("%02x", digest[i]);
	  printf("\n");
	}

	fflush(stdout);

	/* store file size as 4 bytes in big endian order */
	file_obj.size[0] = file_length >> 24;
	file_obj.size[1] = file_length >> 16;
	file_obj.size[2] = file_length >> 8;
	file_obj.size[3] = file_length;

	nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, file_obj.modification_time);

	nwos_create_reference_list(ref, &file_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &file_obj.header.object, sizeof(ObjectHeader), file_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &file_obj.size[0], sizeof(C_struct_File) + (num_blocks * sizeof(ObjRef)) - sizeof(EveryObject), file_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &file_obj, sizeof(file_obj));

	nwos_add_to_references(ref, &file_class_ref);

	nwos_add_to_reference_list(&path_ref, &file_obj.header.object.references);
#if 0
	ptr_file_obj = malloc(sizeof(C_struct_File) + num_blocks * sizeof(ObjRef));
	nwos_read_object_from_disk(ref, ptr_file_obj, sizeof(C_struct_File) + num_blocks * sizeof(ObjRef));
	assert(memcmp(kludge, ptr_file_obj, sizeof(C_struct_File) + num_blocks * sizeof(ObjRef)) == 0);

	memset(kludge, 0, sizeof(kludge));  /* clear it */
	nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_file_object_size);  /* read the other way */
	assert(memcmp(ptr_file_obj, kludge, sizeof(C_struct_File_Path) + length) == 0);
#endif
    }

    nwos_set_sequential_blocks(0);   /* turn off the sequenctial ids */

    return result;
}


/* Check a file's md5sum */

CheckFileResult nwos_check_file_md5sum(char* path)
{
    size_t length;
    struct stat stat_struct;
    FILE* fp;
    size_t bytes_read;
    ObjRef path_ref;
    ObjRef file_ref;
    uint8 buffer[4096];
    uint32 file_length;
    int num_blocks;
    C_struct_File file_obj;
    MD5_CTX context;    /* MD5 checksum context */
    C_struct_MD5sum md5_object;
    uint8 digest[16];
    TimeStamp mod_time;
    CheckFileResult result = File_Not_Found;

    if (stat(path, &stat_struct) != 0)
    {
	perror(path);
	return ERROR_OCCURRED;       /* FIX THIS to handle this more gracefully */
    }

    if (stat_struct.st_size == 0)
    {
	fprintf(stderr, "Cannot handle empty files at this time\n");
	return ERROR_OCCURRED;
    }

    if (stat_struct.st_size > 2147483647)
    {
	fprintf(stderr, "Cannot handle files larger than 2GB at this time\n");
	return ERROR_OCCURRED;
    }

    length = strlen(path);

    assert(length <= 255);
    
    /* first find out if we already have this path */

    if (nwos_find_file_path(path, &path_ref))   /* it's ok, found the file */
    {
	nwos_file_path_to_file(&path_ref, &file_ref);

	nwos_read_object_from_disk(&file_ref, &file_obj, sizeof(file_obj));  /* read the file object */

	file_length = (file_obj.size[0] << 24) |(file_obj.size[1] << 16) | (file_obj.size[2] << 8) | file_obj.size[3];

	printf("file size: %u\n", file_length);
	/* now open the file and start creating the data block objects */

	fp = fopen(path, "rb");

	file_length = 0;
	num_blocks = 0;

	MD5Init(&context);   /* initialize the MD5 checksum context */

	bytes_read = sizeof(buffer);   /* go through loop at least once */

	while (bytes_read == sizeof(buffer))
	{
	    bytes_read = fread(buffer, sizeof(uint8), sizeof(buffer), fp);

	    if (bytes_read < sizeof(buffer))    /* eof or error */
	    {
		if (ferror(fp))
		{
		    perror(path);
		    exit(1);            /* FIX THIS - needs to clean up and exit gracefully!! */
		}
	    }

	    file_length = file_length + bytes_read;

	    if (bytes_read > 0)   /* we have data to store */
	    {
		MD5Update(&context, buffer, (unsigned)bytes_read);    /* include this data in the md5 checksum */
	    }
	}

	fclose(fp);

	printf("length: %u\n", file_length);

	MD5Final(digest, &context);   /* finish computing the md5 sum */


	{
	  int i;
	  printf("MD5 sum: ");
	  for (i = 0; i < 16; i++) printf("%02x", digest[i]);
	  printf("\n");
	}

	fflush(stdout);

	if (file_length != ((file_obj.size[0] << 24) | (file_obj.size[1] << 16) | (file_obj.size[2] << 8) | file_obj.size[3]))
	{
	    printf("file size mismatch\n");
	}

	nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, mod_time);

	if (memcmp(file_obj.modification_time, mod_time, sizeof(TimeStamp)) != 0)
	{
	    printf("time mismatch\n");
	}

	nwos_read_object_from_disk(&file_obj.md5sum, &md5_object, sizeof(md5_object));

	if (memcmp(digest, md5_object.md5sum, sizeof(digest)) == 0)
	{
	    int j;
	    printf("MD5 sum OK: ");
	    for (j = 0; j < 16; j++) printf("%02x", digest[j]);
	    printf("\n");

	    result = MD5_Sum_Match;    /* found file and MD5 was ok */
	}
	else
	{
	    int j;
	    printf("MD5 checksum error, expected: ");
	    for (j = 0; j < 16; j++) printf("%02x", md5_object.md5sum[j]);
	    printf("\n                    received: ");
	    for (j = 0; j < 16; j++) printf("%02x", digest[j]);
	    printf("\n");

	    result = MD5_Sum_Mismatch;    /* found file but MD5 was bad */
	}
    }

    return result;   /* didn't find file */
}



/* Read file */

bool nwos_restore_file(ObjRef* file_ref, char* path)
{
    bool result = true;
    ObjRef file_class_ref;
    ObjRef class_ref;
    FILE* fp;
    int i;
    uint32 file_length;
    uint32 num_blocks;
    C_struct_File file_obj;
    DataObject data_obj;
    struct timeval tv[2];      /* [0] is access time and [1] is modification time, see man utimes */
    MD5_CTX context;           /* MD5 checksum context */
    uint8 digest[16];
    C_struct_MD5sum md5_object;
    int ref_list_size;
    ReferenceList* ref_list_ptr;
    uint8 ivec[IVEC_SIZE];
    uint8 chksum[4];

    memset(ivec, 0, sizeof(ivec));

    assert(nwos_find_class_definition("FILE", &file_class_ref));
    nwos_get_object_class(file_ref, &class_ref);
    assert(is_same_object(&class_ref, &file_class_ref));

    nwos_read_object_from_disk(file_ref, &file_obj, sizeof(file_obj));  /* read the file object */

    file_length = (file_obj.size[0] << 24) |(file_obj.size[1] << 16) | (file_obj.size[2] << 8) | file_obj.size[3];

    printf("file size: %u\n", file_length);

    ref_list_size = nwos_reference_list_size(&file_obj.block_list);

    ref_list_ptr = malloc(ref_list_size);

    assert(ref_list_ptr != NULL);

    nwos_read_reference_list_from_disk(&file_obj.block_list, ref_list_ptr, ref_list_size);

    num_blocks = (file_length + DATA_STORAGE_SIZE - 1) / DATA_STORAGE_SIZE;

    assert(num_blocks == ((ref_list_size - sizeof(ReferenceList)) / sizeof(ObjRef)));

//    printf("num_blocks: %u  block_list: %02x%02x%02x%02x\n", num_blocks,
//	   file_obj.block_list.id[0],
//	   file_obj.block_list.id[1],
//	   file_obj.block_list.id[2],
//	   file_obj.block_list.id[3]);


    /* now open the file and start writing the data block objects */

    fp = fopen(path, "wb");

    if (fp == NULL)
    {
	perror(path);
	exit(1);
    }
    else  /* file opened ok */
    {
	i = 0;

	MD5Init(&context);   /* initialize the MD5 checksum context */

	while (file_length > DATA_STORAGE_SIZE)   /* do all except the last block */
	{
	    nwos_read_object_from_disk_and_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, nwos_random_sequence[i%NUM_STORED_SEQ]);

	    nwos_crc32_calculate((uint8*) &data_obj.storage, sizeof(data_obj.storage), chksum);

	    if (memcmp(chksum, data_obj.checksum, sizeof(chksum)) != 0)
	    {
		printf("checksum error in file block - sequence: %d\n", i);
		exit(1);
	    }

	    MD5Update(&context, data_obj.storage, DATA_STORAGE_SIZE);    /* include this data in the md5 checksum */

	    if (fwrite(data_obj.storage, sizeof(uint8), DATA_STORAGE_SIZE, fp) != DATA_STORAGE_SIZE)
	    {
		perror(path);
		exit(1);
	    }

	    file_length -= DATA_STORAGE_SIZE;

	    i++;
	}

	nwos_read_object_from_disk_and_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, nwos_random_sequence[i%NUM_STORED_SEQ]);

	nwos_crc32_calculate((uint8*) &data_obj.storage, file_length, chksum);

	if (memcmp(chksum, data_obj.checksum, sizeof(chksum)) != 0)
	{
	    printf("checksum error in file block - sequence: %d\n", i);
	    exit(1);
	}

	MD5Update(&context, data_obj.storage, file_length);    /* include this data in the md5 checksum */

	if (fwrite(data_obj.storage, sizeof(uint8), file_length, fp) != file_length)
	{
	    perror(path);
	    exit(1);
	}

	if (fclose(fp) != 0)
	{
	    perror(path);
	    exit(1);
	}

	if (result)   /* the file was ok, restore it's modification time */
	{
	    nwos_convert_time_stamp_to_timeval(file_obj.modification_time, &tv[1]);
	    tv[0] = tv[1];  /* copy modification time into access time since we don't save the access time */
	    utimes(path, tv);
	}

	MD5Final(digest, &context);   /* finish computing the md5 sum */

	if (result)
	{
	    nwos_read_object_from_disk(&file_obj.md5sum, &md5_object, sizeof(md5_object));

	    if (memcmp(digest, md5_object.md5sum, sizeof(digest)) == 0)
	    {
		int j;
		printf("MD5 sum OK: ");
		for (j = 0; j < 16; j++) printf("%02x", digest[j]);
		printf("\n");
	    }
	    else
	    {
		int j;
		printf("MD5 checksum error, expected: ");
		for (j = 0; j < 16; j++) printf("%02x", md5_object.md5sum[j]);
		printf("\n                    received: ");
		for (j = 0; j < 16; j++) printf("%02x", digest[j]);
		printf("\n");

		result = false;
	    }
	}

	free(ref_list_ptr);
	ref_list_ptr = NULL;
    }

    return result;
}


/*-------------------------------------------------------------------------------------------------------------------*/
/* Disc List object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

size_t nwos_get_disc_list_object_size(void* disc_list_obj)
{
    uint32 count = nwos_decode_variable_sized_count(((C_struct_Disc_List*)disc_list_obj)->count);

    assert(0 < count && count <= MAX_FILES_PER_DISC_LIST);

    return sizeof(C_struct_Disc_List) + count * sizeof(ObjRef);
}


bool nwos_find_disc_list(char id[12], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_disc_obj = (C_struct_Disc_List*)kludge;
    ObjRef disc_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;
    int i;

    assert(nwos_find_class_definition("DISC LIST", &disc_class_ref));

    nwos_read_class_definition(&disc_class_ref, &class_def_obj);

    ref_list_size = nwos_reference_list_size(&class_def_obj.header.object.references);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(&class_def_obj.header.object.references, ref_list, ref_list_size);

    num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &disc_class_ref))
	{
	    nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &nwos_get_disc_list_object_size);

	    /* remember ptr_disc_obj points to the kludge buffer */

	    if (memcmp(ptr_disc_obj->id, id, sizeof(ptr_disc_obj->id)) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    free(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}



bool nwos_find_matching_disc_list(ObjRef files[MAX_FILES_PER_DISC_LIST], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_disc_obj = (C_struct_Disc_List*)kludge;
    ObjRef disc_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;
    int i;
    int j;
    int k;
    int count = 0;

    /* first count the files */

    for (i = 0; i < MAX_FILES_PER_DISC_LIST; i++)
    {
	if (!is_void_reference(&files[i])) count++;
    }

    assert(count > 0);

    assert(nwos_find_class_definition("DISC LIST", &disc_class_ref));

    nwos_read_class_definition(&disc_class_ref, &class_def_obj);

    ref_list_size = nwos_reference_list_size(&class_def_obj.header.object.references);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(&class_def_obj.header.object.references, ref_list, ref_list_size);

    num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &disc_class_ref))
	{
	    nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &nwos_get_disc_list_object_size);

	    /* remember ptr_disc_obj points to the kludge buffer */

	    if (nwos_decode_variable_sized_count(ptr_disc_obj->count) == count)   /* found a possible match */
	    {
		for (j = 0; j < MAX_FILES_PER_DISC_LIST; j++)
		{
		    if (!is_void_reference(&files[j]))
		    {
			for (k = 0; k < count; k++)
			{
			    if (is_same_object(&files[j], &ptr_disc_obj->files[k])) break;
			}

			if (k == count) break;   /* didn't find a match this can't be the list */
		    }
		}

		if (j == MAX_FILES_PER_DISC_LIST)   /* found a match for each one */
		{
		    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		    break;
		}
	    }
	}
    }

    free(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}



/* Find existing disc list or create new */

ObjCreateResult nwos_create_disc_list(char id[12], ObjRef files[MAX_FILES_PER_DISC_LIST], ObjRef* ref)
{
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_disc_obj = (C_struct_Disc_List*)kludge;
    ObjRef disc_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    int i;
    int count = 0;


    /* first find out if we already have this disc */

    if (!nwos_find_disc_list(id, ref))   /* didn't find it */
    {
	memset(kludge, 0, sizeof(kludge));  /* zero it out */

	/* remember ptr_path_obj points to the kludge buffer */

	assert(nwos_find_class_definition("DISC LIST", &disc_class_ref));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&ptr_disc_obj->header.common, ref, &disc_class_ref);

	memcpy(&ptr_disc_obj->id, id, sizeof(ptr_disc_obj->id));

	for (i = 0; i < MAX_FILES_PER_DISC_LIST; i++)
	{
	    if (!is_void_reference(&files[i]))
	    {
		copy_reference(&ptr_disc_obj->files[count], &files[i]);
		count++;

		nwos_add_to_references(ref, &files[i]);
	    }
	}

	nwos_encode_variable_sized_count(count, ptr_disc_obj->count);

	assert(nwos_get_disc_list_object_size(ptr_disc_obj) == sizeof(C_struct_Disc_List) + count * sizeof(ObjRef));

	nwos_create_reference_list(ref, &ptr_disc_obj->header.object.references);

	nwos_crc32_calculate((uint8*) &ptr_disc_obj->header.object, sizeof(ObjectHeader), ptr_disc_obj->header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &ptr_disc_obj->id, nwos_get_disc_list_object_size(ptr_disc_obj) - sizeof(EveryObject), ptr_disc_obj->header.common.data_chksum);

	nwos_write_object_to_disk(ref, kludge, nwos_get_disc_list_object_size(ptr_disc_obj));

	nwos_add_to_references(ref, &disc_class_ref);

	ptr_disc_obj = malloc(sizeof(C_struct_Disc_List) + count * sizeof(ObjRef));
	nwos_read_object_from_disk(ref, ptr_disc_obj, sizeof(C_struct_Disc_List) + count * sizeof(ObjRef));
	assert(memcmp(kludge, ptr_disc_obj, sizeof(C_struct_Disc_List) + count * sizeof(ObjRef)) == 0);

	memset(kludge, 0, sizeof(kludge));  /* clear it */
	nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &nwos_get_disc_list_object_size);  /* read the other way */
	assert(memcmp(ptr_disc_obj, kludge, sizeof(C_struct_Disc_List) + count * sizeof(ObjRef)) == 0);

	free(ptr_disc_obj);
	ptr_disc_obj = NULL;

	result = CREATED_NEW;
    }

    return result;
}


/*-------------------------------------------------------------------------------------------------------------------*/
/* Disc Copy object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/


/* find existing disc copy */

bool nwos_find_disc_copy(ObjRef* disc_list, int copy_num, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Disc_Copy disc_obj;
    ObjRef disc_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;
    int i;

    assert(nwos_find_class_definition("DISC COPY", &disc_class_ref));

    nwos_read_class_definition(&disc_class_ref, &class_def_obj);

    ref_list_size = nwos_reference_list_size(&class_def_obj.header.object.references);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(&class_def_obj.header.object.references, ref_list, ref_list_size);

    num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &disc_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &disc_obj, sizeof(disc_obj));

	    if (is_same_object(&disc_obj.disc_list, disc_list) && disc_obj.copy_number == copy_num)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    free(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}



/* Find existing disc copy or create new */

ObjCreateResult nwos_create_disc_copy(ObjRef* disc_list, int copy_num, ObjRef* location, ObjRef* ref)
{
    C_struct_Disc_Copy disc_copy_obj;
    C_struct_Disc_Copy* ptr_disc_obj;
    ObjRef disc_class_ref;
    ObjCreateResult result = FOUND_EXISTING;


    /* first find out if we already have this disc copy */

    if (!nwos_find_disc_copy(disc_list, copy_num, ref))   /* didn't find it */
    {
	assert(nwos_find_class_definition("DISC COPY", &disc_class_ref));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&disc_copy_obj.header.common, ref, &disc_class_ref);

	copy_reference(&disc_copy_obj.disc_list, disc_list);
	copy_reference(&disc_copy_obj.location, location);
	disc_copy_obj.copy_number = copy_num;

	nwos_create_reference_list(ref, &disc_copy_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &disc_copy_obj.header.object, sizeof(ObjectHeader), disc_copy_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &disc_copy_obj.disc_list, sizeof(disc_copy_obj) - sizeof(EveryObject), disc_copy_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &disc_copy_obj, sizeof(disc_copy_obj));

	nwos_add_to_references(ref, &disc_class_ref);

	nwos_add_to_references(ref, disc_list);
	nwos_add_to_references(ref, location);

	ptr_disc_obj = malloc(sizeof(C_struct_Disc_Copy));
	nwos_read_object_from_disk(ref, ptr_disc_obj, sizeof(C_struct_Disc_Copy));
	assert(memcmp(&disc_copy_obj, ptr_disc_obj, sizeof(C_struct_Disc_Copy)) == 0);

	free(ptr_disc_obj);
	ptr_disc_obj = NULL;

	result = CREATED_NEW;
    }

    return result;
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* Storage Location object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/


/* find existing storage location */

bool nwos_find_storage_location(char* location, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Storage_Location location_obj;
    ObjRef location_class_ref;
    ObjRef object_class;
    ObjRef name_ref;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;
    int i;

    if (!nwos_find_name(location, &name_ref))
    {
	return false;
    }

    assert(nwos_find_class_definition("STORAGE LOCATION", &location_class_ref));

    nwos_read_class_definition(&location_class_ref, &class_def_obj);

    ref_list_size = nwos_reference_list_size(&class_def_obj.header.object.references);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(&class_def_obj.header.object.references, ref_list, ref_list_size);

    num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &location_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &location_obj, sizeof(location_obj));

	    if (is_same_object(&location_obj.name, &name_ref))   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    free(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}



/* Find existing disc copy or create new */

ObjCreateResult nwos_create_storage_location(char* location, ObjRef* ref)
{
    C_struct_Storage_Location location_obj;
    C_struct_Storage_Location* ptr_loc_obj;
    ObjRef location_class_ref;
    ObjRef name_ref;
    ObjCreateResult result = FOUND_EXISTING;


    /* first find out if we already have this location */

    if (!nwos_find_storage_location(location, ref))   /* didn't find it */
    {
	nwos_create_name(location, &name_ref);

	assert(nwos_find_class_definition("STORAGE LOCATION", &location_class_ref));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&location_obj.header.common, ref, &location_class_ref);

	copy_reference(&location_obj.name, &name_ref);

	nwos_create_reference_list(ref, &location_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &location_obj.header.object, sizeof(ObjectHeader), location_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &location_obj.name, sizeof(location_obj) - sizeof(EveryObject), location_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &location_obj, sizeof(location_obj));

	nwos_add_to_references(ref, &location_class_ref);

	nwos_add_to_references(ref, &name_ref);

	ptr_loc_obj = malloc(sizeof(C_struct_Storage_Location));
	nwos_read_object_from_disk(ref, ptr_loc_obj, sizeof(C_struct_Storage_Location));
	assert(memcmp(&location_obj, ptr_loc_obj, sizeof(C_struct_Storage_Location)) == 0);

	free(ptr_loc_obj);
	ptr_loc_obj = NULL;

	result = CREATED_NEW;
    }

    return result;
}



