/*             This file is part of the New World OS project
--                 Copyright (C) 2004-2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: objectify.h,v $
-- Revision 1.54  2006/12/01 05:08:19  jsedwards
-- Add an external declaration for strlcpy if compiling on linux, since it
-- doesn't have that function.
--
-- Revision 1.53  2006/11/29 18:30:46  jsedwards
-- Added external reference for file_path_to_file function and changed
-- restore_file function to take a file reference and path.
--
-- Revision 1.52  2006/11/28 14:27:06  jsedwards
-- Added unencrypted_flag.
--
-- Revision 1.51  2006/11/27 13:45:48  jsedwards
-- Upped MAX_FILES_PER_DISC_LIST to 256 (could be larger now, but this is
-- good for the moment so we don't use too much ram).  Also added external
-- definitions for the decode and encode variable sized counts.
--
-- Revision 1.50  2006/11/19 16:38:19  jsedwards
-- Added find_matching_disc_list function.
--
-- Revision 1.49  2006/11/19 15:28:20  jsedwards
-- Made get_disc_list_object_size a global function.
--
-- Revision 1.48  2006/11/19 14:48:47  jsedwards
-- Change check_file_md5sum function to return one of three possible results
-- (file not found, md5sum match, or md5sum mismatch) instead of just a
-- boolean saying file found or not found.
--
-- Revision 1.47  2006/11/18 15:09:10  jsedwards
-- Added "max_size" parameter to read_variable_sized_object_from_disk because
-- objects are no longer limited to one file block.
--
-- Revision 1.46  2006/11/18 14:24:02  jsedwards
-- Changed maximum number of files in a disc list to 127 (this allows bit 7
-- to be used to indicate larger counts if desired in the future).  Also
-- added max size of disc list object define.
--
-- Revision 1.45  2006/11/11 13:29:19  jsedwards
-- Changed version string to 0014 because of addition of next public reference
-- to disk header.
--
-- Revision 1.44  2006/11/11 12:01:05  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.43  2006/11/05 21:30:49  jsedwards
-- Add external declarations for disc_copy and storage_location functions.
--
-- Revision 1.42  2006/11/04 18:53:38  jsedwards
-- Added external declarations for new routines in file.c.
--
-- Revision 1.41  2006/10/27 12:09:28  jsedwards
-- Add external reference for 'find_md5'.
--
-- Revision 1.40  2006/10/26 01:33:32  jsedwards
-- Merged alpha_05_branch into main trunk.  Functions declarations and
-- definitions are all from alpha_05_branch.  The only changes from the
-- alpha_05_branch is the removal of old comments and moving all of the
-- C_structs to a separate file (c_structs.h).
--
-- Revision 1.37.2.23  2006/10/25 12:22:29  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.37.2.22  2006/10/22 12:58:02  jsedwards
-- Change name of parameter for "set_sequential_blocks" to approximate number
-- of blocks instead of block spacing, because now the function computes the
-- spacing.
--
-- Revision 1.37.2.21  2006/10/22 00:23:11  jsedwards
-- Change "flags" in common header to one uint32 instead of 4 uint8s.
-- Fix size of ObjRef in is_void_reference().
--
-- Revision 1.37.2.20  2006/10/12 12:08:15  jsedwards
-- Added external declaration of flush_bit_maps function that allows all of
-- the dirty bit maps in the cache to be written back to the disk.
--
-- Revision 1.37.2.19  2006/10/07 22:22:49  jsedwards
-- Changed version number to 0013.  Removed external declaration of generate
-- new id in range.  Added external declaration of set sequential blocks.
--
-- Revision 1.37.2.18  2006/10/06 04:33:15  jsedwards
-- Added external references for copy reference, void reference, and convert
-- word to ref functions.
--
-- Revision 1.37.2.17  2006/10/03 12:53:07  jsedwards
-- Changed so that instead of calling a separate routine after initialize to
-- change the already opened storage, you call it now with a type of storage
-- parameter and a path to the storage.  The problem with the other way was
-- if you tried reading a compressed file on another machine it tried to open
-- the default file which didn't exist.
--
-- Revision 1.37.2.16  2006/10/02 02:14:00  jsedwards
-- Added external declaration for set_compressed_read_only.
--
-- Revision 1.37.2.15  2006/09/29 12:07:17  jsedwards
-- Add declaration of "list_persons" routine.
--
-- Revision 1.37.2.14  2006/09/29 04:20:40  jsedwards
-- Put MAGIC_NUMBER and VERSION_STRING numbers back in.
--
-- Revision 1.37.2.13  2006/09/18 01:30:10  jsedwards
-- Eliminated block size from file class and now use 4 bytes for file size.
-- Changed so block references are not store in block class but have a
-- reference list for the list of blocks.
--
-- Revision 1.37.2.12  2006/09/13 12:27:32  jsedwards
-- Change "id" in common header to an ObjRef.
--
-- Revision 1.37.2.11  2006/09/02 15:03:03  jsedwards
-- Add reference list class reference to root object because now it has to
-- read the reference list to get it's size and we can't easily read it
-- without the read routine verifying it's class.
--
-- Revision 1.37.2.10  2006/09/01 13:21:11  jsedwards
-- Changed "magic_number" and 'version" in common header to "flags" and "id".
--
-- Revision 1.37.2.9  2006/09/01 03:16:32  jsedwards
-- Merged changes for storing files and changes to store objects in one big
-- sparse file.
--
-- Revision 1.37.2.8  2006/08/26 15:27:30  jsedwards
-- Removed embedded md5sum from file object and replaced it with an object
-- reference to a md5sum object.  Also added a reference to a sha1 object.
--
-- Revision 1.37.2.7  2006/08/20 02:16:03  jsedwards
-- Add declaration for restore_file function.
--
-- Revision 1.37.2.6  2006/08/19 18:47:19  jsedwards
-- Added "media" entry to "File" structure.  Added error return value to
-- ObjCreateResult.  Removed "create_file_path" declaration and replaced
-- it with the "create_file" because only "create_file" should call
-- "create_file_path".
--
-- Revision 1.37.2.5  2006/08/19 12:46:49  jsedwards
-- Moved configuration defines out of objectify.h and into a new file config.h.
--
-- Revision 1.37.2.4  2006/08/18 13:08:54  jsedwards
-- Added file reference to the File_Path structure.
--
-- Revision 1.37.2.3  2006/08/18 13:00:41  jsedwards
-- Added external definitions for the "file" module and changed
--   "File_Identifier" structure to "File_Path" (it's temporary so it
--   really doesn't matter what it is called).
--
-- Revision 1.37.2.2  2006/08/13 16:36:43  jsedwards
-- Made a define for the block size of the file called FILE_BLOCK_SIZE, that
-- is the minimum size of the file for each object.  The amount of disk space
-- used for a one byte file.
--
-- Revision 1.37.2.1  2006/08/13 13:47:39  jsedwards
-- Added Check, File, and File_Identifier structures.
--
-- Revision 1.39  2006/01/21 18:35:58  jsedwards
-- Removed all of the C structures and made a separate file with them, which
-- should be generated from the class definition objects at some point.
-- Moved the version information to the objectify_private.h header file.
-- Changed the object reference to the new format (16 bytes).
--
-- Revision 1.38  2006/01/18 01:06:18  jsedwards
-- Removed kruft that was not used.
--
-- Revision 1.37  2006/01/10 13:55:45  jsedwards
-- Added external reference for "find phone for person".
--
-- Revision 1.36  2006/01/09 03:22:35  jsedwards
-- Added external declarations for "any persons named", "any states named",
-- and "describe state" routines.
--
-- Revision 1.35  2006/01/03 03:20:15  jsedwards
-- Removed 'city' from 'Address' class because the street has city information
-- in it.  Added 'ref to word' inline function taken from 'name.c' file.
-- Added external declarations for "find state by name", "create us city",
-- "add city", and "list cities in state" routines.
--
-- Revision 1.34  2006/01/01 19:46:39  jsedwards
-- Added external declarations for "is quit command", "ask yes or no", and
-- "add mobile phone" routines.
--
-- Revision 1.33  2006/01/01 00:43:54  jsedwards
-- Removed count from month object.  Since it was always 2, why have it?
-- Added external declarations for date, month, and year to string routines.
-- Added external declarations for find_person and describe person routines.
--
-- Revision 1.32  2005/12/31 19:01:09  jsedwards
-- Add "gender" class.  Change find_word paramater from "thing" to "thing
-- class".  Add declaration of "add_person".
--
-- Revision 1.31  2005/12/31 03:05:33  jsedwards
-- Added structures related to houses and addresses.
--
-- Revision 1.30  2005/12/30 14:03:47  jsedwards
-- Removed identifier, drivers license number, and residence from person
-- object, they will be found by reference.  Added social security number
-- object.  Changed password object to variable length.  Added external
-- declarations for "is_leap_year" and "is_valid_ssn".  Added a few comments
-- to help discern what external refernces are for.
--
-- Revision 1.29  2005/12/29 17:04:35  jsedwards
-- Changed all objects that have variable sized data to have [0] sized arrays,
-- this way the objects don't have to deal with filling in the random data.
-- Also added external declarations for read object headers and read variable
-- sized object routines.
--
-- Revision 1.28  2005/12/28 12:59:55  jsedwards
-- Added parameters to initialize_objectify to pass in the blowfish key and
-- the two seeds for the sequence generator.  Changed get reference from
-- password to get a variable length key from the password.   Also changed
-- all of the variable length objects so they use the entire 512 bytes of
-- a file.
--
-- Revision 1.27  2005/12/27 20:43:47  jsedwards
-- Changed is_void_reference and is_same_object routines to work on any size
-- of ObjRef, instead of using the "word" which only works for 4 byte refs.
--
-- Revision 1.26  2005/12/27 19:44:06  jsedwards
-- Added "Root" object class and declaration of routine to set the root object.
--
-- Revision 1.25  2005/12/27 18:12:42  jsedwards
-- Changed ObjRef to be 4 bytes instead of 8 and eliminated the uint8_8
-- routines.  Changed the other routines appropriately and eliminated all
-- the fixed "CLASS" definitions because now all file names are random.
--
-- Revision 1.24  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.23  2005/12/23 18:05:10  jsedwards
-- Changed class_def_def to contain pointers to features and added definition
-- of data_definition.
--
-- Revision 1.22  2005/12/21 23:24:30  jsedwards
-- Added declarations for find name, name to string, and several new phone
-- routines.
--
-- Revision 1.21  2005/12/21 16:57:29  jsedwards
-- Added state_postal_code parameter to create_area_code function and added
-- declaration of find_state_from_postal_code.
--
-- Revision 1.20  2005/12/21 03:47:02  jsedwards
-- Added "count" to "Name" objects and "name" to "US State" objects.
--
-- Revision 1.19  2005/12/18 15:03:43  jsedwards
-- Removed postal code class and made postal code part of state class.
--
-- Revision 1.18  2005/12/17 19:20:43  jsedwards
-- Add state and state postal code.
--
-- Revision 1.17  2005/12/11 16:50:41  jsedwards
-- Added "Area Code" class, changed "Business Phone" class to "Work Phone",
-- added declarations for find language, spelling, and word.
--
-- Revision 1.16  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.15  2005/12/08 18:02:07  jsedwards
-- Split "Entity Phone" class into three different classes "Home Phone",
-- "Work Phone", and "Mobile Phone".
--
-- Revision 1.14  2005/12/05 18:59:00  jsedwards
-- Added nickname to person object.
--
-- Revision 1.13  2005/12/05 04:58:54  jsedwards
-- Added declaration for read_reference_list_from_disk.  Don't call
-- read_object_from_disk anymore to read a reference list.
--
-- Revision 1.12  2005/12/04 14:06:19  jsedwards
-- Added 'birth_place' and 'death_place' fields.
-- Changed 'goes_by_name' to 'goes_by' and 'current_address' to 'residence'.
--
-- Revision 1.11  2005/12/04 04:13:03  jsedwards
-- Added 'nwos' prefix to create_xxxx function names and eliminated the
-- 'referring' object parameter from all of them.
--
-- Revision 1.10  2005/12/04 00:40:43  jsedwards
-- Added declarations for create_date and get_month_ref functions.
--
-- Revision 1.9  2005/12/03 22:14:44  jsedwards
-- Changed to new word, name, spelling layout.
--
-- Revision 1.8  2005/12/03 04:12:07  jsedwards
-- Added abbreviation, thing, and language classes.
--
-- Revision 1.7  2005/12/03 02:01:44  jsedwards
-- Change month so that month is stored as two ascii digits instead of a uint8.
--
-- Revision 1.6  2005/12/02 20:28:59  jsedwards
-- Changed 0xF objects to files and make floating point numbers 0x4.
-- Added the name of the object that the reference list is for as a parameter
-- to create_reference_list.
--
-- Revision 1.5  2005/11/30 13:56:11  jsedwards
-- Rearranged the order, added endian sensitivity to class definition class,
-- and added external definitions for new routines.
--
-- Revision 1.4  2005/11/26 17:18:19  jsedwards
-- Removed "mother" and "father" fields from person, that should be handled
-- through the relationship object.  Added current and mailing address.
--
-- Revision 1.3  2005/11/26 16:36:27  jsedwards
-- Added many C structure types (person, month, phone, e-mail, etc.).
--
-- Revision 1.2  2005/11/26 15:34:45  jsedwards
-- Removed unused structures, added 'nwos' prefix to functions, and added
-- some new class definitions.
--
-- Revision 1.1.1.1  2005/11/25 12:44:27  jsedwards
-- Copied from 'lab'.
--
-- Revision 1.14  2005/11/24 19:21:18  jsedwards
-- Removed base32 stuff.
--
-- Revision 1.13  2005/11/24 15:46:24  jsedwards
-- Fixed syntax errors.
--
-- Revision 1.12  2005/11/24 15:14:24  jsedwards
-- Hopefully this is the one... at least for awhile.
--
-- Revision 1.11  2005/11/24 15:05:08  jsedwards
-- Recombined and rearranged some more.
--
-- Revision 1.10  2005/11/24 14:17:31  jsedwards
-- Recombined simple and complex objects (only difference was versioning).
-- Rearraged everything yet again.
--
-- Revision 1.9  2005/11/24 13:20:52  jsedwards
-- Completely rearranged headers into file, object and version.
--
-- Revision 1.8  2005/11/22 14:07:04  jsedwards
-- Split object identifier (upper 4 bits) into two identifiers, one for
-- simple objects (8) and one for complex objects (9).
--
-- Revision 1.7  2005/11/22 13:41:13  jsedwards
-- Split common header out for first 16 bytes of header.
-- Added "is_void_reference" inline function.
-- Added MD5 object stuff.
-- More external references.
--
-- Revision 1.6  2005/11/12 15:41:32  jsedwards
-- Changed UINT64_TO_UINT8_8 macro to inline function.
-- Added external references for write object to disk and create ref list.
--
-- Revision 1.5  2005/11/12 15:07:45  jsedwards
-- Fixed syntax errors.  Added "identifer" field so we can tell what object
-- it is when it is in memory.  Put 64 bit macros back in that were removed
-- in an earlier change.
--
-- Revision 1.4  2005/11/02 13:52:30  jsedwards
-- Removed the "data_length" fields because we get that from the filesystem
-- in this implementation.
--
-- Revision 1.3  2005/11/02 13:51:21  jsedwards
-- Rearranged the order of the common header.  Split EveryObject into two
-- types of objects SimpleObject and ComplexObject.  Changed file name of
-- class definition object.
--
-- Revision 1.2  2004/11/03 14:30:04  jsedwards
-- Merged 'ObjHeader' into 'EveryObj' structure.  Made separate checksum for
--   'EveryObj' data.
--
-- Revision 1.1  2004/10/31 18:27:52  jsedwards
-- Moved object definitions from 'types.h' to here.
--
*/

#ifndef OBJECTIFY_H
#define OBJECTIFY_H

#include "config.h"
#include "time_stamp.h"
#include "types.h"

#define RELEASE_VERSION "Alpha_14"

#define FILE_BLOCK_SIZE 256

#define MAGIC_NUMBER "NWOS"

#define VERSION_STRING "0014"

#define UNENCRYPTED_FLAG 0x80


typedef union { uint8 id[4]; uint32 word; } ObjRef;


typedef struct {
    uint32    flags;                      /*   4 */  /* flags / version */
    ObjRef    id;                         /*   8 */  /* reference id */
    uint8     header_chksum[4];           /*  12 */  /* CRC32 of header */
    uint8     data_chksum[4];             /*  16 */  /* CRC32 of data   */
    TimeStamp creation_time;              /*  24 */
    TimeStamp access_time;                /*  32 */  /* used when in memory for caching */
    ObjRef    class_definition;           /*  36 */
} CommonHeader;


typedef struct {
    ObjRef    clone_of;                   /*   4 */
    ObjRef    references;                 /*   8 */
    ObjRef    context;                    /*  12 */
    ObjRef    created_by_user;            /*  16 */
    ObjRef    created_by_app;             /*  20 */
    ObjRef    prev_version;               /*  24 */
    ObjRef    next_version;               /*  28 */
} ObjectHeader;


typedef struct {
    CommonHeader common;                  /*  36 */
    ObjectHeader object;                  /*  64 */
} EveryObject;

typedef struct {
    CommonHeader common_header;           /*  36 */
    ObjRef       references[0];
} ReferenceList;


typedef struct {
    CommonHeader  header;                 /*   36 */
    uint32        index[256];             /*  ??? */
    ObjRef        references[256];        /* ???? */
} IndexedReferenceList;


#include "c_structs.h"



static inline bool is_void_reference(ObjRef* ref)
{
    int i;
    for (i = 0; i < sizeof(ObjRef); i++) if (ref->id[i] != 0) return false;
    return true;
}


static inline bool is_same_object(ObjRef* ref1, ObjRef* ref2)
{
    int i;
    for (i = 0; i < sizeof(ObjRef); i++) if (ref1->id[i] != ref2->id[i]) return false;
    return true;
}


static inline void copy_reference(ObjRef* dst, ObjRef* src)
{
    int i;
    for (i = 0; i < sizeof(ObjRef); i++) dst->id[i] = src->id[i];
}


static inline void void_reference(ObjRef* ref)
{
    int i;
    for (i = 0; i < sizeof(ObjRef); i++) ref->id[i] = 0;
}


/* NOTE: these only work when references are 4 bytes, need to fix when size is changed */
static inline uint32 nwos_ref_to_word(ObjRef* ref) 
{
  return ((uint32)ref->id[0] << 24) | ((uint32)ref->id[1] << 16) | ((uint32)ref->id[2] << 8) | (uint32)ref->id[3];
}

static inline void nwos_word_to_ref(uint32 word, ObjRef* ref) 
{
    ref->id[0] = word >> 24;
    ref->id[1] = word >> 16;
    ref->id[2] = word >> 8;
    ref->id[3] = word;
}


/* since linux doesn't seem to have this function */
#ifdef linux
extern size_t strlcpy(char* dst, char* src, size_t len);
#endif

extern void nwos_ref_to_name(ObjRef* ref, char name[]);

/* extern ObjRef store_object(EveryObject* obj, uint64 length); */

extern uint32 nwos_decode_variable_sized_count(uint8 count[4]);
extern void   nwos_encode_variable_sized_count(uint32 count, uint8 result[4]);

extern void nwos_initialize_objectify(uint8 bf_key[16], uint32 linear, uint32 serial, StorageType type, char* path);
extern void nwos_terminate_objectify();
extern void nwos_set_root_object(ObjRef* ref);
extern void nwos_flush_bit_maps();

extern size_t nwos_reference_list_size(ObjRef* ref);

extern void nwos_get_object_class(ObjRef* obj, ObjRef* object_class);
extern void nwos_generate_new_id(ObjRef* ref);
extern void nwos_set_sequential_blocks(uint32 approximate_blocks);

extern void nwos_add_to_references(ObjRef* ref, ObjRef* obj);
extern void nwos_add_to_reference_list(ObjRef* ref, ObjRef* ref_list);

extern bool nwos_find_class_definition(char* name, ObjRef* class_ref);

extern void nwos_create_reference_list(ObjRef* for_obj, ObjRef* ref_list);
extern void nwos_create_object(uint64 id, EveryObject* header);
extern void nwos_read_class_definition(ObjRef* ref, C_struct_Class_Definition* class_def);
extern void nwos_read_object_headers_from_disk(ObjRef* ref, EveryObject* header);
extern void nwos_read_object_from_disk(ObjRef* ref, void* object, size_t size);
extern void nwos_read_variable_sized_object_from_disk(ObjRef* ref, void* obj, size_t max_size, size_t (*size_function)(void*));
extern void nwos_read_reference_list_from_disk(ObjRef* ref, ReferenceList* object, size_t size);

typedef enum { CREATED_NEW, FOUND_EXISTING, ERROR_OCCURRED } ObjCreateResult;



/* word stuff */
extern bool nwos_find_language(char* name, ObjRef* ref);

extern bool nwos_find_spelling(char* spelling, ObjRef* ref);
extern ObjCreateResult nwos_create_spelling(char* spelling, ObjRef* ref);

extern bool nwos_find_word(char* word, ObjRef* thing_class, ObjRef* lang, ObjRef* ref);
extern ObjCreateResult nwos_create_word(char* word, ObjRef* thing, ObjRef* lang, ObjRef* ref);

extern bool nwos_find_abbreviation(char* abbr, ObjRef* thing, ObjRef* lang, ObjRef* ref);
extern ObjCreateResult nwos_create_abbreviation(char* abbr, ObjRef* thing, ObjRef* lang, ObjRef* ref);

extern bool nwos_is_quit_command(char* command);
extern bool nwos_ask_yes_or_no(char* statement, char* question);


/* name stuff */
extern bool            nwos_find_name(char* name, ObjRef* ref);
extern ObjCreateResult nwos_create_name(char* name, ObjRef* ref);
extern bool            nwos_name_to_string(ObjRef* ref, char* string, size_t size);

/* date stuff */
extern bool            nwos_is_leap_year(int year);
extern void            nwos_get_month_ref(uint8 month, ObjRef* ref);
extern ObjCreateResult nwos_create_year(uint16 year, ObjRef* ref);
extern ObjCreateResult nwos_create_date(uint16 year, uint8 month, uint8 day, ObjRef* ref);
extern bool            nwos_year_to_string(ObjRef* year_obj_ref, char* string, size_t size_of_string);
extern bool            nwos_month_number_to_string(ObjRef* month_obj_ref, char* string, size_t size_of_string);
extern bool            nwos_date_to_string(ObjRef* date_obj_ref, char* string, size_t size_of_string);

/* person stuff */
extern void nwos_add_person();
extern void nwos_list_persons();
extern void nwos_describe_person();
extern bool nwos_find_person();
extern bool nwos_any_persons_named(char* name);

/* phone stuff */
extern bool nwos_find_area_code(char* area_code, ObjRef* ref);
extern ObjCreateResult nwos_create_area_code(char* area_code, char* state_postal_code, ObjRef* ref);
extern bool nwos_is_valid_phone_number(char* number);
extern const char* nwos_invalid_phone_number_msg(char* number);
extern bool nwos_find_phone_number(char* area_code, char* phone_number, ObjRef* ref);
extern ObjCreateResult nwos_create_phone_number(char* area_code, char* phone_number, ObjRef* ref);
extern bool nwos_phone_number_to_string(ObjRef* ref, char* string, size_t size);
extern void nwos_add_mobile_phone(char* name, char* phone_number);
extern void nwos_find_phone_for_person(char* name);

/* address stuff */
extern bool nwos_find_state_postal_code(char* state_postal_code, ObjRef* ref);
extern bool nwos_find_state_from_postal_code(char* postal_code, ObjRef* ref);
extern bool nwos_find_state_from_name(char* name, ObjRef* ref);
extern bool nwos_any_states_named(char* name);
extern void nwos_describe_state(char* name);
extern ObjCreateResult nwos_create_us_city(char* city, ObjRef* state_ref, ObjRef* city_ref);
extern void nwos_add_city(ObjRef* city_ref);
extern void nwos_list_cities_in_state(char* state);

/* file stuff */
typedef enum { File_Not_Found, MD5_Sum_Match, MD5_Sum_Mismatch } CheckFileResult;

extern bool nwos_find_file_path(char* path, ObjRef* ref);
extern bool nwos_file_path_to_string(ObjRef* ref, char* string, size_t size);
extern bool nwos_file_path_to_file(ObjRef* ref, ObjRef* file_ref);
extern bool nwos_find_md5(uint8 md5[16], ObjRef* ref);
extern CheckFileResult nwos_check_file_md5sum(char* path);
extern ObjCreateResult nwos_create_file_without_storing_data(char* path, ObjRef* ref);
extern ObjCreateResult nwos_create_file(char* path, ObjRef* ref);
extern bool nwos_restore_file(ObjRef* file_ref, char* path);
#define MAX_FILES_PER_DISC_LIST 256
#define MAX_SIZE_DISC_LIST (sizeof(C_struct_Disc_List) + sizeof(ObjRef) * MAX_FILES_PER_DISC_LIST)
extern size_t nwos_get_disc_list_object_size(void* disc_list_obj);
extern bool nwos_find_disc_list(char id[12], ObjRef* ref);
extern bool nwos_find_matching_disc_list(ObjRef files[MAX_FILES_PER_DISC_LIST], ObjRef* ref);
extern ObjCreateResult nwos_create_disc_list(char id[12], ObjRef disc_list[MAX_FILES_PER_DISC_LIST], ObjRef* ref);
extern bool nwos_find_disc_copy(ObjRef* disc_list, int copy_num, ObjRef* ref);
extern ObjCreateResult nwos_create_disc_copy(ObjRef* disc_list, int copy_num, ObjRef* location, ObjRef* ref);
extern bool nwos_find_storage_location(char* location, ObjRef* ref);
extern ObjCreateResult nwos_create_storage_location(char* location, ObjRef* ref);

extern bool nwos_is_valid_ssn(const char* ssn);

extern void nwos_get_key_from_password(uint8 key[], size_t key_size);

#endif

