/*             This file is part of the New World OS project
--                Copyright (C) 2005, 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: phone.c,v $
-- Revision 1.14  2006/11/11 12:01:06  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.13  2006/10/26 01:51:28  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.12.2.2  2006/10/25 12:22:29  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.12.2.1  2006/09/01 13:27:20  jsedwards
-- Changed "nwos_object_size" to "nwos_reference_list_size" and added the
-- object reference to "nwos_fill_in_common_header" so it can put the "id"
-- in the header now.
--
-- Revision 1.12  2006/01/12 03:43:07  jsedwards
-- Commented out print statement.
--
-- Revision 1.11  2006/01/10 14:09:17  jsedwards
-- Added "find phone for person" routine.
--
-- Revision 1.10  2006/01/01 21:49:01  jsedwards
-- Moved date, phone, us_state, and word class creations out of "big bang"
-- and into the respective files.
--
-- Revision 1.9  2006/01/01 21:08:38  jsedwards
-- Removed debugging print statement and added a blank line before printing
-- give up (can't find person and phone number looks bad) message.
--
-- Revision 1.8  2006/01/01 19:55:24  jsedwards
-- Added "parse phone number" and "parse phone number wo area code" routines
-- to deal with phone numbers and reworked the whole problems with phone
-- numbers section of getting a mobile phone.
--
-- Revision 1.7  2006/01/01 14:52:51  jsedwards
-- Added phone number creation stuff in part taken from "create_phone.c".
--
-- Revision 1.6  2005/12/29 17:50:53  jsedwards
-- Commented out printf debugging statements, that aren't useful now.
--
-- Revision 1.5  2005/12/27 18:32:36  jsedwards
-- Changed to look up class definition instead of using a fixed file name.
-- Also changed so that object id is random instead of based upon contents.
--
-- Revision 1.4  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.3  2005/12/21 23:31:07  jsedwards
-- Added create_phone_number, find_phone_number, check phone number, etc.
--
-- Revision 1.2  2005/12/21 17:10:31  jsedwards
-- Add 'nwos' to 'create_all_area_codes' routine name.
--
-- Revision 1.1  2005/12/21 16:55:01  jsedwards
-- Initial version that adds all US area codes.
--
*/


#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"
#include "objectify_private.h"

/*----------------------------------*/
/* The Phone and Phone Number stuff */
/*----------------------------------*/

void nwos_setup_phone()
{
    ObjRef reference;

    printf ("Creating Area Code class definition.\n");
    nwos_create_class_definition("AREA CODE");

    printf ("Creating all of the Area Code objects.\n");
    nwos_create_all_area_codes();

    printf ("Creating Phone Number class definition.\n");
    nwos_create_class_definition("PHONE NUMBER");

    printf ("Creating Home Phone class definition.\n");
    nwos_create_class_definition("HOME PHONE");
    assert(nwos_find_class_definition("HOME PHONE", &reference));

    /* nwos_create_word("Home", &reference, &english_lang_ref, &phone_ref); */

    printf ("Creating Work Phone class definition.\n");
    nwos_create_class_definition("WORK PHONE");
    assert(nwos_find_class_definition("WORK PHONE", &reference));

    /* nwos_create_word("Work", &reference, &english_lang_ref, &phone_ref); */

    printf ("Creating Mobile Phone class definition.\n");
    nwos_create_class_definition("MOBILE PHONE");
    assert(nwos_find_class_definition("MOBILE PHONE", &reference));

    /* nwos_create_word("Mobile", &reference, &english_lang_ref, &phone_ref); */
    /* nwos_create_word("Cell", &reference, &english_lang_ref, &phone_ref); */
}


bool nwos_find_area_code(char* area_code, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Area_Code area_code_obj;
    ObjRef reference;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_area_codes;
    int i;

    assert(strlen(area_code) == 3);

    assert(isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]));

    /* first find out if we already have this area_code */

    assert(nwos_find_class_definition("AREA CODE", &reference));

    nwos_read_class_definition(&reference, &class_def_obj);

    ref_list_size = nwos_reference_list_size(&class_def_obj.header.object.references);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(&class_def_obj.header.object.references, ref_list, ref_list_size);

    num_area_codes = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    /* printf("num_area_codes: %d\n", num_area_codes); */

    for (i = 0; i < num_area_codes; i++)
    {
	nwos_read_object_from_disk(&ref_list->references[i], &area_code_obj, sizeof(area_code_obj));

	if (strncmp(area_code_obj.storage, area_code, 3) == 0)   /* found a match */
	{
	    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
	    break;
	}
    }

    free(ref_list);
    ref_list = NULL;

    return (i < num_area_codes);  /* return true if we found it */
}


ObjCreateResult nwos_create_area_code(char* area_code, char* state_code, ObjRef* ref)
{
    C_struct_Area_Code area_code_obj;
    ObjRef reference;
    ObjCreateResult result = FOUND_EXISTING;

    assert(strlen(area_code) == 3);
    assert(isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]));
    assert(strlen(state_code) == 2);
    assert(isalpha(state_code[0]) && isalpha(state_code[1]));

    assert(nwos_find_class_definition("AREA CODE", &reference));

#if 0
    if (!nwos_find_area_code(area_code, ref) == false)   /* didn't find it */
#endif

    assert(!nwos_find_area_code(area_code, ref));

    {
	memset(&area_code_obj, 0, sizeof(area_code_obj));  /* zero it out */

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&area_code_obj.header.common, ref, &reference);

	memcpy(area_code_obj.storage, area_code, 3);

	nwos_create_reference_list(ref, &area_code_obj.header.object.references);

	memcpy(&area_code_obj.storage, area_code, 3);

	nwos_find_state_from_postal_code(state_code, &area_code_obj.state);
	nwos_add_to_references(ref, &area_code_obj.state);

	nwos_crc32_calculate((uint8*) &area_code_obj.header.object, sizeof(ObjectHeader), area_code_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &area_code_obj.state, sizeof(area_code_obj) - sizeof(EveryObject), area_code_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &area_code_obj, sizeof(area_code_obj));

        nwos_add_to_references(ref, &reference);

	result = CREATED_NEW;
    }

    return result;
}


bool nwos_is_valid_phone_number(char* number)
{
    return nwos_invalid_phone_number_msg(number) == NULL;
}

const char* nwos_invalid_phone_number_msg(char* number)
{
    static char msg[64];
    int i;
    size_t length = strlen(number);
    bool separator = false;

    for (i = 0; number[i] != '\0'; i++)
    {
	if (!isdigit(number[i]))
	{
	    /* ok to have hyphen or period in the fourth place*/
	    if (number[i] == '-' || number[i] == '.')
	    {
		if (separator)  /* already have one separator */
		{
		    return "multiple separators (- or .)";
		}

		if (i != 3)
		{
		    sprintf(msg, "separator '%c' in the wrong place", number[i]);
		    return msg;
		}

		separator = true;
	    }
	    else
	    {
		if (isgraph(number[i]))
		{
		    sprintf(msg, "unexpected character: %c\n", number[i]);
		}
		else
		{
		    sprintf(msg, "unexpected character: 0x%02x\n", (uint8)number[i]);
		}

		return msg;
	    }
	}
    }

    if (separator)
    {
	if (length < 8) return "not enough digits";
	if (length > 8) return "too many digits";
    }
    else
    {
	if (length < 7) return "not enough digits";
	if (length > 7) return "too many digits";
    }

    return NULL;   /* no errors */
}


static void copy_phone_number(char number[7], char* phone_number)
{
    assert(nwos_is_valid_phone_number(phone_number));

    memcpy(number, phone_number, 3);   /* copy first 3 */
    if (!isdigit(phone_number[3]))
    {
	memcpy(number+3, phone_number+4, 4);
    }
    else
    {
	memcpy(number+3, phone_number+3, 4);
    }
}


bool nwos_find_phone_number(char* area_code, char* phone_number, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Phone_Number phone_number_obj;
    ObjRef reference;
    ObjRef area_code_ref;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_phone_numbers;
    int i;
    char number[7];
    bool result = true;

    assert(nwos_is_valid_phone_number(phone_number));

    copy_phone_number(number, phone_number);

    if (area_code != NULL)
    {
	result = nwos_find_area_code(area_code, &area_code_ref);
    }

    if (result)
    {
	assert(nwos_find_class_definition("PHONE NUMBER", &reference));

	nwos_read_class_definition(&reference, &class_def_obj);

	ref_list_size = nwos_reference_list_size(&class_def_obj.header.object.references);

	ref_list = malloc(ref_list_size);

	if (ref_list == NULL) 
	{
	    perror("reading reference list");
	    exit(1);
	}

	nwos_read_reference_list_from_disk(&class_def_obj.header.object.references, ref_list, ref_list_size);

	num_phone_numbers = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

	/* printf("num_phone_numbers: %d\n", num_phone_numbers); */

	result = false;

	for (i = 0; i < num_phone_numbers; i++)
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &phone_number_obj, sizeof(phone_number_obj));

	    if (area_code == NULL || is_same_object(&phone_number_obj.area_code, &area_code_ref))
	    {
		if (memcmp(phone_number_obj.storage, number, 7) == 0)   /* found a match */
		{
		    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		    result = true;
		    break;
		}
	    }
	}

	free(ref_list);
	ref_list = NULL;
    }

    return result;
}


ObjCreateResult nwos_create_phone_number(char* area_code, char* phone_number, ObjRef* ref)
{
    C_struct_Phone_Number phone_number_obj;
    ObjRef reference;

    assert(nwos_is_valid_phone_number(phone_number));

    assert(!nwos_find_phone_number(area_code, phone_number, ref));

    memset(&phone_number_obj, 0, sizeof(phone_number_obj));  /* zero it out */

    assert(nwos_find_class_definition("PHONE NUMBER", &reference));

    nwos_generate_new_id(ref);

    nwos_fill_in_common_header(&phone_number_obj.header.common, ref, &reference);

    nwos_create_reference_list(ref, &phone_number_obj.header.object.references);

    copy_phone_number(phone_number_obj.storage, phone_number);

    if (!nwos_find_area_code(area_code, &phone_number_obj.area_code))   /* didn't find it */
    {
	fprintf(stderr, "Unknown area code: %s\n", area_code);
	exit(1);
    }
    nwos_add_to_references(ref, &phone_number_obj.area_code);

    nwos_crc32_calculate((uint8*) &phone_number_obj.header.object, sizeof(ObjectHeader), phone_number_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &phone_number_obj.country, sizeof(phone_number_obj) - sizeof(EveryObject), phone_number_obj.header.common.data_chksum);

    nwos_write_object_to_disk(ref, &phone_number_obj, sizeof(phone_number_obj));

    nwos_add_to_references(ref, &reference);

    return CREATED_NEW;
}


bool nwos_phone_number_to_string(ObjRef* ref, char* string, size_t size)
{
    C_struct_Phone_Number number_obj;
    C_struct_Area_Code area_code_obj;
    int i;
    int j;

    nwos_read_object_from_disk(ref, &number_obj, sizeof(number_obj));

    nwos_read_object_from_disk(&number_obj.area_code, &area_code_obj, sizeof(area_code_obj));

    i = 0;

    assert(i < size);
    string[i++] = '(';

    for (j = 0; j < 3; j++) 
    {
	assert(i < size);
	string[i++] = area_code_obj.storage[j];
    }

    assert(i < size);
    string[i++] = ')';

    assert(i < size);
    string[i++] = ' ';

    for (j = 0; j < 3; j++) 
    {
	assert(i < size);
	string[i++] = number_obj.storage[j];
    }

    assert(i < size);
    string[i++] = '-';

    for (j = 3; j < 7; j++) 
    {
	assert(i < size);
	string[i++] = number_obj.storage[j];
    }

    assert(i < size);
    string[i++] = '\0';

    return true;
}




/* Create all known area codes for the US */

typedef struct {
  char state_code[3];
  char area_code[4];
} area_code_struct;

static area_code_struct area_code_info[] = {
  { "AL", "205" },
  { "AL", "251" },
  { "AL", "256" },
  { "AL", "334" },
  { "AK", "907" },
  { "AZ", "480" },
  { "AZ", "520" },
  { "AZ", "602" },
  { "AZ", "623" },
  { "AZ", "928" },
  { "AR", "501" },
  { "AR", "870" },
  { "CA", "209" },
  { "CA", "213" },
  { "CA", "310" },
  { "CA", "323" },
  { "CA", "408" },
  { "CA", "415" },
  { "CA", "510" },
  { "CA", "530" },
  { "CA", "559" },
  { "CA", "562" },
  { "CA", "619" },
  { "CA", "626" },
  { "CA", "650" },
  { "CA", "661" },
  { "CA", "707" },
  { "CA", "714" },
  { "CA", "760" },
  { "CA", "805" },
  { "CA", "818" },
  { "CA", "831" },
  { "CA", "858" },
  { "CA", "909" },
  { "CA", "916" },
  { "CA", "925" },
  { "CA", "949" },
  { "CO", "303" },
  { "CO", "719" },
  { "CO", "720" },
  { "CO", "970" },
  { "CT", "203" },
  { "CT", "860" },
  { "DE", "302" },
  { "FL", "305" },
  { "FL", "321" },
  { "FL", "352" },
  { "FL", "386" },
  { "FL", "407" },
  { "FL", "561" },
  { "FL", "727" },
  { "FL", "754" },
  { "FL", "772" },
  { "FL", "786" },
  { "FL", "813" },
  { "FL", "850" },
  { "FL", "863" },
  { "FL", "904" },
  { "FL", "941" },
  { "FL", "954" },
  { "GA", "229" },
  { "GA", "404" },
  { "GA", "478" },
  { "GA", "678" },
  { "GA", "706" },
  { "GA", "770" },
  { "GA", "912" },
  { "HI", "808" },
  { "ID", "208" },
  { "IL", "217" },
  { "IL", "309" },
  { "IL", "312" },
  { "IL", "618" },
  { "IL", "630" },
  { "IL", "708" },
  { "IL", "773" },
  { "IL", "815" },
  { "IL", "847" },
  { "IN", "219" },
  { "IN", "260" },
  { "IN", "317" },
  { "IN", "574" },
  { "IN", "765" },
  { "IN", "812" },
  { "IA", "319" },
  { "IA", "515" },
  { "IA", "563" },
  { "IA", "641" },
  { "IA", "712" },
  { "KS", "316" },
  { "KS", "620" },
  { "KS", "785" },
  { "KS", "913" },
  { "KY", "270" },
  { "KY", "502" },
  { "KY", "606" },
  { "KY", "859" },
  { "LA", "225" },
  { "LA", "318" },
  { "LA", "337" },
  { "LA", "504" },
  { "LA", "985" },
  { "ME", "207" },
  { "MD", "240" },
  { "MD", "301" },
  { "MD", "410" },
  { "MD", "443" },
  { "MA", "339" },
  { "MA", "351" },
  { "MA", "413" },
  { "MA", "508" },
  { "MA", "617" },
  { "MA", "774" },
  { "MA", "781" },
  { "MA", "857" },
  { "MA", "978" },
  { "MI", "231" },
  { "MI", "248" },
  { "MI", "269" },
  { "MI", "313" },
  { "MI", "517" },
  { "MI", "586" },
  { "MI", "616" },
  { "MI", "734" },
  { "MI", "810" },
  { "MI", "906" },
  { "MI", "989" },
  { "MN", "218" },
  { "MN", "320" },
  { "MN", "507" },
  { "MN", "612" },
  { "MN", "651" },
  { "MN", "763" },
  { "MN", "952" },
  { "MS", "228" },
  { "MS", "601" },
  { "MS", "662" },
  { "MO", "314" },
  { "MO", "417" },
  { "MO", "573" },
  { "MO", "636" },
  { "MO", "660" },
  { "MO", "816" },
  { "MT", "406" },
  { "NE", "308" },
  { "NE", "402" },
  { "NV", "702" },
  { "NV", "775" },
  { "NH", "603" },
  { "NJ", "201" },
  { "NJ", "609" },
  { "NJ", "732" },
  { "NJ", "856" },
  { "NJ", "908" },
  { "NJ", "973" },
  { "NM", "505" },
  { "NY", "212" },
  { "NY", "315" },
  { "NY", "347" },
  { "NY", "516" },
  { "NY", "518" },
  { "NY", "607" },
  { "NY", "631" },
  { "NY", "646" },
  { "NY", "716" },
  { "NY", "718" },
  { "NY", "845" },
  { "NY", "914" },
  { "NY", "917" },
  { "NC", "252" },
  { "NC", "336" },
  { "NC", "704" },
  { "NC", "828" },
  { "NC", "910" },
  { "NC", "919" },
  { "NC", "980" },
  { "ND", "701" },
  { "OH", "216" },
  { "OH", "234" },
  { "OH", "330" },
  { "OH", "419" },
  { "OH", "440" },
  { "OH", "513" },
  { "OH", "614" },
  { "OH", "740" },
  { "OH", "937" },
  { "OK", "405" },
  { "OK", "580" },
  { "OK", "918" },
  { "OR", "503" },
  { "OR", "541" },
  { "OR", "971" },
  { "PA", "215" },
  { "PA", "267" },
  { "PA", "412" },
  { "PA", "484" },
  { "PA", "570" },
  { "PA", "610" },
  { "PA", "717" },
  { "PA", "724" },
  { "PA", "814" },
  { "PA", "878" },
  { "RI", "401" },
  { "SC", "803" },
  { "SC", "843" },
  { "SC", "864" },
  { "SD", "605" },
  { "TN", "423" },
  { "TN", "615" },
  { "TN", "731" },
  { "TN", "865" },
  { "TN", "901" },
  { "TN", "931" },
  { "TX", "210" },
  { "TX", "214" },
  { "TX", "254" },
  { "TX", "281" },
  { "TX", "361" },
  { "TX", "409" },
  { "TX", "469" },
  { "TX", "512" },
  { "TX", "682" },
  { "TX", "713" },
  { "TX", "806" },
  { "TX", "817" },
  { "TX", "830" },
  { "TX", "832" },
  { "TX", "903" },
  { "TX", "915" },
  { "TX", "936" },
  { "TX", "940" },
  { "TX", "956" },
  { "TX", "972" },
  { "TX", "979" },
  { "UT", "435" },
  { "UT", "801" },
  { "VT", "802" },
  { "VA", "276" },
  { "VA", "434" },
  { "VA", "540" },
  { "VA", "571" },
  { "VA", "703" },
  { "VA", "757" },
  { "VA", "804" },
  { "WA", "206" },
  { "WA", "253" },
  { "WA", "360" },
  { "WA", "425" },
  { "WA", "509" },
  { "WV", "304" },
  { "WI", "262" },
  { "WI", "414" },
  { "WI", "608" },
  { "WI", "715" },
  { "WI", "920" },
  { "WY", "307" },
};

#define NUM_AREA_CODES  (sizeof(area_code_info) / sizeof(area_code_struct))

void nwos_create_all_area_codes()
{
  int i;
  ObjRef ref;

  for (i = 0; i < NUM_AREA_CODES; i++)
    {
      nwos_create_area_code(area_code_info[i].area_code, area_code_info[i].state_code, &ref);
    }

  /* need to fix this to create the toll free and 900 numbers */
}



/* FIX THIS! - need a common get input routine, instead of duplicating this in every file! */

static void get_input(char* descr, char* buffer, size_t size)
{
    char *ptr;

    while (1)
    {
	printf("%s: ", descr);
	fflush(stdout);
	fgets(buffer, size, stdin);
	ptr = strchr(buffer, '\n');
	if (ptr != NULL)
	{
	    *ptr = '\0';
	    break;
	}
	do { fgets(buffer, size, stdin); } while (strchr(buffer, '\n') == NULL);
	printf("input too long - try again!\n");
    }
}



static bool parse_phone_number(char* phone_number, char area_code[], char number[])
{
    static char* patterns[] =
    {
	"(AAA) NNN-NNNN",
	"AAA-NNN-NNNN",
	"AAA.NNN.NNNN",
	"AAA NNN NNNN",
    };
    const int num_patterns = sizeof(patterns) / sizeof(char*);
    char* ptrpat;
    char* ptrin;
    char* ptrac;
    char* ptrnum;
    int i;

    for (i = 0; i < num_patterns; i++)
    {
	ptrpat = patterns[i];
	ptrin = phone_number;
	ptrac = area_code;
	ptrnum = number;

	while (ptrpat != NULL && *ptrpat != '\0' && *ptrin != '\0')
	{
	    switch (*ptrpat)
	    {
	      case 'A':
		if (isdigit(*ptrin))
		{
		    *ptrac = *ptrin;
		    ptrin++;
		    ptrac++;
		    ptrpat++;
		}
		else    /* no match */
		{
		    ptrpat = NULL;
		}
		break;

	      case ' ':   /* spaces are optional */
		while (*ptrin == ' ') ptrin++;
		ptrpat++;
		break;

	      case 'N':
		if (isalnum(*ptrin))
		{
		    *ptrnum = *ptrin;
		    ptrin++;
		    ptrnum++;
		    ptrpat++;
		}
		else    /* no match */
		{
		    ptrpat = NULL;
		}
		break;

	      default:    /* verify it matches the pattern exactly */
		if (*ptrin == *ptrpat)
		{
		    ptrin++;
		    ptrpat++;
		}
		else   /* no match */
		{
		    ptrpat = NULL;
		}
		break;
	    }
	}

	if (ptrpat != NULL && *ptrpat == '\0' && *ptrin == '\0')    /* it matched */
	{
	    *ptrac = '\0';
	    *ptrnum = '\0';
	    break;
	}
    }

    assert(i == num_patterns || strlen(area_code) == 3);
    assert(i == num_patterns || strlen(number) == 7);

    return (i < num_patterns);
}


static bool parse_phone_number_wo_area_code(char* phone_number, char number[])
{
    static char* patterns[] =
    {
	"NNN-NNNN",
	"NNN.NNNN",
	"NNN NNNN",
    };
    const int num_patterns = sizeof(patterns) / sizeof(char*);
    char* ptrpat;
    char* ptrin;
    char* ptrnum;
    int i;

    for (i = 0; i < num_patterns; i++)
    {
	ptrpat = patterns[i];
	ptrin = phone_number;
	ptrnum = number;

	while (ptrpat != NULL && *ptrpat != '\0' && *ptrin != '\0')
	{
	    switch (*ptrpat)
	    {
	      case ' ':   /* spaces are optional */
		while (*ptrin == ' ') ptrin++;
		ptrpat++;
		break;

	      case 'N':
		if (isalnum(*ptrin))
		{
		    *ptrnum = *ptrin;
		    ptrin++;
		    ptrnum++;
		    ptrpat++;
		}
		else    /* no match */
		{
		    ptrpat = NULL;
		}
		break;

	      default:    /* verify it matches the pattern exactly */
		if (*ptrin == *ptrpat)
		{
		    ptrin++;
		    ptrpat++;
		}
		else   /* no match */
		{
		    ptrpat = NULL;
		}
		break;
	    }
	}

	if (ptrpat != NULL && *ptrpat == '\0' && *ptrin == '\0')    /* it matched */
	{
	    *ptrnum = '\0';
	    break;
	}
    }

    assert(i == num_patterns || strlen(number) == 7);

    return (i < num_patterns);
}


void nwos_add_mobile_phone(char* name, char* phone_number)
{
    ObjRef reference;
    char area_code[16];
    char number[16];
    C_struct_Area_Code area_code_obj;
    C_struct_US_State state_obj;
    C_struct_Mobile_Phone phone_obj;
    ObjRef area_code_ref;
    ObjRef phone_class_ref;
    ObjRef person_ref;
    char temp[64];
    bool number_ok;
    int i;

    /* parse the number first in case there is something wrong with it */

    number_ok = parse_phone_number(phone_number, area_code, number);

    if (!number_ok)    /* try parsing without the area code */
    {
	if (parse_phone_number_wo_area_code(phone_number, number))
	{
	    while (!number_ok)
	    {
		get_input("What is the area code (enter `quit' to return to main command)", area_code, 16);

		printf("\n");

		if (nwos_is_quit_command(area_code))    /* want's to bail out */
		{
		    printf("Ok, returning to main command.\n");
		    return;
		}

		if (isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]) && area_code[3] == '\0')
		{
		    number_ok = true;
		}
		else
		{
		    printf("Invalid area code: %s.  Please re-enter.\n\n", area_code);
		}
	    }
	}
    }

    while (number_ok && !nwos_find_area_code(area_code, &area_code_ref))
    {
	char* unimplemented_area_codes[] = { "800", "866", "877", "900" };
	const int num_unimplemented_area_codes = sizeof(unimplemented_area_codes) / sizeof(char*);
	for (i = 0; i < num_unimplemented_area_codes; i++)
	{
	    if (strcmp(area_code, unimplemented_area_codes[i]) == 0)
	    {
		printf("I'm sorry, this version of software cannot deal with toll free or pay\n");
		printf("numbers such as %s.  Returning to main command.\n", area_code);
		return;
	    }
	}

	printf("That area code is not in the database.  Unfortunately this version of the\n");
	printf("software does not have the capability to add an area code.  If it was a\n");
	printf("mistake, please re-enter the correct area code.  If not please check for\n");
	printf("a newer version of this software at http://sourceforge.net/projects/nwos/.\n");
	printf("\n");

	number_ok = false;

	while (!number_ok)
	{
	    get_input("What is the area code (enter `quit' to return to main command)", area_code, 16);

	    printf("\n");

	    if (nwos_is_quit_command(area_code))    /* want's to bail out */
	    {
		printf("Ok, returning to main command.\n");
		return;
	    }

	    if (isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]) && area_code[3] == '\0')
	    {
		number_ok = true;
	    }
	    else
	    {
		printf("Invalid area code: %s.  Please re-enter.\n\n", area_code);
	    }
	}
    }

    /* Not sure about the phone number at this point, let's see about the person */

    if (!nwos_find_person(name, &person_ref))
    {
	if (!number_ok)
	{
	    printf("I'm sorry I can't find %s in the database and there is something\n", name);
	    printf("wrong with the phone number given: %s.  Please try again.\n", phone_number);
	}

	return;
    }

    /* At this point we have a person, so make sure the phone number is sane */

    if (!number_ok)
    {
	if (!nwos_ask_yes_or_no("I can't figure out the phone number given", "Would you like to re-enter it"))
	{
	    printf("Ok, returning to main command.\n");
	    return;
	}

	while (!number_ok)
	{
	    get_input("What is the area code", area_code, 16);

	    printf("\n");

	    if (nwos_is_quit_command(area_code))    /* want's to bail out */
	    {
		printf("Ok, returning to main command.\n");
		return;
	    }

	    if (isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]) && area_code[3] == '\0')
	    {
		number_ok = true;
	    }
	    else
	    {
		printf("Invalid area code: %s.  Please re-enter.\n\n", area_code);
	    }
	}

	number_ok = false;

	while (!number_ok)
	{
	    get_input("What is the phone number", number, 16);

	    number_ok = nwos_is_valid_phone_number(number);

	    if (!number_ok)
	    {
		printf("Invalid phone number: %s - %s\n", number, nwos_invalid_phone_number_msg(number));
		printf("Please try again.\n");
	    }
	}
    }

    nwos_read_object_from_disk(&area_code_ref, &area_code_obj, sizeof(area_code_obj));

    nwos_read_object_from_disk(&area_code_obj.state, &state_obj, sizeof(state_obj));

    nwos_name_to_string(&state_obj.name, temp, 64);

    printf("Area code is in state: %s\n", temp);


    assert(nwos_find_class_definition("MOBILE PHONE", &phone_class_ref));

    memset(&phone_obj, 0, sizeof(phone_obj));

    nwos_generate_new_id(&reference);

    nwos_fill_in_common_header(&phone_obj.header.common, &reference, &phone_class_ref);

    memcpy(&phone_obj.person, &person_ref, sizeof(ObjRef));

    if (nwos_find_phone_number(area_code, number, &phone_obj.number))
    {
	printf("Sorry, that phone number already exists!\n");
    }
    else
      {
	nwos_create_phone_number(area_code, number, &phone_obj.number);

	nwos_crc32_calculate((uint8*) &phone_obj.header.object, sizeof(ObjectHeader), phone_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &phone_obj.person, sizeof(phone_obj) - sizeof(EveryObject), phone_obj.header.common.data_chksum);

	nwos_write_object_to_disk(&reference, &phone_obj, sizeof(phone_obj));

	nwos_create_reference_list(&reference, &phone_obj.header.object.references);

	nwos_add_to_references(&reference, &phone_obj.number);
	nwos_add_to_references(&reference, &phone_obj.person);
	    
	nwos_add_to_references(&reference, &phone_class_ref);
	
	printf("Mobile phone created: %02x%02x%02x%02x\n", reference.id[0], reference.id[1], reference.id[2], reference.id[3]);
    }
}


void nwos_find_phone_for_person(char* name)
{
    ObjRef person_ref;
    ObjRef phone_class_ref;
    ObjRef object_class;
    EveryObject header;
    C_struct_Mobile_Phone phone_obj;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;
    int i;
    char temp[20];

    if (!nwos_find_person(name, &person_ref))
    {
	printf("I'm sorry I can't find anyone with the name %s.\n", name);
	return;
    }

    /**********************************************/
    /* Now look for a phone in the reference list */
    /**********************************************/

    nwos_read_object_headers_from_disk(&person_ref, &header);

    ref_list_size = nwos_reference_list_size(&header.object.references);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading name reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(&header.object.references, ref_list, ref_list_size);

    num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    /* printf("person num refs: %d\n", num_refs); */

    assert(nwos_find_class_definition("MOBILE PHONE", &phone_class_ref));

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, &phone_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &phone_obj, sizeof(phone_obj));

	    nwos_phone_number_to_string(&phone_obj.number, temp, 20);
	    printf("%s ", temp);
	}

	printf("\n");
    }
}
