/*             This file is part of the New World OS project
--                   Copyright (C) 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: fix_md5.c,v $
-- Revision 1.2  2006/11/11 12:29:19  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.1  2006/10/27 12:12:36  jsedwards
-- One time use program to repair md5 sums that were calculated incorrectly
-- when the files were imported (on a 64 bit processor).
--
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "crc32.h"
#include "objectify_private.h"


uint8 char_to_hex(char c)
{
    if ('0' <= c && c <= '9') return c - '0';

    if ('a' <= c && c <= 'f') return c - 'a' + 10;

    return 0xff;
}


bool convert_string_to_md5(char* str, uint8 md5[16])
{
    int i;
    uint8 lower;
    uint8 upper;

    if (strlen(str) != 32) return false;

    for (i = 0; i < 16; i++)
    {
	upper = char_to_hex(str[i*2]);
	lower = char_to_hex(str[i*2 + 1]);

	if (upper > 15 || lower > 15) return false;

	md5[i] = (upper << 4) | lower;
    }

    return true;
}


int main(int argc, char* argv[])
{
    ObjRef root_object_ref;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    StorageType type = DEFAULT_TYPE;
    char* path = DEFAULT_FILE;
    uint8 bad[16];
    uint8 good[16];
    int i;
    int j;
    ObjRef ref;
    C_struct_MD5sum md5_object;

    if (argc < 3 || (argc & 1) == 0)
    {
	fprintf(stderr, "usage: %s bad-md5 good-md5 {bad-md5 good-md5}\n", argv[0]);
	exit(1);
    }

    printf("\n");

    nwos_get_key_from_password(big_key, sizeof(big_key));

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_initialize_objectify(bf_key, linear, serial, type, path);

    nwos_set_root_object(&root_object_ref);

    for (i = 1; i < argc; i += 2)
    {
	if (!convert_string_to_md5(argv[i], bad))
	{
	    fprintf(stderr, "bad-md5 invalid: %s\n", argv[i]);
	    exit(1);
	}

	if (!convert_string_to_md5(argv[i+1], good))
	{
	    fprintf(stderr, "good-md5 invalid: %s\n", argv[i+1]);
	    exit(1);
	}

	printf("bad: ");
	for (j = 0; j < 16; j++) printf("%02x", bad[j]);

	if (nwos_find_md5(bad, &ref))
	{
	    printf(": %02x%02x%02x%02x\n", ref.id[0], ref.id[1], ref.id[2], ref.id[3]);

	    nwos_read_object_from_disk(&ref, &md5_object, sizeof(md5_object));

	    if (memcmp(bad, md5_object.md5sum, sizeof(bad)) != 0)
	    {
		fprintf(stderr, "ERROR - object md5 doesn't match: ");
		for (j = 0; j < 16; j++) fprintf(stderr, "%02x", md5_object.md5sum[j]);
		fprintf(stderr, "\n");
		break;
	    }

	    memcpy(md5_object.md5sum, good, sizeof(md5_object.md5sum));

	    nwos_crc32_calculate((uint8*) &md5_object.md5sum,
				 sizeof(C_struct_MD5sum) - sizeof(EveryObject),
				 md5_object.header.common.data_chksum);

	    nwos_overwrite_object_to_disk(&ref, &md5_object, sizeof(md5_object));
	    
	    printf(" -> ");
	    for (j = 0; j < 16; j++) printf("%02x", good[j]);
	    printf("\n");
	}
	else
	{
	    printf(": Not found\n");
	}
    }

    nwos_terminate_objectify();

    return 0;
}


