/*
--             This file is part of the New World OS project
--                 Copyright (C) 2004-2007  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: objectify.h,v $
-- Revision 1.118  2007/07/29 18:08:34  jsedwards
-- Removed VERSION_STRING because it is generated in configure.ac now.
--
-- Revision 1.117  2007/07/23 23:48:59  jsedwards
-- Added CREATED_NEW_REVISION, PATH_NOT_FOUND, and MULTIPLE_FILES to
-- ObjCreateResult enumeration.
--
-- Revision 1.116  2007/07/21 14:53:27  jsedwards
-- Added 'match' parameter to nwos_file_is_identical so it can optionally
-- ignore the time stamp on the file.
--
-- Revision 1.115  2007/07/15 17:03:53  jsedwards
-- Added inline functions byteswap_uint16 and byteswap_uint32 to reverse the
-- bytes in a uint16 and uint32.
--
-- Revision 1.114  2007/07/13 20:57:58  jsedwards
-- Removed RELEASE_VERSION, instead use PACKAGE_VERSION generated by configure
-- script in config.h.
--
-- Revision 1.113  2007/07/12 03:14:58  jsedwards
-- Added external declaration for nwos_add_birthday function.
--
-- Revision 1.112  2007/07/07 20:32:03  jsedwards
-- Moved declaration of nwos_log from objectify_private.h.
--
-- Revision 1.111  2007/07/03 00:43:15  jsedwards
-- Change release version to Alpha_24 and version string to 0024.  Also added
-- nwos_2_uint8_to_uint16 inline function.
--
-- Revision 1.110  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.109  2007/06/26 11:56:17  jsedwards
-- Changed to Version 23 (0023).
--
-- Revision 1.108  2007/06/25 16:16:27  jsedwards
-- Changed Release Version to 22.3.
--
-- Revision 1.107  2007/06/21 01:35:09  jsedwards
-- Add external declaration for nwos_get_object_size and defines for
-- maximum object size.
--
-- Revision 1.106  2007/06/20 00:27:10  jsedwards
-- Changed to version 0022 because header was changed.
--
-- Revision 1.105  2007/06/13 20:55:21  jsedwards
-- Added directory parameter to find_matching_path_and_file_association.
--
-- Revision 1.104  2007/06/13 20:16:50  jsedwards
-- Added a directory parameter to create_file_without_storing_data.
--
-- Revision 1.103  2007/06/04 13:42:25  jsedwards
-- Add 'match' paramter to find_matchine_path_and_file_association.
--
-- Revision 1.102  2007/05/28 15:02:40  jsedwards
-- Changed 'created_by_user' and 'created_by_app' to 'creation' and 'source'.
--
-- Revision 1.101  2007/05/26 14:50:23  jsedwards
-- Add external declaration for find_public_manufacturer.
--
-- Revision 1.100  2007/05/25 12:45:47  jsedwards
-- Added 'device' section and list all manufacturers external declaration.
--
-- Revision 1.99  2007/05/24 11:50:18  jsedwards
-- Add acronym and credit union external declarations.
--
-- Revision 1.98  2007/05/22 02:38:43  jsedwards
-- Change 'create_spelling' to 'create_public_spelling'.
--
-- Revision 1.97  2007/05/06 17:21:37  jsedwards
-- Add external declarations for add_account and list_accounts functions.
--
-- Revision 1.96  2007/04/23 13:35:38  jsedwards
-- Change to allow up to a maximum of 4096 files in a disc_list instead of 256.
--
-- Revision 1.95  2007/04/22 17:02:05  jsedwards
-- Added external declarations for url_to_string and list_public_credit_unions.
--
-- Revision 1.94  2007/04/12 02:31:56  jsedwards
-- Change release_string to Alpha_21.
--
-- Revision 1.93  2007/04/07 14:04:48  jsedwards
-- Split find area code function into public and private versions.
--
-- Revision 1.92  2007/04/07 02:16:33  jsedwards
-- Added external references for date and gender functions.
--
-- Revision 1.91  2007/03/27 11:41:53  jsedwards
-- Change version from 0020 to 0021 because the disk_header now has separate
-- last change times for public and private.
--
-- Revision 1.90  2007/03/23 14:17:48  jsedwards
-- Change find or create feature definition to void instead of bool.
--
-- Revision 1.89  2007/03/23 13:25:36  jsedwards
-- Added external declarations for find_public_feature_definition and
-- find_or_create_public_feature_definition.
--
-- Revision 1.88  2007/03/18 19:55:00  jsedwards
-- Added external definition of file_has_backup routine.
--
-- Revision 1.87  2007/03/09 13:42:33  jsedwards
-- Change version from 0019 to 0020 because using RESERVED_PUBLIC_BLOCKS for
-- the beginning of private ids (0019 subtracted total_public_blocks instead).
--
-- Revision 1.86  2007/03/08 13:06:48  jsedwards
-- Added the 4 base classes when c_structs.h is not available (when we are
-- compling 'export_c_structs').
--
-- Revision 1.85  2007/02/28 14:04:11  jsedwards
-- Added external declaration for is_file_stored function.
--
-- Revision 1.84  2007/02/28 13:37:48  jsedwards
-- Change version from 0018 to 0019.
--
-- Revision 1.83  2007/02/24 15:47:18  jsedwards
-- Changed version from 0017 to 0018.
--
-- Revision 1.82  2007/02/11 15:44:11  jsedwards
-- Add external declaration for strlcat which Linux doesn't seem to have.
--
-- Revision 1.81  2007/01/27 17:09:25  jsedwards
-- Added external declaration for file_is_identical function.
--
-- Revision 1.80  2007/01/27 15:44:04  jsedwards
-- Change check_file_md5sum function to check_file_checksums.
--
-- Revision 1.79  2007/01/27 15:25:21  jsedwards
-- Change CheckFileResult to include SHA1 results as well as MD5.
--
-- Revision 1.78  2007/01/27 14:57:28  jsedwards
-- Added "DUPLICATE_FILE" and "DUPLICATE_PATH" as possible object creation
-- results.
--
-- Revision 1.77  2007/01/17 13:14:00  jsedwards
-- Change Encryption_Fast level name to Encryption_Minimal.
--
-- Revision 1.76  2007/01/14 17:16:08  jsedwards
-- Added external declarations for number_of_files_for_path and changed
-- file_path_to_file to file_path_to_path_and_file_association.
--
-- Revision 1.75  2007/01/14 04:01:14  jsedwards
-- Change to pass modification_time to find path and file association.
--
-- Revision 1.74  2007/01/14 02:35:14  jsedwards
-- Changed version from 16 to 17 (0016 to 0017), because I moved the mod_time
-- from the 'file' object to the 'path_and_file_association' object.
--
-- Revision 1.73  2007/01/07 03:21:02  jsedwards
-- Added new Encryption_Fast and Encryption_Very_Low encryption levels and
-- external declarations for push and pop encryption level.
--
-- Revision 1.72  2007/01/06 20:19:03  jsedwards
-- Split find_name and find_spelling into find_public_name, find_private_name,
-- find_public_spelling, and find_private_spelling.  Added declaration for
-- create_private_spelling.
--
-- Revision 1.71  2007/01/05 13:40:21  jsedwards
-- Add external declarations for find_matching_path_and_file_association,
-- find_path_and_file_association, and find_or_create_path_and_file_association.
--
-- Revision 1.70  2007/01/03 14:26:51  jsedwards
-- Added external reference for find_or_create_private_class_definition
-- function.
--
-- Revision 1.69  2006/12/27 12:25:07  jsedwards
-- Add function to put program arguments into log.
--
-- Revision 1.68  2006/12/21 13:12:35  jsedwards
-- Comment out find_class_definition, at least for now.
--
-- Revision 1.67  2006/12/20 12:36:32  jsedwards
-- Add external definitions for find public and find private class definition
-- functions.
--
-- Revision 1.66  2006/12/16 13:09:21  jsedwards
-- Change version to 0016.
--
-- Revision 1.65  2006/12/15 14:57:33  jsedwards
-- Removed Encrypted_Reference type because now Private_Reference is encrypted
-- and there is no unencrypted private blocks.  Also removed the external
-- declaration of the generate_new_id_of_same_type function becaus it is no
-- longer needed (references are always private except when compiled in
-- PUBLIC_MODE).
--
-- Revision 1.64  2006/12/13 14:22:57  jsedwards
-- Added encryption level stuff.
--
-- Revision 1.63  2006/12/12 03:12:17  jsedwards
-- Added external declaration for generate_new_private_id function.
--
-- Revision 1.62  2006/12/10 19:30:03  jsedwards
-- Added external declarations for convert cardinal and ordinal numbers to
-- string and find year and find date routines.
--
-- Revision 1.61  2006/12/10 15:05:56  jsedwards
-- Change external declarations for ordinal functions to take cardinal
-- reference as parameter instead of number string.
--
-- Revision 1.60  2006/12/10 12:07:36  jsedwards
-- Change number routines to pass in character string instead of uint32.
--
-- Revision 1.59  2006/12/09 18:52:03  jsedwards
-- Add external declarations for number functions.
--
-- Revision 1.58  2006/12/05 04:21:24  jsedwards
-- Removed the UNENCRYPTED_FLAG as it is no longer used.
--
-- Revision 1.57  2006/12/04 14:57:17  jsedwards
-- Added two inline functions to convert from uint8[4] to uin32 and back.
--
-- Revision 1.56  2006/12/04 04:39:51  jsedwards
-- Changed version to 0015 and added external references for reference_type
-- and generate_new_id_of_same_type functions.
--
-- Revision 1.55  2006/12/01 14:27:47  jsedwards
-- Added a union in the CommonHeader to allow the number of references in a
-- reference list to use the header_chksum for storage, since the header_chksum
-- isn't used in a ReferenceList.  Also added external references for the new
-- malloc_reference_list and free_reference_list functions.
--
-- Revision 1.54  2006/12/01 05:08:19  jsedwards
-- Add an external declaration for strlcpy if compiling on linux, since it
-- doesn't have that function.
--
-- Revision 1.53  2006/11/29 18:30:46  jsedwards
-- Added external reference for file_path_to_file function and changed
-- restore_file function to take a file reference and path.
--
-- Revision 1.52  2006/11/28 14:27:06  jsedwards
-- Added unencrypted_flag.
--
-- Revision 1.51  2006/11/27 13:45:48  jsedwards
-- Upped MAX_FILES_PER_DISC_LIST to 256 (could be larger now, but this is
-- good for the moment so we don't use too much ram).  Also added external
-- definitions for the decode and encode variable sized counts.
--
-- Revision 1.50  2006/11/19 16:38:19  jsedwards
-- Added find_matching_disc_list function.
--
-- Revision 1.49  2006/11/19 15:28:20  jsedwards
-- Made get_disc_list_object_size a global function.
--
-- Revision 1.48  2006/11/19 14:48:47  jsedwards
-- Change check_file_md5sum function to return one of three possible results
-- (file not found, md5sum match, or md5sum mismatch) instead of just a
-- boolean saying file found or not found.
--
-- Revision 1.47  2006/11/18 15:09:10  jsedwards
-- Added "max_size" parameter to read_variable_sized_object_from_disk because
-- objects are no longer limited to one file block.
--
-- Revision 1.46  2006/11/18 14:24:02  jsedwards
-- Changed maximum number of files in a disc list to 127 (this allows bit 7
-- to be used to indicate larger counts if desired in the future).  Also
-- added max size of disc list object define.
--
-- Revision 1.45  2006/11/11 13:29:19  jsedwards
-- Changed version string to 0014 because of addition of next public reference
-- to disk header.
--
-- Revision 1.44  2006/11/11 12:01:05  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.43  2006/11/05 21:30:49  jsedwards
-- Add external declarations for disc_copy and storage_location functions.
--
-- Revision 1.42  2006/11/04 18:53:38  jsedwards
-- Added external declarations for new routines in file.c.
--
-- Revision 1.41  2006/10/27 12:09:28  jsedwards
-- Add external reference for 'find_md5'.
--
-- Revision 1.40  2006/10/26 01:33:32  jsedwards
-- Merged alpha_05_branch into main trunk.  Functions declarations and
-- definitions are all from alpha_05_branch.  The only changes from the
-- alpha_05_branch is the removal of old comments and moving all of the
-- C_structs to a separate file (c_structs.h).
--
-- Revision 1.37.2.23  2006/10/25 12:22:29  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.37.2.22  2006/10/22 12:58:02  jsedwards
-- Change name of parameter for "set_sequential_blocks" to approximate number
-- of blocks instead of block spacing, because now the function computes the
-- spacing.
--
-- Revision 1.37.2.21  2006/10/22 00:23:11  jsedwards
-- Change "flags" in common header to one uint32 instead of 4 uint8s.
-- Fix size of ObjRef in is_void_reference().
--
-- Revision 1.37.2.20  2006/10/12 12:08:15  jsedwards
-- Added external declaration of flush_bit_maps function that allows all of
-- the dirty bit maps in the cache to be written back to the disk.
--
-- Revision 1.37.2.19  2006/10/07 22:22:49  jsedwards
-- Changed version number to 0013.  Removed external declaration of generate
-- new id in range.  Added external declaration of set sequential blocks.
--
-- Revision 1.37.2.18  2006/10/06 04:33:15  jsedwards
-- Added external references for copy reference, void reference, and convert
-- word to ref functions.
--
-- Revision 1.37.2.17  2006/10/03 12:53:07  jsedwards
-- Changed so that instead of calling a separate routine after initialize to
-- change the already opened storage, you call it now with a type of storage
-- parameter and a path to the storage.  The problem with the other way was
-- if you tried reading a compressed file on another machine it tried to open
-- the default file which didn't exist.
--
-- Revision 1.37.2.16  2006/10/02 02:14:00  jsedwards
-- Added external declaration for set_compressed_read_only.
--
-- Revision 1.37.2.15  2006/09/29 12:07:17  jsedwards
-- Add declaration of "list_persons" routine.
--
-- Revision 1.37.2.14  2006/09/29 04:20:40  jsedwards
-- Put MAGIC_NUMBER and VERSION_STRING numbers back in.
--
-- Revision 1.37.2.13  2006/09/18 01:30:10  jsedwards
-- Eliminated block size from file class and now use 4 bytes for file size.
-- Changed so block references are not store in block class but have a
-- reference list for the list of blocks.
--
-- Revision 1.37.2.12  2006/09/13 12:27:32  jsedwards
-- Change "id" in common header to an ObjRef.
--
-- Revision 1.37.2.11  2006/09/02 15:03:03  jsedwards
-- Add reference list class reference to root object because now it has to
-- read the reference list to get it's size and we can't easily read it
-- without the read routine verifying it's class.
--
-- Revision 1.37.2.10  2006/09/01 13:21:11  jsedwards
-- Changed "magic_number" and 'version" in common header to "flags" and "id".
--
-- Revision 1.37.2.9  2006/09/01 03:16:32  jsedwards
-- Merged changes for storing files and changes to store objects in one big
-- sparse file.
--
-- Revision 1.37.2.8  2006/08/26 15:27:30  jsedwards
-- Removed embedded md5sum from file object and replaced it with an object
-- reference to a md5sum object.  Also added a reference to a sha1 object.
--
-- Revision 1.37.2.7  2006/08/20 02:16:03  jsedwards
-- Add declaration for restore_file function.
--
-- Revision 1.37.2.6  2006/08/19 18:47:19  jsedwards
-- Added "media" entry to "File" structure.  Added error return value to
-- ObjCreateResult.  Removed "create_file_path" declaration and replaced
-- it with the "create_file" because only "create_file" should call
-- "create_file_path".
--
-- Revision 1.37.2.5  2006/08/19 12:46:49  jsedwards
-- Moved configuration defines out of objectify.h and into a new file config.h.
--
-- Revision 1.37.2.4  2006/08/18 13:08:54  jsedwards
-- Added file reference to the File_Path structure.
--
-- Revision 1.37.2.3  2006/08/18 13:00:41  jsedwards
-- Added external definitions for the "file" module and changed
--   "File_Identifier" structure to "File_Path" (it's temporary so it
--   really doesn't matter what it is called).
--
-- Revision 1.37.2.2  2006/08/13 16:36:43  jsedwards
-- Made a define for the block size of the file called FILE_BLOCK_SIZE, that
-- is the minimum size of the file for each object.  The amount of disk space
-- used for a one byte file.
--
-- Revision 1.37.2.1  2006/08/13 13:47:39  jsedwards
-- Added Check, File, and File_Identifier structures.
--
-- Revision 1.39  2006/01/21 18:35:58  jsedwards
-- Removed all of the C structures and made a separate file with them, which
-- should be generated from the class definition objects at some point.
-- Moved the version information to the objectify_private.h header file.
-- Changed the object reference to the new format (16 bytes).
--
-- Revision 1.38  2006/01/18 01:06:18  jsedwards
-- Removed kruft that was not used.
--
-- Revision 1.37  2006/01/10 13:55:45  jsedwards
-- Added external reference for "find phone for person".
--
-- Revision 1.36  2006/01/09 03:22:35  jsedwards
-- Added external declarations for "any persons named", "any states named",
-- and "describe state" routines.
--
-- Revision 1.35  2006/01/03 03:20:15  jsedwards
-- Removed 'city' from 'Address' class because the street has city information
-- in it.  Added 'ref to word' inline function taken from 'name.c' file.
-- Added external declarations for "find state by name", "create us city",
-- "add city", and "list cities in state" routines.
--
-- Revision 1.34  2006/01/01 19:46:39  jsedwards
-- Added external declarations for "is quit command", "ask yes or no", and
-- "add mobile phone" routines.
--
-- Revision 1.33  2006/01/01 00:43:54  jsedwards
-- Removed count from month object.  Since it was always 2, why have it?
-- Added external declarations for date, month, and year to string routines.
-- Added external declarations for find_person and describe person routines.
--
-- Revision 1.32  2005/12/31 19:01:09  jsedwards
-- Add "gender" class.  Change find_word paramater from "thing" to "thing
-- class".  Add declaration of "add_person".
--
-- Revision 1.31  2005/12/31 03:05:33  jsedwards
-- Added structures related to houses and addresses.
--
-- Revision 1.30  2005/12/30 14:03:47  jsedwards
-- Removed identifier, drivers license number, and residence from person
-- object, they will be found by reference.  Added social security number
-- object.  Changed password object to variable length.  Added external
-- declarations for "is_leap_year" and "is_valid_ssn".  Added a few comments
-- to help discern what external refernces are for.
--
-- Revision 1.29  2005/12/29 17:04:35  jsedwards
-- Changed all objects that have variable sized data to have [0] sized arrays,
-- this way the objects don't have to deal with filling in the random data.
-- Also added external declarations for read object headers and read variable
-- sized object routines.
--
-- Revision 1.28  2005/12/28 12:59:55  jsedwards
-- Added parameters to initialize_objectify to pass in the blowfish key and
-- the two seeds for the sequence generator.  Changed get reference from
-- password to get a variable length key from the password.   Also changed
-- all of the variable length objects so they use the entire 512 bytes of
-- a file.
--
-- Revision 1.27  2005/12/27 20:43:47  jsedwards
-- Changed is_void_reference and is_same_object routines to work on any size
-- of ObjRef, instead of using the "word" which only works for 4 byte refs.
--
-- Revision 1.26  2005/12/27 19:44:06  jsedwards
-- Added "Root" object class and declaration of routine to set the root object.
--
-- Revision 1.25  2005/12/27 18:12:42  jsedwards
-- Changed ObjRef to be 4 bytes instead of 8 and eliminated the uint8_8
-- routines.  Changed the other routines appropriately and eliminated all
-- the fixed "CLASS" definitions because now all file names are random.
--
-- Revision 1.24  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.23  2005/12/23 18:05:10  jsedwards
-- Changed class_def_def to contain pointers to features and added definition
-- of data_definition.
--
-- Revision 1.22  2005/12/21 23:24:30  jsedwards
-- Added declarations for find name, name to string, and several new phone
-- routines.
--
-- Revision 1.21  2005/12/21 16:57:29  jsedwards
-- Added state_postal_code parameter to create_area_code function and added
-- declaration of find_state_from_postal_code.
--
-- Revision 1.20  2005/12/21 03:47:02  jsedwards
-- Added "count" to "Name" objects and "name" to "US State" objects.
--
-- Revision 1.19  2005/12/18 15:03:43  jsedwards
-- Removed postal code class and made postal code part of state class.
--
-- Revision 1.18  2005/12/17 19:20:43  jsedwards
-- Add state and state postal code.
--
-- Revision 1.17  2005/12/11 16:50:41  jsedwards
-- Added "Area Code" class, changed "Business Phone" class to "Work Phone",
-- added declarations for find language, spelling, and word.
--
-- Revision 1.16  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.15  2005/12/08 18:02:07  jsedwards
-- Split "Entity Phone" class into three different classes "Home Phone",
-- "Work Phone", and "Mobile Phone".
--
-- Revision 1.14  2005/12/05 18:59:00  jsedwards
-- Added nickname to person object.
--
-- Revision 1.13  2005/12/05 04:58:54  jsedwards
-- Added declaration for read_reference_list_from_disk.  Don't call
-- read_object_from_disk anymore to read a reference list.
--
-- Revision 1.12  2005/12/04 14:06:19  jsedwards
-- Added 'birth_place' and 'death_place' fields.
-- Changed 'goes_by_name' to 'goes_by' and 'current_address' to 'residence'.
--
-- Revision 1.11  2005/12/04 04:13:03  jsedwards
-- Added 'nwos' prefix to create_xxxx function names and eliminated the
-- 'referring' object parameter from all of them.
--
-- Revision 1.10  2005/12/04 00:40:43  jsedwards
-- Added declarations for create_date and get_month_ref functions.
--
-- Revision 1.9  2005/12/03 22:14:44  jsedwards
-- Changed to new word, name, spelling layout.
--
-- Revision 1.8  2005/12/03 04:12:07  jsedwards
-- Added abbreviation, thing, and language classes.
--
-- Revision 1.7  2005/12/03 02:01:44  jsedwards
-- Change month so that month is stored as two ascii digits instead of a uint8.
--
-- Revision 1.6  2005/12/02 20:28:59  jsedwards
-- Changed 0xF objects to files and make floating point numbers 0x4.
-- Added the name of the object that the reference list is for as a parameter
-- to create_reference_list.
--
-- Revision 1.5  2005/11/30 13:56:11  jsedwards
-- Rearranged the order, added endian sensitivity to class definition class,
-- and added external definitions for new routines.
--
-- Revision 1.4  2005/11/26 17:18:19  jsedwards
-- Removed "mother" and "father" fields from person, that should be handled
-- through the relationship object.  Added current and mailing address.
--
-- Revision 1.3  2005/11/26 16:36:27  jsedwards
-- Added many C structure types (person, month, phone, e-mail, etc.).
--
-- Revision 1.2  2005/11/26 15:34:45  jsedwards
-- Removed unused structures, added 'nwos' prefix to functions, and added
-- some new class definitions.
--
-- Revision 1.1.1.1  2005/11/25 12:44:27  jsedwards
-- Copied from 'lab'.
--
-- Revision 1.14  2005/11/24 19:21:18  jsedwards
-- Removed base32 stuff.
--
-- Revision 1.13  2005/11/24 15:46:24  jsedwards
-- Fixed syntax errors.
--
-- Revision 1.12  2005/11/24 15:14:24  jsedwards
-- Hopefully this is the one... at least for awhile.
--
-- Revision 1.11  2005/11/24 15:05:08  jsedwards
-- Recombined and rearranged some more.
--
-- Revision 1.10  2005/11/24 14:17:31  jsedwards
-- Recombined simple and complex objects (only difference was versioning).
-- Rearraged everything yet again.
--
-- Revision 1.9  2005/11/24 13:20:52  jsedwards
-- Completely rearranged headers into file, object and version.
--
-- Revision 1.8  2005/11/22 14:07:04  jsedwards
-- Split object identifier (upper 4 bits) into two identifiers, one for
-- simple objects (8) and one for complex objects (9).
--
-- Revision 1.7  2005/11/22 13:41:13  jsedwards
-- Split common header out for first 16 bytes of header.
-- Added "is_void_reference" inline function.
-- Added MD5 object stuff.
-- More external references.
--
-- Revision 1.6  2005/11/12 15:41:32  jsedwards
-- Changed UINT64_TO_UINT8_8 macro to inline function.
-- Added external references for write object to disk and create ref list.
--
-- Revision 1.5  2005/11/12 15:07:45  jsedwards
-- Fixed syntax errors.  Added "identifer" field so we can tell what object
-- it is when it is in memory.  Put 64 bit macros back in that were removed
-- in an earlier change.
--
-- Revision 1.4  2005/11/02 13:52:30  jsedwards
-- Removed the "data_length" fields because we get that from the filesystem
-- in this implementation.
--
-- Revision 1.3  2005/11/02 13:51:21  jsedwards
-- Rearranged the order of the common header.  Split EveryObject into two
-- types of objects SimpleObject and ComplexObject.  Changed file name of
-- class definition object.
--
-- Revision 1.2  2004/11/03 14:30:04  jsedwards
-- Merged 'ObjHeader' into 'EveryObj' structure.  Made separate checksum for
--   'EveryObj' data.
--
-- Revision 1.1  2004/10/31 18:27:52  jsedwards
-- Moved object definitions from 'types.h' to here.
--
*/

#ifndef OBJECTIFY_H
#define OBJECTIFY_H

#include "config.h"
#include "time_stamp.h"
#include "types.h"


#define FILE_BLOCK_SIZE 256

#define MAGIC_NUMBER "NWOS"



typedef union { uint8 id[4]; uint32 word; } ObjRef;


typedef struct {
    uint32    flags;                      /*   4 */  /* flags / version */
    ObjRef    id;                         /*   8 */  /* reference id */
    union {
      uint8     header_chksum[4];         /*  12 */  /* CRC32 of header */
      uint32    num_refs;                            /* used by reference lists only */
    };
    uint8     data_chksum[4];             /*  16 */  /* CRC32 of data   */
    TimeStamp creation_time;              /*  24 */
    TimeStamp access_time;                /*  32 */  /* used when in memory for caching */
    ObjRef    class_definition;           /*  36 */
} CommonHeader;


typedef struct {
    ObjRef    clone_of;                   /*   4 */
    ObjRef    references;                 /*   8 */
    ObjRef    context;                    /*  12 */
    ObjRef    creation;                   /*  16 */
    ObjRef    source;                     /*  20 */
    ObjRef    prev_version;               /*  24 */
    ObjRef    next_version;               /*  28 */
} ObjectHeader;


typedef struct {
    CommonHeader common;                  /*  36 */
    ObjectHeader object;                  /*  64 */
} EveryObject;

typedef struct {
    CommonHeader common_header;           /*  36 */
    ObjRef       references[0];
} ReferenceList;


typedef struct {
    CommonHeader  header;                 /*   36 */
    uint32        index[256];             /*  ??? */
    ObjRef        references[256];        /* ???? */
} IndexedReferenceList;


/* If we are making the export_c_structs program, c_structs.h doesn't   */
/* exist yet, so these need to be defined for it to build successfully. */

#ifdef USE_PREDEFINED_STRUCTS

typedef struct {
    EveryObject header;
    ObjRef name;
    uint8 count;
    ObjRef feature[0];
} C_struct_Class_Definition;

typedef struct {
    EveryObject header;
    ObjRef character_set;
    uint8 count;
    char storage[0];
} C_struct_Spelling;

typedef struct {
    EveryObject header;
    uint8 count;
    ObjRef spelling[0];
} C_struct_Name;

typedef struct {
    EveryObject header;
    ObjRef inherit;
    ObjRef class;
    ObjRef label;
    uint8 count;
} C_struct_Feature_Definition;

typedef struct {
    EveryObject header;
    ObjRef class_definition_class;
    ObjRef reference_list_class;
} C_struct_Root;

#else
#include "c_structs.h"
#endif


static inline bool is_void_reference(ObjRef* ref)
{
    int i;
    for (i = 0; i < sizeof(ObjRef); i++) if (ref->id[i] != 0) return false;
    return true;
}


static inline bool is_same_object(ObjRef* ref1, ObjRef* ref2)
{
    int i;
    for (i = 0; i < sizeof(ObjRef); i++) if (ref1->id[i] != ref2->id[i]) return false;
    return true;
}


static inline void copy_reference(ObjRef* dst, ObjRef* src)
{
    int i;
    for (i = 0; i < sizeof(ObjRef); i++) dst->id[i] = src->id[i];
}


static inline void void_reference(ObjRef* ref)
{
    int i;
    for (i = 0; i < sizeof(ObjRef); i++) ref->id[i] = 0;
}


static inline void nwos_4_uint8_to_uint32(uint8* src, uint32* dst) 
{
    *dst = ((uint32)src[0] << 24) | ((uint32)src[1] << 16) | ((uint32)src[2] << 8) | (uint32)src[3];
}

static inline void nwos_2_uint8_to_uint16(uint8* src, uint16* dst) 
{
    *dst = ((uint32)src[0] << 8) | ((uint32)src[1]);
}

static inline void nwos_uint32_to_4_uint8(uint32* src, uint8* dst) 
{
    dst[0] = *src >> 24;
    dst[1] = *src >> 16;
    dst[2] = *src >> 8;
    dst[3] = *src;
}


/* NOTE: these only work when references are 4 bytes, need to fix when size is changed */
static inline uint32 nwos_ref_to_word(ObjRef* ref) 
{
  return ((uint32)ref->id[0] << 24) | ((uint32)ref->id[1] << 16) | ((uint32)ref->id[2] << 8) | (uint32)ref->id[3];
}

static inline void nwos_word_to_ref(uint32 word, ObjRef* ref) 
{
    ref->id[0] = word >> 24;
    ref->id[1] = word >> 16;
    ref->id[2] = word >> 8;
    ref->id[3] = word;
}

static inline uint16 byteswap_uint16(uint16 x)
{
    return (x << 8) | (x >> 8);
}

static inline uint32 byteswap_uint32(uint32 x)
{
  return (x << 24) |
    ((x & 0x0000ff00) << 8) |
    ((x & 0x00ff0000) >> 8) |
    (x >> 24);
}


typedef enum { Public_Reference, Private_Reference } Reference_Type;
typedef enum { Encryption_None=0, Encryption_Minimal, Encryption_Very_Low, Encryption_Low, Encryption_Medium, Encryption_High, Encryption_Extreme } Encryption_Level;

/* since linux doesn't seem to have this function */
#ifdef linux
extern size_t strlcpy(char* dst, char* src, size_t len);
extern size_t strlcat(char* dst, char* src, size_t len);
#endif

extern void nwos_ref_to_name(ObjRef* ref, char name[]);

/* extern ObjRef store_object(EveryObject* obj, uint64 length); */

extern uint32 nwos_decode_variable_sized_count(uint8 count[4]);
extern void   nwos_encode_variable_sized_count(uint32 count, uint8 result[4]);

extern void nwos_log(char* str);
extern void nwos_log_arguments(int argc, char* argv[]);
extern void nwos_initialize_objectify(uint8 bf_key[16], uint32 linear, uint32 serial, StorageType type, char* path);
extern void nwos_terminate_objectify();
extern void nwos_set_root_object(ObjRef* ref);
extern void nwos_flush_bit_maps();

extern size_t nwos_reference_list_size(ObjRef* ref);
extern ReferenceList* nwos_malloc_reference_list(ObjRef* ref);
extern void nwos_free_reference_list(ReferenceList* ref_list);

extern Reference_Type nwos_reference_type(ObjRef* ref);
extern void nwos_get_object_class(ObjRef* obj, ObjRef* object_class);
extern size_t nwos_get_object_size(void* object);
extern void nwos_generate_new_id(ObjRef* ref);
extern void nwos_generate_new_private_id(ObjRef* ref);
extern void nwos_set_encryption_level(Encryption_Level);
extern void nwos_push_encryption_level(Encryption_Level);
extern void nwos_pop_encryption_level();

extern void nwos_add_to_references(ObjRef* ref, ObjRef* obj);
extern void nwos_add_to_reference_list(ObjRef* ref, ObjRef* ref_list);
extern void nwos_remove_from_references(ObjRef* ref, ObjRef* obj);
extern void nwos_remove_from_reference_list(ObjRef* ref, ObjRef* ref_list);

//extern bool nwos_find_class_definition(char* name, ObjRef* class_ref);
extern bool nwos_find_public_class_definition(char* name, ObjRef* class_ref);
extern bool nwos_find_private_class_definition(char* name, ObjRef* class_ref);
extern void nwos_find_or_create_private_class_definition(char* name_of_class, ObjRef* ref);
extern bool nwos_find_public_feature_definition(ObjRef* type_ref, char* name, int count, ObjRef* feat_ref);
extern void nwos_find_or_create_public_feature_definition(ObjRef* type_ref, char* name, int count, ObjRef* feat_ref);

extern void nwos_create_reference_list(ObjRef* for_obj, ObjRef* ref_list);
extern void nwos_create_object(uint64 id, EveryObject* header);
extern void nwos_read_class_definition(ObjRef* ref, C_struct_Class_Definition* class_def);
extern void nwos_read_object_headers_from_disk(ObjRef* ref, EveryObject* header);
extern void nwos_read_object_from_disk(ObjRef* ref, void* object, size_t size);
extern void nwos_read_variable_sized_object_from_disk(ObjRef* ref, void* obj, size_t max_size, size_t (*size_function)(void*));
extern void nwos_read_reference_list_from_disk(ObjRef* ref, ReferenceList* object, size_t size);

typedef enum { CREATED_NEW, CREATED_NEW_REVISION, FOUND_EXISTING, ERROR_OCCURRED, DUPLICATE_PATH, DUPLICATE_FILE, PATH_NOT_FOUND, MULTIPLE_FILES, NUM_OBJ_RESULTS } ObjCreateResult;



/* word stuff */
extern bool nwos_find_language(char* name, ObjRef* ref);

extern bool nwos_find_public_spelling(char* spelling, ObjRef* ref);
extern bool nwos_find_private_spelling(char* spelling, ObjRef* ref);
extern ObjCreateResult nwos_create_public_spelling(char* spelling, ObjRef* ref);
extern ObjCreateResult nwos_create_private_spelling(char* spelling, ObjRef* ref);

extern bool nwos_find_word(char* word, ObjRef* thing_class, ObjRef* lang, ObjRef* ref);
extern ObjCreateResult nwos_create_word(char* word, ObjRef* thing, ObjRef* lang, ObjRef* ref);

extern bool nwos_find_abbreviation(char* abbr, ObjRef* thing, ObjRef* lang, ObjRef* ref);
extern ObjCreateResult nwos_create_abbreviation(char* abbr, ObjRef* thing, ObjRef* lang, ObjRef* ref);

extern bool nwos_find_public_acronym(char* acronym, ObjRef* ref);
extern bool nwos_acronym_to_string(ObjRef* ref, char* string, size_t size);

extern bool nwos_is_quit_command(char* command);
extern bool nwos_ask_yes_or_no(char* statement, char* question);


/* name stuff */
extern bool            nwos_find_public_name(char* name, ObjRef* ref);
extern bool            nwos_find_private_name(char* name, ObjRef* ref);
extern ObjCreateResult nwos_create_name(char* name, ObjRef* ref);
extern bool            nwos_name_to_string(ObjRef* ref, char* string, size_t size);

/* date stuff */
extern bool            nwos_is_leap_year(int year);
extern void            nwos_get_month_ref(uint8 month, ObjRef* ref);
extern bool            nwos_find_public_year(uint16 year, ObjRef* ref);
extern bool            nwos_find_private_year(uint16 year, ObjRef* ref);
extern ObjCreateResult nwos_create_year(uint16 year, ObjRef* ref);
extern void            nwos_find_public_month(uint8 month, ObjRef* ref);
extern bool            nwos_find_private_month(uint8 month, ObjRef* ref);
extern bool            nwos_find_public_date(uint16 year, uint8 month, uint8 day, ObjRef* ref);
extern void            nwos_find_or_create_private_month_and_day(uint8 month, uint8 day, ObjRef* ref);
extern ObjCreateResult nwos_create_date(uint16 year, uint8 month, uint8 day, ObjRef* ref);
extern bool            nwos_year_to_string(ObjRef* year_obj_ref, char* string, size_t size_of_string);
extern bool            nwos_month_number_to_string(ObjRef* month_obj_ref, char* string, size_t size_of_string);
extern bool            nwos_date_to_string(ObjRef* date_obj_ref, char* string, size_t size_of_string);

/* numbers */
extern bool            nwos_find_cardinal_number(char* number, ObjRef* ref);
extern ObjCreateResult nwos_create_cardinal_number(char* number, ObjRef* ref);
extern void            nwos_cardinal_number_to_string(ObjRef* ref, char* string, size_t size);
extern bool            nwos_find_ordinal_number(ObjRef* cardinal_ref, ObjRef* ref);
extern ObjCreateResult nwos_create_ordinal_number(ObjRef* cardinal_ref, ObjRef* ref);
extern void            nwos_ordinal_number_to_string(ObjRef* ref, char* string, size_t size);

/* person stuff */
extern void nwos_add_person();
extern void nwos_add_birthday(char* name, char* date);
extern void nwos_list_persons();
extern void nwos_describe_person();
extern bool nwos_find_person();
extern bool nwos_any_persons_named(char* name);
extern void nwos_find_public_gender(char* gender, ObjRef* ref);

/* phone stuff */
extern bool nwos_find_public_area_code(char* area_code, ObjRef* ref);
extern bool nwos_find_private_area_code(char* area_code, ObjRef* ref);
extern ObjCreateResult nwos_create_area_code(char* area_code, char* state_postal_code, ObjRef* ref);
extern bool nwos_is_valid_phone_number(char* number);
extern const char* nwos_invalid_phone_number_msg(char* number);
extern bool nwos_find_phone_number(char* area_code, char* phone_number, ObjRef* ref);
extern ObjCreateResult nwos_create_phone_number(char* area_code, char* phone_number, ObjRef* ref);
extern bool nwos_phone_number_to_string(ObjRef* ref, char* string, size_t size);
extern void nwos_add_mobile_phone(char* name, char* phone_number);
extern void nwos_find_phone_for_person(char* name);

/* address stuff */
extern bool nwos_find_state_postal_code(char* state_postal_code, ObjRef* ref);
extern bool nwos_find_state_from_postal_code(char* postal_code, ObjRef* ref);
extern bool nwos_find_state_from_name(char* name, ObjRef* ref);
extern bool nwos_any_states_named(char* name);
extern void nwos_describe_state(char* name);
extern ObjCreateResult nwos_create_us_city(char* city, ObjRef* state_ref, ObjRef* city_ref);
extern void nwos_add_city(ObjRef* city_ref);
extern void nwos_list_cities_in_state(char* state);

/* financial stuff */
extern bool nwos_find_public_credit_union(char* name, ObjRef* ref);
extern void nwos_add_account();
extern void nwos_list_accounts();


/* device stuff */
extern bool nwos_find_public_manufacturer(char* name, ObjRef* ref);
extern void nwos_list_public_manufacturers();


/* file stuff */
typedef enum { File_Not_Found, Checksums_Match, MD5_Sum_Mismatch, SHA1_Sum_Mismatch, Checksums_Mismatch } CheckFileResult;

extern bool nwos_url_to_string(ObjRef* ref, char* string, size_t size);

extern void nwos_list_public_credit_unions();

extern bool nwos_find_file_path(char* path, ObjRef* ref);
extern bool nwos_file_path_to_string(ObjRef* ref, char* string, size_t size);
extern int  nwos_number_of_files_for_path(ObjRef* path_ref);
extern bool nwos_file_path_to_path_and_file_association(ObjRef* path_ref, int index, ObjRef* assoc_ref);
extern bool nwos_find_md5(uint8 md5[16], ObjRef* ref);
extern bool nwos_find_path_and_file_association(ObjRef* path_ref, ObjRef* file_ref, TimeStamp mod_time, ObjRef* ref);
typedef enum { MatchAll, IgnoreTime } MatchCode;
extern bool nwos_find_matching_path_and_file_association(char* path, char* file, ObjRef* ref, MatchCode match);
extern CheckFileResult nwos_check_file_checksums(char* path);
extern bool nwos_file_is_stored(ObjRef* association_ref);  /* returns true if data stored in system */
extern bool nwos_file_has_backup(ObjRef* association_ref);  /* returns true if file is on a backup disc */
extern ObjCreateResult nwos_create_file_without_storing_data(char* path, char* file_name, ObjRef* ref);
extern ObjCreateResult nwos_create_file(char* path, ObjRef* ref);
extern ObjCreateResult nwos_add_new_revision_of_file(char* path, ObjRef* ref);
extern ObjCreateResult nwos_find_or_create_path_and_file_association(ObjRef* path_ref, ObjRef* file_ref, TimeStamp mod_time, ObjRef* ref);
extern bool nwos_restore_file(ObjRef* file_ref, char* path);
extern bool nwos_file_is_identical(ObjRef* assoc_ref, char* path, MatchCode match);

#define MAX_FILES_PER_DISC_LIST 4096
#define MAX_SIZE_DISC_LIST (sizeof(C_struct_Disc_List) + sizeof(ObjRef) * MAX_FILES_PER_DISC_LIST)
extern size_t nwos_get_disc_list_object_size(void* disc_list_obj);
extern bool nwos_find_disc_list(char id[12], ObjRef* ref);
extern bool nwos_find_matching_disc_list(ObjRef files[MAX_FILES_PER_DISC_LIST], ObjRef* ref);
extern ObjCreateResult nwos_create_disc_list(char id[12], ObjRef disc_list[MAX_FILES_PER_DISC_LIST], ObjRef* ref);
extern bool nwos_find_disc_copy(ObjRef* disc_list, int copy_num, ObjRef* ref);
extern ObjCreateResult nwos_create_disc_copy(ObjRef* disc_list, int copy_num, ObjRef* location, ObjRef* ref);
extern bool nwos_find_storage_location(char* location, ObjRef* ref);
extern ObjCreateResult nwos_create_storage_location(char* location, ObjRef* ref);

extern bool nwos_is_valid_ssn(const char* ssn);

extern void nwos_get_key_from_password(uint8 key[], size_t key_size);

/* currently the largest object is a disc list */
#define MAX_OBJECT_BLOCKS ((MAX_SIZE_DISC_LIST + FILE_BLOCK_SIZE - 1) / FILE_BLOCK_SIZE)
#define MAX_OBJECT_SIZE (MAX_OBJECT_BLOCKS * FILE_BLOCK_SIZE)

#endif

