/*
--             This file is part of the New World OS project
--                    Copyright (C) 2007 QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: diff_compressed.c,v $
-- Revision 1.12  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.11  2007/06/22 22:07:17  jsedwards
-- Add code to correctly deal with 0021 to 0022 conversion.
--
-- Revision 1.10  2007/06/19 19:06:30  jsedwards
-- Changed feature names in Disk Header because it was change (0022).
--
-- Revision 1.9  2007/03/24 14:08:17  jsedwards
-- Change to not write the next block info into bytes 0-3 when file 1 has
-- reached the end of file, because it tricks the patch program.
--
-- Revision 1.8  2007/03/24 12:32:39  jsedwards
-- Moved the increment of blocks added count outside the if writing output
-- file, so it gets incremented even when not writing an output file.
--
-- Revision 1.7  2007/03/24 12:29:13  jsedwards
-- Fixed to finish adding blocks from file 2 after file file 1 has reached
-- the end of the file.
--
-- Revision 1.6  2007/03/15 13:44:04  jsedwards
-- Changed to zero out the bytes before writing and added counts.
--
-- Revision 1.5  2007/03/15 12:50:25  jsedwards
-- Changed to write both the old header and the new header into the patch file.
--
-- Revision 1.4  2007/03/14 23:53:38  jsedwards
-- Added code to write the differences to a file.
--
-- Revision 1.3  2007/03/14 23:42:01  jsedwards
-- Changed to just look for changed blocks and blocks added in file 2 (assume
-- that no blocks were deleted).
--
-- Revision 1.2  2007/03/14 14:38:12  jsedwards
-- Changed to verify the headers are the same.
--
-- Revision 1.1  2007/02/25 16:00:53  jsedwards
-- This is a quick hack to compare the compressed archives of version 0017
-- and the updated (moved root objects) 0018 archive.
--
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>    /* define sleep() */

#include "objectify_private.h"

uint32 convert_4_uint8_to_uint32(uint8 byte[4])
{
  return ((uint32)byte[0] << 24) | ((uint32)byte[1] << 16) | ((uint32)byte[2] << 8) | (uint32)byte[3];
}


int main(int argc, char* argv[])
{
    FILE* fp1;
    FILE* fp2;
    FILE* fp3 = NULL;
    unsigned char buf1a[FILE_BLOCK_SIZE];
    unsigned char buf2a[FILE_BLOCK_SIZE];
    int block1;
    int block2;
    int changed = 0;
    int added = 0;
    size_t read1;
    size_t read2;
    size_t write3;
    Disk_Header header1;
    Disk_Header header2;
    Disk_Header_0021 old_header;


    if (argc != 3 && argc != 4)
    {
	fprintf(stderr, "usage: %s old_file new_file [outfile]\n", argv[0]);
	exit(1);
    }


    /* Open the first file and check it */

    fp1 = fopen(argv[1], "r");
    if (fp1 == NULL)
    {
	perror(argv[1]);
	exit(1);
    }

    read1 = fread(buf1a, 1, sizeof(buf1a), fp1);

    if (read1 != FILE_BLOCK_SIZE)
    {
	if (ferror(fp1))
	{
	    perror(argv[1]);
	}
	else
	{
	    fprintf(stderr, "Unexpected end of file: %s\n", argv[1]);
	}
	fclose(fp1);
	exit(1);
    }

    memcpy(&header1, buf1a, sizeof(header1));

    if (memcmp(header1.magic_number, "NWOS", 4) != 0)
    {
	fprintf(stderr, "Not an Objectify file: %s\n", argv[1]);
	fclose(fp1);
	exit(1);
    }


    /* Open the second file and check it */

    fp2 = fopen(argv[2], "r");
    if (fp2 == NULL)
    {
	perror(argv[2]);
	exit(1);
    }

    read2 = fread(buf2a, 1, sizeof(buf2a), fp2);

    if (read2 != FILE_BLOCK_SIZE)
    {
	if (ferror(fp2))
	{
	    perror(argv[2]);
	}
	else
	{
	    fprintf(stderr, "Unexpected end of file: %s\n", argv[2]);
	}
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }

    memcpy(&header2, buf2a, sizeof(header2));

    if (memcmp(header2.magic_number, "NWOS", 4) != 0)
    {
	fprintf(stderr, "Not an Objectify file: %s\n", argv[2]);
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }


    /* Verify they are the same version and disk size */

    if (memcmp(header1.version_string, header2.version_string, 4) != 0)
    {
	fprintf(stderr, "WARNING, versions don't match - %s: %c%c%c%c  %s: %c%c%c%c\n",
		argv[1], 
		header1.version_string[0], header1.version_string[1],
		header1.version_string[2], header1.version_string[3],
		argv[2], 
		header2.version_string[0], header2.version_string[1],
		header2.version_string[2], header2.version_string[3]);
	sleep(5);
    }

    /* special code to handle upgrade from 0021 to 0022 */
    if (memcmp(header1.version_string, "0021", 4) == 0 &&
	memcmp(header2.version_string, "0022", 4) == 0)
    {
	memcpy(&old_header, buf1a, sizeof(old_header));

	if (memcmp(old_header.total_private_blocks, header2.total_blocks, 4) != 0)
	{
	    fprintf(stderr, "Disk private blocks size different - %s: %u  %s: %u\n",
		    argv[1], 
		    convert_4_uint8_to_uint32(old_header.total_private_blocks),
		    argv[2], 
		    convert_4_uint8_to_uint32(header2.total_blocks));
	}
    }
    else
      {
    if (memcmp(header1.total_blocks, header2.total_blocks, 4) != 0)
    {
	fprintf(stderr, "Disk private blocks size different - %s: %u  %s: %u\n",
		argv[1], 
		convert_4_uint8_to_uint32(header1.total_blocks),
		argv[2], 
		convert_4_uint8_to_uint32(header2.total_blocks));
    }
      }

    if (argc == 4)
    {
	fp3 = fopen(argv[3], "w");

	if (fp3 == NULL)
	{
	    perror(argv[3]);
	    fclose(fp1);
	    fclose(fp2);
	    exit(1);
	}
    }

    if (fp3 != NULL)   /* write both headers into the patch file */
    {
	write3 = fwrite(buf1a, 1, sizeof(buf1a), fp3);

	if (write3 != FILE_BLOCK_SIZE)
	{
	    perror(argv[3]);
	    fclose(fp1);
	    fclose(fp2);
	    fclose(fp3);
	    exit(1);
	}

	write3 = fwrite(buf2a, 1, sizeof(buf2a), fp3);

	if (write3 != FILE_BLOCK_SIZE)
	{
	    perror(argv[3]);
	    fclose(fp1);
	    fclose(fp2);
	    fclose(fp3);
	    exit(1);
	}
    }

    block1 = 1;
    block2 = 1;

    read1 = fread(buf1a, 1, sizeof(buf1a), fp1);
    read2 = fread(buf2a, 1, sizeof(buf2a), fp2);

    while (!feof(fp1) && !feof(fp2) && read1 == FILE_BLOCK_SIZE && read2 == FILE_BLOCK_SIZE)
    {
	if (buf1a[0] != 0 || buf1a[1] != 0 || buf1a[2] != 0 || buf1a[3] != 0)
	{
	    printf("\n%s block %d - first four bytes not zero: %02x%02x%02x%02x\n",
		   argv[1], block1, buf1a[0], buf1a[1], buf1a[2], buf1a[3]);
	    break;
	}

	if (buf2a[0] != 0 || buf2a[1] != 0 || buf2a[2] != 0 || buf2a[3] != 0)
	{
	    printf("\n%s block %d - first four bytes not zero: %02x%02x%02x%02x\n",
		   argv[2], block2, buf2a[0], buf2a[1], buf2a[2], buf2a[3]);
	    break;
	}

	if (buf1a[4] == buf2a[4] && buf1a[5] == buf2a[5] && buf1a[6] == buf2a[6] && buf1a[7] == buf2a[7])
	{
	    /* blocks have the same id number, just compare them */

	    if (memcmp(buf1a, buf2a, FILE_BLOCK_SIZE) != 0)
	    {
		printf("%02x%02x%02x%02x: changed\n",
		       buf1a[4], buf1a[5], buf1a[6], buf1a[7]);

		if (fp3 != NULL)   /* write the changed block */
		{
		    write3 = fwrite(buf2a, 1, sizeof(buf2a), fp3);

		    if (write3 != FILE_BLOCK_SIZE)
		    {
			perror(argv[2]);
			fclose(fp1);
			fclose(fp2);
			fclose(fp3);
			exit(1);
		    }
		}

		changed++;
	    }

	    read1 = fread(buf1a, 1, sizeof(buf1a), fp1);
	    block1++;
	}
	else   /* id is different, assume this an add, since right now can't delete */
	{
	    printf("%02x%02x%02x%02x: added\n",
		   buf2a[4], buf2a[5], buf2a[6], buf2a[7]);

	    if (fp3 != NULL)   /* write the new block */
	    {
		/* write the id of the next old block into bytes 0-3 of the new block */
		buf2a[0] = buf1a[4];
		buf2a[1] = buf1a[5];
		buf2a[2] = buf1a[6];
		buf2a[3] = buf1a[7];

		write3 = fwrite(buf2a, 1, sizeof(buf2a), fp3);

		if (write3 != FILE_BLOCK_SIZE)
		{
		    perror(argv[3]);
		    fclose(fp1);
		    fclose(fp2);
		    fclose(fp3);
		    exit(1);
		}
	    }

	    added++;
	}

	read2 = fread(buf2a, 1, sizeof(buf2a), fp2);
	block2++;
    }

    if (feof(fp2) && !feof(fp1))
    {
	fprintf(stderr, "WARNING: end of file reached on %s before %s\n",
		argv[2], argv[1]);
    }

    while (!feof(fp2) && read2 == FILE_BLOCK_SIZE)
    {
	if (buf2a[0] != 0 || buf2a[1] != 0 || buf2a[2] != 0 || buf2a[3] != 0)
	{
	    printf("\n%s block %d - first four bytes not zero: %02x%02x%02x%02x\n",
		   argv[2], block2, buf2a[0], buf2a[1], buf2a[2], buf2a[3]);
	    break;
	}

	printf("%02x%02x%02x%02x: added\n",
	       buf2a[4], buf2a[5], buf2a[6], buf2a[7]);

	if (fp3 != NULL)   /* write the new block */
	{
	    write3 = fwrite(buf2a, 1, sizeof(buf2a), fp3);

	    if (write3 != FILE_BLOCK_SIZE)
	    {
		perror(argv[3]);
		fclose(fp1);
		fclose(fp2);
		fclose(fp3);
		exit(1);
	    }
	}

	added++;

	read2 = fread(buf2a, 1, sizeof(buf2a), fp2);
	block2++;
    }

    printf("\n");


    if (memcmp(header1.version_string, header2.version_string, 4) != 0)
    {
	fprintf(stderr, "WARNING, versions don't match - %s: %c%c%c%c  %s: %c%c%c%c\n",
		argv[1], 
		header1.version_string[0], header1.version_string[1],
		header1.version_string[2], header1.version_string[3],
		argv[2], 
		header2.version_string[0], header2.version_string[1],
		header2.version_string[2], header2.version_string[3]);
    }

    printf("Added: %d\n", added);
    printf("Changed: %d\n", changed);

    if (!feof(fp1) && read1 != FILE_BLOCK_SIZE)
    {
	perror(argv[1]);
    }

    fclose(fp1);

    if (!feof(fp2) && read2 != FILE_BLOCK_SIZE)
    {
	perror(argv[2]);
    }

    fclose(fp2);

    return 0;
}
