/*
--             This file is part of the New World OS project
--                Copyright (C) 2006, 2007  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: expand_sparse.c,v $
-- Revision 1.23  2007/08/10 00:04:06  jsedwards
-- Removed defintion of _LARGEFILE64_SOURCE, now using _FILE_OFFSET_BITS=64.
--
-- Revision 1.22  2007/08/01 01:17:50  jsedwards
-- Changed to accept 0025 compressed files, as well as 0023 and 0024.
--
-- Revision 1.21  2007/07/30 16:04:35  jsedwards
-- Changed to allow either 0023 or 0024 compressed files (since they are
-- identical) to be read in.
--
-- Revision 1.20  2007/07/03 01:21:46  jsedwards
-- Changed to allow reading a 0023 compressed file into a 0024 storage device,
-- to allow upgrading.
--
-- Revision 1.19  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.18  2007/06/28 13:46:31  jsedwards
-- Added assert to verify num_blocks stays in sync with nwos_private_blocks_used.
--
-- Revision 1.17  2007/06/28 13:44:56  jsedwards
-- Changed so that final message about blocks used and total blocks mismatch
-- are put in log as well as printed to the screen.
--
-- Revision 1.16  2007/06/26 13:55:23  jsedwards
-- Changes for new (0023) format where chunks need to be allocated.
--
-- Revision 1.15  2007/06/20 22:47:45  jsedwards
-- Add check to make sure disk is empty first, and if restoring from 0021
-- compressed also fix the number of blocks used for bit maps when changing
-- total blocks on disk.
--
-- Revision 1.14  2007/06/20 03:49:41  jsedwards
-- Fix to allow expanding a 0021 compressed file into a 0022 disk.
--
-- Revision 1.13  2007/06/19 19:04:19  jsedwards
-- Removed stuff for public objects because they are now stored separate from
-- the private objects.
--
-- Revision 1.12  2007/03/27 11:34:21  jsedwards
-- Change to deal with separate public and private last modified times in
-- disk header and add code to convert from 0020 archive (single time).
--
-- Revision 1.11  2007/03/06 11:44:32  jsedwards
-- Added code to verify that it is ok to write disk.
--
-- Revision 1.10  2007/02/27 12:27:19  jsedwards
-- Removed functions that were duplicates of functions in objectify.c and
-- changed to use the regular functions in objectify.c.
--
-- Revision 1.9  2007/02/25 13:56:33  jsedwards
-- Added #if'd code for importing a 0017 compressed file into 0018.
--
-- Revision 1.8  2007/02/11 15:15:20  jsedwards
-- Change 'sprintf' calls to 'snprintf' calls so the OpenBSD linker will stop
-- whining.
--
-- Revision 1.7  2007/02/11 14:41:26  jsedwards
-- Change all 'off64_t' and 'lseek64' references to 'off_t' and 'lseek',
-- because BSD doesn't dig the whole brain damaged 64 bit thing.
--
-- Revision 1.6  2007/01/17 13:08:29  jsedwards
-- Change to use Disk_Header structure instead of hard coded addresses.
--
-- Revision 1.5  2006/12/02 17:28:27  jsedwards
-- Changed to write header back to disk so update time is restored too.
--
-- Revision 1.4  2006/11/11 12:01:03  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.3  2006/11/02 11:49:28  jsedwards
-- Fixed all cases where 'z' was used as a format for 'off64_t' values because
-- the older compiler complains.
--
-- Revision 1.2  2006/10/26 01:51:26  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.9  2006/10/24 03:11:43  jsedwards
-- Change to use disk size stored on disk instead of defined constant.
--
-- Revision 1.1.2.8  2006/10/19 01:54:00  jsedwards
-- Fixed format specifiers for off64_t and uint32 which is now an unsigned
-- instead of unsigned long.
--
-- Revision 1.1.2.7  2006/10/11 13:19:04  jsedwards
-- Change so that first block is not written over the first block on the disk
-- because it over writes the disk block size (and version).
--
-- Revision 1.1.2.6  2006/10/10 07:32:36  jsedwards
-- Move bit map defines to objectify_private.h instead of spread all over.
--
-- Revision 1.1.2.5  2006/10/09 13:06:06  jsedwards
-- Change bit map size from 256 to 8192 (to improve speed, disk_usage program
-- took over an hour to run with 256 byte bit maps).
--
-- Revision 1.1.2.4  2006/10/08 13:46:38  jsedwards
-- Fix order of bits within the byte (calling 0 the most significant bit).
--
-- Revision 1.1.2.3  2006/10/07 23:35:58  jsedwards
-- Added code to fill in the block bit maps.
--
-- Revision 1.1.2.2  2006/10/07 22:30:18  jsedwards
-- Changed to convert the reference id to a block number so it can be expanded
-- into a disk drive or partition.
--
-- Revision 1.1.2.1  2006/09/17 13:53:44  jsedwards
-- Program to un-compress a non-sparse file into the regular sparse file.
--
*/


#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "objectify_private.h"



static void print_usage(char *program)
{
    fprintf(stderr, "usage: %s input-file\n", program);
}


int main(int argc, char* argv[])
{
    uint8 block[FILE_BLOCK_SIZE];
    size_t bytes_read;
    FILE* ifp;
    uint32 ref;
    uint32 hash;
    int num_blocks = 0;
    uint32 total_private_blocks;
    uint32 used_private_blocks;
    uint32 used_private_chunks;
    uint32 num_chunks;
    Disk_Header compressed_header;
#if 0
    Disk_Header_0021 old_header;
#endif
    char answer[8];
    char msg[128];


    if (argc != 2)
    {
	print_usage(argv[0]);
	exit(1);
    }

    if (*argv[1] == '-')
    {
	fprintf(stderr, "error: this program doesn't have any options\n");
	print_usage(argv[0]);
	exit(1);
    }

    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(NULL, 0, 0, DEFAULT_TYPE_RW, DEFAULT_FILE);


    /* Verify that is what s/he really wants to do */

    fprintf(stderr, "\n");
    fprintf(stderr, "WARNING: the contents of %s will be OVERWRITTEN!!!\n", DEFAULT_FILE);
    fprintf(stderr, "\n");
    fprintf(stderr, "Do you want to continue? (enter `yes' to write %s) ", DEFAULT_FILE);

    fflush(stderr);

    fgets(answer, sizeof(answer), stdin);

    printf("\n");

    if (strcmp(answer, "yes\n") != 0)
    {
	exit(1);
    }


    /* open the compressed archive */

    ifp = fopen(argv[1], "r");

    if (ifp == NULL)
    {
	perror(argv[1]);
	exit(1);
    }

    bytes_read = fread(block, 1, FILE_BLOCK_SIZE, ifp);

    if (memcmp(&block[0], MAGIC_NUMBER, 4) != 0)
    {
	printf("Compressed file missing magic number: %02x%02x%02x%02x\n",
	       block[0],
	       block[1],
	       block[2],
	       block[3]);
	fclose(ifp);
	exit(1);
    }


    /* verify that the number of used blocks is consistent with empty disk */
    num_chunks = (nwos_total_private_blocks + BLOCKS_IN_CHUNK - 1) / BLOCKS_IN_CHUNK; /* round up */

    if (nwos_used_private_blocks != 0 || nwos_used_private_chunks != 0)
    {
	fprintf(stderr, "ERROR: disk not empty, chunks used: %u  blocks used: %u!.\n", 
		nwos_used_private_blocks, nwos_used_private_chunks);
	exit(1);
    }


    /* Version 0023, 0024 and 0025 archives are identical so allow expanding 0023 or 0024 */

    if (memcmp(&block[4], "0023", 4) == 0 || memcmp(&block[4], "0024", 4) == 0 || memcmp(&block[4], "0025", 4) == 0)
    {
	fprintf(stderr, "WARNING: compressed file is version 002%c, is compatible.\n", block[7]);
	printf("WARNING: compressed file is version 002%c, is compatible.\n", block[7]);

	sleep(5);
    }
    else
    {
	if (memcmp(VERSION_STRING, &block[4], 4) != 0)
	{
	    printf("Version mismatch - disk: %s   file: %c%c%c%c\n",
		   VERSION_STRING,
		   compressed_header.version_string[0],
		   compressed_header.version_string[1],
		   compressed_header.version_string[2],
		   compressed_header.version_string[3]);
	    fclose(ifp);
	    exit(1);
	}

    }  /* remove these two braces if the the blocks and chunks */
    {  /* need to be read differently for the old file format */ 

	memcpy(&compressed_header, block, sizeof(compressed_header));

	nwos_4_uint8_to_uint32(compressed_header.total_blocks, &total_private_blocks);
	nwos_4_uint8_to_uint32(compressed_header.used_blocks, &used_private_blocks);
	nwos_4_uint8_to_uint32(compressed_header.used_chunks, &used_private_chunks);

	printf("compressed file version: %c%c%c%c\n",
	       compressed_header.version_string[0],
	       compressed_header.version_string[1],
	       compressed_header.version_string[2],
	       compressed_header.version_string[3]);
    }

    printf("   total_private_blocks: %u  used_private_blocks: %u\n",
	   total_private_blocks, used_private_blocks);

    if (used_private_chunks > nwos_total_private_chunks)
    {
	fprintf(stderr, "used chunks: %u greater than disk space: %u\n",
		used_private_chunks, nwos_total_private_chunks);
	exit(1);
    }

    /* finally copy the objects */

    bytes_read = fread(block, 1, FILE_BLOCK_SIZE, ifp);

    while (bytes_read == FILE_BLOCK_SIZE)
    {
	nwos_4_uint8_to_uint32(&block[4], &ref);

	hash = nwos_hash_uint32_ref(ref);

	if (hash == 0)
	{
	    nwos_allocate_new_chunk(ref);

	    hash = nwos_hash_uint32_ref(ref);

	    assert(hash != 0);
	}

	printf("id: %08x  block: %08x\n", ref, hash);
	fflush(stdout);

	if (nwos_test_bit_in_map(hash))
	{
	    fprintf(stderr, "ERROR: block is already used!\n");
	    exit(1);
	}

	nwos_set_bit_in_map(hash);

	nwos_write_block((ObjRef*)&block[4], block);

	num_blocks++;

	assert(num_blocks == nwos_used_private_blocks);

	bytes_read = fread(block, 1, FILE_BLOCK_SIZE, ifp);
    }

    if (ferror(ifp))
    {
	perror(argv[1]);
	exit(1);
    }

    if (fclose(ifp) != 0)
    {
	perror(argv[1]);
	exit(1);
    }

    nwos_terminate_objectify();

    snprintf(msg, sizeof(msg), "Number of blocks written: %d  chunks_used: %u",
	     num_blocks, nwos_used_private_chunks);
    nwos_log(msg);
    printf("%s\n", msg);

    if (used_private_blocks != nwos_used_private_blocks)
    {
	snprintf(msg, sizeof(msg), "WARNING: used_private_blocks mismatch, disk: %u  compressed %u\n",
		nwos_used_private_blocks, used_private_blocks);
	fprintf(stderr, "%s\n", msg);
    }

    return 0;
}

