/*
--             This file is part of the New World OS project
--                 Copyright (C) 2006-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: list_discs.c,v $
-- Revision 1.19  2008/02/03 01:05:23  jsedwards
-- Changed DEFAULT_TYPE_RO to READ_ONLY.
--
-- Revision 1.18  2007/07/05 18:24:02  jsedwards
-- Changed to not print copy if there is a newer version of the object.
--
-- Revision 1.17  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.16  2007/06/06 00:19:33  jsedwards
-- Fixed name of storage location class.
--
-- Revision 1.15  2007/06/03 22:02:39  jsedwards
-- Changed so that location printed can be either a Location or a Person.
--
-- Revision 1.14  2007/03/03 15:25:16  jsedwards
-- Changed to list each disc list and then each copy of it.
--
-- Revision 1.13  2007/02/24 01:09:49  jsedwards
-- Add "--contents" option to list all files on disc as well.  Default is
-- now to just list discs.
--
-- Revision 1.12  2007/02/11 16:58:26  jsedwards
-- Changed so DEFAULT_TYPE has to specify RO (Read-Only) or RW (Read-Write).
--
-- Revision 1.11  2007/01/20 13:39:44  jsedwards
-- Change so that it prints a nice error message instead of asserting when
-- private File_Path or Disc_Copy classes don't exist.
--
-- Revision 1.10  2007/01/07 03:24:53  jsedwards
-- Changed to work with new path_and_file_association object.
--
-- Revision 1.9  2006/12/21 13:08:36  jsedwards
-- Hack to make it compile with new split public and private classes,
-- NON-FUNCTIONAL!
--
-- Revision 1.8  2006/12/19 14:52:18  jsedwards
-- Kludged version to compile with new 'file path' structure without 'file'
-- feature.  DOES NOT function correctly!
--
-- Revision 1.7  2006/12/01 14:31:30  jsedwards
-- Changed to use new malloc_reference_list and free_reference_list functions
-- instead of inlining the code.
--
-- Revision 1.6  2006/11/27 13:50:44  jsedwards
-- Changed to use variable sized counts for disc lists (so number of files can
-- be larger than 127).
--
-- Revision 1.5  2006/11/19 15:28:20  jsedwards
-- Made get_disc_list_object_size a global function.
--
-- Revision 1.4  2006/11/18 15:09:10  jsedwards
-- Added "max_size" parameter to read_variable_sized_object_from_disk because
-- objects are no longer limited to one file block.
--
-- Revision 1.3  2006/11/18 14:37:02  jsedwards
-- Change size of kludge buffer for reading disc list to max size of disc
-- list instead of file block size, because now disc list objects can be
-- larger than one file block.
--
-- Revision 1.2  2006/11/11 12:01:04  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.1  2006/11/07 14:04:46  jsedwards
-- Program to list all of the discs logged in the system.
--
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "objectify.h"


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}


int main(int argc, char* argv[])
{
    ObjRef root_object_ref;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    C_struct_Class_Definition class_def_obj;
    C_struct_Disc_Copy copy_obj;
    C_struct_Storage_Location location_obj;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_File file_obj;
    C_struct_Person person_obj;
    uint8 kludge[MAX_SIZE_DISC_LIST];
    uint8 cludge[FILE_BLOCK_SIZE];
    C_struct_Disc_List* ptr_list_obj = (C_struct_Disc_List*)kludge;
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)cludge;
    ObjRef list_class_ref;
    ObjRef copy_class_ref;
    ObjRef path_class_ref;
    ObjRef loc_class_ref;
    ObjRef person_class_ref;
    ObjRef name_ref;
    ObjRef object_class;
    ReferenceList* list_class_ref_list;
    ReferenceList* disc_list_ref_list;
    char name_str[128];
    char id_str[13];
    uint32 file_size;
    int list_class_num_refs;
    int disc_list_num_refs;
    int i;
    int j;
    int count;
    int copies;
    bool do_contents = false;

    if (argc > 1)
    {
	if (argc == 2 && strcmp(argv[1], "--contents") == 0)
	{
	    do_contents = true;
	}
	else
	{
	    fprintf(stderr, "usage: %s [--contents]\n", argv[0]);
	    exit(1);
	}
    }

    printf("\n");

    nwos_get_key_from_password(big_key, sizeof(big_key));

    printf("\n");

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_initialize_objectify(bf_key, linear, serial, READ_ONLY, DEFAULT_FILE);

    nwos_set_root_object(&root_object_ref);

    if (!nwos_find_private_class_definition("FILE PATH", &path_class_ref))
    {
	fprintf(stderr, "No private files found.\n");
	exit(1);
    }

    if (!nwos_find_private_class_definition("DISC LIST", &list_class_ref))
    {
	fprintf(stderr, "No private disc lists found.\n");
	exit(1);
    }

    if (!nwos_find_private_class_definition("DISC COPY", &copy_class_ref))
    {
	fprintf(stderr, "No private disc copies found.\n");
	exit(1);
    }

    if (!nwos_find_private_class_definition("STORAGE LOCATION", &loc_class_ref))
    {
	fprintf(stderr, "Warning: no locations found.\n");
	void_reference(&loc_class_ref);
    }

    if (!nwos_find_private_class_definition("PERSON", &person_class_ref))
    {
	fprintf(stderr, "Warning: no persons found.\n");
	void_reference(&person_class_ref);
    }

    nwos_read_class_definition(&list_class_ref, &class_def_obj);

    list_class_ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    list_class_num_refs = list_class_ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < list_class_num_refs; i++)
    {
	nwos_get_object_class(&list_class_ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &list_class_ref))
	{
	    nwos_read_variable_sized_object_from_disk(&list_class_ref_list->references[i], kludge, sizeof(kludge), &nwos_get_disc_list_object_size);

	    memcpy(id_str, ptr_list_obj->id, 12);
	    id_str[12] = '\0';

	    count = nwos_decode_variable_sized_count(ptr_list_obj->count);

	    printf("disc: %s  files: %u\n", id_str, count);

	    disc_list_ref_list = nwos_malloc_reference_list(&ptr_list_obj->header.object.references);

	    disc_list_num_refs = disc_list_ref_list->common_header.num_refs;

	    copies = 0;

	    for (j = 0; j < disc_list_num_refs; j++)
	    {
		nwos_get_object_class(&disc_list_ref_list->references[j], &object_class);

		if (is_same_object(&object_class, &copy_class_ref))
		{
		    copies++;

		    nwos_read_object_from_disk(&disc_list_ref_list->references[j], &copy_obj, sizeof(copy_obj));

		    nwos_get_object_class(&copy_obj.location, &object_class);

		    if (is_void_reference(&copy_obj.header.object.next_version))
		    {
			if (is_same_object(&object_class, &loc_class_ref))
			{
			    nwos_read_object_from_disk(&copy_obj.location, &location_obj, sizeof(location_obj));
			    copy_reference(&name_ref, &location_obj.name);
			}
			else if (is_same_object(&object_class, &person_class_ref))
			{
			    nwos_read_object_from_disk(&copy_obj.location, &person_obj, sizeof(person_obj));
			    if (is_void_reference(&person_obj.goes_by))
			    {
				copy_reference(&name_ref, &person_obj.first_name);
			    }
			    else
			    {
				copy_reference(&name_ref, &person_obj.goes_by);
			    }
			}
			else
			{
			    fprintf(stderr, "Error: unknown location class: %02x%02x%02x%02x\n",
				    copy_obj.location.id[0],
				    copy_obj.location.id[1],
				    copy_obj.location.id[2],
				    copy_obj.location.id[3]);
			}

			assert(nwos_name_to_string(&name_ref, name_str, sizeof(name_str)));

			printf(" copy number: %d  location: %s\n", copy_obj.copy_number, name_str);
		    }
		}
	    }

	    nwos_free_reference_list(disc_list_ref_list);
	    disc_list_ref_list = NULL;

	    if (copies == 0)
	    {
		printf("  NO copies logged!\n");
	    }

	    if (do_contents)    /* list all the files as well */
	    {
		for (j = 0; j < count; j++)
		{
		    nwos_read_object_from_disk(&ptr_list_obj->files[j], &assoc_obj, sizeof(assoc_obj));

		    nwos_read_variable_sized_object_from_disk(&assoc_obj.path, cludge, sizeof(cludge), &get_path_object_size);

		    memcpy(name_str, ptr_path_obj->storage, ptr_path_obj->count);
		    name_str[ptr_path_obj->count] = '\0';

		    nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj));

		    file_size = (file_obj.size[0] << 24) | (file_obj.size[1] << 16) | (file_obj.size[2] << 8) | (file_obj.size[3]);

		    printf("   %s %u\n", name_str, file_size);
		}
	    }
	}
    }

    nwos_free_reference_list(list_class_ref_list);
    list_class_ref_list = NULL;

    nwos_terminate_objectify();

    return 0;
}


