/*
--             This file is part of the New World OS project
--                   Copyright (C) 2007  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: rename_file.c,v $
-- Revision 1.11  2007/10/04 11:34:16  jsedwards
-- Fixed compiler warning about "no_history" possibly being used uninitialized.
--
-- Revision 1.10  2007/09/23 13:20:03  jsedwards
-- Added code to optionally do the rename the way it was done in Revision 1.5
-- and before, by overwriting the string in the path object.
--
-- Revision 1.9  2007/09/23 13:10:16  jsedwards
-- Changed handling of arguments so that the regular code doesn't explicitly
-- access argv[1] or argv[2] but through a pointer.
--
-- Revision 1.8  2007/09/23 12:23:53  jsedwards
-- Moved code out of main that renamed the file by creating a new path, into
-- new 'create_new_path' function.
--
-- Revision 1.7  2007/09/02 19:47:25  jsedwards
-- Added call to set the block estimate.
--
-- Revision 1.6  2007/08/25 20:16:11  jsedwards
-- Changed completely so that instead of just changing the string in the Path
-- object it creates a new path object with the new string and a new Fath and
-- File Association object and hooks up the next_verision and prev_version
-- links.
--
-- Revision 1.5  2007/08/25 00:45:11  jsedwards
-- Added code to print error and exit if the new file name is already in the
-- system.
--
-- Revision 1.4  2007/08/24 23:59:06  jsedwards
-- Changed so that file_is_used_in_disc_list is only called if a private
-- disc_list class exists (meaning there are private disc_list objects).
--
-- Revision 1.3  2007/08/24 23:48:05  jsedwards
-- Moved code to find out if a path (file name) was used in a disc_list and
-- made it into a new function: file_is_used_in_disc_list().
--
-- Revision 1.2  2007/08/24 19:44:51  jsedwards
-- Added code to print disc_list id instead of reference number.
--
-- Revision 1.1  2007/08/24 19:11:18  jsedwards
-- Initial version.
--
*/

#include "crc32.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify_private.h"


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}



static char* file_is_used_in_disc_list(ObjRef* disc_list_class_ref, C_struct_Path_And_File_Association* assoc_obj)
{
    ObjRef object_class;
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_list_obj = (C_struct_Disc_List*)kludge;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    static char id[sizeof(ptr_list_obj->id) + 1];
    char* result = NULL;


    ref_list = nwos_malloc_reference_list(&assoc_obj->header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("num_refs: %d\n", num_refs);

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, disc_list_class_ref))
	{
	    nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &nwos_get_disc_list_object_size);

	    memcpy(id, ptr_list_obj->id, sizeof(id));
	    id[sizeof(ptr_list_obj->id)] = '\0';

	    result = id;
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return result;
}


/* NOTE: assoc_obj is modified by this function! */

void create_new_path(char* new_path, C_struct_Path_And_File_Association *assoc_obj)
{
    C_struct_Path_And_File_Association temp;
    ObjRef assoc_class_ref;
    ObjRef new_path_ref;
    ObjRef old_assoc_ref;
    ObjRef new_assoc_ref;


    assert(nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref));

    assert(nwos_create_file_path(new_path, &new_path_ref) == CREATED_NEW);

    copy_reference(&old_assoc_ref, &assoc_obj->header.common.id);

    nwos_generate_new_id(&new_assoc_ref);


    /***********************************************************************/
    /* Point to the new association object from the old association object */
    /***********************************************************************/

    assert(is_void_reference(&assoc_obj->header.object.next_version));   /* above code checks this */

    copy_reference(&assoc_obj->header.object.next_version, &new_assoc_ref);

    nwos_crc32_calculate((uint8*) &assoc_obj->header.object, sizeof(ObjectHeader), assoc_obj->header.common.header_chksum);

    nwos_overwrite_object_to_disk(&old_assoc_ref, assoc_obj, sizeof(*assoc_obj));

    /* read back to verify */
    nwos_read_object_from_disk(&old_assoc_ref, &temp, sizeof(temp));
    assert(memcmp(&temp, assoc_obj, sizeof(temp)) == 0);

    void_reference(&assoc_obj->header.object.next_version);   /* remove it before creating the new */


    /*****************************************/
    /* Now create the new association object */
    /*****************************************/

    nwos_fill_in_common_header(&assoc_obj->header.common, &new_assoc_ref, &assoc_class_ref);

    copy_reference(&assoc_obj->header.object.prev_version, &old_assoc_ref);

    copy_reference(&assoc_obj->path, &new_path_ref);

    nwos_create_reference_list(&new_assoc_ref, &assoc_obj->header.object.references);

    nwos_crc32_calculate((uint8*) &assoc_obj->header.object, sizeof(ObjectHeader), assoc_obj->header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &assoc_obj->path, sizeof(C_struct_Path_And_File_Association) - sizeof(EveryObject), assoc_obj->header.common.data_chksum);

    nwos_write_object_to_disk(&new_assoc_ref, assoc_obj, sizeof(*assoc_obj));

    nwos_add_to_references(&new_assoc_ref, &assoc_class_ref);
    nwos_add_to_references(&new_assoc_ref, &assoc_obj->path);
    nwos_add_to_references(&new_assoc_ref, &assoc_obj->file);

    /* read back to verify */
    nwos_read_object_from_disk(&new_assoc_ref, &temp, sizeof(temp));
    assert(memcmp(&temp, assoc_obj, sizeof(temp)) == 0);
}


int main(int argc, char* argv[])
{
    ObjRef root_object_ref;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    C_struct_Path_And_File_Association assoc_obj;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    ObjRef list_class_ref;
    ObjRef old_path_ref;
    ObjRef dummy_ref;
    ObjRef old_assoc_ref;
    size_t length;
    char *id;
    char* old_name = "-";
    char* new_name = "-";
    bool no_history = false;


    if (argc == 3)
    {
	old_name = argv[1];
	new_name = argv[2];
    }
    else if (argc == 4 && strcmp(argv[1], "--no-history") == 0)
    {
	no_history = true;
	old_name = argv[2];
	new_name = argv[3];
    }

    if (*old_name == '-' || *new_name == '-')
    {
	fprintf(stderr, "usage: %s [--no-history] old-file-name new-file-name\n", argv[0]);
	exit(1);
    }

    length = strlen(new_name);

    if (length > 255)
    {
	fprintf(stderr, "Error: maximum length of file name is 255\n");
	exit(1);
    }


    printf("\n");

    nwos_get_key_from_password(big_key, sizeof(big_key));

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_initialize_objectify(bf_key, linear, serial, DEFAULT_TYPE_RW, DEFAULT_FILE);

    nwos_set_root_object(&root_object_ref);

    nwos_set_security_level(Security_Very_Low);

    nwos_set_block_estimate(4);


    /**************************/
    /* Check the names are ok */
    /**************************/

    if (!nwos_find_file_path(old_name, &old_path_ref))
    {
	fprintf(stderr, "Could not find: %s in system.\n", old_name);
	nwos_terminate_objectify();
	exit(1);
    }

    if (nwos_number_of_files_for_path(&old_path_ref) > 1)
    {
	fprintf(stderr, "This version cannot change a file name if there is more than one file referencing it.\n");
	nwos_terminate_objectify();
	exit(1);
    }

    if (nwos_find_file_path(new_name, &dummy_ref))
    {
	fprintf(stderr, "Error: %s already exists in the system\n", new_name);
	nwos_terminate_objectify();
	exit(1);
    }


    nwos_read_variable_sized_object_from_disk(&old_path_ref, kludge, sizeof(kludge), &get_path_object_size);

    assert(nwos_file_path_to_path_and_file_association(&old_path_ref, 0, &old_assoc_ref));

    nwos_read_object_from_disk(&old_assoc_ref, &assoc_obj, sizeof(assoc_obj));

    if (!is_void_reference(&assoc_obj.header.object.next_version))
    {
	fprintf(stderr, "The file that had the name %s has already been renamed.\n", old_name);
	nwos_terminate_objectify();
	exit(1);
    }


    /******************************************************/
    /* before changing verify it is NOT in any disc lists */
    /******************************************************/

    if (nwos_find_private_class_definition("DISC LIST", &list_class_ref))
    {
	id = file_is_used_in_disc_list(&list_class_ref, &assoc_obj);

	if (id != NULL)   /* the file was found in disc_list with this id */
	{
	    fprintf(stderr, "\nError: found file %s in disc list: %s\n\n", old_name, id);
	    fprintf(stderr, "           *** Name NOT changed! ***\n\n");

	    nwos_terminate_objectify();
	    exit(1);
	}
    }

    assert(is_same_object(&ptr_path_obj->header.common.id, &old_path_ref));
    assert(is_same_object(&assoc_obj.header.common.id, &old_assoc_ref));

    if (no_history)
    {
	ptr_path_obj->count = length;

	memcpy(ptr_path_obj->storage, new_name, length);

	nwos_crc32_calculate((uint8*) &ptr_path_obj->count, sizeof(C_struct_File_Path) + length - sizeof(EveryObject), ptr_path_obj->header.common.data_chksum);

	nwos_overwrite_object_to_disk(&old_path_ref, ptr_path_obj, get_path_object_size(ptr_path_obj));
    }
    else
    {
	create_new_path(new_name, &assoc_obj);   /* assoc_obj is modified by this function */
    }

    nwos_terminate_objectify();

    return 0;
}


