/*
--             This file is part of the New World OS project
--                 Copyright (C) 2004-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: security.c,v $
-- Revision 1.13  2008/03/12 04:06:44  jsedwards
-- Changed to use the GNU MD5 context and functions instead of the RSA context
-- and functions.
--
-- Revision 1.12  2007/09/15 14:31:58  jsedwards
-- Added conditional code so that if TEST_PASS_PHRASE is defined, it will use
-- it instead of asking for pass phrase from user.
--
-- Revision 1.11  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.10  2007/01/16 14:12:02  jsedwards
-- Changed to use MD5 and SHA1 to generate key from short pass phrases.
--
-- Revision 1.9  2007/01/16 13:37:22  jsedwards
-- Eliminated get_short_key_from_password and made get_key_from_password call
-- convert_short_pass_phrase_to_reference if the password entered was too
-- short by itself.
--
-- Revision 1.8  2007/01/16 02:27:04  jsedwards
-- Added new routine for short pass phrases.
--
-- Revision 1.7  2006/12/01 14:37:23  jsedwards
-- Fix the year in the copyright.
--
-- Revision 1.6  2006/11/11 12:01:06  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.5  2006/10/26 01:51:29  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.4.2.4  2006/09/09 13:03:33  jsedwards
-- Moved "create_root_object" routine from security.c to objectify.c so that
-- the security module didn't have references to so many other parts of the
-- system.
--
-- Revision 1.4.2.3  2006/09/06 13:15:25  jsedwards
-- Removed nwos_seed_sequence function and instead pass pointers to the values
-- in the call to next_sequence function.
--
-- Revision 1.4.2.2  2006/09/02 15:04:30  jsedwards
-- Add reference list class reference to root object because now it has to
-- read the reference list to get it's size and we can't easily read it
-- without the read routine verifying it's class.
--
-- Revision 1.4.2.1  2006/09/02 01:12:25  jsedwards
-- Add root object reference in call to fill_in_common_header so it can put
-- it into the header.
--
-- Revision 1.4  2005/12/30 03:16:12  jsedwards
-- Changed "Password" prompt to "Pass phrase".
--
-- Revision 1.3  2005/12/28 13:03:24  jsedwards
-- Added sequence generator (from Fine random_number_generator).  Changed the
-- get reference from password routine to get a variable length key from the
-- password.  It's actually more like a pass phrase now because it requires
-- the input to be long enough to get enough bits to fill up the key.
--
-- Revision 1.2  2005/12/27 19:45:27  jsedwards
-- Added routine to create a root object.
--
-- Revision 1.1  2005/12/27 18:02:42  jsedwards
-- Initial version converts a pass phrase into an object reference (ObjRef).
--
*/

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include "crc32.h"
#include "objectify.h"
#include "objectify_private.h"

#include "gnu/md5.h"
#include "gnu/sha1.h"


#define MAX_NUMBER_SIZE 512    /* number can be up to 4096 bits */
#define SHORT_NUMBER_SIZE 36

#define MULTIPLIER 3141592621U
#define ADDEND     101U


uint32 nwos_next_sequence(uint32* linear, uint32* serial)
{
    uint32 feedback;

    *linear = (*linear * MULTIPLIER) + ADDEND;

    /* feedback_mask: BIT 32 is 0x80004000 */
    /* exclusive or of bits 31 and 14 */
    feedback = ((*serial >> 31) ^ (*serial >> 14)) & 1;

    *serial = (*serial << 1) | feedback;

    return *linear ^ *serial;
}


void convert_pass_phrase_to_reference(char* pass_phrase, uint8 key[], size_t key_size)
{
    int i;
    int j;
    uint8 number[MAX_NUMBER_SIZE];
    uint8 mult;
    uint32 acc;
    char *p;

    mult = '~' - ' ' + 2;

#if 0
    printf("mult: %02x\n", mult);
#endif

    memset(&number, 0, MAX_NUMBER_SIZE);

    for (p = pass_phrase; *p != '\0'; p++)
    {
	acc = *p;

	if (' ' <= acc && acc <= '~')
	{
	    acc = acc - ' ' + 1;
	}
	else
	{
	    acc = 0;
	}

	/* multiply and add big endian ([0] is most significant byte) */
	for (i = MAX_NUMBER_SIZE; i > 0; i--)
	{
	    acc = (number[i-1] * mult) + acc;
	    number[i-1] = acc;
	    acc = acc >> 8;
	}
#if 0
	for (i = 0; i < MAX_NUMBER_SIZE; i++) if (number[i] != 0) break;

	while (i < MAX_NUMBER_SIZE)
	  {
	    printf("%02x", number[i++]);
	  }
	printf("\n");
#endif
    }

    /* exclusive or each reference size chunk into the result */

    memset(key, 0, key_size);

    j = key_size - 1;
    for (i = MAX_NUMBER_SIZE; i > 0; i--)
    {
	key[j] = key[j] ^ number[i-1];
	j--;
	if (j < 0) j = key_size - 1;
    }
}


void convert_short_pass_phrase_to_reference(char* pass_phrase, uint8 key[], size_t key_size)
{
#ifndef PUBLIC_MODE
    int i;
    int j;
    uint8 number[SHORT_NUMBER_SIZE];
    uint8 mult;
    uint32 acc;
    char *p;
    struct md5_ctx md5_context;    /* MD5 checksum context */
    struct sha1_ctx sha1_context;
    uint8 md5_digest[16];
    uint8 sha1_digest[20];

    mult = '~' - ' ' + 2;

#if 0
    printf("mult: %02x\n", mult);
#endif

    memset(&number, 0, SHORT_NUMBER_SIZE);

    for (p = pass_phrase; *p != '\0'; p++)
    {
	acc = *p;

	if (' ' <= acc && acc <= '~')
	{
	    acc = acc - ' ' + 1;
	}
	else
	{
	    acc = 0;
	}

	/* multiply and add big endian ([0] is most significant byte) */
	for (i = SHORT_NUMBER_SIZE; i > 0; i--)
	{
	    acc = (number[i-1] * mult) + acc;
	    number[i-1] = acc;
	    acc = acc >> 8;
	}
#if 0
	for (i = 0; i < SHORT_NUMBER_SIZE; i++) if (number[i] != 0) break;

	while (i < SHORT_NUMBER_SIZE)
	  {
	    printf("%02x", number[i++]);
	  }
	printf("\n");
#endif
    }

    for (i = 0; i < SHORT_NUMBER_SIZE; i++) if (number[i] != 0) break;

    md5_init_ctx(&md5_context);   /* initialize the MD5 checksum context */
    md5_process_bytes(&number[i], SHORT_NUMBER_SIZE-i, &md5_context);    /* include this data in the md5 checksum */
    md5_finish_ctx(&md5_context, md5_digest);   /* finish computing the md5 sum */

    sha1_init_ctx(&sha1_context);
    sha1_process_bytes(&number[i], SHORT_NUMBER_SIZE-i, &sha1_context);    /* include this data in the sha1 checksum */
    sha1_finish_ctx(&sha1_context, sha1_digest);

    /* exclusive or each reference size chunk into the result */

    memset(key, 0, key_size);

    j = 0;
    for (i = 0; i < 20; i++)
    {
	key[j] = key[j] ^ sha1_digest[i];
	j++;
	if (j == key_size) j = 0;
    }
    for (i = 0; i < 16; i++)
    {
	key[j] = key[j] ^ md5_digest[i];
	j++;
	if (j == key_size) j = 0;
    }
#endif
}


void nwos_get_key_from_password(uint8 key[], size_t key_size)
{
    bool ok = false;
    char buffer[MAX_NUMBER_SIZE];
    char *p;
    int chars_needed;

    chars_needed = (key_size * 80) / 65; /* we get 6.5 bits for each character entered, we need 8 for each output byte */

    while (chars_needed * 65 < key_size * 80) chars_needed++;   /* adjust up for truncation errors */

#ifdef TEST_PASS_PHRASE
    strncpy(buffer, TEST_PASS_PHRASE, sizeof(buffer));
#else
    while (!ok)
    {
	fprintf(stderr, "Pass phrase: ");
	fflush(stderr);

	fgets(buffer, MAX_NUMBER_SIZE, stdin);

	p = strchr(buffer, '\n');   /* find the newline char */

	if (p == NULL)    /* line was tool long */
	{
	    while (p == NULL) 
	    {
		fgets(buffer, MAX_NUMBER_SIZE, stdin);
		p = strchr(buffer, '\n');   /* find the newline char */
	    }
	    printf("password was too long, must be less than %d characters\n", MAX_NUMBER_SIZE);
	}
	else     /* line was ok */
	{
	    *p = '\0';   /* eliminate the newline character */

	    if (strlen(buffer) < 10)
	    {
		printf("password was too short, must be at least 10 characters for\n"
		       "short password or %d characters for higher security password\n", chars_needed);
	    }
	    else
	    {
		ok = true;   /* we should be good to go */
	    }
	}
    }
#endif

    if (strlen(buffer) < chars_needed)
    {
	convert_short_pass_phrase_to_reference(buffer, key, key_size);
    }
    else
    {
	convert_pass_phrase_to_reference(buffer, key, key_size);
    }
}




#if 0
int main(int argc, char* argv[])
{
  ObjRef key;
  int i;

  key = convert_pass_phrase_to_reference(argv[1]);

  for (i = 0; i < sizeof(ObjRef); i++) printf("%02x", key.id[i]);
  printf("\n");

  return 0;
}
#endif

#if 0
int main(int argc, char* argv[])
{
  uint8 big_key[28];
  int i;

  nwos_get_key_from_password(big_key, sizeof(big_key));

  for (i = 0; i < sizeof(big_key); i++)
  {
      printf("%02x", big_key[i]);
  }

  printf("\n");

  return 0;
}
#endif

