/*
--             This file is part of the New World OS project
--                    Copyright (C) 2008 QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: read_md5_file.c,v $
-- Revision 1.4  2008/05/08 11:58:12  jsedwards
-- Fixed so that it will accept an asterisk before the file name.
--
-- Revision 1.3  2008/02/23 01:40:24  jsedwards
-- Removed print statement accidentally left in previous check in.
--
-- Revision 1.2  2008/02/16 13:11:38  jsedwards
-- Changed to call new nwos_create_file_with_only_md5 function with entries
-- read from file.
--
-- Revision 1.1  2008/02/14 13:05:29  jsedwards
-- Initial version just reads file in and prints it back out.
--
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "../objectify_private.h"


#define MAX_ENTRIES 64

typedef uint8 MD5[16];

typedef struct {
    char* name;
    MD5 md5;
} File_and_MD5;


static uint8 convert_char_to_hex_value(char c)
{
    uint8 result = 0xff;

    if (isxdigit(c))
    {
	if (isdigit(c))
	{
	    result = c - '0';
	}
	else if (isupper(c))
	{
	    result = c - 'A' + 10;
	}
	else
	{
	    result = c - 'a' + 10;
	}
    }

    return result;
}


int main(int argc, char* argv[])
{
    ObjRef root_ref;
    ObjRef ref;
    int i;
    int entry = 0;
    ObjCreateResult result;
    FILE* fp;
    File_and_MD5 entries[MAX_ENTRIES];
    char line[128];
    char *p;
    uint8 upper;
    uint8 lower;
    size_t len;

    if (argc != 2)
    {
	fprintf(stderr, "usage: %s md5_file\n", argv[0]);
	exit(1);
    }

    fp = fopen(argv[1], "r");

    if (fp == NULL)
    {
	perror(argv[1]);
	exit(1);
    }

    while (fgets(line, sizeof(line), fp) != NULL)
    {
	assert(entry < MAX_ENTRIES);

	p = line;

	for (i = 0; i < sizeof(MD5); i++)
	{
	    upper = convert_char_to_hex_value(*p++);

	    if (upper < 16)
	    {
		lower = convert_char_to_hex_value(*p++);
	    }

	    if (upper > 15 || lower > 15)
	    {
		break;
	    }

	    entries[entry].md5[i] = (upper << 4) | lower;
	}

	len = 0;

	if (i == sizeof(MD5) && *p++ == ' ')
	{
	    if (*p == ' ' || *p == '*')
	    {
		p++;                      /* point to the first character of the file name */
		*strchr(p, '\n') = '\0';  /* replace the newline with a null */
		len = strlen(p);
	    }
	}

	if (len == 0)
	{
	    fprintf(stderr, "invalid line: %s\n", line);
	    fclose(fp);
	    exit(1);
	}

	entries[entry].name = malloc(len + 1);

	if (entries[entry].name == NULL)
	{
	    perror("allocating memory for name");
	    fclose(fp);
	    exit(1);
	}

	strlcpy(entries[entry].name, p, len + 1);

	entry++;
    }

    if (ferror(fp))
    {
	perror(argv[1]);
	fclose(fp);
	exit(1);
    }

    fclose(fp);

    if (entry == 0)
    {
	fprintf(stderr, "Empty file: %s\n", argv[1]);
	exit(1);
    }


    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(NULL, 0, 0, PUBLIC, NULL);

    printf("next_reference: %02x%02x%02x%02x\n", 
	   nwos_next_public_ref.id[0],
	   nwos_next_public_ref.id[1],
	   nwos_next_public_ref.id[2],
	   nwos_next_public_ref.id[3]);

    root_ref.id[0] = 0;
    root_ref.id[1] = 0;
    root_ref.id[2] = 0;
    root_ref.id[3] = 1;

    nwos_set_root_object(&root_ref);

    for (i = 0; i < entry; i++)
    {
	int j;

	for (j = 0; j < 16; j++)
	{
	    printf("%02x", entries[i].md5[j]);
	}
	printf("  %s\n", entries[i].name);

	result = nwos_create_file_with_only_md5(entries[i].name, entries[i].md5, &ref);

	assert(!is_void_reference(&ref));

	if (result == CREATED_NEW)
	{
	    printf("created new file path: %02x%02x%02x%02x\n", 
		   ref.id[0], ref.id[1], ref.id[2], ref.id[3]);
	}
	else
	{
	    printf("result: %d\n", result);
	}
    }


    nwos_terminate_objectify();

    return 0;
}


