/*
--             This file is part of the New World OS project
--                 Copyright (C) 2005-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: name.c,v $
-- Revision 1.34  2008/04/26 11:42:18  jsedwards
-- Added 'const' to char* name parameter in some of the functions.
--
-- Revision 1.33  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.32  2007/06/03 17:32:38  jsedwards
-- Fix so when converting to string it changes the third letter ot upper case
-- if the first two letters are "Mc" or "O'".
--
-- Revision 1.31  2007/05/23 03:18:25  jsedwards
-- Removed acronym functions added in r1.30, this version is identical to r1.29.
--
-- Revision 1.30  2007/05/22 03:02:42  jsedwards
-- Added public acronym functions.
--
-- Revision 1.29  2007/05/22 02:38:43  jsedwards
-- Change 'create_spelling' to 'create_public_spelling'.
--
-- Revision 1.28  2007/01/06 20:07:49  jsedwards
-- Split find_name into find_public_name and find_private_name and fixed
-- create name to create according to mode we are in.
--
-- Revision 1.27  2006/12/27 12:23:30  jsedwards
-- Allow an apostrophe in a name (O'Malley).
--
-- Revision 1.26  2006/12/25 12:14:52  jsedwards
-- Fix to only work with public names, this still needs to be fixed.
--
-- Revision 1.25  2006/12/21 12:59:47  jsedwards
-- Hack to make it compile with new split public and private classes,
-- NON-FUNCTIONAL!
--
-- Revision 1.24  2006/12/01 14:31:30  jsedwards
-- Changed to use new malloc_reference_list and free_reference_list functions
-- instead of inlining the code.
--
-- Revision 1.23  2006/11/18 15:09:10  jsedwards
-- Added "max_size" parameter to read_variable_sized_object_from_disk because
-- objects are no longer limited to one file block.
--
-- Revision 1.22  2006/11/11 12:01:04  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.21  2006/10/26 01:51:27  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.20.2.3  2006/09/01 13:27:20  jsedwards
-- Changed "nwos_object_size" to "nwos_reference_list_size" and added the
-- object reference to "nwos_fill_in_common_header" so it can put the "id"
-- in the header now.
--
-- Revision 1.20.2.2  2006/08/18 12:53:48  jsedwards
-- Changed hard coded 512 for block size to FILE_BLOCK_SIZE.
--
-- Revision 1.20.2.1  2006/08/16 12:43:58  jsedwards
-- Changed to use temporary buffers to reconvert to string instead of reusing
-- the "kludge" buffer, so that new versions of GCC don't complain about the
-- signedness of pointers.
--
-- Revision 1.20  2006/01/12 02:56:27  jsedwards
-- Added code to make first letter upper case, because spellings are now
-- always stored as lower case.
--
-- Revision 1.19  2006/01/09 02:45:18  jsedwards
-- Fix bug where it tried to verify the name on objects it didn't find.  Now
-- it only reassembles the object and verifies it, if it finds it.
--
-- Revision 1.18  2006/01/04 18:53:02  jsedwards
-- Fixed bug in matching name object number of words.  It was not correctly
-- comparing the number of words, so it would match one with the wrong number
-- of words.  For example when searching for "Los Angeles" it would match up
-- with "East Los Angeles".
--
-- Revision 1.17  2006/01/04 18:37:44  jsedwards
-- Changed to allow abbreviations (such as St. for Saint) in names.
-- Changed reassembly code to just call "name to string" routine to
-- reassemble the name and then compare it to the original name.
--
-- Revision 1.16  2006/01/02 23:21:56  jsedwards
-- Moved 'ref_to_word' function to an inline function in objectify.h.
-- Changed old 'ref_to_name' calls to use 'ref_to_word' instead.
--
-- Revision 1.15  2006/01/02 21:57:54  jsedwards
-- Fixed two bugs in the finding of names.  The first was it set current_top
-- outside of the compare lists loop, so it didn't get reset back to zero
-- when we started over on list [0].  The second was that when it recomputed
-- the num_spellings after compare lists it indexed the reference list with
-- the "word" index instead of [0] which it was computing.
--
-- Revision 1.14  2006/01/02 21:48:33  jsedwards
-- Removed some of the debugging printf statements and added asserts to check
-- that the new calculated num_spellings is rational.
--
-- Revision 1.13  2006/01/02 21:35:38  jsedwards
-- Added a bunch of debug printf statements.
--
-- Revision 1.12  2005/12/29 17:50:53  jsedwards
-- Commented out printf debugging statements, that aren't useful now.
--
-- Revision 1.11  2005/12/29 17:36:08  jsedwards
-- Changes to make Name and Spelling objects variable sized.  This allows the
-- write object to disk routine to fill in the remainder of the 512 bytes with
-- random data.
--
-- Revision 1.10  2005/12/27 18:26:57  jsedwards
-- Added special routine for the "big_bang" to set the class reference.
-- Changed to look up class definition instead of using a fixed file name.
-- Also changed so that object id is random instead of based upon contents.
--
-- Revision 1.9  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.8  2005/12/21 23:23:03  jsedwards
-- Added 'find_name' and 'name_to_string' routines.
--
-- Revision 1.7  2005/12/21 03:58:00  jsedwards
-- Changed so that a name can have multiple words in it.
--
-- Revision 1.6  2005/12/11 16:32:15  jsedwards
-- Fixed error message to say "name" instead of "year".
--
-- Revision 1.5  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.4  2005/12/05 05:22:46  jsedwards
-- Change to call read_reference_list_from_disk for reference lists instead
-- of read_object_from_disk.  Also moved calculating checksums down to just
-- before writing object, so that no changes occur after computing them.
--
-- Revision 1.3  2005/12/04 04:13:03  jsedwards
-- Added 'nwos' prefix to create_xxxx function names and eliminated the
-- 'referring' object parameter from all of them.
--
-- Revision 1.2  2005/12/04 00:39:28  jsedwards
-- Changed to use new "spelling" class to store the letters that make up the
-- name (copied the create_word class and adjusted accordingly).
--
-- Revision 1.1  2005/12/03 21:46:47  jsedwards
-- Initial version extracted from create_person.c file.
--
*/


#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"
#include "objectify_private.h"


static ObjRef* get_public_name_class_ref()
{
    static ObjRef class_ref;

    if (is_void_reference(&class_ref))
    {
	nwos_find_public_class_definition("NAME", &class_ref);

	assert(!is_void_reference(&class_ref));
    }

    return &class_ref;
}


static ObjRef* get_private_name_class_ref()
{
    static ObjRef class_ref;

    if (is_void_reference(&class_ref))
    {
	if (!nwos_find_private_class_definition("NAME", &class_ref))
	{
	    return NULL;
	}
    }

    return &class_ref;
}


static int split_name(const char* name, char words[21][20])
{
    int result = 0; 
    int i;
    const char* p;

    i = 0;
    for (p = name; *p != '\0'; p++)
    {
	if (*p == ' ')
	{
	    assert(i > 0);
	    words[result][i] = '\0';
	    i = 0;
	    result++;
	}
	else
	{
	    assert(isalnum(*p) || *p == '.' || *p == '\'');   /* allow abbreviation or apostrophe in name */
	    assert(result <= 21);
	    assert(i <= 19);         /* leave room for null */
	    words[result][i] = *p;
	    i++;
	}
    }

    assert(i > 0);
    words[result][i] = '\0';
    result++;

    return result;
}


static size_t get_name_object_size(void* name_obj)
{
    assert(((C_struct_Name*)name_obj)->count > 0);

    return sizeof(C_struct_Name) + (((C_struct_Name*)name_obj)->count * sizeof(ObjRef));
}


bool nwos_find_public_name(const char* name, ObjRef* ref)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Name* ptr_name_obj = (C_struct_Name*)kludge;
    EveryObject spelling_header;
    ObjRef spelling_ref[21];
    ObjRef object_class;
    ReferenceList* spelling_ref_list[21];
    ObjRef* current_top;
    int num_spellings[21];
    int old_num_spellings;
    int word;
    int j;
    int k;
    bool result = true;
    int num_words;
    char words[21][20];
    char temp_name[256];

    num_words = split_name(name, words);

#ifdef DEBUG_NAME
    printf("num_words: %d\n", num_words);
    for (j = 0; j < num_words; j++)
      {
	printf("word[%d]: %s\n", j, words[j]);
      }
#endif

    assert(num_words > 0);

    for (word = 0; word < num_words; word++)
    {
	if (!nwos_find_public_spelling(words[word], &spelling_ref[word]))
	{
#ifdef DEBUG_NAME
	    printf("did not find word: %s\n", words[word]);
#endif
	    result = false;
	    break;
	}
#ifdef DEBUG_NAME
	else
	{
	    printf("word: %s - %08lx\n", words[word], 
		   nwos_ref_to_word(&spelling_ref[word]));
	}
#endif
    }

    if (result)  /* we found all the words */
    {
#ifdef DEBUG_NAME
	printf("found all words\n");
#endif
	for (word = 0; word < num_words; word++)
	{
	    nwos_read_object_headers_from_disk(&spelling_ref[word], &spelling_header);

	    spelling_ref_list[word] = nwos_malloc_reference_list(&spelling_header.object.references);

	    num_spellings[word] = spelling_ref_list[word]->common_header.num_refs;

#ifdef DEBUG_NAME
	    printf("word: %s  num_spellings: %d\n", words[word], num_spellings[word]);
	    for (j = 0; j < num_spellings[word]; j++)
	    {
		printf("spelling_ref_list[%d]->references[%d]: %08lx\n", word, j, nwos_ref_to_word(&spelling_ref_list[word]->references[j]));
	    }
#endif
	}

	/* now go through each reference list and find the name objects and place them at the beginning */

	/* NOTE: We won't have to do this when the reference list is divided up by classes!! */

	for (word = 0; result && word < num_words; word++)
	{
	    current_top = &spelling_ref_list[word]->references[0];   /* create a pointer of where we want to put the good ones */

	    for (j = 0; j < num_spellings[word]; j++)
	    {
		nwos_get_object_class(&spelling_ref_list[word]->references[j], &object_class);   /* find out what kind of object it is */
		if (is_same_object(&object_class, get_public_name_class_ref()))   /* it is a name object */
		{
		    /* found a name, if we have eliminated any, copy this one to the good ones */
		    if (current_top != &spelling_ref_list[word]->references[j])
		    {
			memcpy(current_top, &spelling_ref_list[word]->references[j], sizeof(ObjRef));
		    }
		    current_top++;   /* another good one */
		}
	    }

	    if (current_top != &spelling_ref_list[word]->references[j])   /* recompute how many name objects we have */
	    {
		old_num_spellings = num_spellings[word];   /* save it to make sure the new one is sane */ 
		num_spellings[word] = current_top - &spelling_ref_list[word]->references[0];
		assert(0 <= num_spellings[word] && num_spellings[word] <= old_num_spellings);
	    }

	    if (num_spellings[word] == 0)    /* no use continuing if there are no name objects for this word */
	    {
		result = false;
	    }

#ifdef DEBUG_NAME
	    printf("word: %s  num_name_refs: %d\n", words[word], num_spellings[word]);
#endif
	}

	/* Now go through what we have left and see if there is a name that has all of the words in it.  */
	/* I.E. if there aren't any common names in all of the reference lists, then there isn't a name  */
        /* that conatins all of these words and we can quit looking.  The advantage to this is then we   */
	/* don't have to actually read the objects yet, we can do this with just the reference lists.    */
	/* This can be done two lists at a time, compare two lists and eliminate all of the ones that    */
	/* aren't in both lists.  When it is finished both lists have to be identical and we no longer   */
	/* have to look at the second list and can just compare the 1st list and the 3rd list and so on. */

	for (word = 1; result && word < num_words; word++)
	{
	    current_top = &spelling_ref_list[0]->references[0];   /* create a pointer of where we want to put the good ones */

#ifdef DEBUG_NAME
	    printf("current_top: %p in the beginning\n", current_top);
#endif

	    for (j = 0; j < num_spellings[0]; j++)     /* for each reference in the first word's reference list */
	    {
		for (k = 0; k < num_spellings[word]; k++)     /* for each reference in the i'th word's reference list */
		{
#ifdef DEBUG_NAME
	    printf("word[%d]: %s - refs[%d]: %08lx - refs[%d]: %08lx\n", word, words[word],
		   j, nwos_ref_to_word(&spelling_ref_list[0]->references[j]),
		   k, nwos_ref_to_word(&spelling_ref_list[word]->references[k]));
#endif
		    if (is_same_object(&spelling_ref_list[0]->references[j], &spelling_ref_list[word]->references[k])) /* same */
		    {
			/* found a name in both lists, if we have eliminated any, copy this one to the good ones */
			if (current_top != &spelling_ref_list[0]->references[j])
			{
			    memcpy(current_top, &spelling_ref_list[0]->references[j], sizeof(ObjRef));
			}
			current_top++;   /* another one in both lists */
		    }
		}
	    }

	    if (current_top != &spelling_ref_list[0]->references[j])   /* recompute how many name objects we have */
	    {
		old_num_spellings = num_spellings[0];   /* save it to make sure the new one is sane */ 
		num_spellings[0] = current_top - &spelling_ref_list[0]->references[0];
		assert(0 <= num_spellings[0] && num_spellings[0] <= old_num_spellings);
	    }
#ifdef DEBUG_NAME
	    printf("num_spellings[0]: %d\n", num_spellings[0]);
#endif
	}

	/* We should now have a list in the first words reference list of all the names that contain all of the words. */
	/* We can now open each one of those an see if the words are in the right order. */

	result = false;   /* assume we won't find it */

	/* Note we won't go through this list at all if all of the references were eliminated in the first word */

	for (j = 0; j < num_spellings[0]; j++)
	{
	    nwos_read_variable_sized_object_from_disk(&spelling_ref_list[0]->references[j], kludge, sizeof(kludge), &get_name_object_size);

	    /* remember ptr_name_obj points to the kludge buffer */

	    if (ptr_name_obj->count == num_words)    /* right number of words */
	    {
		for (word = 0; word < num_words; word++)
		{
		    if (!is_same_object(&ptr_name_obj->spelling[word], &spelling_ref[word])) break;  /* it is not the right word */
		}

		if (word == num_words)   /* found it */
		{
		    result = true;
		    memcpy(ref, &spelling_ref_list[0]->references[j], sizeof(ObjRef));
		    break;
		}
	    }
	}

	for (word = 0; word < num_words; word++)
	{
	    nwos_free_reference_list(spelling_ref_list[word]);
	    spelling_ref_list[word] = NULL;
	}
    }

    if (result)  /* assemble the object to make sure it matches the original string */
    {
	nwos_name_to_string(ref, temp_name, sizeof(temp_name));
	assert(strcasecmp(name, temp_name) == 0);
    }

    return result;
}


bool nwos_find_private_name(const char* name, ObjRef* ref)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Name* ptr_name_obj = (C_struct_Name*)kludge;
    EveryObject spelling_header;
    ObjRef spelling_ref[21];
    ObjRef object_class;
    ReferenceList* spelling_ref_list[21];
    ObjRef* current_top;
    int num_spellings[21];
    int old_num_spellings;
    int word;
    int j;
    int k;
    bool result = true;
    int num_words;
    char words[21][20];
    char temp_name[256];

    if (get_private_name_class_ref() == NULL) return false;

    num_words = split_name(name, words);

#ifdef DEBUG_NAME
    printf("num_words: %d\n", num_words);
    for (j = 0; j < num_words; j++)
      {
	printf("word[%d]: %s\n", j, words[j]);
      }
#endif

    assert(num_words > 0);

    for (word = 0; word < num_words; word++)
    {
	if (!nwos_find_private_spelling(words[word], &spelling_ref[word]))
	{
#ifdef DEBUG_NAME
	    printf("did not find word: %s\n", words[word]);
#endif
	    result = false;
	    break;
	}
#ifdef DEBUG_NAME
	else
	{
	    printf("word: %s - %08lx\n", words[word], 
		   nwos_ref_to_word(&spelling_ref[word]));
	}
#endif
    }

    if (result)  /* we found all the words */
    {
#ifdef DEBUG_NAME
	printf("found all words\n");
#endif
	for (word = 0; word < num_words; word++)
	{
	    nwos_read_object_headers_from_disk(&spelling_ref[word], &spelling_header);

	    spelling_ref_list[word] = nwos_malloc_reference_list(&spelling_header.object.references);

	    num_spellings[word] = spelling_ref_list[word]->common_header.num_refs;

#ifdef DEBUG_NAME
	    printf("word: %s  num_spellings: %d\n", words[word], num_spellings[word]);
	    for (j = 0; j < num_spellings[word]; j++)
	    {
		printf("spelling_ref_list[%d]->references[%d]: %08lx\n", word, j, nwos_ref_to_word(&spelling_ref_list[word]->references[j]));
	    }
#endif
	}

	/* now go through each reference list and find the name objects and place them at the beginning */

	/* NOTE: We won't have to do this when the reference list is divided up by classes!! */

	for (word = 0; result && word < num_words; word++)
	{
	    current_top = &spelling_ref_list[word]->references[0];   /* create a pointer of where we want to put the good ones */

	    for (j = 0; j < num_spellings[word]; j++)
	    {
		nwos_get_object_class(&spelling_ref_list[word]->references[j], &object_class);   /* find out what kind of object it is */
		if (is_same_object(&object_class, get_private_name_class_ref()))   /* it is a name object */
		{
		    /* found a name, if we have eliminated any, copy this one to the good ones */
		    if (current_top != &spelling_ref_list[word]->references[j])
		    {
			memcpy(current_top, &spelling_ref_list[word]->references[j], sizeof(ObjRef));
		    }
		    current_top++;   /* another good one */
		}
	    }

	    if (current_top != &spelling_ref_list[word]->references[j])   /* recompute how many name objects we have */
	    {
		old_num_spellings = num_spellings[word];   /* save it to make sure the new one is sane */ 
		num_spellings[word] = current_top - &spelling_ref_list[word]->references[0];
		assert(0 <= num_spellings[word] && num_spellings[word] <= old_num_spellings);
	    }

	    if (num_spellings[word] == 0)    /* no use continuing if there are no name objects for this word */
	    {
		result = false;
	    }

#ifdef DEBUG_NAME
	    printf("word: %s  num_name_refs: %d\n", words[word], num_spellings[word]);
#endif
	}

	/* Now go through what we have left and see if there is a name that has all of the words in it.  */
	/* I.E. if there aren't any common names in all of the reference lists, then there isn't a name  */
        /* that conatins all of these words and we can quit looking.  The advantage to this is then we   */
	/* don't have to actually read the objects yet, we can do this with just the reference lists.    */
	/* This can be done two lists at a time, compare two lists and eliminate all of the ones that    */
	/* aren't in both lists.  When it is finished both lists have to be identical and we no longer   */
	/* have to look at the second list and can just compare the 1st list and the 3rd list and so on. */

	for (word = 1; result && word < num_words; word++)
	{
	    current_top = &spelling_ref_list[0]->references[0];   /* create a pointer of where we want to put the good ones */

#ifdef DEBUG_NAME
	    printf("current_top: %p in the beginning\n", current_top);
#endif

	    for (j = 0; j < num_spellings[0]; j++)     /* for each reference in the first word's reference list */
	    {
		for (k = 0; k < num_spellings[word]; k++)     /* for each reference in the i'th word's reference list */
		{
#ifdef DEBUG_NAME
	    printf("word[%d]: %s - refs[%d]: %08lx - refs[%d]: %08lx\n", word, words[word],
		   j, nwos_ref_to_word(&spelling_ref_list[0]->references[j]),
		   k, nwos_ref_to_word(&spelling_ref_list[word]->references[k]));
#endif
		    if (is_same_object(&spelling_ref_list[0]->references[j], &spelling_ref_list[word]->references[k])) /* same */
		    {
			/* found a name in both lists, if we have eliminated any, copy this one to the good ones */
			if (current_top != &spelling_ref_list[0]->references[j])
			{
			    memcpy(current_top, &spelling_ref_list[0]->references[j], sizeof(ObjRef));
			}
			current_top++;   /* another one in both lists */
		    }
		}
	    }

	    if (current_top != &spelling_ref_list[0]->references[j])   /* recompute how many name objects we have */
	    {
		old_num_spellings = num_spellings[0];   /* save it to make sure the new one is sane */ 
		num_spellings[0] = current_top - &spelling_ref_list[0]->references[0];
		assert(0 <= num_spellings[0] && num_spellings[0] <= old_num_spellings);
	    }
#ifdef DEBUG_NAME
	    printf("num_spellings[0]: %d\n", num_spellings[0]);
#endif
	}

	/* We should now have a list in the first words reference list of all the names that contain all of the words. */
	/* We can now open each one of those an see if the words are in the right order. */

	result = false;   /* assume we won't find it */

	/* Note we won't go through this list at all if all of the references were eliminated in the first word */

	for (j = 0; j < num_spellings[0]; j++)
	{
	    nwos_read_variable_sized_object_from_disk(&spelling_ref_list[0]->references[j], kludge, sizeof(kludge), &get_name_object_size);

	    /* remember ptr_name_obj points to the kludge buffer */

	    if (ptr_name_obj->count == num_words)    /* right number of words */
	    {
		for (word = 0; word < num_words; word++)
		{
		    if (!is_same_object(&ptr_name_obj->spelling[word], &spelling_ref[word])) break;  /* it is not the right word */
		}

		if (word == num_words)   /* found it */
		{
		    result = true;
		    memcpy(ref, &spelling_ref_list[0]->references[j], sizeof(ObjRef));
		    break;
		}
	    }
	}

	for (word = 0; word < num_words; word++)
	{
	    nwos_free_reference_list(spelling_ref_list[word]);
	    spelling_ref_list[word] = NULL;
	}
    }

    if (result)  /* assemble the object to make sure it matches the original string */
    {
	nwos_name_to_string(ref, temp_name, sizeof(temp_name));
	assert(strcasecmp(name, temp_name) == 0);
    }

    return result;
}


/* NOTE: the find part could be extracted from the create name below, but */
/* I'm not sure exactly how to do it right now so I am leaving it as is.  */

ObjCreateResult nwos_create_name(const char* name, ObjRef* ref)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Name* ptr_name_obj = (C_struct_Name*)kludge;
    EveryObject spelling_header;
    ObjRef* name_class_ref_ptr;
    ObjRef spelling_ref[21];
    ObjRef object_class;
    ReferenceList* spelling_ref_list[21];
    ObjRef* current_top;
    int num_spellings[21];
    int old_num_spellings;
    int word;
    int j;
    int k;
    ObjCreateResult spelling_result = FOUND_EXISTING;
    ObjCreateResult result = CREATED_NEW;
    int num_words;
    char words[21][20];
    char temp_name[256];
#ifdef DEBUG_NAME
    uint8 *p;
#endif
#if 0
    int i;
    char reassemble[256];
    char *p;
#endif

    num_words = split_name(name, words);

#if 0
    printf("words: %d\n", num_words);
    for (i = 0; i < num_words; i++)
      {
	printf("%d: %s\n", i, words[i]);
      }
#endif

#ifdef PUBLIC_MODE
    name_class_ref_ptr = get_public_name_class_ref();
#else
    nwos_find_or_create_private_class_definition("NAME", &object_class);
    name_class_ref_ptr = get_private_name_class_ref();
#endif

    assert(num_words > 0);

    for (word = 0; word < num_words; word++)
    {
#ifdef PUBLIC_MODE
	if (nwos_create_public_spelling(words[word], &spelling_ref[word]) != FOUND_EXISTING)
#else
	if (nwos_create_private_spelling(words[word], &spelling_ref[word]) != FOUND_EXISTING)
#endif
	{
	    spelling_result = CREATED_NEW;
	}
#ifdef DEBUG_NAME
	printf("spelling_ref[%d]: %08lx\n", word, nwos_ref_to_word(&spelling_ref[word]));
#endif
    }

    if (spelling_result == FOUND_EXISTING)     /* all words existed */
    {
	for (word = 0; word < num_words; word++)
	{
	    nwos_read_object_headers_from_disk(&spelling_ref[word], &spelling_header);

	    spelling_ref_list[word] = nwos_malloc_reference_list(&spelling_header.object.references);

	    num_spellings[word] = spelling_ref_list[word]->common_header.num_refs;

#ifdef DEBUG_NAME
	    printf("num_spellings: %d - %d\n", word, num_spellings[word]);
	    for (j = 0; j < num_spellings[word]; j++)
	      {
		p = (uint8*)spelling_ref_list[word];
		for (k = 0; k < ref_list_size; k++) printf("%02x%c", *p++, (k%16 == 15) ? '\n' : ' ');
		printf("spelling_references[%d][%d]: %08lx\n", word, j, nwos_ref_to_word(&spelling_ref_list[word]->references[j]));
	      }
#endif
	}

	/* now go through each reference list and find the name objects and place them at the beginning */

	/* NOTE: We won't have to do this when the reference list is divided up by classes!! */

	for (word = 0; spelling_result == FOUND_EXISTING && word < num_words; word++)
	{
	    current_top = &spelling_ref_list[word]->references[0];   /* create a pointer of where we want to put the good ones */

	    for (j = 0; j < num_spellings[word]; j++)
	    {
		nwos_get_object_class(&spelling_ref_list[word]->references[j], &object_class);   /* find out what kind of object it is */
#ifdef DEBUG_NAME
		printf("object_class[%d][%d]: %08lx\n", word, j, nwos_ref_to_word(&object_class));
#endif
		if (is_same_object(&object_class, name_class_ref_ptr))   /* it is a name object */
		{
#ifdef DEBUG_NAME
		  printf("is a name object\n");
#endif
		    /* found a name, if we have eliminated any, copy this one to the good ones */
		    if (current_top != &spelling_ref_list[word]->references[j])
		    {
#ifdef DEBUG_NAME
		      printf("moving it from %p to %p\n", &spelling_ref_list[word]->references[j], current_top);
#endif
			memcpy(current_top, &spelling_ref_list[word]->references[j], sizeof(ObjRef));
		    }
		    current_top++;   /* another good one */
		}
	    }

	    if (current_top != &spelling_ref_list[word]->references[j])   /* recompute how many name objects we have */
	    {
		old_num_spellings = num_spellings[word];   /* save it to make sure the new one is sane */ 
		num_spellings[word] = current_top - &spelling_ref_list[word]->references[0];
		assert(0 <= num_spellings[word] && num_spellings[word] <= old_num_spellings);
#ifdef DEBUG_NAME
		printf("new size of num_spellings[%d]: %d\n", word, num_spellings[word]);
#endif
	    }

	    if (num_spellings[word] == 0)    /* no use continuing if there are no name objects for this word */
	    {
		spelling_result = CREATED_NEW;
	    }
	}

	/* Now go through what we have left and see if there is a name that has all of the words in it.  */
	/* I.E. if there aren't any common names in all of the reference lists, then there isn't a name  */
        /* that conatins all of these words and we can quit looking.  The advantage to this is then we   */
	/* don't have to actually read the objects yet, we can do this with just the reference lists.    */
	/* This can be done two lists at a time, compare two lists and eliminate all of the ones that    */
	/* aren't in both lists.  When it is finished both lists have to be identical and we no longer   */
	/* have to look at the second list and can just compare the 1st list and the 3rd list and so on. */

	for (word = 1; spelling_result == FOUND_EXISTING && word < num_words; word++)
	{
	    current_top = &spelling_ref_list[0]->references[0];   /* create a pointer of where we want to put the good ones */

#ifdef DEBUG_NAME
	  printf("checking word: %d\n", word);
#endif
	    for (j = 0; j < num_spellings[0]; j++)     /* for each reference in the first word's reference list */
	    {
#ifdef DEBUG_NAME
	    printf("compare spelling_ref_list[0]->references[%d]: %08lx\n", j, nwos_ref_to_word(&spelling_ref_list[0]->references[j]));
#endif
		for (k = 0; k < num_spellings[word]; k++)     /* for each reference in the i'th word's reference list */
		{
#ifdef DEBUG_NAME
	    printf("with    spelling_ref_list[%d]->references[%d]: %08lx\n", word, k, nwos_ref_to_word(&spelling_ref_list[word]->references[k]));
#endif
		    if (is_same_object(&spelling_ref_list[0]->references[j], &spelling_ref_list[word]->references[k])) /* same */
		    {
			/* found a name in both lists, if we have eliminated any, copy this one to the good ones */
			if (current_top != &spelling_ref_list[0]->references[j])
			{
			    memcpy(current_top, &spelling_ref_list[0]->references[j], sizeof(ObjRef));
			}
			current_top++;   /* another one in both lists */
		    }
		}
	    }

	    if (current_top != &spelling_ref_list[0]->references[j])   /* recompute how many name objects we have */
	    {
		old_num_spellings = num_spellings[0];   /* save it to make sure the new one is sane */ 
		num_spellings[0] = current_top - &spelling_ref_list[0]->references[0];
		assert(0 <= num_spellings[0] && num_spellings[0] <= old_num_spellings);
	    }

	    if (num_spellings[0] == 0)    /* no use continuing if there are no more name objects for this word */
	    {
		spelling_result = CREATED_NEW;
	    }
	}

	/* We should now have a list in the first words reference list of all the names that contain all of the words. */
	/* We can now open each one of those an see if the words are in the right order. */

	for (j = 0; j < num_spellings[0]; j++)
	{
#ifdef DEBUG_NAME
	    printf("reading spelling_ref_list[0]->references[%d]: %08lx\n", j, nwos_ref_to_word(&spelling_ref_list[0]->references[j]));
#endif
	    nwos_read_variable_sized_object_from_disk(&spelling_ref_list[0]->references[j], kludge, sizeof(kludge), &get_name_object_size);

	    /* remember ptr_name_obj points to the kludge buffer */

	    if (ptr_name_obj->count == num_words)    /* right number of words */
	    {
		for (word = 0; word < num_words; word++)
		{
		    if (!is_same_object(&ptr_name_obj->spelling[word], &spelling_ref[word])) break;  /* it is not the right word */
		}

		if (word == num_words)   /* found it */
		{
		    memcpy(ref, &spelling_ref_list[0]->references[j], sizeof(ObjRef));
		    result = FOUND_EXISTING;
		    break;
		}
	    }
	}

	for (word = 0; word < num_words; word++)
	{
	    nwos_free_reference_list(spelling_ref_list[word]);
	    spelling_ref_list[word] = NULL;
	}
    }

    if (result != FOUND_EXISTING)   /* didn't find it */
    {
	memset(kludge, 0, sizeof(kludge));  /* zero it out */

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&ptr_name_obj->header.common, ref, name_class_ref_ptr);

	ptr_name_obj->count = num_words;
	for (word = 0; word < num_words; word++)
	{
	    memcpy(&ptr_name_obj->spelling[word], &spelling_ref[word], sizeof(ObjRef));
	    nwos_add_to_references(ref, &spelling_ref[word]);
	}

	nwos_create_reference_list(ref, &ptr_name_obj->header.object.references);

	nwos_crc32_calculate((uint8*) &ptr_name_obj->header.object, sizeof(ObjectHeader), ptr_name_obj->header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &ptr_name_obj->count, sizeof(C_struct_Name) + (num_words * sizeof(ObjRef)) - sizeof(EveryObject), ptr_name_obj->header.common.data_chksum);


	nwos_write_object_to_disk(ref, kludge, sizeof(C_struct_Name) + (num_words * sizeof(ObjRef)));

#if 0
	{
	int i;
	for (i = 0; i < sizeof(C_struct_Name) + (num_words * sizeof(ObjRef)); i++)
	  {
	    printf("%02x%c", kludge[i], (i % 16 == 15) ? '\n' : ' ');
	  }
	}
#endif
        nwos_add_to_references(ref, name_class_ref_ptr);
    }

    /* reassemble the object to make sure it matches the original string */

    nwos_name_to_string(ref, temp_name, sizeof(temp_name));
    assert(strcasecmp(name, temp_name) == 0);
#if 0
    printf("reassemble: %s\n", kludge);
#endif

    return result;
}



/* This is horrible - FIX THIS!! */

static size_t get_spelling_object_size(void* spelling_obj)
{
    assert(((C_struct_Spelling*)spelling_obj)->count > 0);

    return sizeof(C_struct_Spelling) + ((C_struct_Spelling*)spelling_obj)->count;
}


bool nwos_name_to_string(ObjRef* ref, char* string, size_t size)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Name* ptr_name_obj;
    C_struct_Spelling* ptr_spelling_obj;
    int word;
    int i;
    int j;
    size_t name_obj_size;

    /* read the name object into the kludge buffer and then after we know what size it is malloc space and copy it there */
    nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_name_object_size);
    ptr_name_obj = (C_struct_Name*)kludge;
    name_obj_size = sizeof(C_struct_Name) + (ptr_name_obj->count * sizeof(ObjRef));
    /* printf("name_obj_size: %d\n", name_obj_size); */
    assert(name_obj_size > sizeof(C_struct_Name));
    ptr_name_obj = malloc(name_obj_size);
    memcpy(ptr_name_obj, kludge, name_obj_size);

    ptr_spelling_obj = (C_struct_Spelling*)kludge;

    i = 0;
    for (word = 0; word < ptr_name_obj->count; word++)
    {
	assert(!is_void_reference(&ptr_name_obj->spelling[word]));
	nwos_read_variable_sized_object_from_disk(&ptr_name_obj->spelling[word], kludge, sizeof(kludge), &get_spelling_object_size);

	string[i++] = toupper(ptr_spelling_obj->storage[0]);    /* first letter is upper case */

	for (j = 1; j < ptr_spelling_obj->count; j++) 
	{
	    assert(i < size);

	    /* if third letter and first two were McX or O'X then make the third letter upper case */
            if (j == 2 &&
		((ptr_spelling_obj->storage[0] == 'm' && ptr_spelling_obj->storage[1] == 'c') ||
		 (ptr_spelling_obj->storage[0] == 'o' && ptr_spelling_obj->storage[1] == '\'')))
	    {
		string[i++] = toupper(ptr_spelling_obj->storage[j]);
	    }
	    else
	    {
		string[i++] = ptr_spelling_obj->storage[j];
	    }
	}

	assert(i < size);

	if (word + 1 == ptr_name_obj->count)   /* it's the last word */
	{
	    string[i++] = '\0';
	}
	else
	{
	    string[i++] = ' ';
	}
    }

    free(ptr_name_obj);

    return true;
}


