/*
--             This file is part of the New World OS project
--                 Copyright (C) 2004-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: disk_io.c,v $
-- Revision 1.93  2008/09/01 18:08:19  jsedwards
-- Added chunk_info_reverse_index to get from chunk number back to an index
-- into the chunk_info table.
--
-- Revision 1.92  2008/09/01 16:18:10  jsedwards
-- Removed nwos_allocate_all_chunks_hack, no longer used.  Removing the
-- --allocate-all option from prep_disk.
--
-- Revision 1.91  2008/09/01 15:44:01  jsedwards
-- Fix bug in nwos_allocate_new_chunk where it wasn't searching the chunk_info
-- table to insert new chunk.
--
-- Revision 1.90  2008/09/01 02:57:40  jsedwards
-- Fix so that if hash isn't found or if block isn't in compressed index
-- nwos_read_block returns false.
--
-- Revision 1.89  2008/08/10 15:16:38  jsedwards
-- Added printing of path in "Missing magic number" error message.
--
-- Revision 1.88  2008/08/10 15:07:26  jsedwards
-- Added path in which error occurred to "Incorrect version string" message.
--
-- Revision 1.87  2008/07/19 13:34:12  jsedwards
-- Added new disk_io_is_read_only and disk_io_is_read_write functions.
--
-- Revision 1.86  2008/07/16 14:23:47  jsedwards
-- Fixed copy and paste error, where chunk_info .used and .index were written
-- into .ref when moving entry.
--
-- Revision 1.85  2008/06/26 14:54:30  jsedwards
-- Change so that if initialize backup fails, it shuts disk_io down again and
-- exits.
--
-- Revision 1.84  2008/06/12 02:16:29  jsedwards
-- Fixed so that read_only flag is not set in PUBLIC_MODE, because if it is
-- set no backup file gets written.
--
-- Revision 1.83  2008/06/07 16:26:21  jsedwards
-- Fix Bug #1987367 - moved setting of 'read_only' variable outside of if
-- private objects statement so that it is always set, even if there are no
-- private objects.
--
-- Revision 1.82  2008/05/22 11:33:51  jsedwards
-- Made disk_header global and renamed to nwos_disk_header.
--
-- Revision 1.81  2008/05/05 12:41:15  jsedwards
-- Changed so that if fopen of index file fails that it just prints a warning
-- and continues without writing the index file.
--
-- Revision 1.80  2008/05/05 12:15:33  jsedwards
-- Fix Bug #1957451 - added test to make sure fopen didn't return NULL before
-- calling fwrite for index file.
--
-- Revision 1.79  2008/04/13 23:06:57  jsedwards
-- Changed to conditionally use gettimeofday instead of clock_gettime if the
-- OS doesn't support it (like Mac OS X).
--
-- Revision 1.78  2008/04/02 03:06:21  jsedwards
-- Added file locking.
--
-- Revision 1.77  2008/03/24 05:04:10  jsedwards
-- Change to not close private (or public) private descriptor if it is not
-- open.
--
-- Revision 1.76  2008/02/25 14:31:20  jsedwards
-- Changed nwos_allocate_new_chunk function to allow creating a new chunk for
-- any reference and not just greater than the existing references.
--
-- Revision 1.75  2008/02/24 21:00:18  jsedwards
-- Added code to allocate more chunks if the existing chunks are more than 90%
-- full.
--
-- Revision 1.74  2008/02/24 20:30:44  jsedwards
-- Fix Bug #1900855 - changed to just pick a block at random and if that fails
-- to just do a linear search.
--
-- Revision 1.73  2008/02/11 04:33:37  jsedwards
-- Changed to set 'read_only' flag for PUBLIC AccessType too.
--
-- Revision 1.72  2008/02/11 03:59:57  jsedwards
-- Added "PUBLIC" AccessType for opening only the Public Objects.
--
-- Revision 1.71  2008/02/09 17:43:19  jsedwards
-- Moved backup code out of "if (private..." so that backup files will be made
-- in PUBLIC_MODE as well.
--
-- Revision 1.70  2008/02/06 03:25:58  jsedwards
-- Added initializer for compressed_file boolean variable so compiler won't
-- complain about it being uninitialized.
--
-- Revision 1.69  2008/02/04 03:10:19  jsedwards
-- Changed to get backup directory path from user_config.
--
-- Revision 1.68  2008/02/03 02:11:22  jsedwards
-- Change to use the new nwos_get_private_objects_path function that checks
-- for environment variable and configuration file.
--
-- Revision 1.67  2008/02/03 02:05:42  jsedwards
-- Fixed wrong path for public objects broken in previous check in.
--
-- Revision 1.66  2008/02/03 01:46:34  jsedwards
-- Reverted some of the changes made in revision 1.64 for the nwos_public_path
-- back to the way they were in 1.63.
--
-- Revision 1.65  2008/02/03 01:39:50  jsedwards
-- Fix to compile without warnings in PUBLIC_MODE.
--
-- Revision 1.64  2008/01/22 14:55:25  jsedwards
-- Changed to call nwos_get_public_objects_path() in new user_config.c file,
-- instead of having a global variable.
--
-- Revision 1.63  2008/01/17 15:04:02  jsedwards
-- Moved progress bar code to prep_disk.c and instead pass a function pointer
-- in that is called (if non-null) with progress.
--
-- Revision 1.62  2008/01/17 14:42:59  jsedwards
-- Fixed the progress bar so it works correctly even for small sizes.
--
-- Revision 1.61  2008/01/17 14:33:36  jsedwards
-- Took code added in previous revision back out because in the sparse file
-- case the entire file has already been written, all we have to do is
-- allocate the chunks.  This revision should be the same as Revision 1.59.
--
-- Revision 1.60  2008/01/17 13:44:45  jsedwards
-- Added code to allocate all hack that checks to see if it is a sparse file
-- or a partition and writes the whole chunk if it is a sparse file.  This is
-- necessary for the time being because if we only write the bit map to a
-- sparse file it really will create a sparse file which wastes a lot of disk
-- space.
--
-- Revision 1.59  2008/01/17 05:13:55  jsedwards
-- Added call to flush data after each bit map block write and usleep 10 mS.
--
-- Revision 1.58  2008/01/17 05:05:44  jsedwards
-- Changed to just write the bit map bytes into each chunk allocated instead
-- of writing the entire chunk with zeros.  Should no longer be necessary to
-- write all the zeros because the it map determines if the block is used or
-- not.
--
-- Revision 1.57  2008/01/17 04:55:59  jsedwards
-- Added code to print the progress in the allocate_all_hack function.
--
-- Revision 1.56  2007/12/09 16:56:14  jsedwards
-- Changed assert to more descriptive prints, when block isn't found in
-- storage_index.
--
-- Revision 1.55  2007/12/09 16:49:29  jsedwards
-- Comment out statement in terminate_disk_io that nulled out the public_path,
-- so we can start objectify back up again if necessary.
--
-- Revision 1.54  2007/11/13 15:01:33  jsedwards
-- Added #ifndef NO_BACKUP around calls to backup module.  This way we don't
-- have to include it when compiling export_c_structs.
--
-- Revision 1.53  2007/11/09 15:53:47  jsedwards
-- Moved variables and code that deal with backup file to new file backup.c.
--
-- Revision 1.52  2007/11/08 12:32:57  jsedwards
-- Split code to finish writing the backup file out into a separate function.
--
-- Revision 1.51  2007/11/08 05:23:17  jsedwards
-- Added code to create and write backup file.
--
-- Revision 1.50  2007/11/03 15:15:10  jsedwards
-- Fix to handle the case where the block_estimate is smaller than what will
-- fit in one chunk.
--
-- Revision 1.49  2007/11/03 15:11:03  jsedwards
-- Renamed 'max_usable' variable in nwos_generate_closely_spaced_id to
-- 'max_used' because it makes more sense.
--
-- Revision 1.48  2007/11/03 14:18:46  jsedwards
-- Added nwos_allocate_all_chunks_hack function for prep_disk to allocate all
-- available space for chunks, until I can get the real allocation working.
--
-- Revision 1.47  2007/10/25 12:35:53  jsedwards
-- Moved the nwos_restart_id_generation and nwos_check_blocks_available
-- functions from the beginning to a location farther into the file.  NO code
-- changes.
--
-- Revision 1.46  2007/10/25 04:26:45  jsedwards
-- Moved last_ref and nwos_restart_id_generation from objectify.c.
--
-- Revision 1.45  2007/10/21 13:21:40  jsedwards
-- Added code to nwos_check_blocks_available to allocate more chunks (at the
-- end) if there are any available.  This version is limited to chunks after
-- chunks that are already allocated.
--
-- Revision 1.44  2007/10/19 01:56:06  jsedwards
-- Added global variables to store parameters to control chunk allocation and
-- a function to set them.
--
-- Revision 1.43  2007/10/13 18:20:00  jsedwards
-- Changed the 'calculate_chunk_density' function to just search back from the
-- end and renamed to 'find_first_low_density_in_chunk'.
--
-- Revision 1.42  2007/10/13 11:06:39  jsedwards
-- Changed the name of the CHUNK_FACTOR define to BYTES_PER_SLICE.
--
-- Revision 1.41  2007/10/11 13:05:07  jsedwards
-- Added code to allow older compatible compressed files to be read in when
-- the version doesn't match exactly.
--
-- Revision 1.40  2007/10/08 14:12:39  jsedwards
-- Added 'calculate_chunk_density' and 'find_starting_chunk' functions, not
-- used yet.
--
-- Revision 1.39  2007/10/07 03:31:21  jsedwards
-- Changed 'nwos_set_block_estimate' function to 'nwos_check_blocks_available'
-- and to return a boolean value if there is room or not.
--
-- Revision 1.38  2007/09/02 19:36:31  jsedwards
-- Added variables and functions to save and log the block estimate.
--
-- Revision 1.37  2007/08/29 15:38:17  jsedwards
-- Added #ifdef LOG_BIT_MAP around log messages for the bit map read and
-- writes, so by default they are not logged.
--
-- Revision 1.36  2007/08/14 00:43:14  jsedwards
-- Fix value passed to find bit map to include offset to chunks.
--
-- Revision 1.35  2007/08/13 19:18:42  jsedwards
-- Fixed two bugs in calculating start block in generating closely spaced id.
--
-- Revision 1.34  2007/08/13 18:01:11  jsedwards
-- Removed a couple more debugging print statements accidentally left in.
--
-- Revision 1.33  2007/08/13 00:25:58  jsedwards
-- Fix bug in the case where more than one chunk is allocated and the first
-- chunk is not used.
--
-- Revision 1.32  2007/08/12 20:05:30  jsedwards
-- Removed debugging print statements accidentally left in last check in.
--
-- Revision 1.31  2007/08/12 19:37:04  jsedwards
-- Fixed previous log entry which was completely wrong (NO code changes).
--
-- Revision 1.30  2007/08/12 19:31:39  jsedwards
-- Rewrite generate_new_closely_spaced_id to work with new chunk stuff.
--
-- Revision 1.29  2007/08/10 00:03:35  jsedwards
-- Removed defintion of _LARGEFILE64_SOURCE, now using _FILE_OFFSET_BITS=64.
-- Also removed using O_LARGEFILE from open call.
--
-- Revision 1.28  2007/08/07 05:09:22  jsedwards
-- Changed to store index into chunk_info table in bit map cache instead of
-- the chunk block number.
--
-- Revision 1.27  2007/08/07 02:34:50  jsedwards
-- Changed variable name from 'block' to 'ref' because block was misleading.
--
-- Revision 1.26  2007/08/07 01:52:37  jsedwards
-- Added 'in_use' element to bit map cache to determine if an entry is in use
-- instead of testing for 'chunk' being non-zero.
--
-- Revision 1.25  2007/08/06 04:12:19  jsedwards
-- Moved binary search of chunk_info table for reference into separate function.
--
-- Revision 1.24  2007/08/06 02:15:57  jsedwards
-- Changed 1_in_N algorithm to pick a random incrment between 1 and N, instead
-- of picking 1 in N number of blocks.
--
-- Revision 1.23  2007/08/02 18:41:06  jsedwards
-- Change to use the new index in the chunk_info table to comupute the address
-- of the chunk in storage.
--
-- Revision 1.22  2007/08/01 00:50:09  jsedwards
-- Fixed to byte swap chunk_info index values on little endian machines when
-- the chunk_info table is read and written.  Also added code to allocate chunk
-- function to calculate index and store it in chunk_info table (not used yet).
--
-- Revision 1.21  2007/07/15 20:32:36  jsedwards
-- Added chunk_used_index and code to sort chunk_info[].used and put indexes
-- into chunk_used_index.
--
-- Revision 1.20  2007/07/15 17:19:49  jsedwards
-- Changed to use WORDS_BIGENDING instead of __BYTE_ORDER == __LITTLE_ENDIAN
-- to determine endianess and byteswap_uint16 and 32 functions in objectify.h
-- instead of bswap_16 and 32 to make more platform independant.  Also added
-- initializer to 'i' variable so compiler wouldn't complaing that it was
-- possibly uninitialized.
--
-- Revision 1.19  2007/07/06 15:26:13  jsedwards
-- Changes to work in PUBLIC_MODE with new separate public objects file.
--
-- Revision 1.18  2007/07/04 12:42:38  jsedwards
-- Fix bug in read_block searching in index table, doesn't need to use hash
-- anymore to search storage_index, just convert the reference to uint32.
--
-- Revision 1.17  2007/07/03 01:19:18  jsedwards
-- Changed from chunk_index which was a uint32 to chunk_info which is a
-- structure containing the old uint32 reference and a uint32 used blocks
-- count.
--
-- Revision 1.16  2007/07/02 15:06:26  jsedwards
-- Removed special_update routine that was used to import 0021 compressed into
-- 0022 storage.  No longer needed.
--
-- Revision 1.15  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.14  2007/06/30 04:28:59  jsedwards
-- Added #ifdef around log message for write block, now off by default.
--
-- Revision 1.13  2007/06/30 00:49:04  jsedwards
-- Fix bug in generate_new_completely_random_id where the byte offset into
-- the bit map was off by 4 (BIT_MAP_BLOCKS / 8 bits), so it was testing the
-- wrong byte in the bit map for used or not.  Assert failed because sometimes
-- the block was already used.
--
-- Revision 1.12  2007/06/28 19:06:09  jsedwards
-- Add log message when generating a random ID or 1 in N ID.
--
-- Revision 1.11  2007/06/28 13:26:06  jsedwards
-- Removed line in allocate_new_chunk that added the blocks used for the bit
-- map to the used_blocks count.  In 0023 the blocks used count doesn't
-- include the bit map blocks.
--
-- Revision 1.10  2007/06/28 04:37:32  jsedwards
-- Fix the stupid binary search thing yet again.
--
-- Revision 1.9  2007/06/28 02:43:41  jsedwards
-- Tweak the binary search for id a bit more.
--
-- Revision 1.8  2007/06/28 02:36:37  jsedwards
-- Make binary search for reference more robust.
--
-- Revision 1.7  2007/06/27 01:13:42  jsedwards
-- Fix nwos_generate_new_1_in_N_id to generate new id if current one doesn't
-- fit in a chunk.
--
-- Revision 1.6  2007/06/26 20:02:37  jsedwards
-- Changed algorithm to generate random id to generate id in existing chunk
-- instead of completely random.
--
-- Revision 1.5  2007/06/26 16:46:06  jsedwards
-- Moved code to compute number of blocks used in an entry to the read
-- cache entry function.
--
-- Revision 1.4  2007/06/26 14:31:44  jsedwards
-- Rename private_generate_new_completely_random_id to
-- generate_new_completely_random_id and make it static.
--
-- Revision 1.3  2007/06/26 14:21:34  jsedwards
-- Moved random id generation functions from objectify.c.
--
-- Revision 1.2  2007/06/26 13:20:53  jsedwards
-- Changed to use new chunk index to convert reference ids to block numbers on
-- disk, instead of wrapping them around the disk size.
--
-- Revision 1.1  2007/06/25 05:28:54  jsedwards
-- New file created from variables, functions and code related to disk I/O,
--  taken from the objectfy.c file.
--
*/

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>


#include "objectify_private.h"


Disk_Header nwos_disk_header;  /* this should NOT be accessed by normal programs */

const char*  nwos_public_path;
const char*  nwos_private_path;

uint32 nwos_total_private_blocks;
uint32 nwos_used_public_blocks;
uint32 nwos_used_private_blocks;

uint32 nwos_block_offset_to_chunks;
uint32 nwos_total_private_chunks = 0;
uint32 nwos_used_private_chunks;

ObjRef nwos_next_public_ref;

static int    public_file_desc;
static int    private_file_desc;
static bool    read_only;
static bool    modified;
static ObjRef* storage_index;
static int     storage_count;

static Chunk_Info* chunk_info;
static int* chunk_info_reverse_index;  /* index from chunk index to info table */
static bool chunk_info_modified;
static uint16* chunk_used_index;  /* order of chunks sorted by blocks used, fewest to most used */

static uint32 block_estimate;
static uint32 last_used_blocks;

#ifndef PUBLIC_MODE
static ObjRef last_ref;
#endif


#define BIT_MAP_CACHE_SIZE 256

typedef struct {
  bool8 in_use;
  bool8 dirty;
  int index;    /* index into chunk_info table */
  int age;
  uint8* map;
} Bit_Map_Cache_Entry;

static Bit_Map_Cache_Entry bit_map_cache[BIT_MAP_CACHE_SIZE];

static int bit_map_tick = 0;

static void write_bit_map(Bit_Map_Cache_Entry* entry);

#ifndef PUBLIC_MODE
static uint32 chunk_density_target;
static int32  chunk_skip_backward;
static int32  chunk_skip_forward;
#endif


/*************************/
/* Block estimate stuff. */
/*************************/

static void log_estimate_results()
{
    char log_msg[128];

    uint32 actual = nwos_used_private_blocks - last_used_blocks;

    snprintf(log_msg, sizeof(log_msg), "  block estimate: %u  actual: %u", block_estimate, actual);

    nwos_log(log_msg);
}



/*****************************************/
/* Function that indexes compressed file */
/*****************************************/

#define INDEX_ALLOC_SIZE 262144 

void index_compressed_file()
{
    int i;
    int size_of_index = INDEX_ALLOC_SIZE;
    size_t bytes_read;
    uint8 buffer[FILE_BLOCK_SIZE];
    uint8 header[FILE_BLOCK_SIZE];
    char* index_path = NULL;
    FILE* index_fp;

#ifdef NO_INDEX_FILE
    index_fp = NULL;
#else
    assert(private_file_desc > 0);

    /* first see if there is an existing index file */

    i = strlen(nwos_private_path);
    index_path = nwos_malloc(i+5);  /* 4 for .ndx and 1 for null */
    strcpy(index_path, nwos_private_path);

    if (index_path[i-4] == '.' && index_path[i-3] == 'o' && index_path[i-2] == 'b' && index_path[i-1] == 'j')
    {
	i = i - 4;   /* write over the old .obj */
    }

    index_path[i++] = '.';
    index_path[i++] = 'n';
    index_path[i++] = 'd';
    index_path[i++] = 'x';
    index_path[i++] = '\0';

    fprintf(stderr, "index_path: %s\n", index_path);

    index_fp = fopen(index_path, "r");
#endif

    storage_index = malloc(size_of_index * sizeof(ObjRef));   /* allocate space for the first batch */

    assert(storage_index != NULL);

    /* read the first block, it contains the header */
    bytes_read = read(private_file_desc, header, sizeof(header));

    assert(bytes_read == sizeof(header));

    /* if there is an index file see if the header matches */
    if (index_fp != NULL)
    {
	bytes_read = fread(buffer, 1, FILE_BLOCK_SIZE, index_fp);
	assert(bytes_read == sizeof(buffer));

	fprintf(stderr, "Index file found: ");
	fflush(stderr);

	if (memcmp(header, buffer, FILE_BLOCK_SIZE) == 0)   /* we have a winner */
	{
	    fprintf(stderr, "match\n");
	}
	else
	{
	    fprintf(stderr, "no match\n");
	    fclose(index_fp);
	    index_fp = NULL;
	}
    }
    else /* no index file */
    {
	errno = 0;   /* clear errno for later */
    }

    fprintf(stderr, "Indexing");
    fflush(stderr);

    i = 1;
    while (1)
    {
	if (i % 262144 == 1)
	{
	    fprintf(stderr, ".");
	    fflush(stderr);
	}

	if (i == size_of_index)   /* need to allocate more space */
	{
	    size_of_index += INDEX_ALLOC_SIZE;
	    storage_index = realloc(storage_index, size_of_index * sizeof(ObjRef));
	    assert(storage_index != NULL);
	}

	assert(i < size_of_index);

	if (index_fp != NULL)   /* read index from file */
	{
	    bytes_read = fread(&storage_index[i], sizeof(ObjRef), 1, index_fp);

	    if (bytes_read != 1) break;
	}
	else
	{
	    bytes_read = read(private_file_desc, buffer, sizeof(buffer));

	    if (bytes_read != sizeof(buffer)) break;

	    memcpy(&storage_index[i], &buffer[4], sizeof(ObjRef));
	}

	i++;
    }

    storage_count = i;

    fprintf(stderr, "\nblocks: %d\n", i);

    if (index_fp != NULL)
    {
	if (ferror(index_fp))
	{
	    perror(index_path);
	    exit(1);
	}
	fclose(index_fp);
    }
    else
    {
	if (errno != 0)
	{
	    perror(nwos_private_path);
	    exit(1);
	}

#ifndef NO_INDEX_FILE
	index_fp = fopen(index_path, "w");

	if (index_fp == NULL)
	{
	    fprintf(stderr, "Could not create index file: ");
	    perror(index_path);
	    fprintf(stderr, "Index will not be saved!\n");
	}
	else
	{
	    fprintf(stderr, "Writing index file: %s\n", index_path);

	    if (fwrite(header, 1, sizeof(header), index_fp) != sizeof(header))
	    {
		perror(index_path);
		exit(1);
	    }

	    if (fwrite(&storage_index[1], sizeof(ObjRef), i, index_fp) != i)
	    {
		perror(index_path);
		exit(1);
	    }

	    if (fclose(index_fp))
	    {
		perror(index_path);
		exit(1);
	    }
	}
#endif
    }

    assert(bytes_read == 0);

    if (index_path != NULL)
    {
	nwos_free(index_path);
    }
}



/**********************************************/
/* Function to initialize the disk I/O module */
/**********************************************/

void initialize_random_number_generator()
{
#ifdef HAVE_CLOCK_GETTIME
    struct timespec ts;

    clock_gettime(CLOCK_REALTIME, &ts);
    srandom((uint32)ts.tv_sec ^ (uint32)ts.tv_nsec);
#else
    struct timeval tv;

    gettimeofday(&tv, NULL);
    srandom((uint32)tv.tv_sec ^ (uint32)tv.tv_usec);
#endif

#ifdef DISABLE_SECURITY_FEATURES
    randomxxx = 0x00000100;     /* start objects here */
#endif
}

/*************************************************************************************/
/* This updates the chunk_info_reverse_index table to point to the chunk_info table. */
/*                                                                                   */
/*  chunk_info   reverse                                                             */
/*    0 [5]       0 [4]                                                              */
/*    1 [3]       1 [5]                                                              */
/*    2 [4]       2 [3]                                                              */
/*    3 [2]       3 [1]                                                              */
/*    4 [0]       4 [2]                                                              */
/*    5 [1]       5 [0]                                                              */
/*                                                                                   */
/*************************************************************************************/

static void update_chunk_info_reverse_index()
{
    int i;

    assert(nwos_total_private_chunks > 0);

    if (chunk_info_reverse_index == NULL)
    {
	chunk_info_reverse_index = nwos_malloc(nwos_total_private_chunks * sizeof(int));
    }

    for (i = 0; i < nwos_used_private_chunks; i++)
    {
	assert(chunk_info[i].index < nwos_used_private_chunks);
	chunk_info_reverse_index[chunk_info[i].index] = i;
    }

    /* verify it is correct */
    for (i = 0; i < nwos_used_private_chunks; i++)
    {
	assert(chunk_info[chunk_info_reverse_index[i]].index == i);
    }
}


void nwos_initialize_disk_io(AccessType type, const char* path)
{
    size_t chunk_info_size_in_bytes;
    char log_msg[128];
    uint32 used_blocks;
    int i;
    int j;
    int k;
    uint16 left;
    uint16 right;
    uint16 save;
    bool compressed_file = false;
    struct flock lock;
#ifndef NO_BACKUP
    const char* backup_dir;
#endif

    /* make sure the storage is something we can deal with */
#ifdef PUBLIC_MODE
    assert(type == PUBLIC);
#else
    assert(type == PUBLIC || type == READ_ONLY || type == READ_WRITE);
#endif

    assert(nwos_private_path == NULL);   /* make sure this is only called once */
    assert(private_file_desc == 0);
    assert(public_file_desc == 0);

    initialize_random_number_generator();

    nwos_public_path = nwos_get_public_objects_path();

    if (type == PUBLIC)
    {
	assert(path == NULL);
    }
    else
    {
	if (path == NULL)    /* no compressed file was passed in */
	{
	    nwos_private_path = nwos_get_private_objects_path();
	    compressed_file = (strstr(nwos_private_path, "compressed") != NULL);
	}
	else
	{
	    assert(type == READ_ONLY);
	    nwos_private_path = path;
	    compressed_file = true;
	}
    }

    /********************************/
    /* open the public objects file */
    /********************************/

#ifdef PUBLIC_MODE
    public_file_desc = open(nwos_public_path, O_RDWR);
#else
    public_file_desc = open(nwos_public_path, O_RDONLY);
#endif

    if (public_file_desc < 0)
    {
	perror(nwos_public_path);
	exit(1);
    }

#ifdef PUBLIC_MODE
    lock.l_type = F_WRLCK;
    lock.l_whence = SEEK_SET;
    lock.l_start = 0;
    lock.l_len = 0;

    if (fcntl(public_file_desc, F_SETLK, &lock) != 0)
    {
	perror(nwos_public_path);
	exit(1);
    }
#endif

    if (read(public_file_desc, &nwos_disk_header, sizeof(nwos_disk_header)) != sizeof(nwos_disk_header))
    {
	snprintf(log_msg, sizeof(log_msg), "reading disk header from: %s", nwos_public_path);
	perror(log_msg);
	exit(1);
    }

    if (memcmp(nwos_disk_header.magic_number, MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header: %s\n", nwos_public_path);
	exit(1);
    }

    if (memcmp(nwos_disk_header.version_string, VERSION_STRING, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header: %s\n", nwos_public_path);
	exit(1);
    }

    nwos_4_uint8_to_uint32(nwos_disk_header.used_blocks, &nwos_used_public_blocks);

    snprintf(log_msg, sizeof(log_msg), " used public blocks: %9u", nwos_used_public_blocks);
    nwos_log(log_msg);

#ifdef PUBLIC_MODE
    nwos_word_to_ref(nwos_used_public_blocks, &nwos_next_public_ref);

    /* don't set read_only flag if in PUBLIC_MODE */
#else
    read_only = (type == PUBLIC || type == READ_ONLY);
#endif


    /***************************************************/
    /* if a private objects file was specified open it */
    /***************************************************/

    if (nwos_private_path != NULL)
    {
	if (read_only)
	{
	    private_file_desc = open(nwos_private_path, O_RDONLY);
	}
	else
	{
	    private_file_desc = open(nwos_private_path, O_RDWR);
	}

	if (private_file_desc < 0)
	{
	    perror(nwos_private_path);
	    exit(1);
	}

	lock.l_type = read_only ? F_RDLCK : F_WRLCK;
	lock.l_whence = SEEK_SET;
	lock.l_start = 0;
	lock.l_len = 0;

	if (fcntl(private_file_desc, F_SETLK, &lock) != 0)
	{
	    perror(nwos_private_path);
	    exit(1);
	}


	if (compressed_file)
	{
	    index_compressed_file();   /* build an index into the compressed file */

	    if (lseek(private_file_desc, 0LL, SEEK_SET) < 0)
	    {
		perror("rewinding after indexing");
		exit(1);
	    }
	}

	if (read(private_file_desc, &nwos_disk_header, sizeof(nwos_disk_header)) != sizeof(nwos_disk_header))
	{
	    snprintf(log_msg, sizeof(log_msg), "reading disk header from: %s", nwos_private_path);
	    perror(log_msg);
	    exit(1);
	}

	if (memcmp(nwos_disk_header.magic_number, MAGIC_NUMBER, 4) != 0)
	{
	    fprintf(stderr, "Missing magic number in disk header: %s\n", nwos_private_path);
	    exit(1);
	}

	if (memcmp(nwos_disk_header.version_string, VERSION_STRING, 4) != 0)
	{
	    // allow more compressed files to be other compatible versions

	    if ((compressed_file && memcmp(nwos_disk_header.version_string, OLDEST_COMPATIBLE_COMPRESSED_VERSION, 4) < 0) ||
		(!compressed_file && memcmp(nwos_disk_header.version_string, VERSION_STRING, 4) != 0))
	    {
		fprintf(stderr, "Incorrect version string in disk header: %s\n", nwos_private_path);
		exit(1);
	    }
	}

	nwos_4_uint8_to_uint32(nwos_disk_header.block_offset_to_chunks, &nwos_block_offset_to_chunks);
	nwos_4_uint8_to_uint32(nwos_disk_header.total_blocks, &nwos_total_private_blocks);
	nwos_4_uint8_to_uint32(nwos_disk_header.used_blocks, &nwos_used_private_blocks);
	nwos_4_uint8_to_uint32(nwos_disk_header.used_chunks, &nwos_used_private_chunks);

	nwos_total_private_chunks = nwos_total_private_blocks / BLOCKS_IN_CHUNK;

	if (lseek(private_file_desc, FILE_BLOCK_SIZE, SEEK_SET) < 0)
	{
	    perror("chunk index");
	    exit(1);
	}

	chunk_info_size_in_bytes = nwos_total_private_chunks * sizeof(Chunk_Info);
	chunk_info = nwos_malloc(chunk_info_size_in_bytes);

	if (read(private_file_desc, chunk_info, chunk_info_size_in_bytes) != chunk_info_size_in_bytes)
	{
	    snprintf(log_msg, sizeof(log_msg), "reading chunk info from: %s", nwos_private_path);
	    perror(log_msg);
	    exit(1);
	}

	used_blocks = 0;
	for (i = 0; i < (int)nwos_total_private_chunks; i++)
	{
#ifndef WORDS_BIGENDIAN
	    chunk_info[i].ref = byteswap_uint32(chunk_info[i].ref);
	    chunk_info[i].used = byteswap_uint16(chunk_info[i].used);
	    chunk_info[i].index = byteswap_uint16(chunk_info[i].index);
#endif
	    used_blocks += chunk_info[i].used;
	}

	if (used_blocks != nwos_used_private_blocks)
	{
	    snprintf(log_msg, sizeof(log_msg), 
		     "Warning: calculated sum of used blocks (%u) doesn't match stored (%u)",
		     used_blocks, nwos_used_private_blocks);
	    nwos_log(log_msg);
	    fprintf(stderr, "%s\n", log_msg);
	}

	chunk_info_modified = false;

	update_chunk_info_reverse_index();


	/* now create the chunk_used_index */

	chunk_info_size_in_bytes = nwos_total_private_chunks * sizeof(uint16);
	chunk_used_index = nwos_malloc(chunk_info_size_in_bytes);

	/* insert them into a heap to do a heap sort */
	for (i = 0; i < (int)nwos_used_private_chunks; i++)
	{
	    j = i + 1;
	    while (j / 2 > 0 && chunk_info[chunk_used_index[(j / 2) - 1]].used < chunk_info[i].used)
	    {
		/* parent is smaller, move it lower in the tree */
		chunk_used_index[j - 1] = chunk_used_index[(j / 2) - 1];
		j = j / 2;
	    }

	    chunk_used_index[j - 1] = i;
	}

	for (i = (int)nwos_used_private_chunks; i > 1; i--)
	{
	    /* save the index at the top of the tree (smallest) */
	    save = chunk_used_index[i - 1];

	    /* move the root of the the (largest) to the top */
	    chunk_used_index[i - 1] = chunk_used_index[0];

	    j = 1;
	    while (j < (i / 2) - 1)
	    {
		k = j * 2;

		/* if there is a right child */
		if (k < i - 1)
		{
		    left = chunk_info[chunk_used_index[k - 1]].used;
		    right = chunk_info[chunk_used_index[k + 1 - 1]].used;

		    if (left < right)
		    {
			k++;
		    }
		}

		if (chunk_info[save].used >= chunk_info[chunk_used_index[k-1]].used) break;

		chunk_used_index[j - 1] = chunk_used_index[k - 1];

		j = k;
	    }

	    chunk_used_index[j - 1] = save;
	}

	if (!compressed_file)
	{
	    assert(nwos_total_private_blocks > 0);
	}

	snprintf(log_msg, sizeof(log_msg), "   private blocks: %9u  used: %9u", nwos_total_private_blocks, nwos_used_private_blocks);
	nwos_log(log_msg);
    }

#ifndef NO_BACKUP
    /* Do a backup file? */

    backup_dir = nwos_get_backup_directory_path();

    if (backup_dir != NULL && !read_only)
    {
	if (!nwos_initialize_backup(backup_dir, &nwos_disk_header))    /* creating backup failed, shut 'er back down */
	{
	    nwos_terminate_disk_io();
	    exit(1);
	}
    }
#endif
}


/****************/
/* Open status. */
/****************/

bool nwos_disk_io_is_read_only()
{
    return read_only;
}

bool nwos_disk_io_is_read_write()
{
    return !read_only;
}


/*********************************************/
/* Function to terminate the disk I/O module */
/*********************************************/

void nwos_terminate_disk_io()
{
    int i;
#ifndef PUBLIC_MODE
    size_t chunk_info_size_in_bytes;
#endif
    char log_msg[128];

    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].in_use && bit_map_cache[i].dirty)
	{
	    write_bit_map(&bit_map_cache[i]);

	    bit_map_cache[i].in_use = false;

	    free(bit_map_cache[i].map);
	    bit_map_cache[i].map = NULL;
	}
    }

    if (modified)
    {
#ifdef PUBLIC_MODE
	snprintf(log_msg, sizeof(log_msg), "  used public blocks: %9u",
		 nwos_ref_to_word(&nwos_next_public_ref));
	nwos_log(log_msg);

	copy_reference((ObjRef*)&nwos_disk_header.used_blocks, &nwos_next_public_ref);

	nwos_get_time_stamp(nwos_disk_header.last_change);
#else
	nwos_uint32_to_4_uint8(&nwos_used_private_blocks, nwos_disk_header.used_blocks);

	snprintf(log_msg, sizeof(log_msg), "  private blocks: %9u  used: %9u", nwos_total_private_blocks, nwos_used_private_blocks);
	nwos_log(log_msg);

	nwos_get_time_stamp(nwos_disk_header.last_change);

	if (chunk_info_modified)
	{
	    snprintf(log_msg, sizeof(log_msg), "  chunk index modified - new size: %u",
		     nwos_used_private_chunks);
	    nwos_log(log_msg);

	    chunk_info_size_in_bytes = nwos_total_private_chunks * sizeof(Chunk_Info);

#ifndef WORDS_BIGENDIAN
	    for (i = 0; i < (int)nwos_total_private_chunks; i++)
	    {
		chunk_info[i].ref = byteswap_uint32(chunk_info[i].ref);
		chunk_info[i].used = byteswap_uint16(chunk_info[i].used);
		chunk_info[i].index = byteswap_uint16(chunk_info[i].index);
	    }
#endif
	    if (lseek(private_file_desc, (off_t)FILE_BLOCK_SIZE, SEEK_SET) < 0)
	    {
		perror(nwos_private_path);
		exit(1);
	    }

	    if (write(private_file_desc, chunk_info, chunk_info_size_in_bytes) != chunk_info_size_in_bytes)
	    {
		perror(nwos_private_path);
		exit(1);
	    }

	    nwos_uint32_to_4_uint8(&nwos_used_private_chunks, nwos_disk_header.used_chunks);

	    chunk_info_modified = false;
	}
#endif

#ifdef PUBLIC_MODE
	if (lseek(public_file_desc, (off_t)0, SEEK_SET) < 0)
#else
	if (lseek(private_file_desc, (off_t)0, SEEK_SET) < 0)
#endif
	{
	    perror(nwos_private_path);
	    exit(1);
	}

#ifdef PUBLIC_MODE
	if (write(public_file_desc, &nwos_disk_header, sizeof(nwos_disk_header)) != sizeof(nwos_disk_header))
#else
	if (write(private_file_desc, &nwos_disk_header, sizeof(nwos_disk_header)) != sizeof(nwos_disk_header))
#endif
	{
	    perror(nwos_private_path);
	    exit(1);
	}
    }
    else
    {
	nwos_log("  No changes made");
    }

    if (private_file_desc > 0 && close(private_file_desc) < 0)
    {
	perror(nwos_private_path);
    }

    private_file_desc = 0;
    nwos_private_path = NULL;

#ifndef NO_BACKUP
    nwos_terminate_backup(modified, &nwos_disk_header);
#endif

    if (public_file_desc > 0 && close(public_file_desc) < 0)
    {
	perror(nwos_public_path);
    }

    public_file_desc = 0;

    log_estimate_results();
}


/*************************************/
/* This function creates a new chunk */
/*************************************/

void nwos_allocate_new_chunk(uint32 ref)
{
    uint32 offset;
    static uint8* chunk;
    int i;

    assert(nwos_used_private_chunks < nwos_total_private_chunks);

    assert(RESERVED_PUBLIC_BLOCKS <= ref && ref <= MAXIMUM_VALID_PRIVATE_REFERENCE);

    assert(nwos_hash_uint32_ref(ref) == 0);

    /* figure out where this ref goes in the chunk_info table */
    for (i = nwos_used_private_chunks - 1; i >= 0; i--)
    {
	if (chunk_info[i].ref < ref) break;

	chunk_info[i+1].ref = chunk_info[i].ref;
	chunk_info[i+1].used = chunk_info[i].used;
	chunk_info[i+1].index = chunk_info[i].index;
    }
    i++;

    offset = (ref - RESERVED_PUBLIC_BLOCKS) % USABLE_BLOCKS_PER_CHUNK;

    chunk_info[i].ref = ref - offset;

    assert(i == 0 || chunk_info[i-1].ref < chunk_info[i].ref);
    assert(i == nwos_used_private_chunks || chunk_info[i+1].ref > chunk_info[i].ref);

    /* make sure chunk_info.ref is an even multiple of USABLE_BLOCKS_PER_CHUNK */

    offset = chunk_info[i].ref - RESERVED_PUBLIC_BLOCKS;

    chunk_info[i].used = 0;
    chunk_info[i].index = nwos_used_private_chunks;

    assert(chunk_info[i].index < nwos_total_private_chunks);

    assert((offset % USABLE_BLOCKS_PER_CHUNK) == 0);

    if (chunk == NULL)   /* first time this has been called, allocate memory */
    {
	chunk = malloc(CHUNK_SIZE);

	if (chunk == NULL)
	{
	    perror("allocate memory for creating new chunk");
	    exit(1);
	}

	memset(chunk, 0, CHUNK_SIZE);

	chunk[0] = 0xff;
	chunk[1] = 0xff;
	chunk[2] = 0xff;
	chunk[3] = 0xff;
    }

    offset = nwos_block_offset_to_chunks + (chunk_info[i].index * BLOCKS_IN_CHUNK);

    if (lseek(private_file_desc, (off_t)offset << 8, SEEK_SET) < 0)
    {
	perror(nwos_private_path);
	exit(1);
    }

    if (write(private_file_desc, chunk, CHUNK_SIZE) != CHUNK_SIZE)
    {
	perror(nwos_private_path);
	exit(1);
    }

    nwos_used_private_chunks += 1;

    update_chunk_info_reverse_index();

    chunk_info_modified = true;
}


/**********************************************************************************************/
/* This function returns amount of private space used on disk 0.1 = 10%, 0.5 = 50%, 0.9 = 90% */
/**********************************************************************************************/

float nwos_private_space_used()
{
    float result;

    result = (float)nwos_used_private_blocks / (float)nwos_total_private_blocks;

    assert(0.0f <= result && result <= 1.0f);

    return result;
}


/**********************************************************************************************/
/* This function returns the index into the chunk_info table for the reference passed in or   */
/* -1 if not found.                                                                           */
/**********************************************************************************************/

#ifndef PUBLIC_MODE
static int uint32_ref_to_info_index(uint32 ref)
{
    int lwr;
    int upr;
    int mid = 0;

    assert(nwos_used_private_chunks > 0);
    assert(RESERVED_PUBLIC_BLOCKS <= ref && ref <= MAXIMUM_VALID_PRIVATE_REFERENCE);

    lwr = 1;

    upr = nwos_used_private_chunks;

    while (lwr <= upr)
    {
	mid = (upr + lwr) / 2;

	if (ref < chunk_info[mid - 1].ref)
	{
	    upr = mid - 1;
	}
	else if (ref >= chunk_info[mid - 1].ref + USABLE_BLOCKS_PER_CHUNK)
	{
	    lwr = mid + 1;
	}
	else
	{
	    mid = mid - 1;  /* adjust to 0 based index */
	    break;
	}
    }

    if (lwr <= upr)
    {
	assert(0 <= mid && mid < nwos_used_private_chunks);
	assert(chunk_info[mid].ref <= ref && ref < chunk_info[mid].ref + USABLE_BLOCKS_PER_CHUNK);

	return mid;
    }

    return -1;   /* not found */
}
#endif


/*********************************************************************************************/
/* this function maps a uint32 reference to a block number                                   */
/*                                                                                           */
/* The new formula is to skip over the bit maps instead of leaving holes for them, so every  */
/* 32 bit number is available for use.                                                       */
/*********************************************************************************************/

uint32 nwos_hash_uint32_ref(uint32 ref)
{
#ifdef PUBLIC_MODE
    assert(ref < nwos_used_public_blocks);
    return ref;
#else
    int index;
    uint32 result = 0;
    uint32 chunk;
    uint32 offset;

    if (ref < RESERVED_PUBLIC_BLOCKS)
    {
	result = ref;
    }
    else if (ref <= MAXIMUM_VALID_PRIVATE_REFERENCE && nwos_used_private_chunks > 0)
    {
	index = uint32_ref_to_info_index(ref);

	if (index >= 0)    /* found it */
	{
	    chunk = (uint32) chunk_info[index].index;

	    offset = (ref - RESERVED_PUBLIC_BLOCKS) % USABLE_BLOCKS_PER_CHUNK;

	    result = nwos_block_offset_to_chunks + (chunk * BLOCKS_IN_CHUNK) + BIT_MAP_BLOCKS + offset;
	}

//	printf("total blocks: %u chunks: %u  chunk: %u  offset: %u  result: %u (%08x)\n",
//	       nwos_total_private_blocks, nwos_total_private_chunks, chunk, offset, result, result);
    }

    return result;
#endif
}


uint32 nwos_hash_ref(ObjRef* ref)
{
    uint32 uint32_ref;

    uint32_ref = ((uint32)(ref->id[0]) << 24) | ((uint32)(ref->id[1]) << 16) | 
                                  ((uint32)(ref->id[2]) << 8) | ((uint32)(ref->id[3]));

    return nwos_hash_uint32_ref(uint32_ref);
}


static inline off_t ref_to_offset(ObjRef* ref) 
{
    uint32 hash;

    assert(FILE_BLOCK_SIZE == 256);

    hash = nwos_hash_ref(ref);

    return hash == 0 ? (off_t) -1 : (off_t) hash << 8;
}


/*********************************************/
/* Functions to manipulate the bit map cache */
/*********************************************/

static int8 bits_in_byte[256] = 
{
/*        0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f */
/* 0 */   0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4,
/* 1 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 2 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 3 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 4 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 5 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 6 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 7 */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* 8 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 9 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* a */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* b */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* c */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* d */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* e */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* f */   4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
};

void write_bit_map(Bit_Map_Cache_Entry* entry)
{
    int i;
    uint32 chunk;
    uint32 blocks_used = 0;
    char log_msg[80];

    assert(0 <= entry->index && entry->index < nwos_used_private_chunks);

    chunk = nwos_block_offset_to_chunks + chunk_info[entry->index].index * BLOCKS_IN_CHUNK;

#ifdef LOG_BIT_MAP
    snprintf(log_msg, sizeof(log_msg), "write bit_map: %d - %08x", entry->index, chunk);
    nwos_log(log_msg);
#endif

    assert(chunk < nwos_block_offset_to_chunks + nwos_total_private_blocks);

    /* don't count the blocks used by the bit map itself */
    for (i = BIT_MAP_BLOCKS / 8; i < BIT_MAP_BYTES; i++)
    {
	blocks_used += bits_in_byte[entry->map[i]];
    }

    assert(blocks_used <= BLOCKS_IN_CHUNK - BIT_MAP_BLOCKS);

    if (blocks_used != chunk_info[entry->index].used)
    {
	snprintf(log_msg, sizeof(log_msg),
		 "Warning: blocks used in chunk mismatch - map: %u  stored: %u",
		 blocks_used, chunk_info[entry->index].used);
	perror(log_msg);
	fprintf(stderr, "%s\n", log_msg);

	chunk_info[entry->index].used = blocks_used;
	chunk_info_modified = true;
    }

    if (lseek(private_file_desc, (off_t)chunk * FILE_BLOCK_SIZE, SEEK_SET) < 0)
    {
	snprintf(log_msg, sizeof(log_msg), "write_bit_map lseek chunk:%08x", chunk);
	perror(log_msg);
	exit(1);
    }

    if (write(private_file_desc, entry->map, BIT_MAP_BYTES) != BIT_MAP_BYTES)
    {
	snprintf(log_msg, sizeof(log_msg), "write_bit_map write chunk:%08x", chunk);
	perror(log_msg);
	exit(1);
    }

    entry->dirty = false;

    modified = true;
}


void read_bit_map_into_cache(Bit_Map_Cache_Entry* entry)
{
    int i;
    uint32 chunk;
    uint32 blocks_used = 0;
    char log_msg[80];

#ifdef LOG_BIT_MAP
    snprintf(log_msg, sizeof(log_msg), "read bit_map: %08x\n", entry->chunk);
    nwos_log(log_msg);
#endif

    chunk = nwos_block_offset_to_chunks + chunk_info[entry->index].index * BLOCKS_IN_CHUNK;

    assert(chunk < nwos_block_offset_to_chunks + nwos_total_private_blocks);

    if (lseek(private_file_desc, (off_t)chunk * FILE_BLOCK_SIZE, SEEK_SET) < 0)
    {
	snprintf(log_msg, sizeof(log_msg), "read_bit_map lseek chunk:%08x", chunk);
	perror(log_msg);
	exit(1);
    }

    if (entry->map == NULL)
    {
	entry->map = malloc(BIT_MAP_BYTES);
	assert(entry->map != NULL);
    }

    if (read(private_file_desc, entry->map, BIT_MAP_BYTES) != BIT_MAP_BYTES)
    {
	snprintf(log_msg, sizeof(log_msg), "read_bit_map write chunk:%08x", chunk);
	perror(log_msg);
	exit(1);
    }

    /* don't count the blocks used by the bit map itself */
    for (i = BIT_MAP_BLOCKS / 8; i < BIT_MAP_BYTES; i++)
    {
	blocks_used += bits_in_byte[entry->map[i]];
    }

    assert(blocks_used <= BLOCKS_IN_CHUNK - BIT_MAP_BLOCKS);


    if (blocks_used != chunk_info[entry->index].used)
    {
	snprintf(log_msg, sizeof(log_msg),
		 "Warning: blocks used in chunk mismatch - map: %u  stored: %u",
		 blocks_used, chunk_info[entry->index].used);
	perror(log_msg);
	fprintf(stderr, "%s\n", log_msg);

	chunk_info[entry->index].used = blocks_used;
	chunk_info_modified = true;
    }

    entry->in_use = true;
    entry->dirty = false;
}


/**********************************************************************************************/
/* This function returns the index into the chunk_info table for the chunk index passed in or */
/* -1 if not found.                                                                           */
/**********************************************************************************************/

static int chunk_index_to_info_index(uint16 chunk_index)
{
    assert(chunk_index < nwos_used_private_chunks);

    return chunk_info_reverse_index[chunk_index];
}


static Bit_Map_Cache_Entry* find_bit_map_in_cache(uint32 block)
{
    int i;
    int info_index;
    uint16 chunk_index;
    Bit_Map_Cache_Entry* result = NULL;

    assert(nwos_block_offset_to_chunks <= block && block < nwos_block_offset_to_chunks + nwos_total_private_blocks);

    /* assert((block % CHUNK_SIZE) != 0);  what was this for? */

    chunk_index = (block - nwos_block_offset_to_chunks) / BLOCKS_IN_CHUNK;

    info_index = chunk_index_to_info_index(chunk_index);

    assert(info_index >= 0);   /* should never call this if the chunk this block is in isn't used */

    /* see if it's already in the cache */
    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].in_use && bit_map_cache[i].index == info_index) break;
    }

    if (i < BIT_MAP_CACHE_SIZE)   /* found it */
    {
	result = &bit_map_cache[i];
    }
    else                           /* didn't find it */
    {
	/* find an empty one or find the oldest */

	result = bit_map_cache;   /* so we have an age to compare to */

	for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
	{
	    if (!bit_map_cache[i].in_use)
	    {
		result = &bit_map_cache[i];
		break;
	    }

	    if (bit_map_cache[i].age < result->age)
	    {
		result = &bit_map_cache[i];
	    }
	}

	if (i == BIT_MAP_CACHE_SIZE && result->dirty)    /* didn't find an empty one, write the oldest one out */
	{
	    write_bit_map(result);
	}

	result->index = info_index;

	read_bit_map_into_cache(result);
    }

    bit_map_tick++;
    result->age = bit_map_tick;

    assert(result != NULL);

    return result;
}


void nwos_set_bit_in_map(uint32 block)
{
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    if ((entry->map[byte_num] & (0x80 >> bit_num)) == 0)    /* don't count a block that was already used */
    {
	entry->map[byte_num] |= (0x80 >> bit_num);

/* printf("set_bit_in_map block: %08x  chunk: %08x  byte: %d  bit: %d\n", block, entry->chunk, byte_num, bit_num); */

	entry->dirty = true;

	chunk_info[entry->index].used++;
	chunk_info_modified = true;

	nwos_used_private_blocks++;
    }
}


void nwos_clear_bit_in_map(uint32 block)
{
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    if ((entry->map[byte_num] & (0x80 >> bit_num)) != 0)    /* don't count a block that was already clear */
    {
	entry->map[byte_num] &= ~(0x80 >> bit_num);

/* printf("clear_bit_in_map block: %08x  chunk: %08x  byte: %d  bit: %d\n", block, entry->chunk, byte_num, bit_num); */

	entry->dirty = true;

	chunk_info[entry->index].used--;
	chunk_info_modified = true;

	nwos_used_private_blocks--;
    }
}


bool nwos_test_bit_in_map(uint32 block)
{
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    return (entry->map[byte_num] & (0x80 >> bit_num)) != 0;
}


void nwos_flush_bit_maps()
{
    int i;

    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].in_use && bit_map_cache[i].dirty)
	{
	    write_bit_map(&bit_map_cache[i]);
	}
    }
}


#ifndef PUBLIC_MODE

//static float chunk_usage(Bit_Map_Cache_Entry* entry)
//{
//    float result;

//    result = (float)chunk_info[entry->index].used / (float)USABLE_BLOCKS_PER_CHUNK;

//    assert(0.0f <= result && result <= 1.0f);

//    return result;
//}
#endif


bool nwos_block_used(ObjRef* ref)
{
#ifdef PUBLIC_MODE
    uint8 block[FILE_BLOCK_SIZE];

    return  nwos_read_block(ref, block) && !is_void_reference((ObjRef*)&block[4]);
#else
    uint32 block;
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    block = nwos_hash_ref(ref);

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    return ((entry->map[byte_num] & (0x80 >> bit_num)) != 0);
#endif
}



/**********************************/
/* Functions to generate new id's */
/**********************************/

static void generate_new_completely_random_id(ObjRef* ref)
{
    int i;
    int bit;
    int byte;
    int chunk = 0;
    Bit_Map_Cache_Entry* entry;
    uint32 word;
#ifdef LOG_ID_GENERATION
    char msg[128];
#endif

    assert(nwos_used_private_chunks > 0);

    if (nwos_used_private_chunks > 1)
    {
#define UBPC USABLE_BLOCKS_PER_CHUNK

	/* first try to find one with some empties at random */
	for (i = 0; i < nwos_used_private_chunks; i++)
	{
	    chunk = random() % nwos_used_private_chunks;

	    if (chunk_info[chunk].used < USABLE_BLOCKS_PER_CHUNK) break;
	}

	if (i == nwos_used_private_chunks)   /* didn't find one, just do a search */
	{
	    for (chunk = 0; chunk < nwos_used_private_chunks; chunk++)
	    {
		if (chunk_info[chunk].used < USABLE_BLOCKS_PER_CHUNK) break;
	    }
	}
    }

    assert(chunk_info[chunk].used < USABLE_BLOCKS_PER_CHUNK);

    entry = find_bit_map_in_cache(nwos_block_offset_to_chunks +
				 ((uint32)chunk_info[chunk].index * BLOCKS_IN_CHUNK) + BIT_MAP_BLOCKS);

    /* don't pick one of the bytes (currently 0-3) used by the bit map */
    byte = (random() % (BIT_MAP_BYTES - (BIT_MAP_BLOCKS / 8))) + (BIT_MAP_BLOCKS / 8);

    if (entry->map[byte] == 0xff)   /* didn't get lucky */
    {
	for (i = 1; i < BIT_MAP_BYTES; i++)   /* find the nearest empty */
	{
	    if (byte - i >= 0 && entry->map[byte - i] != 0xff)
	    {
		byte -= i;
		break;
	    }

	    if (byte + i < BIT_MAP_BYTES && entry->map[byte + i] != 0xff)
	    {
		byte += i;
		break;
	    }

	    assert(0 <= byte - i || byte + i < BIT_MAP_BYTES);
	}
    }

    assert(BIT_MAP_BLOCKS / 8 <= byte && byte < BIT_MAP_BYTES);
    assert(entry->map[byte] != 0xff);

    bit = random() % 8;

    if ((entry->map[byte] & (0x80 >> bit)) != 0)
    {
	for (i = 1; i < 8; i++)   /* find the nearest empty */
	{
	    if (bit - i >= 0 && (entry->map[byte] & (0x80 >> (bit - i))) == 0)
	    {
		bit -= i;
		break;
	    }

	    if (bit + i < 8 && (entry->map[byte] & (0x80 >> (bit + i))) == 0)
	    {
		bit += i;
		break;
	    }

	    assert(0 <= bit - i || bit + i < 8);
	}
    }

    assert(0 <= bit && bit < 8);

    word = chunk_info[chunk].ref + byte * 8 - BIT_MAP_BLOCKS + bit;
    nwos_word_to_ref(word, ref);

#ifdef LOG_ID_GENERATION
    snprintf(msg, sizeof(msg), "generate_new_completely_random_id(%08x)", word);
    nwos_log(msg);
#endif

    assert(!nwos_block_used(ref));
}


void nwos_generate_new_completely_random_id(ObjRef* ref)
{
    generate_new_completely_random_id(ref);
    nwos_set_bit_in_map(nwos_hash_ref(ref));
}


#ifndef PUBLIC_MODE

/* This function searches back from the end of a chunk looking */
/* for an area that has lower usage than the max_density.      */

static uint32 find_first_low_density_in_chunk(int info_index, uint8 max_density)
{
    Bit_Map_Cache_Entry* entry;
    uint16 used;
    int i;
    uint32 result = BLOCKS_IN_CHUNK;

    assert(1 <= max_density && max_density <= 7);
    assert(chunk_info[info_index].used < USABLE_BLOCKS_PER_CHUNK);

    entry = find_bit_map_in_cache(nwos_block_offset_to_chunks +
			     ((uint32)chunk_info[info_index].index * BLOCKS_IN_CHUNK) + BIT_MAP_BLOCKS);

    used = bits_in_byte[entry->map[BIT_MAP_BYTES - 1]];

    for (i = 2; i < (USABLE_BLOCKS_PER_CHUNK / 8); i++)
    {
	used += bits_in_byte[entry->map[BIT_MAP_BYTES - i]];
	if (used / (8 * i)  > max_density) break;
    }

    while (bits_in_byte[entry->map[BIT_MAP_BYTES - i]] > max_density) i--;

    /* Note: if we are in the first usable byte this will back up into bit map blocks which is ok. */
    if (entry->map[BIT_MAP_BYTES - i] == 0)
    {
	i++;  /* look for last used block in previous byte */
    }

    result = BIT_MAP_BYTES - i;

    for (i = 7; i > 0; i--)
    {
	if (entry->map[result] & (0x80 >> i)) break;
    }

    return (result * 8) + i + 1;
}


/* This function returns an index into the chunk_info table for the first chunk */
/* If there is no space that meets the critera it returns -1.                   */
/*                                                                              */
/* needed_blocks    - total number of blocks needed                             */
/* max_used         - max number of blocks used in chunk to be acceptable       */
/* chunk_skip       - number of chunks that can be skipped                      */

static int find_starting_chunk(uint32 needed_blocks, uint32 max_used, int chunk_skip)
{
    uint16 last_chunk_index;
    uint32 blocks_available = 0;
    int result;
    int i;

    assert(nwos_used_private_chunks * USABLE_BLOCKS_PER_CHUNK - nwos_used_private_blocks > needed_blocks);

    for (result = 0; result < nwos_used_private_chunks; result++)
    {
	if (chunk_info[result].used <= max_used)    /* possible starting place */
	{
	    blocks_available = USABLE_BLOCKS_PER_CHUNK - chunk_info[result].used;
	    last_chunk_index = chunk_info[result].index;

	    for (i = result + 1; i < nwos_used_private_chunks; i++)
	    {
		if (blocks_available >= needed_blocks) break;

		if (chunk_info[i].used > max_used) break;

		if (chunk_info[i].index - last_chunk_index - 1 > chunk_skip)
		{
		    i--;   /* this block had enough space so reconsider it as the first block */
		    break;
		}

		blocks_available = blocks_available + USABLE_BLOCKS_PER_CHUNK - chunk_info[i].used;
	    }

	    if (blocks_available >= needed_blocks) break;

	    result = i;
	}
    }

    if (blocks_available < needed_blocks)   /* didn't make it */
    {
	result = -1;
    }

    return result;
}



void nwos_restart_id_generation()
{
    void_reference(&last_ref);
}


void nwos_set_allocation_parameters(uint32 density, int32 skip_backward, int32 skip_forward)
{
    assert(1 <= density && density <= USABLE_BLOCKS_PER_CHUNK);
    assert(-(int)RESERVED_PRIVATE_CHUNKS <= skip_backward && skip_backward <= 0);
    assert(1 <= skip_forward && skip_forward <= RESERVED_PRIVATE_CHUNKS);

    chunk_density_target = density;
    chunk_skip_backward = skip_backward;
    chunk_skip_forward = skip_forward;
}


bool nwos_check_blocks_available(uint32 estimated_blocks)
{
    uint32 total_blocks;
    uint32 allocated_blocks;
    uint32 skip;
    uint32 max;
    uint32 ref;
    char msg[80];
    bool need_to_allocate_more_chunks;

    assert(estimated_blocks > 0);

    assert(1 <= chunk_density_target && chunk_density_target <= USABLE_BLOCKS_PER_CHUNK);
    assert(-(int)RESERVED_PRIVATE_CHUNKS <= chunk_skip_backward && chunk_skip_backward <= 0);
    assert(1 <= chunk_skip_forward && chunk_skip_forward <= RESERVED_PRIVATE_CHUNKS);

    if (block_estimate > 0)   /* there was a previous estimate */
    {
	log_estimate_results();
    }

    block_estimate = estimated_blocks;

    last_used_blocks = nwos_used_private_blocks;

    /* make sure we have enough space */

    total_blocks = nwos_total_private_chunks * USABLE_BLOCKS_PER_CHUNK;

#if 0
    printf("Total chunks: %u  blocks: %u\n", nwos_total_private_chunks, total_blocks);
    printf("Used chunks: %u  blocks: %u\n", nwos_used_private_chunks, nwos_used_private_blocks);
#endif

    if (total_blocks <= nwos_used_private_blocks + block_estimate)
    {
	fprintf(stderr, "\n");
	fprintf(stderr, "Archive is too full:\n");
	fprintf(stderr, "      Total blocks: %u\n", total_blocks);
	fprintf(stderr, "       Used blocks: %u\n", nwos_used_private_blocks);
	fprintf(stderr, "  Available blocks: %u\n", total_blocks - nwos_used_private_blocks);
	fprintf(stderr, "  Estimated blocks: %u\n", block_estimate);
	fprintf(stderr, "\n");

	return false;
    }

    allocated_blocks = nwos_used_private_chunks * USABLE_BLOCKS_PER_CHUNK;

    /* if more than 90% of the allocated blocks are used allocate some more */
    if (nwos_total_private_chunks > nwos_used_private_chunks && 
	nwos_used_private_blocks > (nwos_used_private_chunks * USABLE_BLOCKS_PER_CHUNK / 10) * 9)
    {
	need_to_allocate_more_chunks = true;
	nwos_log("NOTE: more than 90%% full, allocating more chunks");
    }
    else
    {
	need_to_allocate_more_chunks = allocated_blocks <= nwos_used_private_blocks + block_estimate;
    }

    while (need_to_allocate_more_chunks)
    {
	/* allocate can currently only deal with adding chunks at the end */
	if (chunk_info[nwos_used_private_chunks-1].index + 1 >= nwos_total_private_chunks)
	{
	    fprintf(stderr, "This version cannot allocate chunks below what has already been allocated,\n");
	    fprintf(stderr, "since no more blocks are available at the end, cannot import this file.\n");

	    return false;
	}

	if (chunk_skip_forward == 1)
	{
	    skip = 1;
	}
	else if (chunk_skip_forward == RESERVED_PRIVATE_CHUNKS)
	{
	    max = MAXIMUM_VALID_PRIVATE_REFERENCE - chunk_info[nwos_used_private_chunks-1].ref;
	    skip = 1 + random() % (max / USABLE_BLOCKS_PER_CHUNK);
	}
	else
	{
	    skip = 1 + random() % chunk_skip_forward;
	}

	ref = chunk_info[nwos_used_private_chunks-1].ref + (USABLE_BLOCKS_PER_CHUNK * skip);

	snprintf(msg, sizeof(msg), "Allocating chunk: %08x\n", ref);
	nwos_log(msg);

	nwos_allocate_new_chunk(ref);

	/* make sure it increased */
	assert(nwos_used_private_chunks * USABLE_BLOCKS_PER_CHUNK > allocated_blocks);

	allocated_blocks = nwos_used_private_chunks * USABLE_BLOCKS_PER_CHUNK;

	need_to_allocate_more_chunks = allocated_blocks <= nwos_used_private_blocks + block_estimate;
    }

    return true;
}


/* density is how many blocks to use 1-7 out of 8 blocks. */

void nwos_generate_new_closely_spaced_id(int density, ObjRef* new_ref)
{
    Bit_Map_Cache_Entry* entry = NULL;
    uint32 ref;
    uint32 block;
    int    byte_num = 0;   /* in case first chunk is usable (don't go through loop) */
    int    bit_num;
    int    bits_used;
    uint8  byte;
    int    max_used;
    uint16 chunk_index;
    static int info_index = 0;
    static int skipped = 0;
#ifdef LOG_ID_GENERATION
    char   msg[128];
#endif

    assert(1 <= density && density <= 7);
    assert(block_estimate > 0);

    max_used = (USABLE_BLOCKS_PER_CHUNK * (8 - density)) / 8;

    /* if the estimated blocks will fit in one chunk, reduce it to what we need */
    if (block_estimate <= USABLE_BLOCKS_PER_CHUNK - max_used)
    {
	max_used = USABLE_BLOCKS_PER_CHUNK - block_estimate;
    }

    if (is_void_reference(&last_ref))
    {
	block = nwos_block_offset_to_chunks + chunk_info[info_index].index * BLOCKS_IN_CHUNK;
    }
    else
    {
	block = nwos_hash_ref(&last_ref) + 1;
    }


    /* at this point 'block' should have the block number we are starting the search at */

    while (true)
    {
	/* first see if block has gone past the end of this chunk */

	if ((block - nwos_block_offset_to_chunks) % BLOCKS_IN_CHUNK < BIT_MAP_BLOCKS)
	{
	    chunk_index = (block - nwos_block_offset_to_chunks) / BLOCKS_IN_CHUNK;

	    info_index = chunk_index_to_info_index(chunk_index);

	    while (chunk_info[info_index].used > max_used)
	    {
		/* see if the chunk after this one is usable, and if so back into this chunk */
		if (info_index + 1 == nwos_total_private_chunks)
		{
		    /* need to allocate a new chunk */
		    bool need_to_fix_allocation_code = false;
		    assert(need_to_fix_allocation_code);
		}
		else if (chunk_info[info_index + 1].used <= max_used)
		{
		    entry = find_bit_map_in_cache(nwos_block_offset_to_chunks + chunk_info[info_index].index * BLOCKS_IN_CHUNK + BIT_MAP_BLOCKS);

		    bits_used = 0;
		    for (byte_num = BIT_MAP_BYTES - 1; byte_num > 0; byte_num--)
		    {
			bits_used += bits_in_byte[entry->map[byte_num]];

			if (bits_used / (BIT_MAP_BYTES - byte_num) > density) break;
		    }

		    if (BIT_MAP_BYTES - byte_num > 1) break;

		    byte_num = 0;    /* reset this to what it was */
		}

		info_index++;
	    }

	    block = nwos_block_offset_to_chunks + chunk_info[info_index].index * BLOCKS_IN_CHUNK + BIT_MAP_BLOCKS + (byte_num * 8);
	}

	if (entry == NULL || chunk_info[entry->index].index != (block - nwos_block_offset_to_chunks) / BLOCKS_IN_CHUNK)
	{
	    entry = find_bit_map_in_cache(block);
	}

	assert(chunk_info[entry->index].index == (block - nwos_block_offset_to_chunks) / BLOCKS_IN_CHUNK);

	byte_num = (block % BLOCKS_IN_CHUNK) / 8;

	assert(byte_num < BIT_MAP_BYTES);
	bit_num = block % 8;

	byte = entry->map[byte_num];

	if (byte == 0xff)     /* this byte is full */
	{
	    block += (8 - bit_num);     /* skip over it */
	    skipped += (8 - bit_num);
	}
	else if ((byte & (0x80 >> bit_num)) == 0)  /* this block is empty */
	{
	    if (skipped > 0)                          /* and we've already skipped over blocks */
	    {
		skipped--;                               /* use one skipped */
		break;                                   /* and use this empty */
	    }

	    if ((random() % 8) < density)             /* randomly skip some blocks */
	    {
		break;                                   /* use this empty */
	    }

	    bit_num++;                                /* check the next bit */
	    if (bit_num > 7)
	    {
		bit_num = 0;
		byte_num++;
		if (byte_num == BIT_MAP_BYTES)
		{
		    break;                             /* use this empty because at end of this chunk */
		}

		byte = entry->map[byte_num];
	    }

	    if ((byte & (0x80 >> bit_num)) != 0)  /* next block is used */
	    {
		break;                                   /* so use this empty instead of skipping it */
	    }

	    /* if we get to here we are skipping this empty block for density reasons */
	}

	block++;
	skipped++;
    }

    assert(nwos_test_bit_in_map(block) == false);

    nwos_set_bit_in_map(block);

    assert(chunk_info[entry->index].index == (block - nwos_block_offset_to_chunks) / BLOCKS_IN_CHUNK);

    ref = chunk_info[entry->index].ref + (block - nwos_block_offset_to_chunks - chunk_info[entry->index].index * BLOCKS_IN_CHUNK - BIT_MAP_BLOCKS);

    nwos_word_to_ref(ref, new_ref);

    copy_reference(&last_ref, new_ref);

#ifdef LOG_ID_GENERATION
    snprintf(msg, sizeof(msg), "nwos_generate_new_closely_spaced_id(%08x)", ref);
    nwos_log(msg);
#endif
}


/* generate approximately 1 block out of N */

void nwos_generate_new_1_in_N_id(uint32 n, ObjRef* new_ref)
{
    Bit_Map_Cache_Entry* entry = NULL;
    uint32 ref;
    uint32 hash;
    int    byte_num;
    int    bit_num;
    uint8  byte;
#ifdef LOG_ID_GENERATION
    char   msg[128];
#endif


    if (is_void_reference(&last_ref))
    {
	generate_new_completely_random_id(new_ref);

	ref = nwos_ref_to_word(new_ref);
    }
    else
    {
	ref = nwos_ref_to_word(&last_ref) + (random() % n) + 1;
    }

    while (true)
    {
	hash = nwos_hash_uint32_ref(ref);

	if (hash == 0)  /* continue this somewhere else */
	{
	    generate_new_completely_random_id(new_ref);

	    ref = nwos_ref_to_word(new_ref);

	    hash = nwos_hash_uint32_ref(ref);
	}	    

	if (entry == NULL || chunk_info[entry->index].index != (hash - nwos_block_offset_to_chunks) / BLOCKS_IN_CHUNK)
	{
	    entry = find_bit_map_in_cache(hash);
	}

	assert(chunk_info[entry->index].index == (hash - nwos_block_offset_to_chunks) / BLOCKS_IN_CHUNK);

	byte_num = (hash % BLOCKS_IN_CHUNK) / 8;
	assert(byte_num < BIT_MAP_BYTES);
	bit_num = hash % 8;

	byte = entry->map[byte_num];
	
	if ((byte & (0x80 >> bit_num)) == 0) break;

	ref++;
    }

    nwos_set_bit_in_map(hash);

    nwos_word_to_ref(ref, new_ref);

    copy_reference(&last_ref, new_ref);

#ifdef LOG_ID_GENERATION
    snprintf(msg, sizeof(msg), "nwos_generate_new_1_in_N_id(%08x)", ref);
    nwos_log(msg);
#endif
}
#endif


#ifdef PUBLIC_MODE
void nwos_generate_new_public_id(ObjRef* ref)
{
    int i;
    int acc;
    uint8 diff[4];

    copy_reference(ref, &nwos_next_public_ref);

    for (i = sizeof(nwos_next_public_ref)-1; i > -1; i--)
    {
	nwos_next_public_ref.id[i]++;
	if (nwos_next_public_ref.id[i] != 0) break;
    }

    acc = (int)nwos_next_public_ref.id[3] - (int)ref->id[3];
    diff[3] = acc;
    acc = (int)nwos_next_public_ref.id[2] - (int)ref->id[2] + (acc >> 8);
    diff[2] = acc;
    acc = (int)nwos_next_public_ref.id[1] - (int)ref->id[1] + (acc >> 8);
    diff[1] = acc;
    acc = (int)nwos_next_public_ref.id[0] - (int)ref->id[0] + (acc >> 8);
    diff[0] = acc;

    assert(diff[0] == 0 && diff[1] == 0 && diff[2] == 0 && diff[3] == 1);

    nwos_used_public_blocks++;
}
#endif


/*****************************************/
/* Function to read a block from storage */
/*****************************************/

bool nwos_read_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE])
{
    char ref_str[64];
    off_t offset;
    int i = 0;
    int lower;
    int upper;
    int file_desc;
    uint32 ref_hash;
    uint32 mid_hash;

    assert(!is_void_reference(ref));

    /*    printf("Read block: %02x%02x%02x%02x\n", ref->id[0], ref->id[1], ref->id[2], ref->id[3]); */


#if NO_DISK_ACCESS

    for (i = 0; i < 4096 && cache[i] != NULL; i++)
    {
	if (is_same_object(ref, (ObjRef*)&cache[i][4]))
	{
	    break;
	}
    }

    assert(i < 4096);

    if (cache[i] != NULL)
    {
	memcpy(block, cache[i], FILE_BLOCK_SIZE);
    }

    return cache[i] != NULL;

#else
    if (nwos_ref_to_word(ref) < RESERVED_PUBLIC_BLOCKS)
    {
	file_desc = public_file_desc;
	offset = (off_t) nwos_ref_to_word(ref) * FILE_BLOCK_SIZE;
    }
    else if (storage_index == NULL)     /* normal storage */
    {
	file_desc = private_file_desc;
	offset = ref_to_offset(ref);
    }
    else
    {
	file_desc = private_file_desc;

	lower = 1;
	upper = storage_count;

	ref_hash = nwos_ref_to_word(ref);

	while (lower <= upper)
	{
	    i = (lower + upper) / 2;

	    mid_hash = nwos_ref_to_word(&storage_index[i-1]);

	    if (mid_hash > ref_hash)
	    {
		upper = i - 1;
	    }
	    else if (mid_hash < ref_hash)
	    {
		lower = i + 1;
	    }
	    else
	    {
		i = i - 1;
		break;
	    }
	}

	if (lower > upper)
	{
	    int i;

	    fprintf(stderr, "Can't find %08x in index table:\n", ref_hash);
	    fflush(stderr);

	    if (ref_hash < nwos_ref_to_word(&storage_index[0]))
	    {
		i = 0;
	    }
	    else
	    {
		for (i = 1; i < storage_count; i++)
		{
		    if (nwos_ref_to_word(&storage_index[i-1]) < ref_hash && ref_hash < nwos_ref_to_word(&storage_index[i])) break;
		}
	    }

	    if (i > 0)
	    {
		fprintf(stderr, "  %d: %08x\n", i-1, nwos_ref_to_word(&storage_index[i-1]));
	    }

	    if (i < storage_count)
	    {
		fprintf(stderr, "  %d: %08x\n", i, nwos_ref_to_word(&storage_index[i]));
	    }

	    offset = (off_t) -1;
	}
	else
	{
	    assert(is_same_object(&storage_index[i], ref));

	    offset = (off_t)i * FILE_BLOCK_SIZE;
	}
    }

    if (offset < 0) return false;

    if (lseek(file_desc, offset, SEEK_SET) < 0)
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	snprintf(ref_str, sizeof(ref_str), "nwos_read_block lseek ref:%02x%02x%02x%02x offset:%x%08x",
		ref->id[0], ref->id[1], ref->id[2], ref->id[3], offset_upr, offset_lwr);

	perror(ref_str);
	exit(1);
    }

    return read(file_desc, block, FILE_BLOCK_SIZE) == FILE_BLOCK_SIZE;

#endif
}


/****************************************/
/* Function to write a block to storage */
/****************************************/

void nwos_write_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE])
{
    char log_msg[128];
    off_t offset;

#ifdef PUBLIC_MODE
    assert(nwos_reference_type(ref) == Public_Reference);
#else
    assert(nwos_reference_type(ref) != Public_Reference);
#endif

    assert(!is_void_reference(ref));
    assert((block[0] & 0x7f) == 0 && block[1] == 0 && block[2] == 0 && block[3] == 0);
    assert(is_same_object((ObjRef*)block + 1, ref));

#ifdef LOG_WRITE_BLOCK
    snprintf(log_msg, sizeof(log_msg), "write_block - ref:%02x%02x%02x%02x block:%08x",
	    ref->id[0], ref->id[1], ref->id[2], ref->id[3], nwos_hash_ref(ref));
    nwos_log(log_msg);
#endif

#if NO_DISK_ACCESS
{
    int i;
    for (i = 0; i < 4096 && cache[i] != NULL; i++)
    {
	if (is_same_object(ref, (ObjRef*)&cache[i][4]))
	{
	    printf("WARNING - rewritting the same block: %02x%02x%02x%02x\n", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	    break;
	}
    }

    assert(i < 4096);

    if (cache[i] == NULL)
    {
	cache[i] = malloc(FILE_BLOCK_SIZE);
	assert(cache[i] != NULL);
    }

    memcpy(cache[i], block, FILE_BLOCK_SIZE);
}
#else
    assert(storage_index == NULL);   /* at this time if there is a storage index it is read only */

    offset = ref_to_offset(ref);

    assert(offset >= 0);

#ifdef PUBLIC_MODE
    if (lseek(public_file_desc, offset, SEEK_SET) < 0)
#else
    if (lseek(private_file_desc, offset, SEEK_SET) < 0)
#endif
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	snprintf(log_msg, sizeof(log_msg), "write_block lseek ref:%02x%02x%02x%02x offset:%x%08x",
		ref->id[0], ref->id[1], ref->id[2], ref->id[3], offset_upr, offset_lwr);

	perror(log_msg);
	exit(1);
    }

#ifdef PUBLIC_MODE
    if (write(public_file_desc, block, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE)
#else
    if (write(private_file_desc, block, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE)
#endif
    {
	snprintf(log_msg, sizeof(log_msg), "write_block write ref:%02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);

	perror(log_msg);
	exit(1);
    }

#ifndef NO_BACKUP
    nwos_backup_write_block(block);
#endif

    modified = true;
#endif
}


/************************************************/
/* Remove an object (by writing zero's over it) */
/************************************************/

void nwos_remove_object(ObjRef* ref)
{
    char log_msg[128];
    uint8 block[FILE_BLOCK_SIZE];
    off_t offset;

    assert(nwos_object_exists(ref));

    snprintf(log_msg, sizeof(log_msg), "nwos_remove_object - %02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
    nwos_log(log_msg);

    memset(block, 0, sizeof(block));

    assert(sizeof(block) == FILE_BLOCK_SIZE);

    offset = ref_to_offset(ref);

    assert(offset >= 0);

    if (lseek(private_file_desc, offset, SEEK_SET) < 0)
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	snprintf(log_msg, sizeof(log_msg), "remove_object lseek ref:%02x%02x%02x%02x offset:%x%08x",
		ref->id[0], ref->id[1], ref->id[2], ref->id[3], offset_upr, offset_lwr);

	perror(log_msg);
	exit(1);
    }

    if (write(private_file_desc, block, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE)
    {
	snprintf(log_msg, sizeof(log_msg), "nwos_remove_object write ref:%02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);

	perror(log_msg);
	exit(1);
    }

    nwos_clear_bit_in_map(nwos_hash_ref(ref));

    modified = true;

    assert(!nwos_object_exists(ref));
}



