/*
--             This file is part of the New World OS project
--                 Copyright (C) 2005-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: person.c,v $
-- Revision 1.25  2008/09/01 00:10:54  jsedwards
-- Fix copyright year.  NO code changes.
--
-- Revision 1.24  2008/08/31 20:33:18  jsedwards
-- Add assert around calls to nwos_read_object_from_disk because it now
-- returns false when it fails, whereas before it would assert itself.
--
-- Revision 1.23  2007/07/29 19:46:00  jsedwards
-- Fix so variable isn't compiled because code that uses it isn't compiled to
-- eliminate compiler warning.
--
-- Revision 1.22  2007/07/12 03:11:47  jsedwards
-- Add nwos_add_birthday function to add a birthday to a person that doesn't
-- have one.
--
-- Revision 1.21  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.20  2007/05/05 14:43:45  jsedwards
-- Changed add_person to verify public names first.
--
-- Revision 1.19  2007/04/02 03:50:57  jsedwards
-- Added routines to do private gender objects and made add_person function
-- deal with creating private person objects.
--
-- Revision 1.18  2007/01/06 20:13:32  jsedwards
-- Changed to find_private_name instead of find_name.
--
-- Revision 1.17  2006/12/21 12:59:47  jsedwards
-- Hack to make it compile with new split public and private classes,
-- NON-FUNCTIONAL!
--
-- Revision 1.16  2006/12/07 14:08:46  jsedwards
-- Moved setup routine and associated functions to attic/big_bang.c.
--
-- Revision 1.15  2006/12/01 14:31:30  jsedwards
-- Changed to use new malloc_reference_list and free_reference_list functions
-- instead of inlining the code.
--
-- Revision 1.14  2006/11/11 12:01:06  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.13  2006/11/03 04:17:12  jsedwards
-- Change "nick_name" to "nickname" (in the dictionary it is one word).
--
-- Revision 1.12  2006/10/26 01:51:28  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.11.2.6  2006/10/25 12:22:29  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.11.2.5  2006/10/01 11:43:18  jsedwards
-- Add asserts to make sure it finds the gender objects.
--
-- Revision 1.11.2.4  2006/09/29 12:10:07  jsedwards
-- Add list_persons function.
--
-- Revision 1.11.2.3  2006/09/01 13:27:20  jsedwards
-- Changed "nwos_object_size" to "nwos_reference_list_size" and added the
-- object reference to "nwos_fill_in_common_header" so it can put the "id"
-- in the header now.
--
-- Revision 1.11.2.2  2006/08/16 12:50:12  jsedwards
-- Removed unused variable "create".
--
-- Revision 1.11.2.1  2006/07/31 12:48:59  jsedwards
-- Changed to use ask_yes_or_no function so that it will accept 'y' instead of
-- just 'yes'.
--
-- Revision 1.11  2006/01/09 13:52:54  jsedwards
-- Moved all "add to references" calls after person object is created.  This
-- way we don't get a bunch of non-existent objects in the reference lists
-- if it crashes before the person object is created.
--
-- Revision 1.10  2006/01/09 03:25:15  jsedwards
-- Added "any persons named", "full name to string", and "upgrade persons
-- from 0004 to 0005" routines.
--
-- Revision 1.9  2006/01/08 18:37:36  jsedwards
-- Fixed a bug in find person where it assumed that if it found the name it
-- would find a person, which of course isn't true.
--
-- Revision 1.8  2006/01/08 16:43:48  jsedwards
-- Fixed size of buffer passed to 'get_input' so that it is based upon the
-- actual 'sizeof' the buffer, instead of #define which could be (and was)
-- wrong.
--
-- Revision 1.7  2006/01/02 23:02:35  jsedwards
-- Fixed bug where fixed define was passed to fgets instead of the actual
-- 'size' parameter.  Added an assert to verify class was found.
--
-- Revision 1.6  2006/01/01 21:17:57  jsedwards
-- Changed "find person" routine to not print out the "cannot find" message,
-- and just return false.  It is now up to the caller to print the "not found"
-- message.
--
-- Revision 1.5  2006/01/01 21:13:25  jsedwards
-- Removed Social Security Number stuff from "add person" again.  This will
-- be done with a separate command.
--
-- Revision 1.4  2006/01/01 19:42:29  jsedwards
-- Fixed bug when entering a male person, there was garbage in the maiden name.
-- Changed the wording for more than one person.
--
-- Revision 1.3  2006/01/01 00:47:05  jsedwards
-- Added "find_person" and "describe_person" routines.
--
-- Revision 1.2  2005/12/31 17:51:14  jsedwards
-- Added gender stuff.  Moved creation of classes related to persons from
-- big_bang.  Added social security stuff back in (still not working).
--
-- Revision 1.1  2005/12/31 14:57:05  jsedwards
-- Copied "create_person.c" into "computer.c" and "person.c" and then removed
-- "create_person.c".
--
-- Revision 1.15  2005/12/30 14:13:39  jsedwards
-- Removed Social Security Number stuff, not ready yet.
--
-- Revision 1.14  2005/12/30 05:12:57  jsedwards
-- Tweaked the birth date stuff and added social security number.
--
-- Revision 1.13  2005/12/29 18:40:38  jsedwards
-- Added code for the new long key and encryption stuff.
--
-- Revision 1.12  2005/12/27 19:52:21  jsedwards
-- Added code to generate the root object from a password.  Changed to find
-- class definition instead of hardcoded file name and to generate a random
-- person object id instead of generating it based on contents.
--
-- Revision 1.11  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.10  2005/12/21 23:28:15  jsedwards
-- Fixed so that it adds the references to the name and date objects.
--
-- Revision 1.9  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.8  2005/12/05 19:06:31  jsedwards
-- Moved calculation of header checksum down just before writing object to
-- disk to ensure no changes were made after the checksum was calculated.
-- Also added the "nickname" field.
--
-- Revision 1.7  2005/12/04 14:14:46  jsedwards
-- Changed to actually get input and create a person (finally)!
--
-- Revision 1.6  2005/12/04 04:13:02  jsedwards
-- Added 'nwos' prefix to create_xxxx function names and eliminated the
-- 'referring' object parameter from all of them.
--
-- Revision 1.5  2005/12/04 00:35:49  jsedwards
-- Changed to just create a date.
--
-- Revision 1.4  2005/12/04 00:23:46  jsedwards
-- Removed create year, date, and name stuff and put it in the date.c and
-- the name.c files.
--
-- Revision 1.3  2005/12/02 20:33:11  jsedwards
-- Changed so that name objects have "NAME" in the filename and added new
-- parameter to the create_reference_list calls.
--
-- Revision 1.2  2005/12/02 19:30:32  jsedwards
-- Changed to create Year object filename from the year itself.
--
-- Revision 1.1  2005/12/02 13:05:17  jsedwards
-- Initial version that only creates a name and a year.
--
*/

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify.h"
#include "objectify_private.h"
#include "crc32.h"

#define MAX_NAME_SIZE 64
#define MAX_DATE_SIZE 16
#define MAX_SSN_SIZE  12
#define MAX_WORD_SIZE 12


bool nwos_is_valid_gender(char* str)
{
    if (strcasecmp(str, "female") == 0) return true;
    if (strcasecmp(str, "male") == 0) return true;
    if (strlen(str) == 1)
    {
	if (*str == 'F' || *str == 'f') return true;
	if (*str == 'm' || *str == 'M') return true;
    }

    return false;
}


void nwos_find_public_gender(char* gender, ObjRef* ref)
{
    C_struct_Word word_obj;
    ObjRef gender_class_ref;
    ObjRef english_lang_ref;
    ObjRef word_ref;

    assert(nwos_is_valid_gender(gender));

    assert(nwos_find_public_class_definition("GENDER", &gender_class_ref));

    nwos_find_language("English", &english_lang_ref);

    if (toupper(*gender) == 'F')
    {
	assert(nwos_find_word("female", &gender_class_ref, &english_lang_ref, &word_ref));
    }
    else if (tolower(*gender) == 'm')
    {
	assert(nwos_find_word("male", &gender_class_ref, &english_lang_ref, &word_ref));
    }

    assert(nwos_read_object_from_disk(&word_ref, &word_obj, sizeof(word_obj)));

    memcpy(ref, &word_obj.thing, sizeof(ObjRef));
}


bool nwos_find_private_gender(char* gender, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Gender gender_obj;
    ObjRef gender_class_ref;
    ObjRef public_gender_ref;
    ObjRef class_ref;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    bool result = false;

    assert(nwos_is_valid_gender(gender));

    nwos_find_public_gender(gender, &public_gender_ref);

    if (nwos_find_private_class_definition("GENDER", &gender_class_ref))
    {
	nwos_read_class_definition(&gender_class_ref, &class_def_obj);

	ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

	num_refs = ref_list->common_header.num_refs;

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &class_ref);

	    if (is_same_object(&class_ref, &gender_class_ref))
	    {
		assert(nwos_read_object_from_disk(&ref_list->references[i], &gender_obj, sizeof(gender_obj)));

		if (is_same_object(&gender_obj.header.object.clone_of, &public_gender_ref))
		{
		    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		    result = true;
		    break;
		}
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    return result;
}


/****************************************************************************************/
/* This routine finds a private gender object or creates one by cloning the public one. */
/**************************************(*************************************************/

void nwos_find_or_create_private_gender(char* gender, ObjRef* ref)
{
    ObjRef public_gender_ref;
    ObjRef private_gender_class_ref;
    C_struct_Gender gender_obj;
    char msg[128];

    assert(nwos_is_valid_gender(gender));

    if (!nwos_find_private_gender(gender, ref))
    {
	nwos_find_public_gender(gender, &public_gender_ref);

	nwos_find_or_create_private_class_definition("GENDER", &private_gender_class_ref);

	assert(nwos_read_object_from_disk(&public_gender_ref, &gender_obj, sizeof(gender_obj)));

	nwos_generate_new_completely_random_id(ref);

	snprintf(msg, sizeof(msg), "clone_gender(%02x%02x%02x%02x) -> %02x%02x%02x%02x\n",
		 public_gender_ref.id[0], public_gender_ref.id[1], public_gender_ref.id[2], public_gender_ref.id[3],
		 ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	nwos_log(msg);

	copy_reference(&gender_obj.header.common.id, ref);

	nwos_get_time_stamp(gender_obj.header.common.creation_time);

	copy_reference(&gender_obj.header.common.class_definition, &private_gender_class_ref);

	copy_reference(&gender_obj.header.object.clone_of, &public_gender_ref);

	nwos_create_reference_list(ref, &gender_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &gender_obj.header.object, sizeof(ObjectHeader), gender_obj.header.common.header_chksum);

	nwos_write_object_to_disk(ref, &gender_obj, sizeof(gender_obj));

	nwos_add_to_references(ref, &private_gender_class_ref);
    }
}


bool nwos_is_valid_ssn(const char* ssn)
{
    const char *ptr;
    bool result;
    bool dash = false;

    ptr = ssn;

    result = (isdigit(*ptr++) && isdigit(*ptr++) && isdigit(*ptr++));

    if (result && *ptr == '-')
    {
	ptr++;
	dash = true;
    }

    if (result) result = (isdigit(*ptr++) && isdigit(*ptr++));

    if (result && dash) result = (*ptr++ == '-');

    if (result) result = (isdigit(*ptr++) && isdigit(*ptr++) && isdigit(*ptr++) && isdigit(*ptr++));

    return result;
}

#if 0
ObjCreateResult nwos_create_ssn(ObjRef* person, char* number, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Social_Security_Number ssn;
    ObjRef reference;
    ObjRef class_ref;
    ReferenceList* ref_list;
    int num_ssns;
    char just_digits[9];
    ObjCreateResult result = CREATED_NEW;


    /* first find out if we already have this number */

    assert(nwos_find_private_class_definition("SOCIAL SECURITY NUMBER", &reference));

    nwos_read_class_definition(&reference, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_years = ref_list->common_header.num_refs;

    /* printf("num_years: %d\n", num_years); */

    for (i = 0; i < num_years; i++)
    {
	assert(nwos_read_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge)));

	/* remember the ptr_year_obj points to the kludge buffer */

	if (ptr_year_obj->count == 4 && memcmp(ptr_year_obj->storage, year_ascii, 4) == 0)   /* found a match */
	{
	    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
	    result = FOUND_EXISTING;
	    break;
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    if (result != FOUND_EXISTING)   /* didn't find it */
    {
	memset(kludge, 0, sizeof(kludge));  /* zero it out */

	/* remember the ptr_year_obj points to the kludge buffer */

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&ptr_year_obj->header.common, ref, &reference);

	ptr_year_obj->count = 4;
	memcpy(ptr_year_obj->storage, year_ascii, 4);

	nwos_create_reference_list(ref, &ptr_year_obj->header.object.references);

	nwos_crc32_calculate((uint8*) &ptr_year_obj->header.object, sizeof(ObjectHeader), ptr_year_obj->header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &ptr_year_obj->count, sizeof(kludge) - sizeof(EveryObject), ptr_year_obj->header.common.data_chksum);

	nwos_write_object_to_disk(ref, kludge, sizeof(kludge));

        nwos_add_to_reference_list(ref, &class_def_obj.header.object.references);
    }

    return result;
}
#endif


static void get_input(char* descr, char* buffer, size_t size)
{
    char *ptr;

    while (1)
    {
	printf("%s: ", descr);
	fflush(stdout);
	fgets(buffer, size, stdin);
	ptr = strchr(buffer, '\n');
	if (ptr != NULL)
	{
	    *ptr = '\0';
	    break;
	}
	do { fgets(buffer, size, stdin); } while (strchr(buffer, '\n') == NULL);
	printf("input too long - try again!\n");
    }
}


static void check_public(char* name)
{
    ObjRef ref;

    if (nwos_find_public_name(name, &ref))
    {
	printf("public: %02x%02x%02x%02x\n", ref.id[0], ref.id[1], ref.id[2], ref.id[3]); 
    }
    else
    {
	printf("Warning: %s not in public names\n", name);
    }
}

void nwos_add_person()
{
    ObjRef reference;
    char gender[MAX_WORD_SIZE];
    char first_name[MAX_NAME_SIZE];
    char middle_name[MAX_NAME_SIZE];
    char last_name[MAX_NAME_SIZE];
    char goes_by[MAX_NAME_SIZE];
    char maiden_name[MAX_NAME_SIZE];
    char nickname[MAX_NAME_SIZE];
    char birth_date[MAX_DATE_SIZE];
    ObjRef person_class_ref;
    int month, day, year;
    bool ok;
    C_struct_Person person_obj;

    nwos_find_or_create_private_class_definition("PERSON", &person_class_ref);

    ok = false;
    while (!ok)
    {
	get_input("gender", gender, sizeof(gender));
	ok = nwos_is_valid_gender(gender);
	if (!ok) printf("must input either 'male' or 'female' or simply 'm' or 'f'\n");
    }

    get_input("last name", last_name, sizeof(last_name));
    if (*last_name == '\0')
    {
	printf("Error: this version cannot create a person without the last name.\n");
	return;
    }
    check_public(last_name);

    get_input("first name", first_name, sizeof(first_name));
    if (*first_name == '\0')
    {
	printf("Error: this version cannot create a person without the first name.\n");
	return;
    }
    check_public(first_name);

    get_input("middle name", middle_name, sizeof(middle_name));
    if (*middle_name != '\0')
    {
	check_public(middle_name);
    }

    get_input("goes by", goes_by, sizeof(goes_by));
    if (*goes_by != '\0')
    {
	check_public(goes_by);
    }

    get_input("nick name", nickname, sizeof(nickname));
    if (*nickname != '\0')
    {
	check_public(nickname);
    }


    if (toupper(*gender) == 'F')
    {
	get_input("maiden name", maiden_name, sizeof(maiden_name));
	if (*maiden_name != '\0')
	{
	    check_public(maiden_name);
	}
    }
    else
    {
	maiden_name[0] = '\0';
    }

    ok = false;
    while (!ok)
    {
	get_input("birth date M/D/YYYY", birth_date, sizeof(birth_date));
	if (*birth_date == '\0')
	{
	    ok = true;    /* none given */
	}
	else
	{
	    ok = (sscanf(birth_date, "%d/%d/%d", &month, &day, &year) == 3);
	    if (!ok) ok = (sscanf(birth_date, "%d-%d-%d", &month, &day, &year) == 3);
	    if (ok) ok = (1582 <= year && year <= 9999);    /* 1582 start of Gregorian calendar */
	    if (ok) ok = (1 <= month && month <= 12);
	    if (ok) switch (month)
	    {
	      case 2:
		if (nwos_is_leap_year(year))
		{
		    ok = (day <= 29);
		}
		else
		{
		    ok = (day <= 28);
		}
		break;

	      case 4:
	      case 6:
	      case 9:
	      case 11:
		ok = (day <= 30);
		break;

	      default:
		ok = (day <= 31);
		break;
	    }
	}
    }

    if (nwos_ask_yes_or_no(NULL, "Create person"))
    {
	memset(&person_obj, 0, sizeof(person_obj));

	nwos_generate_new_id(&reference);

	nwos_fill_in_common_header(&person_obj.header.common, &reference, &person_class_ref);

	nwos_find_or_create_private_gender(gender, &person_obj.gender);

	nwos_create_name(first_name, &person_obj.first_name);

	if (*middle_name != '\0')
	{
	    nwos_create_name(middle_name, &person_obj.middle_name);
	}

	nwos_create_name(last_name, &person_obj.last_name);

	if (*goes_by != '\0') 
	{
	    nwos_create_name(goes_by, &person_obj.goes_by);
	}

	if (*nickname != '\0') 
	{
	    nwos_create_name(nickname, &person_obj.nickname);
	}

	if (*maiden_name != '\0')
	{
	    nwos_create_name(maiden_name, &person_obj.maiden_name);
	}

	if (*birth_date != '\0')
	{
	    nwos_create_date(year, month, day, &person_obj.birth_date);
	}

	nwos_create_reference_list(&reference, &person_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &person_obj.header.object, sizeof(ObjectHeader), person_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &person_obj.gender, sizeof(person_obj) - sizeof(EveryObject), person_obj.header.common.data_chksum);

	nwos_write_object_to_disk(&reference, &person_obj, sizeof(person_obj));

        nwos_add_to_references(&reference, &person_class_ref);
	nwos_add_to_references(&reference, &person_obj.gender);
	nwos_add_to_references(&reference, &person_obj.first_name);
	nwos_add_to_references(&reference, &person_obj.last_name);

	if (!is_void_reference(&person_obj.middle_name))
	{
	    nwos_add_to_references(&reference, &person_obj.middle_name);
	}

	if (!is_void_reference(&person_obj.goes_by))
	{
	    nwos_add_to_references(&reference, &person_obj.goes_by);
	}

	if (!is_void_reference(&person_obj.nickname))
	{
	    nwos_add_to_references(&reference, &person_obj.nickname);
	}

	if (!is_void_reference(&person_obj.maiden_name))
	{
	    nwos_add_to_references(&reference, &person_obj.maiden_name);
	}

	if (!is_void_reference(&person_obj.birth_date))
	{
	    nwos_add_to_references(&reference, &person_obj.birth_date);
	}
	
	printf("Object created: %02x%02x%02x%02x\n", reference.id[0], reference.id[1], reference.id[2], reference.id[3]);
    }
}


void nwos_add_birthday(char* name, char* date)
{
    bool date_ok;
    ObjRef person_ref;
    int month, day, year;
    C_struct_Person person_obj;
#if 0
    char temp[16];
#endif

    date_ok = (sscanf(date, "%d/%d/%d", &month, &day, &year) == 3);
    if (!date_ok) date_ok = (sscanf(date, "%d-%d-%d", &month, &day, &year) == 3);

    if (!date_ok)
    {
	printf("Date must be of the format MM/DD/YYYY or MM-DD-YYYY\n");
    }
    else
    {
	date_ok = (1582 <= year && year <= 9999);    /* 1582 start of Gregorian calendar */

	if (!date_ok)
	{
	    printf("Year must be in the range 1582 to 9999\n");
	}
	else
	{
	    date_ok = (1 <= month && month <= 12);

	    if (!date_ok)
	    {
		printf("Month must be in the range 1 to 12\n");
	    }
	    else
	    {
		switch (month)
		{
		  case 2:
		    if (nwos_is_leap_year(year))
		    {
			date_ok = (1 <= day && day <= 29);

			if (!date_ok)
			{
			    printf("Day must be in the range 1 to 29 because %d is a leap year\n", year);
			}
		    }
		    else
		    {
			date_ok = (1 <= day && day <= 28);

			if (!date_ok)
			{
			    printf("Day must be in the range 1 to 28 because %d is not a leap year\n", year);
			}
		    }
		    break;

		  case 4:
		  case 6:
		  case 9:
		  case 11:
		    date_ok = (1 <= day && day <= 30);
		    if (!date_ok)
		    {
			printf("Day must be in the range 1 to 30\n");
		    }
		    break;

		  default:
		    date_ok = (1 <= day && day <= 31);
		    if (!date_ok)
		    {
			printf("Day must be in the range 1 to 31\n");
		    }
		    break;
		}
	    }
	}
    }

    if (!date_ok) return;

    if (!nwos_find_person(name, &person_ref))
    {
	printf("I'm sorry I can't find %s in the database.\n", name);
	return;
    }

    assert(nwos_read_object_from_disk(&person_ref, &person_obj, sizeof(person_obj)));

    if (!is_void_reference(&person_obj.birth_date))
    {
	printf("This version cannot change a birthday, SORRY!\n");
	return;
#if 0
	nwos_date_to_string(&person_obj.birth_date, temp, sizeof(temp));

	printf("%s already has a birthday set: %s\n", name, temp);

	if (nwos_ask_yes_or_no(NULL, "overwrite"))
	{
	    /* need to make sure we can delete reference from old date */
	}
#endif
    }  
	
    nwos_create_date(year, month, day, &person_obj.birth_date);

    nwos_crc32_calculate((uint8*) &person_obj.gender, sizeof(person_obj) - sizeof(EveryObject), person_obj.header.common.data_chksum);

    nwos_overwrite_object_to_disk(&person_ref, &person_obj, sizeof(person_obj));

    nwos_add_to_references(&person_ref, &person_obj.birth_date);
}


bool nwos_any_persons_named(char* name)
{
    ObjRef name_ref;
    ObjRef object_class;
    EveryObject name_header;
    ObjRef person_class_ref;
    int person_count;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    bool result = false;

  /* first find the name */

    if (nwos_find_private_name(name, &name_ref))
    {
	assert(nwos_find_private_class_definition("PERSON", &person_class_ref));

	nwos_read_object_headers_from_disk(&name_ref, &name_header);

	ref_list = nwos_malloc_reference_list(&name_header.object.references);

	num_refs = ref_list->common_header.num_refs;

	/* printf("name num refs: %d\n", num_refs); */

	person_count = 0;

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	    if (is_same_object(&object_class, &person_class_ref))
	    {
		result = true;
		break;
	    }
	}
    }

    return result;
}


#define MAX_NAMES 20

bool nwos_find_person(char* name, ObjRef* person_ref)
{
    C_struct_Person person_obj;
    ObjRef name_ref;
    ObjRef object_class;
    EveryObject name_header;
    ObjRef person_class_ref;
    ObjRef person_refs[MAX_NAMES];
    int person_count;
    ReferenceList* ref_list;
    int num_refs;
    int selection;
    char temp[64];
    char buffer[4];
    bool ok;
    int i;
    int j;
    bool result = false;

  /* first find the name */

    if (nwos_find_private_name(name, &name_ref))
    {
	assert(nwos_find_private_class_definition("PERSON", &person_class_ref));

	nwos_read_object_headers_from_disk(&name_ref, &name_header);

	ref_list = nwos_malloc_reference_list(&name_header.object.references);

	num_refs = ref_list->common_header.num_refs;

	/* printf("name num refs: %d\n", num_refs); */

	person_count = 0;

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	    if (is_same_object(&object_class, &person_class_ref))
	    {
		for (j = 0; j < person_count; j++)    /* see if we already have this one */
		{
		    if (is_same_object(&ref_list->references[i], &person_refs[j])) break;
		}

		if (j == person_count)    /* not already in the list, so add it */
		{
		    memcpy(&person_refs[person_count], &ref_list->references[i], sizeof(ObjRef));
		    person_count++;
		}
	    }

	    if (person_count == MAX_NAMES)
	    {
		printf("/nSorry, right now this program can only deal with a maximum of %d possible names.\n", MAX_NAMES);
		break;
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;

	selection = 0;    /* default to first person in case there is only one */

	if (person_count > 1)  /* ask which one we want */
	{
	    printf("\nThere is more than one person with that name:\n\n");

	    for (j = 0; j < person_count; j++)    /* see if we already have this one */
	    {
		assert(nwos_read_object_from_disk(&person_refs[j], &person_obj, sizeof(person_obj)));

		nwos_name_to_string(&person_obj.first_name, temp, sizeof(temp));
		printf("  %d: %s ", j, temp);
		if (!is_void_reference(&person_obj.middle_name))
		{
		    nwos_name_to_string(&person_obj.middle_name, temp, sizeof(temp));
		    printf("%s ", temp);
		}
		nwos_name_to_string(&person_obj.last_name, temp, sizeof(temp));
		printf("%s\n", temp);
	    }

	    printf("Which %s are you referring to?\n\n", name);

	    ok = false;
	    while (!ok)
	    {
		get_input("selection number", buffer, sizeof(buffer));

		printf("\n");

		if (isdigit(buffer[0]) && buffer[1] == '\0')   /* single digit */
		{
		    selection = buffer[0] - '0';
		    ok = (selection <= person_count);
		}
		else if (isdigit(buffer[0]) && buffer[2] == '\0')   /* two digits */
		{
		    selection = (buffer[0] - '0') * 10 + (buffer[1] - '0');
		    ok = (selection < person_count);
		}

		if (!ok) printf("Please enter the number of one of the people listed.\n");
	    }
	}

	if (person_count > 0)
	{
	    memcpy(person_ref, &person_refs[selection], sizeof(ObjRef));
	    result = true;
	}
    }

    return result;
}


void nwos_list_persons()
{
    C_struct_Person person_obj;
    ObjRef object_class;
    EveryObject class_ref_header;
    ObjRef person_class_ref;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    char name[32];

    assert(nwos_find_private_class_definition("PERSON", &person_class_ref));

    nwos_read_object_headers_from_disk(&person_class_ref, &class_ref_header);

    ref_list = nwos_malloc_reference_list(&class_ref_header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("person class num refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, &person_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &person_obj, sizeof(person_obj)));

	    nwos_name_to_string(&person_obj.last_name, name, sizeof(name));

	    printf("%s, ", name);

	    nwos_name_to_string(&person_obj.first_name, name, sizeof(name));

	    printf("%s ", name);

	    if (!is_void_reference(&person_obj.middle_name))
	    {
		nwos_name_to_string(&person_obj.middle_name, name, sizeof(name));

		printf("%s", name);
	    }

	    printf("\n");
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;
}


void nwos_describe_person(char* name)
{
    C_struct_Person person_obj;
    ObjRef person_ref;
    char temp[64];
    
    if (!nwos_find_person(name, &person_ref))
    {
	printf("\nI'm sorry I don't know anyone named %s.\n\n", name);
    }
    else
    {
	assert(nwos_read_object_from_disk(&person_ref, &person_obj, sizeof(person_obj)));

	printf("full name: ");

	nwos_name_to_string(&person_obj.first_name, temp, sizeof(temp));
	printf("%s ", temp);

	if (!is_void_reference(&person_obj.middle_name))
	{
	    nwos_name_to_string(&person_obj.middle_name, temp, sizeof(temp));
	    printf("%s ", temp);
	}

	nwos_name_to_string(&person_obj.last_name, temp, sizeof(temp));
	printf("%s\n", temp);


	if (!is_void_reference(&person_obj.goes_by))
	{
	    printf("goes by: ");
	    nwos_name_to_string(&person_obj.goes_by, temp, sizeof(temp));
	    printf("%s\n", temp);
	}

	if (!is_void_reference(&person_obj.nickname))
	{
	    printf("nickname: ");
	    nwos_name_to_string(&person_obj.nickname, temp, sizeof(temp));
	    printf("%s\n", temp);
	}

	if (!is_void_reference(&person_obj.maiden_name))
	{
	    printf("maiden_name: ");
	    nwos_name_to_string(&person_obj.maiden_name, temp, sizeof(temp));
	    printf("%s\n", temp);
	}

	if (!is_void_reference(&person_obj.birth_date))
	{
	    printf("born: ");
	    nwos_date_to_string(&person_obj.birth_date, temp, sizeof(temp));
	    printf("%s\n", temp);
	}
    }

    printf("\n");
}


void nwos_full_name_to_string(ObjRef* first_name_ref, ObjRef* middle_name_ref, ObjRef* last_name_ref, char* name, size_t size)
{
    char first_name[32];
    char middle_name[32];
    char last_name[32];

    nwos_name_to_string(first_name_ref, first_name, sizeof(first_name));

    if (is_void_reference(middle_name_ref))
    {
	middle_name[0] = '\0';
    }
    else
    {
	nwos_name_to_string(middle_name_ref, middle_name, sizeof(middle_name));
    }

    nwos_name_to_string(last_name_ref, last_name, sizeof(last_name));

    snprintf(name, size, "%s %s %s", first_name, middle_name, last_name);
}


