/*
--             This file is part of the New World OS project
--                 Copyright (C) 2005-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: us_state.c,v $
-- Revision 1.27  2008/09/01 00:10:55  jsedwards
-- Fix copyright year.  NO code changes.
--
-- Revision 1.26  2008/08/31 21:47:01  jsedwards
-- Add assert arounds call to nwos_read_object_from_disk because it now
-- returns false when it fails, whereas before it would assert itself.
--
-- Revision 1.25  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.24  2007/01/06 20:14:26  jsedwards
-- Change to find_public_name and find_public_spelling instead of find_name
-- and find spelling.
--
-- Revision 1.23  2006/12/25 12:17:00  jsedwards
-- Removed assert from create_city.  For now cities have to be public, need
-- to fix this.
--
-- Revision 1.22  2006/12/21 13:08:37  jsedwards
-- Hack to make it compile with new split public and private classes,
-- NON-FUNCTIONAL!
--
-- Revision 1.21  2006/12/11 03:33:21  jsedwards
-- Changed US_State number to ordinal object instead of uint8.
--
-- Revision 1.20  2006/12/08 10:41:07  jsedwards
-- Changed 'create_city' to 'nwos_private_create_city' because the big_bang
-- needs to use it.
--
-- Revision 1.19  2006/12/07 14:08:46  jsedwards
-- Moved setup routine and associated functions to attic/big_bang.c.
--
-- Revision 1.18  2006/12/01 14:31:31  jsedwards
-- Changed to use new malloc_reference_list and free_reference_list functions
-- instead of inlining the code.
--
-- Revision 1.17  2006/11/11 12:01:07  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.16  2006/10/26 01:51:29  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.15.2.2  2006/10/19 01:47:08  jsedwards
-- Fix format specifier for uint32 which is unsigned instead of long unsigned
-- now.
--
-- Revision 1.15.2.1  2006/09/01 13:27:20  jsedwards
-- Changed "nwos_object_size" to "nwos_reference_list_size" and added the
-- object reference to "nwos_fill_in_common_header" so it can put the "id"
-- in the header now.
--
-- Revision 1.15  2006/01/09 13:56:53  jsedwards
-- Restored call to create reference list for state object that was
-- accidentally deleted in revision 1.12.
--
-- Revision 1.14  2006/01/09 03:45:29  jsedwards
-- Added assert to verify that the capital in 0004 state objects was void.
--
-- Revision 1.13  2006/01/09 03:09:37  jsedwards
-- Added test function to return true if there is a state with the given name.
-- Added a routine to describe a state.  Added code to the upgrade routine
-- to create the "CITY" class and to print out more information about the
-- upgrade.
--
-- Revision 1.12  2006/01/06 14:12:49  jsedwards
-- Added code to upgrade a 0004 state object to a 0005 state object (which
-- entails adding the capitol city).
--
-- Revision 1.11  2006/01/04 18:56:43  jsedwards
-- Added routine to add a bunch of additional major cities (only Alaska to
-- California are entered in this version).  Moved print of "City created" to
-- interactive creation so it doesn't print during "additional city" creation.
--
-- Revision 1.10  2006/01/03 02:58:45  jsedwards
-- Fixed bug in "create us city" routine, where it wasn't reading the class
-- reference list.
--
-- Revision 1.9  2006/01/02 19:43:37  jsedwards
-- Added creation of the US City class.  Added "find state by name", "create
-- us city", "add city", and "list cities in state" routines.  NOTE: there is
-- something wrong with this version, it can't find the name Salt Lake City or
-- Oklahoma City (yet it can find Jefferson City and Carson City).
--
-- Revision 1.8  2006/01/02 19:24:16  jsedwards
-- Moved remaining state information (capital city and slogan) from states.txt
-- file and into state_info.  Changed debugging printf to print reference id.
--
-- Revision 1.7  2006/01/01 21:49:01  jsedwards
-- Moved date, phone, us_state, and word class creations out of "big bang"
-- and into the respective files.
--
-- Revision 1.6  2005/12/29 17:50:53  jsedwards
-- Commented out printf debugging statements, that aren't useful now.
--
-- Revision 1.5  2005/12/29 16:52:19  jsedwards
-- Reordered the state info entries so that they are in order of states
-- joining instead of alphabetical by postal code.
--
-- Revision 1.4  2005/12/29 16:38:07  jsedwards
-- Changed to just read spelling obj header to get reference list instead of
-- the whole variable sized object.
--
-- Revision 1.3  2005/12/27 18:32:36  jsedwards
-- Changed to look up class definition instead of using a fixed file name.
-- Also changed so that object id is random instead of based upon contents.
--
-- Revision 1.2  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.1  2005/12/21 04:03:42  jsedwards
-- Initial version that creates all 50 states and finds state from postal code.
--
*/


#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"
#include "objectify_private.h"



bool nwos_find_state_from_postal_code(char* postal_code, ObjRef* ref)
{
    EveryObject spelling_header;
    ObjRef state_class_ref;
    ObjRef spelling_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_spellings;
    int i;
    bool result = false;

    assert(strlen(postal_code) == 2);

    assert(nwos_find_public_class_definition("US STATE", &state_class_ref));

    if (nwos_find_public_spelling(postal_code, &spelling_ref))
    {
	nwos_read_object_headers_from_disk(&spelling_ref, &spelling_header);

	ref_list = nwos_malloc_reference_list(&spelling_header.object.references);

	num_spellings = ref_list->common_header.num_refs;

	/* printf("num_spellings (in find_postal_code): %d\n", num_spellings); */

	for (i = 0; i < num_spellings; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	    if (is_same_object(&object_class, &state_class_ref))   /* it is a state object */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		result = true;
		break;
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    return result;
}


bool nwos_find_state_from_name(char* name, ObjRef* ref)
{
    EveryObject name_header;
    ObjRef state_class_ref;
    ObjRef name_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_names;
    int i;
    bool result = false;

    assert(nwos_find_public_class_definition("US STATE", &state_class_ref));

    if (nwos_find_public_name(name, &name_ref))
    {
	nwos_read_object_headers_from_disk(&name_ref, &name_header);

	ref_list = nwos_malloc_reference_list(&name_header.object.references);

	num_names = ref_list->common_header.num_refs;

	/* printf("num_names (in find_postal_code): %d\n", num_names); */

	for (i = 0; i < num_names; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	    if (is_same_object(&object_class, &state_class_ref))   /* it is a state object */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		result = true;
		break;
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    return result;
}


bool nwos_any_states_named(char* name)
{
    ObjRef state_ref;

    return nwos_find_state_from_name(name, &state_ref);
}


/* local routine that does the actual city object creation, needed by create all 50 states to create capital cities */
/* WARNING: DOES NOT add city to state references because if it is "create all 50 states" calling, the state object */
/* has not been written to disk yet and so we can't call add_to_references yet.  So the caller must add it to state refs. */

void nwos_private_create_city(ObjRef* city_class_ref, ObjRef* name_ref, ObjRef* state_ref, ObjRef* city_ref)
{
    C_struct_US_City city_obj;

    memset(&city_obj, 0, sizeof(city_obj));  /* zero it out */

    nwos_generate_new_id(city_ref);

    nwos_fill_in_common_header(&city_obj.header.common, city_ref, city_class_ref);

    nwos_create_reference_list(city_ref, &city_obj.header.object.references);

    memcpy(&city_obj.name, name_ref, sizeof(ObjRef));
    memcpy(&city_obj.state, state_ref, sizeof(ObjRef));

    nwos_crc32_calculate((uint8*) &city_obj.header.object, sizeof(ObjectHeader), city_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &city_obj.name, sizeof(city_obj) - sizeof(EveryObject), city_obj.header.common.data_chksum);

    nwos_write_object_to_disk(city_ref, &city_obj, sizeof(city_obj));

    nwos_add_to_references(city_ref, city_class_ref);
    nwos_add_to_references(city_ref, name_ref);
}



ObjCreateResult nwos_create_us_city(char* city, ObjRef* state_ref, ObjRef* city_ref)
{
    EveryObject name_header;
    C_struct_US_City city_obj;
    ObjRef city_class_ref;
    ObjRef name_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    ObjCreateResult result = CREATED_NEW;


    assert(!is_void_reference(state_ref));

    /* first find out if we already have this city in this state */

    assert(nwos_find_public_class_definition("US CITY", &city_class_ref));

    if (nwos_create_name(city, &name_ref) == FOUND_EXISTING)    /* see if this city already exists */
    {
	nwos_read_object_headers_from_disk(&name_ref, &name_header);

	ref_list = nwos_malloc_reference_list(&name_header.object.references);

	num_refs = ref_list->common_header.num_refs;

	/* printf("num_refs: %d\n", num_refs); */

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);

	    if (is_same_object(&city_class_ref, &object_class))
	    {
		assert(nwos_read_object_from_disk(&ref_list->references[i], &city_obj, sizeof(city_obj)));

		if (is_same_object(&city_obj.state, state_ref))   /* found a match */
		{
		    memcpy(city_ref, &ref_list->references[i], sizeof(ObjRef));
		    result = FOUND_EXISTING;
		    break;
		}
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    if (result != FOUND_EXISTING)   /* didn't find it */
    {
	nwos_private_create_city(&city_class_ref, &name_ref, state_ref, city_ref);
	nwos_add_to_references(city_ref, state_ref);
    }

    return result;
}


static void get_input(char* descr, char* buffer, size_t size)
{
    char *ptr;

    while (1)
    {
	printf("%s: ", descr);
	fflush(stdout);
	fgets(buffer, size, stdin);
	ptr = strchr(buffer, '\n');
	if (ptr != NULL)
	{
	    *ptr = '\0';
	    break;
	}
	do { fgets(buffer, size, stdin); } while (strchr(buffer, '\n') == NULL);
	printf("input too long - try again!\n");
    }
}


#define MAX_NAMES 20
#define MAX_NAME_SIZE 64

void nwos_add_city(ObjRef* city_ref)
{
    C_struct_US_City city_obj;
    C_struct_US_State state_obj;
    ObjRef name_ref;
    ObjRef state_ref;
    ObjRef object_class;
    EveryObject name_header;
    ObjRef city_class_ref;
    ObjRef city_refs[MAX_NAMES];
    int city_count;
    ReferenceList* ref_list;
    int num_refs;
    int selection;
    char temp[80];
    char buffer[4];
    bool ok;
    int i;
    int j;
    char city[MAX_NAME_SIZE];
    char state[MAX_NAME_SIZE];

    assert(false);

    nwos_find_public_class_definition("US CITY", &city_class_ref);

    get_input("city", city, sizeof(city));

  /* first find the name */

    assert(nwos_find_public_class_definition("US CITY", &city_class_ref));

    selection = -1;    /* default to create new city */
    city_count = 0;

    if (nwos_find_public_name(city, &name_ref))
    {
	nwos_read_object_headers_from_disk(&name_ref, &name_header);

	ref_list = nwos_malloc_reference_list(&name_header.object.references);

	num_refs = ref_list->common_header.num_refs;

	/* printf("name num refs: %d\n", num_refs); */

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	    if (is_same_object(&object_class, &city_class_ref))    /* add it to the list */
	    {
		memcpy(&city_refs[city_count], &ref_list->references[i], sizeof(ObjRef));
		city_count++;
	    }

	    if (city_count == MAX_NAMES)
	    {
		printf("/nSorry, right now this program can only deal with a maximum of %d possible names.\n", MAX_NAMES);
		break;
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;

	if (city_count > 0)  /* ask which one we want */
	{
	    printf("\nI have found cities in the following states with that name\n\n");

	    for (j = 0; j < city_count; j++)    /* see if we already have this one */
	    {
		assert(nwos_read_object_from_disk(&city_refs[j], &city_obj, sizeof(city_obj)));

		assert(nwos_read_object_from_disk(&city_obj.state, &state_obj, sizeof(state_obj)));

		nwos_name_to_string(&state_obj.name, temp, sizeof(temp));

		printf("  %d: %s\n", j, temp);
	    }

	    printf("\nIf it is in one of these states, enter the number, otherwise just hit enter to create a new city.\n\n");

	    ok = false;
	    while (!ok)
	    {
		get_input("selection number", buffer, sizeof(buffer));

		printf("\n");

		if (buffer[0] == '\0')
		{
		    ok = true;   /* create a new one */
		}
		else if (isdigit(buffer[0]) && buffer[1] == '\0')   /* single digit */
		{
		    selection = buffer[0] - '0';
		    ok = (selection <= city_count);
		}
		else if (isdigit(buffer[0]) && buffer[2] == '\0')   /* two digits */
		{
		    selection = (buffer[0] - '0') * 10 + (buffer[1] - '0');
		    ok = (selection < city_count);
		}

		if (!ok) printf("Please enter the number of one of the cities listed.\n");
	    }
	}
    }

    if (selection != -1)     /* selected an existing city */
    {
	memcpy(city_ref, &city_refs[selection], sizeof(ObjRef));
    }
    else                     /* need to add a new one */
    {
	snprintf(temp, sizeof(temp), "What state is %s in (name or 2 letter code)", city);

	ok = false;
	while (!ok)
	{
	    get_input(temp, state, sizeof(state));

	    printf("\n");

	    if (strlen(state) == 2)
	    {
		ok = nwos_find_state_from_postal_code(state, &state_ref);
	    }
	    else
	    {
		ok = nwos_find_state_from_name(state, &state_ref);
	    }

	    if (!ok) printf("Can't find state: %s, please re-enter.\n\n", state);
	}

	nwos_create_us_city(city, &state_ref, city_ref);
	printf("Created city: %08x\n", nwos_ref_to_word(city_ref));
    }
}


void nwos_describe_state(char* state)
{
    C_struct_US_State state_obj;
    C_struct_US_City city_obj;
    ObjRef state_ref;
    char temp[80];
    bool ok;

    if (strlen(state) == 2)
    {
	ok = nwos_find_state_from_postal_code(state, &state_ref);
    }
    else
    {
        ok = nwos_find_state_from_name(state, &state_ref);
    }

    if (!ok)
    {
	printf("I can't find the state of %s\n", state);
	return;
    }
    else
    {
	assert(nwos_read_object_from_disk(&state_ref, &state_obj, sizeof(state_obj)));

	nwos_name_to_string(&state_obj.name, temp, sizeof(temp));
	printf("Name: %s\n", temp);

	nwos_ordinal_number_to_string(&state_obj.number, temp, sizeof(temp));
	printf("Number %s\n", temp);

	nwos_date_to_string(&state_obj.date, temp, sizeof(temp));
	printf("Joined: %s\n", temp);

	assert(nwos_read_object_from_disk(&state_obj.capital, &city_obj, sizeof(city_obj)));

	nwos_name_to_string(&city_obj.name, temp, sizeof(temp));
	printf("Capital: %s\n", temp);
    }

    printf("\n");
}


void nwos_list_cities_in_state(char* state)
{
    C_struct_US_City city_obj;
    ObjRef state_ref;
    ObjRef object_class;
    EveryObject state_header;
    ObjRef city_class_ref;
    ReferenceList* ref_list;
    int num_refs;
    char temp[80];
    bool ok;
    int i;

    nwos_find_public_class_definition("US CITY", &city_class_ref);

    if (strlen(state) == 2)
    {
	ok = nwos_find_state_from_postal_code(state, &state_ref);
    }
    else
    {
        ok = nwos_find_state_from_name(state, &state_ref);
    }

    if (!ok)
    {
	printf("I can't find the state of %s\n", state);
	return;
    }

    nwos_read_object_headers_from_disk(&state_ref, &state_header);

    ref_list = nwos_malloc_reference_list(&state_header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("state num refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, &city_class_ref))    /* add it to the list */
	{
	    /* printf("found a city: %08lx\n", nwos_ref_to_word(&ref_list->references[i])); */
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &city_obj, sizeof(city_obj)));
	    nwos_name_to_string(&city_obj.name, temp, sizeof(temp));
	    /* printf("name: %08lx - %s\n", nwos_ref_to_word(&city_obj.name), temp); */
	    printf("%s\n", temp);
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;
}




