/*
--             This file is part of the New World OS project
--                 Copyright (C) 2005-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: computer.c,v $
-- Revision 1.34  2008/09/01 03:13:41  jsedwards
-- Change for new nwos_initialize_objectify calling convention (doesn't pass
-- back root_object_reference anymore) and removed call to nwos_set_root_object
-- because initialize calls it now.
--
-- Revision 1.33  2008/08/30 12:46:07  jsedwards
-- Removed code and variables to read pass phrase and pass it to initialize,
-- and change parameters passed to initialize.
--
-- Revision 1.32  2008/02/03 01:00:16  jsedwards
-- Changed DEFAULT_TYPE_RW to READ_WRITE.
--
-- Revision 1.31  2007/10/07 03:51:35  jsedwards
-- Added new function 'space_available' and don't execute command if it
-- returns false.
--
-- Revision 1.30  2007/09/02 19:45:31  jsedwards
-- Added calls to set the block estimate.
--
-- Revision 1.29  2007/08/12 20:45:00  jsedwards
-- Change all of the "Encryption Level" stuff to "Security Level" because it
-- doesn't really change the encryption at all, all it does is change the
-- randomization of where objects are stored.
--
-- Revision 1.28  2007/07/12 03:12:42  jsedwards
-- Add a command to add a birthday to a person.
--
-- Revision 1.27  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.26  2007/05/24 13:27:13  jsedwards
-- Added list all manufacturers command.
--
-- Revision 1.25  2007/05/06 17:24:34  jsedwards
-- Add 'add account' and 'list accounts' commands.
--
-- Revision 1.24  2007/04/22 15:17:44  jsedwards
-- Added "list all credit unions" command.
--
-- Revision 1.23  2007/03/17 11:43:30  jsedwards
-- Added code to set the encryption level to extreme.
--
-- Revision 1.22  2007/02/11 16:58:26  jsedwards
-- Changed so DEFAULT_TYPE has to specify RO (Read-Only) or RW (Read-Write).
--
-- Revision 1.21  2007/02/11 16:20:27  jsedwards
-- Changed calls to "strncat" to "strlcat" because it makes more sense.
--
-- Revision 1.20  2007/01/06 20:20:36  jsedwards
-- Changed to call both find_private_name and find_public_name instead of
-- find_name (which no longer exists).
--
-- Revision 1.19  2006/12/15 11:42:11  jsedwards
-- Removed kludge added in previous version, this version should be the same
-- as r1.17.
--
-- Revision 1.18  2006/12/15 11:29:40  jsedwards
-- Kludged to read root object from private area.
--
-- Revision 1.17  2006/11/30 00:51:12  jsedwards
-- Changed to use strlcpy instead of strcpy to stop warnings from linker.
--
-- Revision 1.16  2006/11/11 12:01:01  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.15  2006/10/26 01:51:26  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.14.2.3  2006/10/03 12:53:07  jsedwards
-- Changed so that instead of calling a separate routine after initialize to
-- change the already opened storage, you call it now with a type of storage
-- parameter and a path to the storage.  The problem with the other way was
-- if you tried reading a compressed file on another machine it tried to open
-- the default file which didn't exist.
--
-- Revision 1.14.2.2  2006/09/29 12:09:24  jsedwards
-- Added "list_persons" command.
--
-- Revision 1.14.2.1  2006/09/02 15:09:41  jsedwards
-- Add call to terminate objectify so reference list cache gets flushed back
-- to disk, etc.
--
-- Revision 1.14  2006/01/12 03:42:37  jsedwards
-- Changed the "what is number" routine to accept "phone number", "mobile
-- number", and "cell number" as well.
--
-- Revision 1.13  2006/01/12 03:35:22  jsedwards
-- Added help for "what is so and so's number".
--
-- Revision 1.12  2006/01/10 13:57:20  jsedwards
-- Added a phone number query command.
--
-- Revision 1.11  2006/01/09 03:27:13  jsedwards
-- Changed "describe" command to deal with persons or states.
--
-- Revision 1.10  2006/01/08 15:25:26  jsedwards
-- Added some more information to the help print out.
--
-- Revision 1.9  2006/01/08 15:12:46  jsedwards
-- Added "help" information.
--
-- Revision 1.8  2006/01/04 19:00:08  jsedwards
-- Made a separate subroutine to execute commands, instead of doing it in main
-- routine.  Changed main routine to check for parameters passed on command
-- line and reassemble them into a command string, which is passed to the
-- execute command subroutine.  This way a command can be passed to it via
-- the command line.
--
-- Revision 1.7  2006/01/04 15:15:41  jsedwards
-- Added "add city" and "list cities in state" commands.
--
-- Revision 1.6  2006/01/02 01:57:52  jsedwards
-- Fixed bug in get_input routine that was using a fixed size for the buffer
-- instead of the size parameter.
--
-- Revision 1.5  2006/01/01 21:19:30  jsedwards
-- Removed debugging print statement.
--
-- Revision 1.4  2006/01/01 19:50:50  jsedwards
-- Moved all of the tests for quitting (stop, exit, etc.) into a function
-- (is quit command) in objectify.c.
-- Added code to add mobile phone number to database.
--
-- Revision 1.3  2006/01/01 00:45:42  jsedwards
-- Added a command loop where it asks what to do and then tries to interpret
-- your request.
--
-- Revision 1.2  2005/12/31 19:03:33  jsedwards
-- Moved code to create a person object to the "person.c" file.
--
-- Revision 1.1  2005/12/31 14:49:08  jsedwards
-- Copied "create_person.c" into "computer.c" and "person.c" and then removed
-- "create_person.c".
--
-- Revision 1.15  2005/12/30 14:13:39  jsedwards
-- Removed Social Security Number stuff, not ready yet.
--
-- Revision 1.14  2005/12/30 05:12:57  jsedwards
-- Tweaked the birth date stuff and added social security number.
--
-- Revision 1.13  2005/12/29 18:40:38  jsedwards
-- Added code for the new long key and encryption stuff.
--
-- Revision 1.12  2005/12/27 19:52:21  jsedwards
-- Added code to generate the root object from a password.  Changed to find
-- class definition instead of hardcoded file name and to generate a random
-- person object id instead of generating it based on contents.
--
-- Revision 1.11  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.10  2005/12/21 23:28:15  jsedwards
-- Fixed so that it adds the references to the name and date objects.
--
-- Revision 1.9  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.8  2005/12/05 19:06:31  jsedwards
-- Moved calculation of header checksum down just before writing object to
-- disk to ensure no changes were made after the checksum was calculated.
-- Also added the "nickname" field.
--
-- Revision 1.7  2005/12/04 14:14:46  jsedwards
-- Changed to actually get input and create a person (finally)!
--
-- Revision 1.6  2005/12/04 04:13:02  jsedwards
-- Added 'nwos' prefix to create_xxxx function names and eliminated the
-- 'referring' object parameter from all of them.
--
-- Revision 1.5  2005/12/04 00:35:49  jsedwards
-- Changed to just create a date.
--
-- Revision 1.4  2005/12/04 00:23:46  jsedwards
-- Removed create year, date, and name stuff and put it in the date.c and
-- the name.c files.
--
-- Revision 1.3  2005/12/02 20:33:11  jsedwards
-- Changed so that name objects have "NAME" in the filename and added new
-- parameter to the create_reference_list calls.
--
-- Revision 1.2  2005/12/02 19:30:32  jsedwards
-- Changed to create Year object filename from the year itself.
--
-- Revision 1.1  2005/12/02 13:05:17  jsedwards
-- Initial version that only creates a name and a year.
--
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify.h"
#include "objectify_private.h"
#include "crc32.h"

#define MAX_NAME_SIZE 64
#define MAX_DATE_SIZE 16


static void get_input(char* descr, char* buffer, size_t size)
{
    char *ptr;

    while (1)
    {
	printf("%s: ", descr);
	fflush(stdout);
	fgets(buffer, size, stdin);
	ptr = strchr(buffer, '\n');
	if (ptr != NULL)
	{
	    *ptr = '\0';
	    break;
	}
	do { fgets(buffer, size, stdin); } while (strchr(buffer, '\n') == NULL);
	printf("input too long - try again!\n");
    }
}


static bool is_what_is_number(char* command)
{
    static char* what_is = "what is ";
    char* ptr;
    bool result = false;

    if (strncasecmp(command, what_is, strlen(what_is)) == 0)
    {
	ptr = strchr(command, '\'');
	if (ptr != NULL)
	{
	    ptr++;
	    if (*ptr == 's') ptr++;
	    result = (strcasecmp(ptr, " number") == 0 || strcasecmp(ptr, " phone number") == 0
		      || strcasecmp(ptr, " cell number") == 0 || strcasecmp(ptr, " mobile number") == 0);
	}
    }

    return result;
}


static bool space_available(uint32 estimated_blocks)
{
    if (!nwos_check_blocks_available(estimated_blocks))
    {
	printf("Command not completed\n");

	return false;
    }

    return true;
}


void execute_command(char *command)
{
    char* ptr1;
    char* ptr2;
    char* name;
    ObjRef ref;

    if (strcasecmp(command, "add person") == 0)
    {
	if (space_available(36))
	{
	    nwos_add_person();
	}
    }
    else if (strcasecmp(command, "list persons") == 0)
    {
	nwos_list_persons();
    }
    else if (strcasecmp(command, "list all credit unions") == 0)
    {
	nwos_list_public_credit_unions();
    }
    else if (strcasecmp(command, "list all manufacturers") == 0)
    {
	nwos_list_public_manufacturers();
    }
    else if (strcasecmp(command, "add account") == 0)
    {
	if (space_available(12))
	{
	    nwos_add_account();
	}
    }
    else if (strcasecmp(command, "list accounts") == 0)
    {
	nwos_list_accounts();
    }
    else if (strncasecmp(command, "describe ", strlen("describe ")) == 0 && strlen(command) > strlen("describe "))
    {
	/* copy just the name part into name buffer */
	name = malloc(strlen(command));   /* allocate enough room for entire command even though we won't use it all */
	assert(strlcpy(name, command + strlen("describe "), strlen(command)) < strlen(command));

	if (!nwos_find_private_name(name, &ref) && !nwos_find_public_name(name, &ref))
	{
	    printf("I don't know of anyone or anything by the name of: %s\n", name);
	}
	else
	{
	    if (nwos_any_persons_named(name))
	    {
		nwos_describe_person(name);
	    }

	    if (nwos_any_states_named(name))
	    {
		nwos_describe_state(name);
	    }
	}
    }
    else if (strncasecmp(command, "list cities in ", strlen("list cities in ")) == 0 && strlen(command) > strlen("list cities in "))
    {
	nwos_list_cities_in_state(command + strlen("list cities in "));
    }
    else if (strcasecmp(command, "add city") == 0)
    {
	if (space_available(8))
	{
	    ObjRef city_ref;

	    nwos_add_city(&city_ref);
	}
    }
    else if (strstr(command, " birthday is ") != NULL)
    {
	ptr1 = strstr(command, " birthday is ");
	ptr2 = ptr1 + strlen(" birthday is ");      /* point to the start of the phone number */
	if (ptr1[-2] == '\'' && ptr1[-1] == 's')
	{
	    ptr1 -= 2;
	}
	else if (ptr1[-1] == '\'')
	{
	    ptr1 -= 1;
	}
	*ptr1 = '\0';    /* terminate the name here */

	if (space_available(4))
	{
	    nwos_add_birthday(command, ptr2);
	}
    }
    else if (strstr(command, " mobile phone is ") != NULL)
    {
	ptr1 = strstr(command, " mobile phone is ");
	ptr2 = ptr1 + strlen(" mobile phone is ");      /* point to the start of the phone number */
	if (ptr1[-2] == '\'' && ptr1[-1] == 's')
	{
	    ptr1 -= 2;
	}
	else if (ptr1[-1] == '\'')
	{
	    ptr1 -= 1;
	}
	*ptr1 = '\0';    /* terminate the name here */

	if (space_available(4))
	{
	    nwos_add_mobile_phone(command, ptr2);
	}
    }
    else if (is_what_is_number(command))
    {
	*strchr(command, '\'') = '\0';
	nwos_find_phone_for_person(command + strlen("what is "));
    }
    else
    {
	printf("I'm sorry I don't understand\n");
    }
}


int main(int argc, char* argv[])
{
    char command[80];
    int i;

    printf("\n");

    nwos_initialize_objectify(READ_WRITE, DEFAULT_FILE);

    nwos_set_security_level(Security_Extreme);

    if (argc == 1)   /* nothing on the command line */
    {
	while (1)
	{
	    printf("\n");

	    get_input("How may I help you?", command, sizeof(command));

	    printf("\n");

	    if (nwos_is_quit_command(command)) break;

	    if (strcasecmp(command, "help") == 0 || strcasecmp(command, "commands") == 0 || strcasecmp(command, "?") == 0)
	    {
		printf("The only commands that are available in this version are (without quotes):\n\n");

		printf("   \"add person\" - adds a new person to the database.\n");
		printf("   \"list persons\" - list persons to the database.\n");
		printf("   \"add account\" - adds a new checking account to the database.\n");
		printf("   \"list accounts\" - list checking accounts in the database.\n");
		printf("   \"add city\" - adds a new city to the database (not of much use in this version).\n");
		printf("   \"describe <name>\" - describes any person(s) with the name <name> in the database.\n");
		printf("   \"list cities in <state>\" - lists the cities in the database for state <state>.\n");
		printf("   \"<name>'s birthday is <date>\" - adds the birthday <date> to the person <name>.\n");
		printf("   \"<name>'s mobile phone is <number>\" - adds the phone number <number> to the person <name>.\n");
		printf("   \"what is <name>'s number\" - prints out the phone number for the person <name>.\n");
		printf("   \"list all credit unions\" - list all credit unions in the database.\n");
		printf("   \"list all manufacturers\" - list all manufacturers in the database.\n");

		printf("\n   \"commands\" or \"help\" - print this information.\n");
		printf("   \"done\", \"exit\", \"finish\", \"quit\", \"stop\", or \"terminate\" - exit the program.\n");

		printf("\nPlease note that this version is picky about the spacing, you need to have exactly one space\n");
		printf("as shown in the commands above.\n");
	    }
	    else
	    {
		execute_command(command);
	    }
	}

	printf("Goodbye!\n\n");
    }
    else
    {
	command[0] = '\0';
	for (i = 1; i < argc; i++)
	{
	    if (strlen(command) + strlen(argv[i]) + 1 > sizeof(command))
	    {
		fprintf(stderr, "The word %s exceeds size of input buffer\n", argv[i]);
		exit(1);
	    }
	    if (command[0] != '\0')     /* there are already words in command so add a space in front of this word */
	    {
		strlcat(command, " ", sizeof(command));
	    }
	    strlcat(command, argv[i], sizeof(command));
	}

	execute_command(command);
    }

    nwos_terminate_objectify();

    return 0;
}

