/*
--             This file is part of the New World OS project
--                 Copyright (C) 2006-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- NOTE: This app should only be run one time by me to convert the Alpha_05
--       objects to the new format of objects.  The plan after this is that
--       those objects become the "official" objects and everyone just uses
--       them.  This eliminates the whole chicken and egg problem.
--
-- CHANGES TO OBJECT FORMAT:
--
-- Previously objects had a header checksum and a data checksum.  That has been
-- replaced with a GPG signature.
--
-- $Log: convert_to_new_format.c,v $
-- Revision 1.4  2008/09/01 00:22:58  jsedwards
-- Fix year in copyright.  NO code changes.
--
-- Revision 1.3  2008/08/31 16:42:01  jsedwards
-- Added assert around calls to nwos_read_object_from_disk because now it
-- returns false if it fails instead of asserting itself.
--
-- Revision 1.2  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.1  2007/07/01 14:16:04  jsedwards
-- Move to attic.
--
-- Revision 1.3  2006/12/01 14:31:30  jsedwards
-- Changed to use new malloc_reference_list and free_reference_list functions
-- instead of inlining the code.
--
-- Revision 1.2  2006/11/11 12:01:02  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.1  2006/01/16 03:17:36  jsedwards
-- One time program to convert Alpha_04/Alpha_05 objects to the new format.
-- This program should not ever be needed again after that.
--
*/

#include <assert.h>
#include <stdio.h>
#include <string.h>   /* define memset */
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "objectify.h"
#include "objectify_private.h"
#include "time_stamp.h"
#include "crc32.h"


#define PUBLIC_DIR_1  "/obj/00000000/"
#define PUBLIC_DIR_2  "/obj/00000000/00000000/"
#define PUBLIC_DIR_3  "/obj/00000000/00000000/00000000/"
#define PRIVATE_DIR_1 "/obj/ffffffff/"
#define PRIVATE_DIR_2 "/obj/ffffffff/ffffffff/"
#define PRIVATE_DIR_3 "/obj/ffffffff/ffffffff/ffffffff/"

#define SIZE_OBJ_REF 16

typedef uint8 NewObjRef[SIZE_OBJ_REF];


NewObjRef Void_Ref        = { 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0,  0 };   /* all zeros is the void reference */
NewObjRef Root_Ref        = { 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0,  1 };   /* one is the root of all (evil?) */
NewObjRef Root_Class_Ref  = { 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0,  2 };   /* the root class definition */
NewObjRef Class_Def_Ref   = { 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0,  3 };   /* the class definition class */
NewObjRef Class_Name_Ref  = { 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0,  4 };   /* name of the class */
NewObjRef Class_Feat_Ref  = { 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0,  5 };   /* the list of features */
NewObjRef Class_List_Ref  = { 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0,  6 };   /* the list of classes */
NewObjRef Any_Class_Ref   = { 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0,  7 };   /* the any class definition */
NewObjRef Array_Class_Ref = { 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0,  8 };   /* the array class definition */
NewObjRef User_Ref        = { 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0,  9 };   /* user (me) that created the objects */
NewObjRef Big_Bang_4_Ref  = { 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 10 };   /* big_bang app version alpha 04 */
NewObjRef Big_Bang_5_Ref  = { 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 11 };   /* big_bang app version alpha 05 */

NewObjRef Next_Ref        = { 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 11 };


typedef struct {
    uint8     magic_number[4];            /*   4 */  /* nwos */
    uint8     version[4];                 /*   8 */  /* version of header 1-9999 */
    TimeStamp creation_time;              /*  16 */
    NewObjRef class_definition;           /*  32 */
    NewObjRef created_by_user;            /*  48 */
    NewObjRef created_by_app;             /*  64 */
    NewObjRef old_version;                /*  80 */
    NewObjRef new_version;                /*  96 */
} AnyObject;


typedef struct {
    AnyObject any;
    NewObjRef class_definition_class;
} New_Root;


typedef struct {
    AnyObject any;
    NewObjRef class_name;
    NewObjRef feature_list;             /* inheritance is in here too */
    NewObjRef derived_list;             /* classes that derive from this class */
    NewObjRef object_list;              /* objects of this type */
} New_Class_Definition;

/* Need to have different class definitions for expanded and deferred classes? */


typedef struct {
    AnyObject header;
    NewObjRef inherited;
    NewObjRef label;
    NewObjRef class;
} C_struct_Data_Definition;



typedef struct {
    AnyObject any;
    NewObjRef class_of_objects;         /* the class of objects stored in this array */
    uint8     size_of_count;            /* number of bytes in the count */
    uint8     count[15];                /* the number of objects stored (most significant byte first) */
    uint8     storage[0];
} New_Array;




void increment_next_reference()   /* bump the counter */
{
    int i;

    i = SIZE_OBJ_REF;

    do
    {
	i = i - 1;
	assert(i >= 0);
	Next_Ref[i] = Next_Ref[i] + 1;
    }
    while (Next_Ref[i] == 0);   /* if it rolled over to zero, go back, Jack, and increment the next byte */
}


typedef struct 
{
  ObjRef old;
  uint8  new[4];     /* only 4 bytes of new stored */
} XlatRef;

XlatRef xlat_table[4096];   /* table to translate all of the old references to the new references */

void add_xlat(ObjRef *old, NewObjRef new)
{
    int i;

    for (i = 0; i < 4096; i++)
    {
	if (is_void_reference(&xlat_table[i].old)) break;  /* found empty slot */
	assert(!is_same_object(&xlat_table[i].old, old));
    }

    assert(i < 4096);

    memcpy(&xlat_table[i].old, old, 4);
    memcpy(&xlat_table[i].new, new+12, 4);

    assert(new[12] == xlat_table[i].new[0]);
    assert(new[13] == xlat_table[i].new[1]);
    assert(new[14] == xlat_table[i].new[2]);
    assert(new[15] == xlat_table[i].new[3]);

    assert(old->id[0] == xlat_table[i].old.id[0]);
    assert(old->id[1] == xlat_table[i].old.id[1]);
    assert(old->id[2] == xlat_table[i].old.id[2]);
    assert(old->id[3] == xlat_table[i].old.id[3]);
}


void add_xlat_private(ObjRef *old)
{
    int i;

    for (i = 0; i < 4096; i++)
    {
       if (is_void_reference(&xlat_table[i].old)) break;  /* found empty slot */
       assert(!is_same_object(&xlat_table[i].old, old));
    }

    assert(i < 4096);

    memcpy(&xlat_table[i].old, old, 4);
    xlat_table[i].new[0] = 0xff;
    xlat_table[i].new[1] = 0xff;
    xlat_table[i].new[2] = 0xff;
    xlat_table[i].new[3] = 0xff;
}


int find_xlat_old(ObjRef *old)
{
    int i;

    for (i = 0; i < 4096; i++)
    {
	if (is_void_reference(&xlat_table[i].old)) return -1;  /* found empty slot */
	if (is_same_object(&xlat_table[i].old, old)) break;
    }

    assert(i < 4096);

    return i;
}


int find_xlat_new(NewObjRef new)
{
    int i;

    for (i = 0; i < 4096; i++)
    {
	if (is_void_reference(&xlat_table[i].old)) return -1;  /* found empty slot */
	if (memcmp(xlat_table[i].new, &new[12], 4) == 0) break;
    }

    assert(i < 4096);

    return i;
}


void write_public_object_to_disk(NewObjRef ref, void* object, size_t size)
{
    int i;
    char filename[64];
    FILE* fp;

    for (i = 0; i < 12; i++) assert(ref[i] == 0);

    sprintf(filename, "%s%02x%02x%02x%02x", PUBLIC_DIR_3, ref[12], ref[13], ref[14], ref[15]);

    fp = fopen(filename, "w");

    if (fp == NULL)
    {
	perror(filename);
    }

    if (fwrite(object, 1, size, fp) != size)
    {
	perror(filename);
    }

    if (fclose(fp) != 0)
    {
	perror(filename);
    }
}


#if 0
void convert_class_def_class_reference_list_to_array(ObjRef* ref_list_ref, NewObjRef* array_ref)
{
    C_struct_class_definition class_def_obj;
    ReferenceList* ref_list;
    New_Array* array_obj;
    size_t array_size;
    int num_classes;
    char buffer[128];
    int i;
    bool result = false;

    printf("Converting reference list for class def object: %08lx\n", nwos_ref_to_word(ref_list_ref));

    assert(nwos_read_object_from_disk(ref_list_ref, &class_def_obj, sizeof(class_def_obj)));

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_classes = ref_list->common_header.num_refs;

    printf("number of objects of this class: %d\n", num_classes);

    array_size = sizeof(New_Array) + sizeof(NewObjRef) * num_classes;

    array_obj = malloc(array_size);

    if (array_obj == NULL) 
    {
	perror("convert_class_def_reference_list, reading reference list");
	exit(1);
    }

    for (i = 0; i < num_classes; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	assert(is_same_object(&object_class, &nwos_class_definition_class_ref));   /* it is the correct kind of object */

    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    free(array_obj);
    array_obj = NULL;

    nwos_remove_object(ref_list_ref);

    return result;
}
#endif

void make_new_directories()
{
    printf("first make the new directories...\n");

    /* for now just make this as easy as possible and */
    /* assume all objects will be less than 4GB of both */
    /* public and private. */

    printf("mkdir %s\n", PUBLIC_DIR_1);
    if (mkdir(PUBLIC_DIR_1, 0755) != 0)
    {
	perror(PUBLIC_DIR_1);
	exit(1);
    }

    printf("mkdir %s\n", PUBLIC_DIR_2);
    if (mkdir(PUBLIC_DIR_2, 0755) != 0)
    {
	perror(PUBLIC_DIR_2);
	exit(1);
    }

    printf("mkdir %s\n", PUBLIC_DIR_3);
    if (mkdir(PUBLIC_DIR_3, 0755) != 0)
    {
	perror(PUBLIC_DIR_3);
	exit(1);
    }

    printf("mkdir %s\n", PRIVATE_DIR_1);
    if (mkdir(PRIVATE_DIR_1, 0755) != 0)
    {
	perror(PRIVATE_DIR_1);
	exit(1);
    }

    printf("mkdir %s\n", PRIVATE_DIR_2);
    if (mkdir(PRIVATE_DIR_2, 0755) != 0)
    {
	perror(PRIVATE_DIR_2);
	exit(1);
    }

    printf("mkdir %s\n", PRIVATE_DIR_3);
    if (mkdir(PRIVATE_DIR_3, 0755) != 0)
    {
	perror(PRIVATE_DIR_3);
	exit(1);
    }
}

void convert_root_object(ObjRef* root_obj_ref)
{
    ObjRef class_def;
    C_struct_Root old_root_obj;
    New_Root new_root_obj;

    printf("Converting root object: %08lx...\n", nwos_ref_to_word(root_obj_ref));

    add_xlat(root_obj_ref, Root_Ref);

    assert(nwos_read_object_from_disk(root_obj_ref, &old_root_obj, sizeof(old_root_obj)));

    memcpy(&new_root_obj.any.magic_number, &old_root_obj.header.common.magic_number, 4);

    assert(memcmp(old_root_obj.header.common.version, "0004", 4) == 0
	   || memcmp(old_root_obj.header.common.version, "0005", 4) == 0);

    memcpy(&new_root_obj.any.version, "0006", 4);

    memcpy(&new_root_obj.any.creation_time, &old_root_obj.header.common.creation_time, sizeof(TimeStamp));

    /* sanity check */
    nwos_find_class_definition("ROOT", &class_def);
    assert(is_same_object(&old_root_obj.header.common.class_definition, &class_def));

    add_xlat(&old_root_obj.header.common.class_definition, Root_Class_Ref);

    memcpy(new_root_obj.any.class_definition, Root_Class_Ref, sizeof(NewObjRef));

    assert(is_void_reference(&old_root_obj.header.object.clone_of));

    assert(!is_void_reference(&old_root_obj.header.object.references));

    printf("Deleting reference list for root object: %08lx\n", nwos_ref_to_word(&old_root_obj.header.object.references));

    nwos_remove_object(&old_root_obj.header.object.references);

    assert(is_void_reference(&old_root_obj.header.object.context));
    assert(is_void_reference(&old_root_obj.header.object.created_by_user));
    assert(is_void_reference(&old_root_obj.header.object.created_by_app));
    assert(is_void_reference(&old_root_obj.header.object.prev_version));
    assert(is_void_reference(&old_root_obj.header.object.next_version));

    memcpy(new_root_obj.any.created_by_user, User_Ref, sizeof(NewObjRef));

    if (memcmp(old_root_obj.header.common.version, "0004", 4) == 0)
    {
	memcpy(new_root_obj.any.created_by_app, Big_Bang_4_Ref, sizeof(NewObjRef));
    }
    else
    {
	memcpy(new_root_obj.any.created_by_app, Big_Bang_5_Ref, sizeof(NewObjRef));
    }

    memcpy(new_root_obj.any.old_version, Void_Ref, sizeof(NewObjRef));
    memcpy(new_root_obj.any.new_version, Void_Ref, sizeof(NewObjRef));

    add_xlat(&old_root_obj.class_definition_class, Class_Def_Ref);

    memcpy(&new_root_obj.class_definition_class, &Class_Def_Ref, sizeof(NewObjRef));

    write_public_object_to_disk(Root_Ref, &new_root_obj, sizeof(new_root_obj));

    nwos_remove_object(root_obj_ref);
}


void convert_class_def_object()
{
    C_struct_class_definition old_cdc_obj;
    New_Class_Definition new_cdc_obj;

    int index;

    printf("Converting class definition object: %08lx...\n", nwos_ref_to_word(&nwos_class_definition_class_ref));

    index = find_xlat_new(Class_Def_Ref);

    assert(index >= 0);

    assert(is_same_object(&nwos_class_definition_class_ref, &xlat_table[index].old));

    assert(nwos_read_object_from_disk(&nwos_class_definition_class_ref, &old_cdc_obj, sizeof(old_cdc_obj)));

    memcpy(&new_cdc_obj.any.magic_number, &old_cdc_obj.header.common.magic_number, 4);

    assert(memcmp(old_cdc_obj.header.common.version, "0004", 4) == 0
	   || memcmp(old_cdc_obj.header.common.version, "0005", 4) == 0);

    memcpy(&new_cdc_obj.any.version, "0006", 4);

    memcpy(&new_cdc_obj.any.creation_time, &old_cdc_obj.header.common.creation_time, sizeof(TimeStamp));

    /* sanity check */
    assert(is_same_object(&old_cdc_obj.header.common.class_definition, &nwos_class_definition_class_ref));

    assert(is_void_reference(&old_cdc_obj.header.object.clone_of));

    assert(!is_void_reference(&old_cdc_obj.header.object.references));

#if 0
    /* don't do this yet */
    convert_class_def_reference_list_to_array(&old_cdc_obj.header.object.references, Class_Ref_List);
#endif

    assert(is_void_reference(&old_cdc_obj.header.object.context));
    assert(is_void_reference(&old_cdc_obj.header.object.created_by_user));
    assert(is_void_reference(&old_cdc_obj.header.object.created_by_app));
    assert(is_void_reference(&old_cdc_obj.header.object.prev_version));
    assert(is_void_reference(&old_cdc_obj.header.object.next_version));

    memcpy(new_cdc_obj.any.class_definition, Class_Def_Ref, sizeof(NewObjRef));

    memcpy(new_cdc_obj.any.created_by_user, User_Ref, sizeof(NewObjRef));

    if (memcmp(old_cdc_obj.header.common.version, "0004", 4) == 0)
    {
	memcpy(new_cdc_obj.any.created_by_app, Big_Bang_4_Ref, sizeof(NewObjRef));
    }
    else
    {
	memcpy(new_cdc_obj.any.created_by_app, Big_Bang_5_Ref, sizeof(NewObjRef));
    }

    memcpy(new_cdc_obj.any.old_version, Void_Ref, sizeof(NewObjRef));
    memcpy(new_cdc_obj.any.new_version, Void_Ref, sizeof(NewObjRef));

    add_xlat(&old_cdc_obj.name, Class_Name_Ref);

    memcpy(&new_cdc_obj.class_name,   &Class_Name_Ref, sizeof(NewObjRef));
    memcpy(&new_cdc_obj.feature_list, &Class_Feat_Ref, sizeof(NewObjRef));
    memcpy(&new_cdc_obj.object_list,  &Class_List_Ref, sizeof(NewObjRef));
    memcpy(&new_cdc_obj.derived_list, &Void_Ref, sizeof(NewObjRef));

    write_public_object_to_disk(Class_Def_Ref, &new_cdc_obj, sizeof(new_cdc_obj));

    nwos_remove_object(&nwos_class_definition_class_ref);
}


int main(int argc, char* argv[])
{
    ObjRef root_object_reference;
    ObjRef reference;
    CommonHeader dummy;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;


    /* to avoid any accidents, first check to make sure the old objects are there */

    reference.id[0] = 0x12;
    reference.id[1] = 0x34;
    reference.id[2] = 0x56;
    reference.id[3] = 0x78;

    assert(nwos_object_exists(&reference));

    nwos_read_reference_list_from_disk(&reference, &dummy, sizeof(dummy));

    if (memcmp(&dummy.version, HEADER_VERSION, sizeof(dummy.version)) != 0)
    {
	fprintf(stderr, "Objects are not version 0005!\n");
	exit(1);
    }

    nwos_get_key_from_password(big_key, sizeof(big_key));

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_reference.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_initialize_objectify(bf_key, linear, serial);

    nwos_set_root_object(&root_object_reference);

    /* looks like we are good to go... */

    make_new_directories();

    convert_root_object(&root_object_reference);

    convert_class_def_object();

    return 0;
}

