/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2005, 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  I have retained the CVS log from
--   this file when it was in the root directory.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
--
-- Revision 1.27  2006/11/11 12:01:01  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.26  2006/10/26 01:07:50  jsedwards
-- Replaced main truck version (1.25) with the latest version from the
-- alpha_05_branch.
--
-- Revision 1.16.2.11  2006/10/25 12:22:26  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.16.2.10  2006/10/03 12:53:07  jsedwards
-- Changed so that instead of calling a separate routine after initialize to
-- change the already opened storage, you call it now with a type of storage
-- parameter and a path to the storage.  The problem with the other way was
-- if you tried reading a compressed file on another machine it tried to open
-- the default file which didn't exist.
--
-- Revision 1.16.2.9  2006/09/29 04:21:57  jsedwards
-- Changed to use magic number and version string definitions instead of
-- hard coding the values.
--
-- Revision 1.16.2.8  2006/09/26 13:26:49  jsedwards
-- Commented out code to creat the file, because using a whole disk drive or
-- partition now.
--
-- Revision 1.16.2.7  2006/09/22 02:30:10  jsedwards
-- Add code to make sure open succeeded.
--
-- Revision 1.16.2.6  2006/09/03 13:57:24  jsedwards
-- Added code to create the one file.
--
-- Revision 1.16.2.5  2006/09/03 11:02:15  jsedwards
-- Moved the setting of spelling class and name class references to just after
-- they are generated so that if we are debugging they are known as early as
-- possible.  Also changed so that when rewriting the 4 initial classes,
-- instead of removing and then writing the object, there is a routine that
-- allows an object to be overwritten.
--
-- Revision 1.16.2.4  2006/09/02 15:09:41  jsedwards
-- Add call to terminate objectify so reference list cache gets flushed back
-- to disk, etc.
--
-- Revision 1.16.2.3  2006/09/02 01:13:30  jsedwards
-- Added reference in calls to fill_in_common_header because it puts it in
-- the header now.
--
-- Revision 1.16.2.2  2006/08/19 14:32:59  jsedwards
-- Moved creation of MD5sum class to file_setup routine (in file.c) and
-- replaced it with a call to the file_setup routine.
--
-- Revision 1.16.2.1  2006/08/18 12:53:48  jsedwards
-- Changed hard coded 512 for block size to FILE_BLOCK_SIZE.
--
-- Revision 1.16  2006/01/12 03:00:23  jsedwards
-- Added code to ask if he/she wants to upgrade objects from 0004.
-- Added call to upgrade spelling objects from 0004 to 0005.
--
-- Revision 1.15  2006/01/09 03:21:00  jsedwards
-- Added more print statements to describe what was happening.
-- Added a call to upgrade persons from 0004 to 0005 version.
--
-- Revision 1.14  2006/01/06 14:15:58  jsedwards
-- Changed to handle upgrading from alpha_04 (0004) to alpha_05 (0005).
--
-- Revision 1.13  2006/01/04 15:13:57  jsedwards
-- Added call to setup address classes.
--
-- Revision 1.12  2006/01/03 03:24:44  jsedwards
-- Updated copyright year in version information printed on start up.
--
-- Revision 1.11  2006/01/01 21:49:01  jsedwards
-- Moved date, phone, us_state, and word class creations out of "big bang"
-- and into the respective files.
--
-- Revision 1.10  2005/12/31 17:53:43  jsedwards
-- Moved class creation routine to a separate file: "class_definition.c".
-- Moved creation of "person" related classes to "person.c".
--
-- Revision 1.9  2005/12/31 15:40:58  jsedwards
-- Added creation of "social security number" class.
--
-- Revision 1.8  2005/12/31 03:07:16  jsedwards
-- Changed wording in disclaimer.
--
-- Revision 1.7  2005/12/30 17:11:46  jsedwards
-- Added disclaimers about no warranty and security issues.
--
-- Revision 1.6  2005/12/29 18:15:18  jsedwards
-- Added code to read back objects after they are written to verify that the
-- write worked correctly.
--
-- Revision 1.5  2005/12/29 18:06:57  jsedwards
-- Added print statements to print the object ID's of class defintions and
-- reference lists.
--
-- Revision 1.4  2005/12/29 18:02:37  jsedwards
-- Changed to call create_reference_list for class_definition_class object
-- instead of rolling our own.  Rolling our own was is bad when regular
-- objects are encrypted and reference lists are not because if we call the
-- write object to disk routine it encrypts the reference list.  Also changed
-- it so that the dummy object has a whole CommonHeader.
--
-- Revision 1.3  2005/12/28 13:06:29  jsedwards
-- Changed to use the new variable length key from password stuff and call the
-- new initialize_objectify routine with the blowfish key and seeds for the
-- sequence generator.
--
-- Revision 1.2  2005/12/27 19:49:36  jsedwards
-- Added code to create the root class and the root object.
--
-- Revision 1.1  2005/12/27 18:45:31  jsedwards
-- Renamed file from create_class_def.c to big_bang.c.  Changed all of the
-- class definitions from hardcoded file names to random ids.  Had to modify
-- to deal with chicken and egg problems.
--
-- Revision 1.20  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.19  2005/12/21 17:10:31  jsedwards
-- Add 'nwos' to 'create_all_area_codes' routine name.
--
-- Revision 1.18  2005/12/21 17:06:48  jsedwards
-- Add call to create the US area codes.
--
-- Revision 1.17  2005/12/21 03:55:48  jsedwards
-- Added creation of abbreviation and US states classes.
--
-- Revision 1.16  2005/12/21 03:52:27  jsedwards
-- Moved US state objects creation from 'create_class_def.c' into 'date.c'.
--
-- Revision 1.15  2005/12/11 17:07:59  jsedwards
-- Added creation of "Area Code" class and the "Home", "Work", and "Mobile"
-- phone classes.
--
-- Revision 1.14  2005/12/11 17:01:37  jsedwards
-- Moved create english language function to the language.c file.  Rearranged
-- the order of the class definitions so that spelling and word classes are
-- defined before the english language definitions, otherwise when the create
-- english language function tries to create the word "English" it fails.
-- Also added code when the create english language function is not called
-- because it already exists to search for the english language object so
-- that other words we create will be able to access it.
--
-- Revision 1.13  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.12  2005/12/08 18:05:04  jsedwards
-- Added creation of Phone Number class definition.
--
-- Revision 1.11  2005/12/05 19:04:51  jsedwards
-- Changed month structure member to the correct one for computing the data
-- checksum.
--
-- Revision 1.10  2005/12/05 05:28:02  jsedwards
-- Moved calculation of headers down to just before object is written to make
-- sure no changes occur after the checksums have been calculated.
--
-- Revision 1.9  2005/12/04 14:11:27  jsedwards
-- Added creation of the Person class.
--
-- Revision 1.8  2005/12/04 04:13:01  jsedwards
-- Added 'nwos' prefix to create_xxxx function names and eliminated the
-- 'referring' object parameter from all of them.
--
-- Revision 1.7  2005/12/03 22:09:37  jsedwards
-- Updated for new style of words and added spelling and language classes.
--
-- Revision 1.6  2005/12/03 02:02:28  jsedwards
-- Added routine to create the month objects (all 12).
--
-- Revision 1.5  2005/12/02 20:31:49  jsedwards
-- Added new parameter to the calls to create_reference_list.
--
-- Revision 1.4  2005/11/30 14:00:53  jsedwards
-- Added creation of date related classes.
--
-- Revision 1.3  2005/11/30 13:49:49  jsedwards
-- Added 'Name' class and made create class definition routine generic.
--
-- Revision 1.2  2005/11/26 15:21:12  jsedwards
-- Modified to use newer data layout and routines.
--
-- Revision 1.1  2005/11/25 13:33:48  jsedwards
-- Copied from 'lab'.
--
-- Revision 1.2  2005/11/24 16:10:13  jsedwards
-- This is broken, needs to be fixed.
--
-- Revision 1.1  2005/11/13 00:10:11  jsedwards
-- Program to create initial class definitions.
--
*/

#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>   /* define memset */
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "../objectify_private.h"
#include "../time_stamp.h"
#include "../crc32.h"



/*******************************************************************************/
/* English Language                                                            */
/*******************************************************************************/

void nwos_create_english_language(ObjRef* ref)
{
    ObjRef lang_class_ref;
    C_struct_Language lang_obj;
    ObjRef word_ref;

    nwos_log("Creating English language");

    memset(&lang_obj, 0, sizeof(lang_obj));  /* zero it out */

    assert(nwos_find_public_class_definition("LANGUAGE", &lang_class_ref));

    nwos_generate_new_public_id(ref);

    nwos_fill_in_common_header(&lang_obj.header.common, ref, &lang_class_ref);

    nwos_create_reference_list(ref, &lang_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &lang_obj.header.object, sizeof(ObjectHeader), lang_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &lang_obj.definition, sizeof(lang_obj) - sizeof(EveryObject), lang_obj.header.common.data_chksum);

    nwos_write_object_to_disk(ref, &lang_obj, sizeof(lang_obj));

    nwos_add_to_references(ref, &lang_class_ref);

    nwos_create_word("English", ref, ref, &word_ref);
}


/*******************************************************************************/
/* Numbers                                                                     */
/*******************************************************************************/

static void create_cardinal_and_ordinal(char* number, ObjRef* english_lang_ref, char* cardinal, char* ordinal)
{
    ObjRef cardinal_ref;
    ObjRef ordinal_ref;
    ObjRef word_ref;
    char msg[128];

    snprintf(msg, sizeof(msg), "Creating cardinal and ordinal numbers: %s, %s, %s", number, cardinal, ordinal);
    nwos_log(msg);

    nwos_create_cardinal_number(number, &cardinal_ref);
    nwos_create_word(cardinal, &cardinal_ref, english_lang_ref, &word_ref);

    nwos_create_ordinal_number(&cardinal_ref, &ordinal_ref);
    nwos_create_word(ordinal, &ordinal_ref, english_lang_ref, &word_ref);
}


static void create_large_cardinal(int number_of_zeros, ObjRef* english_lang_ref, char* cardinal)
{
    ObjRef cardinal_ref;
    ObjRef word_ref;
    char number[65];   /* leave room for the 1 and the null terminator */
    int i;
    char msg[128];

    assert(number_of_zeros < 64);

    number[0] = '1';
    for (i = 0; i < number_of_zeros; i++) number[i+1] = '0';
    number[i] = '\0';

    snprintf(msg, sizeof(msg), "Creating large cardinal number: %s, %s", number, cardinal);
    nwos_log(msg);

    nwos_create_cardinal_number(number, &cardinal_ref);
    nwos_create_word(cardinal, &cardinal_ref, english_lang_ref, &word_ref);
}


static void create_numbers(ObjRef* english_lang_ref)
{
    create_cardinal_and_ordinal("0", english_lang_ref, "zero",  "zeroth");
    create_cardinal_and_ordinal("1", english_lang_ref, "one",   "first");
    create_cardinal_and_ordinal("2", english_lang_ref, "two",   "second");
    create_cardinal_and_ordinal("3", english_lang_ref, "three", "third");
    create_cardinal_and_ordinal("4", english_lang_ref, "four",  "fourth");
    create_cardinal_and_ordinal("5", english_lang_ref, "five",  "fifth");
    create_cardinal_and_ordinal("6", english_lang_ref, "six",   "sixth");
    create_cardinal_and_ordinal("7", english_lang_ref, "seven", "seventh");
    create_cardinal_and_ordinal("8", english_lang_ref, "eight", "eighth");
    create_cardinal_and_ordinal("9", english_lang_ref, "nine",  "ninth");

    create_cardinal_and_ordinal("10", english_lang_ref, "ten",       "tenth");
    create_cardinal_and_ordinal("11", english_lang_ref, "eleven",    "eleventh");
    create_cardinal_and_ordinal("12", english_lang_ref, "twelve",    "twelfth");
    create_cardinal_and_ordinal("13", english_lang_ref, "thirteen",  "thirteenth");
    create_cardinal_and_ordinal("14", english_lang_ref, "fourteen",  "fourteenth");
    create_cardinal_and_ordinal("15", english_lang_ref, "fifteen",   "fifteenth");
    create_cardinal_and_ordinal("16", english_lang_ref, "sixteen",   "sixteenth");
    create_cardinal_and_ordinal("17", english_lang_ref, "seventeen", "seventeenth");
    create_cardinal_and_ordinal("18", english_lang_ref, "eighteen",  "eighteenth");
    create_cardinal_and_ordinal("19", english_lang_ref, "nineteen",  "nineteenth");

    create_cardinal_and_ordinal("20", english_lang_ref, "twenty",  "twentieth");
    create_cardinal_and_ordinal("30", english_lang_ref, "thirty",  "thirtieth");
    create_cardinal_and_ordinal("40", english_lang_ref, "fourty",  "fourtieth");
    create_cardinal_and_ordinal("50", english_lang_ref, "fifty",   "fiftieth");
    create_cardinal_and_ordinal("60", english_lang_ref, "sixty",   "sixtieth");
    create_cardinal_and_ordinal("70", english_lang_ref, "seventy", "seventieth");
    create_cardinal_and_ordinal("80", english_lang_ref, "eighty",  "eightieth");
    create_cardinal_and_ordinal("90", english_lang_ref, "ninety",  "ninetieth");

    create_cardinal_and_ordinal("100",           english_lang_ref, "hundred",  "hundredth");
    create_cardinal_and_ordinal("1000",          english_lang_ref, "thousand", "thousandth");
    create_cardinal_and_ordinal("1000000",       english_lang_ref, "million",  "millionth");
    create_cardinal_and_ordinal("1000000000",    english_lang_ref, "billion",  "billionth");
    create_cardinal_and_ordinal("1000000000000", english_lang_ref, "trillion", "trillionth");

    /* now this is a mess, for now just going with the American system, this needs to be fixed at some point! */

    create_large_cardinal(15, english_lang_ref, "quadrillion");
    create_large_cardinal(18, english_lang_ref, "quintillion");
    create_large_cardinal(21, english_lang_ref, "sextillion");
    create_large_cardinal(24, english_lang_ref, "septillion");
    create_large_cardinal(27, english_lang_ref, "octillion");
    create_large_cardinal(30, english_lang_ref, "nonillion");
    create_large_cardinal(33, english_lang_ref, "decillion");
    create_large_cardinal(36, english_lang_ref, "undecillion");
    create_large_cardinal(39, english_lang_ref, "duodecillion");
    create_large_cardinal(42, english_lang_ref, "tredecillion");
    create_large_cardinal(45, english_lang_ref, "quattuordecillion");
    create_large_cardinal(48, english_lang_ref, "quindecillion");
    create_large_cardinal(51, english_lang_ref, "secdecillion");
    create_large_cardinal(54, english_lang_ref, "septendecillion");
    create_large_cardinal(57, english_lang_ref, "octodecillion");
    create_large_cardinal(60, english_lang_ref, "novemdecillion");
    create_large_cardinal(63, english_lang_ref, "vigintillion");
}


/*******************************************************************************/
/* Create all the years from 1582 to 2020                                      */
/*******************************************************************************/

static void create_years()
{
    uint16 year;
    ObjRef ref;
    char msg[128];

    for (year = 1582; year <= 2020; year++)
    {
	snprintf(msg, sizeof(msg), "Creating year: %u", year);
	nwos_log(msg);

	nwos_create_year(year, &ref);
    }
}


/*******************************************************************************/
/* Months                                                                      */
/*******************************************************************************/

static void create_month_and_day(ObjRef* month_ref, uint8 dom, uint16 year_offset, uint8 leap, ObjRef* ref)
{
    C_struct_Month_And_Day month_and_day_obj;
    ObjRef class_ref;
    ObjRef cardinal_ref;
    ObjRef day_of_month_ref;
    ObjRef day_of_year_ref;
    ObjRef day_of_leap_year_ref;
    uint16 doy;
    char ascii[4];
    char msg[128];

    ascii[0] = '0' + (dom / 10);
    ascii[1] = '0' + (dom % 10);
    ascii[2] = '\0';

    nwos_create_cardinal_number(ascii, &cardinal_ref);
    nwos_create_ordinal_number(&cardinal_ref, &day_of_month_ref);

    doy = year_offset + dom;

    ascii[0] = '0' + (doy / 100);
    ascii[2] = '0' + ((doy / 10) % 10);
    ascii[3] = '0' + (doy % 10);
    ascii[4] = '\0';

    nwos_create_cardinal_number(ascii, &cardinal_ref);
    nwos_create_ordinal_number(&cardinal_ref, &day_of_year_ref);

    if (leap == 0)
    {
	copy_reference(&day_of_leap_year_ref, &day_of_year_ref);
    }
    else
    {
	doy++;

	ascii[0] = '0' + (doy / 100);
	ascii[2] = '0' + ((doy / 10) % 10);
	ascii[3] = '0' + (doy % 10);
	ascii[4] = '\0';

	nwos_create_cardinal_number(ascii, &cardinal_ref);
	nwos_create_ordinal_number(&cardinal_ref, &day_of_leap_year_ref);
    }

    snprintf(msg, sizeof(msg), "Creating month and day %s/%u %u %u", ascii, dom, doy - leap, doy);
    nwos_log(msg);


    /* get the class definition object so we have the reference list */

    assert(nwos_find_public_class_definition("MONTH AND DAY", &class_ref));

    memset(&month_and_day_obj, 0, sizeof(month_and_day_obj));  /* zero it out */

    nwos_generate_new_public_id(ref);

    nwos_fill_in_common_header(&month_and_day_obj.header.common, ref, &class_ref);

    copy_reference(&month_and_day_obj.month, month_ref);
    copy_reference(&month_and_day_obj.day, &day_of_month_ref);
    copy_reference(&month_and_day_obj.day_of_year, &day_of_year_ref);
    copy_reference(&month_and_day_obj.day_of_leap_year, &day_of_leap_year_ref);

    nwos_create_reference_list(ref, &month_and_day_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &month_and_day_obj.header.object, sizeof(ObjectHeader), month_and_day_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &month_and_day_obj.month, sizeof(month_and_day_obj) - sizeof(EveryObject), month_and_day_obj.header.common.data_chksum);

    nwos_write_object_to_disk(ref, &month_and_day_obj, sizeof(month_and_day_obj));

    nwos_add_to_references(ref, &class_ref);
    nwos_add_to_references(ref, month_ref);
    nwos_add_to_references(ref, &day_of_month_ref);
    nwos_add_to_references(ref, &day_of_year_ref);
    nwos_add_to_references(ref, &day_of_leap_year_ref);

    nwos_month_number_to_string(month_ref, ascii, sizeof(ascii));
}


static void create_month(uint8 month, uint8 min, uint8 max, ObjRef* ref)
{
    C_struct_Month month_obj;
    ObjRef class_ref;
    ObjRef cardinal_ref;
    ObjRef ordinal_ref;
    ObjRef month_day_ref;
    char month_ascii[3];
    uint8 dom;
    uint8 leap;
    char msg[128];
    uint16 year_offset[13] = { 0, 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 };

    assert(1 <= month && month <= 12);

    month_ascii[0] = '0' + (month / 10);
    month_ascii[1] = '0' + (month % 10);
    month_ascii[2] = '\0';

    snprintf(msg, sizeof(msg), "Creating month %s", month_ascii);
    nwos_log(msg);

    assert(nwos_find_cardinal_number(month_ascii, &cardinal_ref));
    assert(nwos_find_ordinal_number(&cardinal_ref, &ordinal_ref));

    /* get the class definition object so we have the reference list */

    assert(nwos_find_public_class_definition("MONTH", &class_ref));

    memset(&month_obj, 0, sizeof(month_obj));  /* zero it out */

    nwos_generate_new_public_id(ref);

    nwos_fill_in_common_header(&month_obj.header.common, ref, &class_ref);

    copy_reference(&month_obj.number, &ordinal_ref);
    month_obj.minimum_days = min;
    month_obj.maximum_days = max;

    nwos_create_reference_list(ref, &month_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &month_obj.header.object, sizeof(ObjectHeader), month_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &month_obj.definition, sizeof(month_obj) - sizeof(EveryObject), month_obj.header.common.data_chksum);

    nwos_write_object_to_disk(ref, &month_obj, sizeof(month_obj));

    nwos_add_to_references(ref, &class_ref);
    nwos_add_to_references(ref, &ordinal_ref);

    leap = (month > 2);

    for (dom = 1; dom <= min; dom++)
    {
	create_month_and_day(ref, dom, year_offset[month], leap, &month_day_ref);
    }

    if (month == 2)
    {
	create_month_and_day(ref, max, year_offset[month], 1, &month_day_ref);
    }
}


static void create_all_twelve_months(ObjRef* english_lang_ref)
{
    ObjRef reference;
    ObjRef month_ref;

    create_month(1,  31, 31, &reference);
    nwos_create_word("January", &reference, english_lang_ref, &month_ref);

    create_month(2,  28, 29, &reference);
    nwos_create_word("February", &reference, english_lang_ref, &month_ref);

    create_month(3,  31, 31, &reference);
    nwos_create_word("March", &reference, english_lang_ref, &month_ref);

    create_month(4,  30, 30, &reference);
    nwos_create_word("April", &reference, english_lang_ref, &month_ref);

    create_month(5,  31, 31, &reference);
    nwos_create_word("May", &reference, english_lang_ref, &month_ref);

    create_month(6,  30, 30, &reference);
    nwos_create_word("June", &reference, english_lang_ref, &month_ref);

    create_month(7,  31, 31, &reference);
    nwos_create_word("July", &reference, english_lang_ref, &month_ref);

    create_month(8,  31, 31, &reference);
    nwos_create_word("August", &reference, english_lang_ref, &month_ref);

    create_month(9,  30, 30, &reference);
    nwos_create_word("September", &reference, english_lang_ref, &month_ref);

    create_month(10, 31, 31, &reference);
    nwos_create_word("October", &reference, english_lang_ref, &month_ref);

    create_month(11, 30, 30, &reference);
    nwos_create_word("November", &reference, english_lang_ref, &month_ref);

    create_month(12, 31, 31, &reference);
    nwos_create_word("December", &reference, english_lang_ref, &month_ref);
}


static void create_all_366_days(ObjRef* english_lang_ref)
{
    create_all_twelve_months(english_lang_ref);
}


/*******************************************************************************/
/* US States                                                                   */
/*******************************************************************************/

typedef struct 
{
  char  code[3];
  uint8 number;
  uint8 month;
  uint8 day;
  uint16 year;
  char* abbr;
  char* name;
  char* capital;
  char* slogan;
} state_info_struct;

#define  Jan  1
#define  Feb  2
#define  Mar  3
#define  Apr  4
#define  May  5
#define  Jun  6
#define  Jul  7
#define  Aug  8
#define  Sep  9
#define  Oct  10
#define  Nov  11
#define  Dec  12


static state_info_struct state_info[] =
  {
    { "DE",  1, Dec,  7, 1787, "Del.",   "Delaware",       "Dover",          "The First State" },
    { "PA",  2, Dec, 12, 1787, "Pa.",    "Pennsylvania",   "Harrisburg",     "Memories Last A Lifetime" },
    { "NJ",  3, Dec, 18, 1787, "N.J.",   "New Jersey",     "Trenton",        "New Jersey and You Are Perfect Together" },
    { "GA",  4, Jan,  2, 1788, "Ga.",    "Georgia",        "Atlanta",        "Georgia On My Mind" },
    { "CT",  5, Jan,  9, 1788, "Conn.",  "Connecticut",    "Hartford",       "We're Full Of Surprises" },
    { "MA",  6, Feb,  6, 1788, "Mass.",  "Massachusetts",  "Boston",         "We'd Love To Show You Around" },
    { "MD",  7, Apr, 28, 1788, "Md.",    "Maryland",       "Annapolis",      "So Many Things To Do, So Close Together" },
    { "SC",  8, May, 23, 1788, "S.C.",   "South Carolina", "Columbia",       "Smiling Faces Beautiful Places" },
    { "NH",  9, Jun, 21, 1788, "N.H.",   "New Hampshire",  "Concord",        "The Road Less Traveled" },
    { "VA", 10, Jun, 25, 1788, "Va.",    "Virginia",       "Richmond",       "Virginia Is For Lovers" },
    { "NY", 11, Jul, 26, 1788, "N.Y.",   "New York",       "Albany",         "I Love New York" },
    { "NC", 12, Nov, 21, 1789, "N.C.",   "North Carolina", "Raleigh",        "A Better Place To Be" },
    { "RI", 13, May, 29, 1790, "R.I.",   "Rhode Island",   "Providence",     "And Providence Plantations" },
    { "VT", 14, Mar,  4, 1791, "Vt.",    "Vermont",        "Montpelier",     "Green Mountain State" },
    { "KY", 15, Jun,  1, 1792, "Ky.",    "Kentucky",       "Frankfort",      "Always In Season" },
    { "TN", 16, Jun,  1, 1796, "Tenn.",  "Tennessee",      "Nashville",      "Sounds Good To Me" },
    { "OH", 17, Mar,  1, 1803, NULL,     "Ohio",           "Columbus",       "Heart Of It All" },
    { "LA", 18, Apr, 30, 1812, "La.",    "Louisiana",      "Baton Rouge",    "Come As You Are, Leave Different" },
    { "IN", 19, Dec, 11, 1816, "Ind.",   "Indiana",        "Indianapolis",   "The Welcome Mats Always Out", },
    { "MS", 20, Dec, 10, 1817, "Miss.",  "Mississippi",    "Jackson",        "The South's Warmest Welcome" },
    { "IL", 21, Dec,  3, 1818, "Ill.",   "Illinois",       "Springfield",    "A Million Miles From Monday" },
    { "AL", 22, Dec, 14, 1819, "Ala.",   "Alabama",        "Montgomery",     "Unforgettable" },
    { "ME", 23, Mar, 15, 1820, NULL,     "Maine",          "Augusta",        "Maine Is On The Move" },
    { "MO", 24, Aug, 10, 1821, "Mo.",    "Missouri",       "Jefferson City", "Show Me Missouri" },
    { "AR", 25, Jun, 15, 1836, "Ark.",   "Arkansas",       "Little Rock",    "The Natural State" },
    { "MI", 26, Jan, 26, 1837, "Mich.",  "Michigan",       "Lansing",        "Great Things To See And Do" },
    { "FL", 27, Mar,  3, 1845, "Fla.",   "Florida",        "Tallahassee",    "Sunshine State" },
    { "TX", 28, Dec, 29, 1845, "Tex.",   "Texas",          "Austin",         "It's Like A Whole Other Country" },
    { "IA", 29, Dec, 28, 1846, NULL,     "Iowa",           "Des Moines",     "Come Be Our Guest" },
    { "WI", 30, May, 29, 1848, "Wis.",   "Wisconsin",      "Madison",        "Stay Just A Little Bit Longer" },
    { "CA", 31, Sep,  9, 1850, "Calif.", "California",     "Sacramento",     "Find Yourself Here" },
    { "MN", 32, May, 11, 1858, "Minn.",  "Minnesota",      "Saint Paul",     "Explore Minnesota" },
    { "OR", 33, Feb, 14, 1859, "Ore.",   "Oregon",         "Salem",          "Things Look Different Here" },
    { "KS", 34, Jan, 29, 1861, "Kans.",  "Kansas",         "Topeka",         "Heart Of America's West" },
    { "WV", 35, Jun, 20, 1863, "W.Va.",  "West Virginia",  "Charleston",     "Wild And Wonderful" },
    { "NV", 36, Oct, 31, 1864, "Nev.",   "Nevada",         "Carson City",    "Discover Both Sides Of Nevada" },
    { "NE", 37, Mar,  1, 1867, "Nebr.",  "Nebraska",       "Lincoln",        "Genuine" },
    { "CO", 38, Aug,  1, 1875, "Colo.",  "Colorado",       "Denver",         "Colorful Colorado" },
    { "ND", 39, Nov,  2, 1889, "N.D.",   "North Dakota",   "Bismarck",       "Discover The Spirit" },
    { "SD", 40, Nov,  2, 1889, "S.D.",   "South Dakota",   "Pierre",         "Great Faces Great Places" },
    { "MT", 41, Nov,  8, 1889, "Mont.",  "Montana",        "Helena",         "Big Sky Country" },
    { "WA", 42, Nov, 11, 1889, "Wash.",  "Washington",     "Olympia",        "A Little Trip To The Extraordinary" },
    { "ID", 43, Jul,  3, 1890, NULL,     "Idaho",          "Boise",          "Discover Idaho" },
    { "WY", 44, Jul, 10, 1890, "Wyo.",   "Wyoming",        "Cheyenne",	     "Like No Place On Earth", },
    { "UT", 45, Jan,  4, 1896, NULL,     "Utah",           "Salt Lake City", "The Greatest Snow On Earth" },
    { "OK", 46, Nov, 16, 1907, "Okla.",  "Oklahoma",       "Oklahoma City",  "Native America" },
    { "NM", 47, Jan,  6, 1912, "N.M.",   "New Mexico",     "Santa Fe",       "Land of Enchantment" },
    { "AZ", 48, Feb, 14, 1912, "Ariz.",  "Arizona",        "Phoenix",        "Grand Canyon State" },
    { "AK", 49, Jan,  3, 1959, NULL,     "Alaska",         "Juneau",         "North! To Alaska" },
    { "HI", 50, Aug, 21, 1959, NULL,     "Hawaii",         "Honolulu",       "The Islands Of Aloha" },
  };

#define NUM_STATES (sizeof(state_info)/sizeof(state_info_struct))

static void create_all_fifty_states()
{
    C_struct_US_State state_obj;
    ObjRef state_ref;
    ObjRef state_class_ref;
    ObjRef city_class_ref;
    ObjRef city_name_ref;
    ObjRef cardinal_ref;
    ObjRef ordinal_ref;
    size_t length;
    char ascii[4];
    int i;
    char msg[128];

    assert(nwos_find_public_class_definition("US STATE", &state_class_ref));
    assert(nwos_find_public_class_definition("US CITY", &city_class_ref));

    for (i = 0; i < NUM_STATES; i++)
    {
	memset(&state_obj, 0, sizeof(state_obj));  /* zero it out */

	nwos_generate_new_public_id(&state_ref);

	nwos_fill_in_common_header(&state_obj.header.common, &state_ref, &state_class_ref);

	length = strlen(state_info[i].name);

	snprintf(msg, sizeof(msg), "Creating postal code: %s", state_info[i].code);
	nwos_log(msg);

	/* create postal code, first make sure spelling exists */
	if (nwos_find_public_spelling(state_info[i].code, &state_obj.postal_code))
	{
	    /* make sure we haven't screwed up and the state doesn't already exist */
	    assert(!nwos_find_state_from_postal_code(state_info[i].code, &state_ref));
	}
	else
	{
	    nwos_create_spelling(state_info[i].code, &state_obj.postal_code);
	}

	snprintf(msg, sizeof(msg), "Creating date: %d-%d-%d", state_info[i].year, state_info[i].month, state_info[i].day);
	nwos_log(msg);

	nwos_create_date(state_info[i].year, state_info[i].month, state_info[i].day, &state_obj.date);

	ascii[0] = '0' + state_info[i].number / 10;
	ascii[1] = '0' + state_info[i].number % 10;
	ascii[2] = '\0';

	nwos_create_cardinal_number(ascii, &cardinal_ref);
	nwos_create_ordinal_number(&cardinal_ref, &ordinal_ref);
	copy_reference(&state_obj.number, &ordinal_ref);

	nwos_create_name(state_info[i].name, &state_obj.name);

	nwos_create_reference_list(&state_ref, &state_obj.header.object.references);

	nwos_create_name(state_info[i].capital, &city_name_ref);

	snprintf(msg, sizeof(msg), "Creating capital city: %s", state_info[i].capital);
	nwos_log(msg);

	nwos_private_create_city(&city_class_ref, &city_name_ref, &state_ref, &state_obj.capital);

	snprintf(msg, sizeof(msg), "Creating state: %s - %02x%02x%02x%02x", state_info[i].name,
		 state_ref.id[0], state_ref.id[1], state_ref.id[2], state_ref.id[3]);
	nwos_log(msg);

	nwos_add_to_reference_list(&state_obj.capital, &state_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &state_obj.header.object, sizeof(ObjectHeader), state_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &state_obj.name, sizeof(state_obj) - sizeof(EveryObject), state_obj.header.common.data_chksum);

	nwos_write_object_to_disk(&state_ref, &state_obj, sizeof(state_obj));

	assert(nwos_object_exists(&state_ref));

	nwos_add_to_references(&state_ref, &state_obj.postal_code);
	nwos_add_to_references(&state_ref, &state_obj.date);
	nwos_add_to_references(&state_ref, &state_obj.name);
	nwos_add_to_references(&state_ref, &state_class_ref);
    }
}



/*******************************************************************************/
/* Additional US Cities                                                        */
/*******************************************************************************/

typedef struct 
{
  char  code[3];
  char* name;
} city_info_struct;


static city_info_struct additional_city_info[] =
  {
    { "AK", "Anchorage" },
    { "AK", "Bethel" },
    { "AK", "Fairbanks" },

    { "AL", "Anniston" },
    { "AL", "Auburn" },
    { "AL", "Birmingham" },
    { "AL", "Decatur" },
    { "AL", "Dothan" },
    { "AL", "Florence" },
    { "AL", "Gadsden" },
    { "AL", "Huntsville" },
    { "AL", "Madison" },
    { "AL", "Mobile" },
    { "AL", "Tuscaloosa" },

    { "AZ", "Bullhead City" },
    { "AZ", "Casa Grande" },
    { "AZ", "Chandler" },
    { "AZ", "Flagstaff" },
    { "AZ", "Florence" },
    { "AZ", "Gilbert" },
    { "AZ", "Glendale" },
    { "AZ", "Kingman" },
    { "AZ", "Lake Havasu City" },
    { "AZ", "Mesa" },
    { "AZ", "Page" },
    { "AZ", "Phoenix" },
    { "AZ", "Scottsdale" },
    { "AZ", "Tempe" },
    { "AZ", "Prescott" },
    { "AZ", "Marana" },
    { "AZ", "Tuscon" },
    { "AZ", "Yuma" },

    { "AR", "Arkadelphia" },
    { "AR", "Conway" },
    { "AR", "Fayetteville" },
    { "AR", "Fort Smith" },
    { "AR", "Hot Springs" },
    { "AR", "Jonesboro" },
    { "AR", "Pine Bluff" },
    { "AR", "Springdale" },
    { "AR", "Texarkana" },
    { "AR", "Van Buren" },

    { "CA", "Alhambra" },
    { "CA", "Anaheim" },
    { "CA", "Apple Valley" },
    { "CA", "Atascadero" },
    { "CA", "Bakersfield" },
    { "CA", "Barstow" },
    { "CA", "Berkeley" },
    { "CA", "Beverly Hills" },
    { "CA", "Burbank" },
    { "CA", "Carlsbad" },
    { "CA", "Chula Vist" },
    { "CA", "Concord" },
    { "CA", "Corona" },
    { "CA", "Coronado" },
    { "CA", "Cupertino" },
    { "CA", "Daly City" },
    { "CA", "East Los Angeles" },
    { "CA", "El Cajon" },
    { "CA", "El Centro" },
    { "CA", "El Monte" },
    { "CA", "Encinitas" },
    { "CA", "Eureka" },
    { "CA", "Folsom" },
    { "CA", "Foster City" },
    { "CA", "Freemont" },
    { "CA", "Fullerton" },
    { "CA", "Glendale" },
    { "CA", "Grass Valley" },
    { "CA", "Hesperia" },
    { "CA", "Huntington Beach" },
    { "CA", "Indio" },
    { "CA", "Inglewood" },
    { "CA", "Irvine" },
    { "CA", "Hayward" },
    { "CA", "Escondido" },
    { "CA", "La Jolla" },
    { "CA", "Lancaster" },
    { "CA", "Lompoc" },
    { "CA", "Long Beach" },
    { "CA", "Los Angeles" },
    { "CA", "Los Banos" },
    { "CA", "Los Gatos" },
    { "CA", "Milpitas" },
    { "CA", "Marina" },
    { "CA", "Menlo Park" },
    { "CA", "Merced" },
    { "CA", "Monterey" },
    { "CA", "Modesto" },
    { "CA", "Moreno Valley" },
    { "CA", "Morgan Hill" },
    { "CA", "Moutain View" },
    { "CA", "National City" },
    { "CA", "Oakland" },
    { "CA", "Oceanside" },
    { "CA", "Oxnard" },
    { "CA", "Palm Springs" },
    { "CA", "Palmdale" },
    { "CA", "Pasadena" },
    { "CA", "Redding" },
    { "CA", "Redwood City" },
    { "CA", "Riverside" },
    { "CA", "San Bernardino" },
    { "CA", "San Bruno" },
    { "CA", "San Diego" },
    { "CA", "San Francisco" },
    { "CA", "San Jose" },
    { "CA", "San Luis Obispo" },
    { "CA", "San Mateo" },
    { "CA", "San Rafael" },
    { "CA", "Santa Ana" },
    { "CA", "Santa Barbara" },
    { "CA", "Santa Clara" },
    { "CA", "Santa Clarita" },
    { "CA", "Santa Cruz" },
    { "CA", "Santa Maria" },
    { "CA", "Santa Rosa" },
    { "CA", "Seaside" },
    { "CA", "Simi Valley" },
    { "CA", "Sonoma" },
    { "CA", "Sunnyvale" },
    { "CA", "South Lake Tahoe" },
    { "CA", "Spring Valley" },
    { "CA", "Stockton" },
    { "CA", "Temecula" },
    { "CA", "Thousand Oaks" },
    { "CA", "Vallejo" },
    { "CA", "Ventura" },
    { "CA", "Victorville" },
    { "CA", "Walnut Creek" },
    { "CA", "Watsonville" },
    { "CA", "West Covina" },
    { "CA", "West Hollywood" },
    { "CA", "West Sacramento" },
    { "CA", "Yuba City" },

    { "KS", "Kansas City" },
    { "MA", "Boston" },

    { "MO", "Aurora" },
    { "MO", "Columbia" },
    { "MO", "East St. Louis" },
    { "MO", "Edwardsville" },
    { "MO", "Independence" },
    { "MO", "Joplin" },
    { "MO", "Kansas City" },
    { "MO", "Macon" },
    { "MO", "Marshfield" },
    { "MO", "Moberly" },
    { "MO", "Nixa" },
    { "MO", "Ozark" },
    { "MO", "Springfield" },
    { "MO", "St. Charles" },
    { "MO", "St. Louis" },

    { "NV", "Las Vegas" },
    { "NV", "Wendover" },
    { "NY", "New York" },
    { "PA", "Philidelphia" },
    { "UT", "Bountiful" },
    { "UT", "Cedar City" },
    { "UT", "Coalville" },
    { "UT", "Clearfield" },
    { "UT", "Draper" },
    { "UT", "Heber" },
    { "UT", "Kaysville" },
    { "UT", "Layton" },
    { "UT", "Ogden" },
    { "UT", "Orem" },
    { "UT", "North Salt Lake" },
    { "UT", "Park City" },
    { "UT", "Provo" },
    { "UT", "Roy" },
    { "UT", "South Ogden" },
    { "UT", "South Salt Lake" },
    { "UT", "St. George" },
    { "UT", "Tooele" },
    { "UT", "Wendover" },
    { "VA", "Chantilly" },
    { "VA", "Fairfax" },
    { "VA", "Haymarket" },
    { "VA", "Manassas" },
    { "WY", "Evanston" },
    { "WY", "Rock Springs" },
  };
static const int num_additional_cities = sizeof(additional_city_info) / sizeof(city_info_struct);


void create_additional_cities()
{
    int i;
    ObjRef state_ref;
    ObjRef city_ref;
    char msg[128];

    for (i = 0; i < num_additional_cities; i++)
    {
	snprintf(msg, sizeof(msg), "Creating city: %s, %s", additional_city_info[i].name, additional_city_info[i].code);
	nwos_log(msg);

	/* if this is the first one or if the state is different from the previous, find the state reference */
	if (i == 0 || strcmp(additional_city_info[i].code, additional_city_info[i-1].code) != 0)
	{
	    assert(nwos_find_state_from_postal_code(additional_city_info[i].code, &state_ref));
	}
	nwos_create_us_city(additional_city_info[i].name, &state_ref, &city_ref);
    }
}


/*******************************************************************************/
/* Genders                                                                     */
/*******************************************************************************/

static void create_gender(ObjRef* ref)
{
    C_struct_Gender gender_obj;
    ObjRef gender_class_ref;


    assert(nwos_find_public_class_definition("GENDER", &gender_class_ref));

    memset(&gender_obj, 0, sizeof(gender_obj));  /* zero it out */

    nwos_generate_new_public_id(ref);

    nwos_fill_in_common_header(&gender_obj.header.common, ref, &gender_class_ref);

    nwos_create_reference_list(ref, &gender_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &gender_obj.header.object, sizeof(ObjectHeader), gender_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &gender_obj.definition, sizeof(gender_obj) - sizeof(EveryObject), gender_obj.header.common.data_chksum);

    nwos_write_object_to_disk(ref, &gender_obj, sizeof(gender_obj));

    nwos_add_to_references(ref, &gender_class_ref);
}



/*******************************************************************************/
/* US Area Codes                                                               */
/*******************************************************************************/

/* Create all known area codes for the US */

typedef struct {
  char state_code[3];
  char area_code[4];
} area_code_struct;

static area_code_struct area_code_info[] = {
  { "AL", "205" },
  { "AL", "251" },
  { "AL", "256" },
  { "AL", "334" },
  { "AK", "907" },
  { "AZ", "480" },
  { "AZ", "520" },
  { "AZ", "602" },
  { "AZ", "623" },
  { "AZ", "928" },
  { "AR", "501" },
  { "AR", "870" },
  { "CA", "209" },
  { "CA", "213" },
  { "CA", "310" },
  { "CA", "323" },
  { "CA", "408" },
  { "CA", "415" },
  { "CA", "510" },
  { "CA", "530" },
  { "CA", "559" },
  { "CA", "562" },
  { "CA", "619" },
  { "CA", "626" },
  { "CA", "650" },
  { "CA", "661" },
  { "CA", "707" },
  { "CA", "714" },
  { "CA", "760" },
  { "CA", "805" },
  { "CA", "818" },
  { "CA", "831" },
  { "CA", "858" },
  { "CA", "909" },
  { "CA", "916" },
  { "CA", "925" },
  { "CA", "949" },
  { "CO", "303" },
  { "CO", "719" },
  { "CO", "720" },
  { "CO", "970" },
  { "CT", "203" },
  { "CT", "860" },
  { "DE", "302" },
  { "FL", "305" },
  { "FL", "321" },
  { "FL", "352" },
  { "FL", "386" },
  { "FL", "407" },
  { "FL", "561" },
  { "FL", "727" },
  { "FL", "754" },
  { "FL", "772" },
  { "FL", "786" },
  { "FL", "813" },
  { "FL", "850" },
  { "FL", "863" },
  { "FL", "904" },
  { "FL", "941" },
  { "FL", "954" },
  { "GA", "229" },
  { "GA", "404" },
  { "GA", "478" },
  { "GA", "678" },
  { "GA", "706" },
  { "GA", "770" },
  { "GA", "912" },
  { "HI", "808" },
  { "ID", "208" },
  { "IL", "217" },
  { "IL", "309" },
  { "IL", "312" },
  { "IL", "618" },
  { "IL", "630" },
  { "IL", "708" },
  { "IL", "773" },
  { "IL", "815" },
  { "IL", "847" },
  { "IN", "219" },
  { "IN", "260" },
  { "IN", "317" },
  { "IN", "574" },
  { "IN", "765" },
  { "IN", "812" },
  { "IA", "319" },
  { "IA", "515" },
  { "IA", "563" },
  { "IA", "641" },
  { "IA", "712" },
  { "KS", "316" },
  { "KS", "620" },
  { "KS", "785" },
  { "KS", "913" },
  { "KY", "270" },
  { "KY", "502" },
  { "KY", "606" },
  { "KY", "859" },
  { "LA", "225" },
  { "LA", "318" },
  { "LA", "337" },
  { "LA", "504" },
  { "LA", "985" },
  { "ME", "207" },
  { "MD", "240" },
  { "MD", "301" },
  { "MD", "410" },
  { "MD", "443" },
  { "MA", "339" },
  { "MA", "351" },
  { "MA", "413" },
  { "MA", "508" },
  { "MA", "617" },
  { "MA", "774" },
  { "MA", "781" },
  { "MA", "857" },
  { "MA", "978" },
  { "MI", "231" },
  { "MI", "248" },
  { "MI", "269" },
  { "MI", "313" },
  { "MI", "517" },
  { "MI", "586" },
  { "MI", "616" },
  { "MI", "734" },
  { "MI", "810" },
  { "MI", "906" },
  { "MI", "989" },
  { "MN", "218" },
  { "MN", "320" },
  { "MN", "507" },
  { "MN", "612" },
  { "MN", "651" },
  { "MN", "763" },
  { "MN", "952" },
  { "MS", "228" },
  { "MS", "601" },
  { "MS", "662" },
  { "MO", "314" },
  { "MO", "417" },
  { "MO", "573" },
  { "MO", "636" },
  { "MO", "660" },
  { "MO", "816" },
  { "MT", "406" },
  { "NE", "308" },
  { "NE", "402" },
  { "NV", "702" },
  { "NV", "775" },
  { "NH", "603" },
  { "NJ", "201" },
  { "NJ", "609" },
  { "NJ", "732" },
  { "NJ", "856" },
  { "NJ", "908" },
  { "NJ", "973" },
  { "NM", "505" },
  { "NY", "212" },
  { "NY", "315" },
  { "NY", "347" },
  { "NY", "516" },
  { "NY", "518" },
  { "NY", "607" },
  { "NY", "631" },
  { "NY", "646" },
  { "NY", "716" },
  { "NY", "718" },
  { "NY", "845" },
  { "NY", "914" },
  { "NY", "917" },
  { "NC", "252" },
  { "NC", "336" },
  { "NC", "704" },
  { "NC", "828" },
  { "NC", "910" },
  { "NC", "919" },
  { "NC", "980" },
  { "ND", "701" },
  { "OH", "216" },
  { "OH", "234" },
  { "OH", "330" },
  { "OH", "419" },
  { "OH", "440" },
  { "OH", "513" },
  { "OH", "614" },
  { "OH", "740" },
  { "OH", "937" },
  { "OK", "405" },
  { "OK", "580" },
  { "OK", "918" },
  { "OR", "503" },
  { "OR", "541" },
  { "OR", "971" },
  { "PA", "215" },
  { "PA", "267" },
  { "PA", "412" },
  { "PA", "484" },
  { "PA", "570" },
  { "PA", "610" },
  { "PA", "717" },
  { "PA", "724" },
  { "PA", "814" },
  { "PA", "878" },
  { "RI", "401" },
  { "SC", "803" },
  { "SC", "843" },
  { "SC", "864" },
  { "SD", "605" },
  { "TN", "423" },
  { "TN", "615" },
  { "TN", "731" },
  { "TN", "865" },
  { "TN", "901" },
  { "TN", "931" },
  { "TX", "210" },
  { "TX", "214" },
  { "TX", "254" },
  { "TX", "281" },
  { "TX", "361" },
  { "TX", "409" },
  { "TX", "469" },
  { "TX", "512" },
  { "TX", "682" },
  { "TX", "713" },
  { "TX", "806" },
  { "TX", "817" },
  { "TX", "830" },
  { "TX", "832" },
  { "TX", "903" },
  { "TX", "915" },
  { "TX", "936" },
  { "TX", "940" },
  { "TX", "956" },
  { "TX", "972" },
  { "TX", "979" },
  { "UT", "435" },
  { "UT", "801" },
  { "VT", "802" },
  { "VA", "276" },
  { "VA", "434" },
  { "VA", "540" },
  { "VA", "571" },
  { "VA", "703" },
  { "VA", "757" },
  { "VA", "804" },
  { "WA", "206" },
  { "WA", "253" },
  { "WA", "360" },
  { "WA", "425" },
  { "WA", "509" },
  { "WV", "304" },
  { "WI", "262" },
  { "WI", "414" },
  { "WI", "608" },
  { "WI", "715" },
  { "WI", "920" },
  { "WY", "307" },
};

#define NUM_AREA_CODES  (sizeof(area_code_info) / sizeof(area_code_struct))

void nwos_create_all_area_codes()
{
  int i;
  ObjRef ref;
  char msg[128];

  for (i = 0; i < NUM_AREA_CODES; i++)
    {
      snprintf(msg, sizeof(msg), "Creating area code: %s, %s", area_code_info[i].area_code, area_code_info[i].state_code);
      nwos_log(msg);

      nwos_create_area_code(area_code_info[i].area_code, area_code_info[i].state_code, &ref);
    }

  /* need to fix this to create the toll free and 900 numbers */
}


/*******************************************************************************/
/* Compass Directions                                                          */
/*******************************************************************************/

static void create_direction(ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Direction direction_obj;
    ObjRef direction_class_ref;


    /* get the class definition object so we have the reference list */

    assert(nwos_find_public_class_definition("DIRECTION", &direction_class_ref));

    nwos_read_class_definition(&direction_class_ref, &class_def_obj);

    memset(&direction_obj, 0, sizeof(direction_obj));  /* zero it out */

    nwos_generate_new_public_id(ref);

    nwos_fill_in_common_header(&direction_obj.header.common, ref, &direction_class_ref);

    /* since we don't have the whole definition thing going yet there isn't anything to put in this object */

    nwos_create_reference_list(ref, &direction_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &direction_obj.header.object, sizeof(ObjectHeader), direction_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &direction_obj.definition, sizeof(direction_obj) - sizeof(EveryObject), direction_obj.header.common.data_chksum);

    nwos_write_object_to_disk(ref, &direction_obj, sizeof(direction_obj));

    nwos_add_to_reference_list(ref, &class_def_obj.header.object.references);
}


static void create_all_directions()
{
    ObjRef reference;
    ObjRef english_lang_ref;
    ObjRef dir_ref;

    nwos_find_language("English", &english_lang_ref);

    nwos_log("Creating direction: North");
    create_direction(&reference);
    nwos_create_word("North", &reference, &english_lang_ref, &dir_ref);

    nwos_log("Creating direction: South");
    create_direction(&reference);
    nwos_create_word("South", &reference, &english_lang_ref, &dir_ref);

    nwos_log("Creating direction: East");
    create_direction(&reference);
    nwos_create_word("East", &reference, &english_lang_ref, &dir_ref);

    nwos_log("Creating direction: West");
    create_direction(&reference);
    nwos_create_word("West", &reference, &english_lang_ref, &dir_ref);

    nwos_log("Creating direction: Northeast");
    create_direction(&reference);
    nwos_create_word("Northeast", &reference, &english_lang_ref, &dir_ref);

    nwos_log("Creating direction: Northwest");
    create_direction(&reference);
    nwos_create_word("Northwest", &reference, &english_lang_ref, &dir_ref);

    nwos_log("Creating direction: Southeast");
    create_direction(&reference);
    nwos_create_word("Southeast", &reference, &english_lang_ref, &dir_ref);

    nwos_log("Creating direction: Southwest");
    create_direction(&reference);
    nwos_create_word("Southwest", &reference, &english_lang_ref, &dir_ref);
}


/* for now just do the spelling needed for the storage locations */

static char* extra_spellings[] = 
{
  "backup", "binder", "box", "deposit", "failed", "iso", "missing", "music", "safe",
};

#define NUM_EXTRA_SPELLINGS (sizeof(extra_spellings) / sizeof(char*))

static void create_extra_spellings()
{
    int i;
    ObjRef ref;
    char msg[128];

    for (i = 0; i < NUM_EXTRA_SPELLINGS; i++)
    {
	snprintf(msg, sizeof(msg), "Creating spelling: %s", extra_spellings[i]);
	nwos_log(msg);

	nwos_create_spelling(extra_spellings[i], &ref);
    }
}


static char* extra_names[] = 
{
  #include "names.h"
};

#define NUM_EXTRA_NAMES (sizeof(extra_names) / sizeof(char*))

static void create_extra_names()
{
    int i;
    ObjRef ref;
    char msg[128];

    for (i = 0; i < NUM_EXTRA_NAMES; i++)
    {
	snprintf(msg, sizeof(msg), "Creating name: %s", extra_names[i]);
	nwos_log(msg);

	nwos_create_name(extra_names[i], &ref);
    }
}


#if 0
static char* extra_words[] = 
{
  #include "words.h"
};

#define NUM_EXTRA_WORDS (sizeof(extra_words) / sizeof(char*))

static void create_extra_words(ObjRef* english_lang_ref)
{
    int i;
    ObjRef ref;
    char msg[128];

    for (i = 0; i < NUM_EXTRA_WORDS; i++)
    {
	snprintf(msg, sizeof(msg), "Creating word: %s", extra_words[i]);
	nwos_log(msg);

	nwos_create_word(extra_words[i], NULL, english_lang_ref, &ref);
    }
}
#endif


/*****************************************************************************/
/* The big bang main routine.                                                */
/*                                                                           */
/* 1) First double check to make sure this program wasn't run before.  Since */
/*    it no longer has fixed filenames for the classes, it could easily run  */
/*    a second time and create a whole new set of objects.  This is done by  */
/*    creating a dummy object (filename: 12345678) that always has the same  */
/*    name and making sure it doesn't exist before doing anything else.      */
/*                                                                           */
/* 2) Create the four critical class definition objects (class definitions,  */
/*    reference list, spelling, and name).  This has to be done a special    */
/*    way because class definitions have names and to create the names, the  */
/*    spelling and name classes must exist.  A chicken and egg problem.      */
/*                                                                           */
/* 3) Create the root class and the root object.                             */
/*                                                                           */
/* 4) Create the "word" and "language" classes since they are intertwined    */
/*    as well.                                                               */
/*                                                                           */
/* 5) Create date classes, "year", "month" and "date".                       */
/*                                                                           */
/* 6) Create US states classes.                                              */
/*                                                                           */
/* 7) Create the classes for people.                                         */
/*                                                                           */
/* 8) Create Phone classes, area code, phone number, and phones.             */
/*****************************************************************************/

int main(int argc, char* argv[])
{
    ObjRef root_obj_ref;
    ObjRef english_lang_ref;
    ObjRef male_ref;
    ObjRef female_ref;
    ObjRef dummy;

    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(NULL, 0, 0, DEFAULT_TYPE, DEFAULT_FILE);

    root_obj_ref.id[0] = 0;
    root_obj_ref.id[1] = 0;
    root_obj_ref.id[2] = 0;
    root_obj_ref.id[3] = 1;

    nwos_set_root_object(&root_obj_ref);

    printf ("Creating English Language object\n");
    nwos_create_english_language(&english_lang_ref);

    printf("Creating Numbers\n");
    create_numbers(&english_lang_ref);

    printf("Creating Years since 1582\n");
    create_years();

    printf ("Creating 366 days of the year objects\n");
    create_all_366_days(&english_lang_ref);

    printf ("Creating 50 US State objects\n");
    create_all_fifty_states();

    printf ("Creating additional US city objects\n");
    create_additional_cities();

    nwos_log("Creating male gender");
    create_gender(&male_ref);
    nwos_create_word("male", &male_ref, &english_lang_ref, &dummy);

    nwos_log("Creating female gender");
    create_gender(&female_ref);
    nwos_create_word("female", &female_ref, &english_lang_ref, &dummy);

    /* probably should create ones for the more unusual variants... */

    printf ("Creating all of the Area Code objects.\n");
    nwos_create_all_area_codes();

    printf ("Creating Direction objects\n");
    create_all_directions();

    printf("Adding extra spellings\n");
    create_extra_spellings();

    printf("Adding names\n");
    create_extra_names();

    nwos_terminate_objectify();

    return 0;
}

